/*H***************************************************************************
*
* $Archive:: /boards/dsk6455_v2/lib/dsk6455bsl/dsk6455.c                      $
* $Revision:: 2                                                               $
* $Date:: 6/27/06 9:33a                                                       $
*
* DESCRIPTION: DSK6455 board initializion implementation.
* 
* NOTES :  BSL Version# 3.24 
*
* (C) Copyright 2005 by Spectrum Digital Incorporated
* All rights reserved
*
*H***************************************************************************/

#include "dsk6455.h"

#define EMIFA_BASE_ADDR (0x70000000)

#define EMIFA_MIDR     (*(volatile Uint32*)(EMIFA_BASE_ADDR + 0x00000000))
#define EMIFA_STAT     (*(volatile Uint32*)(EMIFA_BASE_ADDR + 0x00000004))
#define EMIFA_BPRIO    (*(volatile Uint32*)(EMIFA_BASE_ADDR + 0x00000020))
#define EMIFA_CE2CFG   (*(volatile Uint32*)(EMIFA_BASE_ADDR + 0x00000080))
#define EMIFA_CE3CFG   (*(volatile Uint32*)(EMIFA_BASE_ADDR + 0x00000084))
#define EMIFA_CE4CFG   (*(volatile Uint32*)(EMIFA_BASE_ADDR + 0x00000088))
#define EMIFA_CE5CFG   (*(volatile Uint32*)(EMIFA_BASE_ADDR + 0x0000008C))
#define EMIFA_AWCC     (*(volatile Uint32*)(EMIFA_BASE_ADDR + 0x000000A0))

#define DDR_BASE_ADDR (0x78000000)

#define DDR_MIDR     (*(volatile Uint32*)(DDR_BASE_ADDR + 0x00000000))
#define DDR_SDCFG    (*(volatile Uint32*)(DDR_BASE_ADDR + 0x00000008))
#define DDR_SDRFC    (*(volatile Uint32*)(DDR_BASE_ADDR + 0x0000000C))
#define DDR_SDTIM1   (*(volatile Uint32*)(DDR_BASE_ADDR + 0x00000010))
#define DDR_SDRIM2   (*(volatile Uint32*)(DDR_BASE_ADDR + 0x00000014))
#define DDR_DDRPHYC  (*(volatile Uint32*)(DDR_BASE_ADDR + 0x000000E4))


/*--------------------------------------------------------------*/
/* init_PLL()                                                   */
/* PLL initialization                                           */
/*--------------------------------------------------------------*/
#define PLLCTL_1	0x029A0100	// PLL1 control register
#define PLLM_1		0x029A0110	// PLL1 multiplier control register
#define PREDIV_1	0x029A0114	// PLL1 pre-divider control register
#define PLLCMD_1	0x029A0138	// PLL1 controller command register
#define PLLSTAT_1	0x029A013C	// PLL1 controller status register
#define DCHANGE_1	0x029A0144	// PLL1 PLLDIV ratio change status register
#define SYSTAT_1	0x029A0150	// PLL1 SYSCLK status register
#define PLLDIV4_1	0x029A0160	// PLL1 controller divider 4 register
#define PLLDIV5_1	0x029A0164	// PLL1 controller divider 5 register

#define PLLDIV1_2	0x029C0118	// PLL2 controller divider 1 register
#define PLLCMD_2	0x029C0138	// PLL2 controller command register
#define PLLSTAT_2	0x029C013C	// PLL2 controller status register
#define DCHANGE_2	0x029C0144	// PLL2 PLLDIV ratio change status register
#define SYSTAT_2	0x029C0150	// PLL2 SYSCLK status register

void init_PLL()
{
	volatile int i;

	int PLLM_val =    20; 
	int PREDIV_val =  1;
	int PLLDIV4_val = 6;
	int PLLDIV5_val = 4;

	/* In PLLCTL, write PLLENSRC = 0 (enable PLLEN bit).*/
	*(volatile int *)PLLCTL_1 &= ~(0x00000020);
	/* In PLLCTL, write PLLEN = 0 (bypass mode).*/
	*(volatile int *)PLLCTL_1 &= ~(0x00000001);
	/* Wait 4 cycles of the slowest of PLLOUT or reference clock source (CLKIN).*/
	for (i=0 ; i<100 ; i++);
	/*In PLLCTL, write PLLRST = 1 (PLL is reset).*/
	*(volatile int *)PLLCTL_1 |= 0x00000008;
	/*If necessary, program PREDIV and PLLM.*/
	*(volatile int *)PLLM_1 = PLLM_val - 1;
	*(volatile int *)PREDIV_1 = (PREDIV_val - 1) | 0x8000;	/* set PLLDIV0 */

	/*If necessary, program PLLDIV1n. Note that you must apply the GO operation
	to change these dividers to new ratios.*/

	/* Check that the GOSTAT bit in PLLSTAT is cleared to show that no GO 
	operation is currently in progress.*/
	while( (*(volatile int *)PLLSTAT_1) & 0x00000001);


	/* Program the RATIO field in PLLDIVn to the desired new divide-down rate. 
	If the RATIO field changed, the PLL controller will flag the change
	in the corresponding bit of DCHANGE.*/
	*(volatile int *)PLLDIV4_1 = (PLLDIV4_val - 1) | 0x8000;	/* set PLLDIV4 */
	*(volatile int *)PLLDIV5_1 = (PLLDIV5_val - 1) | 0x8000;	/* set PLLDIV5 */

	/* Set the GOSET bit in PLLCMD to initiate the GO operation to change
	the divide values and align the SYSCLKs as programmed.*/
	*(volatile int *)PLLCMD_1 |= 0x00000001;

	/* Read the GOSTAT bit in PLLSTAT to make sure the bit returns to 0
	to indicate that the GO operation has completed.*/
	while( (*(volatile int *)PLLSTAT_1) & 0x00000001);

	/* Wait for PLL to properly reset.(128 CLKIN1 cycles).*/
	for (i=0 ; i<1000 ; i++);

	/* In PLLCTL, write PLLRST = 0 to bring PLL out of reset.*/
	*(volatile int *)PLLCTL_1 &= ~(0x00000008);

	/* Wait for PLL to lock (2000 CLKIN1 cycles). */
	for (i=0 ; i<4000 ; i++);

	/* In PLLCTL, write PLLEN = 1 to enable PLL mode. */
	*(volatile int *)PLLCTL_1 |= (0x00000001);

}

/*--------------------------------------------------------------*/
/* init_PLL2RGMII()                                             */
/* Clock setup for the RGMII interface                          */
/*--------------------------------------------------------------*/

#define PLLCTL_2	0x029C0100	// PLL2 control register
#define PLLM_2	0x029C0110	// PLL2 multiplier control register
#define PREDIV_2	0x029C0114	// PLL2 pre-divider control register
#define PLLDIV1_2	0x029C0118	// PLL2 controller divider 1 register
#define PLLCMD_2	0x029C0138	// PLL2 controller command register
#define PLLSTAT_2	0x029C013C	// PLL2 controller status register
#define DCHANGE_2	0x029C0144	// PLL2 PLLDIV ratio change status register
#define SYSTAT_2	0x029C0150	// PLL2 SYSCLK status register


void init_PLL2GMII()
{
//    int i;

    *(volatile int *)PLLDIV1_2 = (2 - 1) | 0x8000;        /* set PLLDIV1 */

    /* Check that the GOSTAT bit in PLLSTAT is cleared to show that no GO operation is currently in progress.*/
    while( (*(volatile int *)PLLSTAT_2) & 0x00000001);

    /* Set the GOSET bit in PLLCMD to initiate the GO operation to change the divide values and align the SYSCLKs as programmed.*/
    *(volatile int *)PLLCMD_2 |= 0x00000001;

    /* Read the GOSTAT bit in PLLSTAT to make sure the bit returns to 0 to indicate that the GO operation has completed.*/
    while( (*(volatile int *)PLLSTAT_2) & 0x00000001);
}


/* Initialize the board */
void DSK6455_init()
{
    volatile Uint32 test;
	volatile Uint32 *pcfg;

    /* This section is coded such that the next two statements will execute in */
	/* under 16 SYSCLK3 cycles */
	pcfg = (volatile Uint32 *)0x2ac0004;

    /* Unlock PERCFG0 through PERLOCK */
	*pcfg++ = 0x0f0a0b00;

    /* Enable EMAC, Timers, GPIO in PERCFG0 */
    *pcfg = 0xC0010550;

    /* Read back PERSTAT0 */
    test = *((Uint32 *)0x02AC0014);

    /* Wait at least 128 CPU cycles */
    DSK6455_wait(128);

	/* Enable DDR and EMIFA in PERCFG1 */
	*((Uint32 *)0x02AC002C) = 0x3;

    init_PLL();

	init_PLL2GMII();

	/* Configure async EMIF */
    EMIFA_CE2CFG = 0x00240120; /* 8-bit async, 10 cycle read/write strobe */
    EMIFA_CE3CFG = 0x00240120; /* 8-bit async, 10 cycle read/write strobe */
   // EMIFA_CE4CFG = 0x00240122; /* 32-bit async, 10 cycle read/write strobe */

   /*
	-------------------------------------------------------------------------------------
	|31   |30----11|---10---|--9---|---8----|-7--6--|--5--4--|--3--2-|--1-0-|
	-------------------------------------------------------------------------------------
	|SSEL|Reserved|RD_BE_EN|CE_EXT|R_ENABLE|W_LTNCY|Reserved|R_LTNCY|SBSIZE|
	-------------------------------------------------------------------------------------
	*/
   // EMIFA_CE4CFG = 0x80000307; /* 16-bit sync,  309 */
//	EMIFA_CE4CFG = 0x80000302; /* 32-bit sync,  306 */
	EMIFA_CE4CFG = 0x80000306; /* 32-bit sync,  306 */

    EMIFA_BPRIO  = 0x000000FE; /* Enable priority based starvation control SPRU971A sec. 7.2 */

	/* Configure DDR for 500MHz operation (sequence is order dependent) */
    DDR_SDCFG    = 0x00D38822; /* Unlock boot + timing, CAS4, 4 banks, 10 bit column */
	DSK6455_waitusec(200);
    DDR_SDRFC    = 0x000007A2; /* Refresh */
    DDR_SDTIM1   = 0x3EDB4B91; /* Timing 1 */
    DDR_SDRIM2   = 0x00A2C722; /* Timing 2 */
    DDR_DDRPHYC  = 0x00000005; /* PHY read latency for CAS 4 is 4 + 2 - 1 */
    DDR_SDCFG    = 0x00538822; /* Lock, CAS4, 4 banks, 10 bit column, lock timing */
    DSK6455_waitusec(200);

//	DSK6455_I2C_init();
}

/* Read an 8-bit value from a CPLD register */
Uint8 DSK6455_rget(Int16 regnum)
{
    Uint8 *pdata;

    /* Return lower 8 bits of register */
    pdata = (Uint8 *)(DSK6455_CPLD_BASE + regnum);
    return (*pdata & 0xff);
}

/* Write an 8-bit value to a CPLD register */
void DSK6455_rset(Int16 regnum, Uint8 regval)
{
    Uint8 *pdata;

    /* Write lower 8 bits of register */
    pdata = (Uint8 *)(DSK6455_CPLD_BASE + regnum);
    *pdata = (regval & 0xff);
}

/* Spin in a delay loop for delay iterations */
void DSK6455_wait(Uint32 delay)
{
    volatile Uint32 i, n;

    n = 0;
    for (i = 0; i < delay; i++)
    {
        n = n + 1;
    }
}

/* Spin in a delay loop for delay microseconds */
void DSK6455_waitusec(Uint32 delay)
{
    DSK6455_wait(delay * 39);
}
