#include "trackcfg.h"

#include <stdio.h>
#include <csl.h>
#include <csl_gpio.h>
#include <csl_edma3.h>
#include <soc.h>
#include <hwi.h>

#include "dsk6455.h"

#define BUFFSIZE 4096
#define PING     0
#define PONG     1

Uint8 gBufferRcvPing[BUFFSIZE] = {0}; // Receive PING buffer
Uint8 gBufferRcvPong[BUFFSIZE] = {0}; // Receive PONG buffer
Uint32 srcBuff[2] = {4, 7};

CSL_Edma3HwDmaChannelSetup  dmahwSetup[CSL_EDMA3_NUM_DMACH] = CSL_EDMA3_DMACHANNELSETUP_DEFAULT;


/* ---------------------- Interrupt Service Routine -------------------- */

/*
 *  edmaHwi() - Interrupt service routine for the DMA transfer.  It is
 *              triggered when a complete DMA receive frame has been
 *              transferred.   The edmaHwi ISR is inserted into the interrupt
 *              vector table at compile time through a setting in the DSP/BIOS
 *              configuration under Scheduling --> HWI --> HWI_INT8.  edmaHwi
 *              uses the DSP/BIOS Dispatcher to save register state and make
 *              sure the ISR co-exists with other DSP/BIOS functions.
 */
void edmaHwi(void)
{
	Uint32 intrh = 0;
    static Uint32       pingOrPong = PING;
	static Int16        rcvdone = 0;

	/* Check for pending EDMA event interrupts (IPRH) */
	intrh = *((volatile Uint32*)0x02a0226C);

	if (intrh & 0x00200000)
	{
	    rcvdone = 1;
	}

	if (rcvdone)
	{
	    if (pingOrPong == PING)
		{
	        pingOrPong = PONG;
		}
		else
		{
		    pingOrPong = PING;
		}

		rcvdone = 0;
	}

    /* Clear CPU interrupt event 72 (EVTCLR2) */
	*((volatile Uint32*)0x1800048) = 0x00000100;

	/* Clear processed EDMA event interrupts (ICRH) */
	*((volatile Uint32*)0x02a02274) = 0x00200000; 

}


void gpioInit()
{  
	CSL_GpioHandle           	hGpio;
    CSL_GpioPinConfig          	config;
    CSL_Status              	status;
    CSL_GpioContext         	pContext;
    CSL_GpioObj             	gpioObj;
    CSL_GpioHwSetup             hwSetup;

    /* Initialize the GPIO CSL module */
    status = CSL_gpioInit(&pContext);
    if (status != CSL_SOK) {
        printf("GPIO: Initialization error.\n");
        return;
    }

    /* Open the CSL module */
    hGpio = CSL_gpioOpen(&gpioObj, CSL_GPIO, NULL, &status);
    if ((hGpio == NULL) || (status != CSL_SOK)) {
        printf("GPIO: Error opening the instance.\n");
        return;
    }

	/* Setup hardware parameters */
    hwSetup.extendSetup = NULL;
    
    /* Setup the General Purpose IO */
    status = CSL_gpioHwSetup(hGpio, &hwSetup);

    /* Configure pin 5 to generate an interrupt on Rising Edge, and
     * configure it as an input. 
     */
    config.pinNum = CSL_GPIO_PIN5;
    config.trigger = CSL_GPIO_TRIG_RISING_EDGE;
    config.direction = CSL_GPIO_DIR_INPUT;
    
	/* Enable the bank interrupt */
    status = CSL_gpioHwControl(hGpio, CSL_GPIO_CMD_BANK_INT_ENABLE, NULL);
    if (status != CSL_SOK) {
        printf("GPIO: Command to enable bank interrupt... Failed.\n");
    }

    /* configure the gpio pin 5 */
    status = CSL_gpioHwControl(hGpio, CSL_GPIO_CMD_CONFIG_BIT, &config);
    if (status != CSL_SOK) {
        printf("GPIO: GPIO pin configuration error.\n");
        return;
    }
}


void edmaInit()
{
	CSL_Edma3Handle        		hModule; 
    CSL_Edma3Obj                edmaObj;
    CSL_Edma3ParamHandle        hParamPing;
    CSL_Edma3ParamHandle        hParamPong;
    CSL_Edma3ParamHandle        hParamBasic;
    CSL_Edma3ChannelObj         chObj;
    CSL_Edma3CmdIntr            regionIntr;
    CSL_Edma3CmdDrae            regionAccess;
    CSL_Edma3ChannelHandle      hChannel;
	CSL_Edma3ParamSetup         myParamSetup;
    CSL_Edma3Context            context;
    CSL_Edma3ChannelAttr        chAttr;
    CSL_Status                  status;
	CSL_Edma3HwSetup            hwSetup = {&dmahwSetup[0],NULL};

    /* Module initialization */
    status = CSL_edma3Init(&context);
    if (status != CSL_SOK) {
        printf ("Edma module initialization failed\n");   
        return;
    }
    
    /* Edma module open */
    hModule = CSL_edma3Open(&edmaObj,CSL_EDMA3,NULL,&status);
    if ( (hModule == NULL) || (status != CSL_SOK)) {
        printf ("Edma module open failed\n");    
        return;
    }

    /* DMA chanel --> PaRAM mapping, queue assignment */
    dmahwSetup[53].paramNum = 53;     /* GPINT5 -> EDMA chanel 53 */
	dmahwSetup[53].que      = CSL_EDMA3_QUE_1;
    status = CSL_edma3HwSetup(hModule,&hwSetup);
    if (status != CSL_SOK) {
         printf ("Hardware setup failed\n");
         CSL_edma3Close (hModule);
         return;
    }

    /* Setup the DRAE masks
     * DRAE enable(Bits 0-63) for the shadow region 1.
     */
    regionAccess.region = CSL_EDMA3_REGION_1 ;
    regionAccess.drae =   0xFFFFFFFF ;   
    regionAccess.draeh =  0xFFFFFFFF ;
    status = CSL_edma3HwControl(hModule,CSL_EDMA3_CMD_DMAREGION_ENABLE, \
                                &regionAccess); 
    if (status != CSL_SOK) {
        printf("Edma region enable command failed\n");
        return;    
    } 

    /* Channel open */
    chAttr.regionNum = CSL_EDMA3_REGION_1;
    chAttr.chaNum = CSL_EDMA3_CHA_GPINT5;
    hChannel = CSL_edma3ChannelOpen(&chObj, CSL_EDMA3, &chAttr, &status);   
    if ( (hChannel == NULL) || (status != CSL_SOK)) {
        printf ("Edma channel open failed\n");    
        return;
    } 

    /* Obtain a handle to parameter set 53 */
    hParamBasic = CSL_edma3GetParamHandle(hChannel,53,NULL);
    if (hParamBasic == NULL) {
        printf ("Edma get param handle for param entry 53 failed\n");    
        return;
    }

	/* Obtain a handle to parameter set 69 */
    hParamPing = CSL_edma3GetParamHandle(hChannel,69,NULL);
    if (hParamPing == NULL) {
        printf ("Edma get param handle for param entry 69 failed\n");    
        return;
    }
    
    /* Obtain a handle to parameter set 68 */
    hParamPong = CSL_edma3GetParamHandle(hChannel,68,NULL);
    if (hParamPong == NULL) {
        printf ("Edma get param handle for param entry 68 failed\n");        
        return;
    }

    /* Setup the parameter entry parameters (Ping buffer) */
    myParamSetup.option = CSL_EDMA3_OPT_MAKE(CSL_EDMA3_ITCCH_DIS, \
                                             CSL_EDMA3_TCCH_DIS, \
                                             CSL_EDMA3_ITCINT_DIS, \
                                             CSL_EDMA3_TCINT_EN, \
                                             53,CSL_EDMA3_TCC_NORMAL,\
                                             CSL_EDMA3_FIFOWIDTH_NONE, \
                                             CSL_EDMA3_STATIC_DIS, \
                                             CSL_EDMA3_SYNC_AB, \
                                             CSL_EDMA3_ADDRMODE_INCR, \
                                             CSL_EDMA3_ADDRMODE_INCR );
    myParamSetup.srcAddr = (Uint32)srcBuff/*0xc0100000*/;         
    myParamSetup.aCntbCnt = CSL_EDMA3_CNT_MAKE(4,64/*256*/);       
    myParamSetup.dstAddr = (Uint32)gBufferRcvPing;        
    myParamSetup.srcDstBidx = CSL_EDMA3_BIDX_MAKE(0,4);     
    myParamSetup.linkBcntrld = CSL_EDMA3_LINKBCNTRLD_MAKE(hParamPong,0);     
    myParamSetup.srcDstCidx = CSL_EDMA3_CIDX_MAKE(0,256/*1024*/);     
    myParamSetup.cCnt = 16;
    status = CSL_edma3ParamSetup(hParamBasic,&myParamSetup);
    if (status != CSL_SOK) {
        printf("Edma parameter entry setup is failed\n");
        return;
    }

	/* Ping setup */
    status = CSL_edma3ParamSetup(hParamPing,&myParamSetup);
    if (status != CSL_SOK) {
        printf("Edma ping parameter entry setup is failed\n");
        return;
    }
    
    /* Pong setup */    
    myParamSetup.linkBcntrld = CSL_EDMA3_LINKBCNTRLD_MAKE(hParamPing,0);     
    myParamSetup.option = CSL_EDMA3_OPT_MAKE(CSL_EDMA3_ITCCH_DIS, \
                                             CSL_EDMA3_TCCH_DIS, \
                                             CSL_EDMA3_ITCINT_DIS, \
                                             CSL_EDMA3_TCINT_EN,\
                                             53,CSL_EDMA3_TCC_NORMAL,\
                                             CSL_EDMA3_FIFOWIDTH_NONE, \
                                             CSL_EDMA3_STATIC_EN, \
                                             CSL_EDMA3_SYNC_AB, \
                                             CSL_EDMA3_ADDRMODE_INCR, \
                                             CSL_EDMA3_ADDRMODE_INCR );
    myParamSetup.srcAddr = (Uint32)srcBuff/*0xc0100000*/;
    myParamSetup.dstAddr = (Uint32)gBufferRcvPong;    
    status = CSL_edma3ParamSetup(hParamPong,&myParamSetup);
    if (status != CSL_SOK) {
        printf("Edma ping parameter entry setup is failed\n");
        return;
    }

	/* Set up channel and queue relationships */
    CSL_edma3HwChannelSetupParam(hChannel, 53);
	CSL_edma3HwChannelSetupQue(hChannel, CSL_EDMA3_QUE_1);

	/* Enable EDMA region interrupts */
    regionIntr.region = CSL_EDMA3_REGION_1;
    regionIntr.intr  = 0x00000000;
    regionIntr.intrh = 0x00200000;
    status = CSL_edma3HwControl(hModule,CSL_EDMA3_CMD_INTR_ENABLE,&regionIntr);    
    if (status != CSL_SOK) {
        printf ("Edma interrupt enable command failed\n");
        return;
    }

	/* Receive event clear and enable */
	CSL_edma3HwChannelControl(hChannel,CSL_EDMA3_CMD_CHANNEL_CLEAR,NULL);
    CSL_edma3HwChannelControl(hChannel,CSL_EDMA3_CMD_CHANNEL_ENABLE,NULL);

}


void main()
{

	memset(gBufferRcvPing, 0, BUFFSIZE);
	memset(gBufferRcvPong, 0, BUFFSIZE);

	/* Initialize the board support library, must be called first */
    DSK6455_init();

    /* Disable global interrupts */
    HWI_disable();

	/* Install HWI handler */
	HWI_dispatchPlug(4, &edmaHwi, 0, NULL);
    HWI_eventMap(4, 72);
	C64_enableIER(0x10);

	/* Initialize the GPIO controller */ 
	gpioInit();

	/* Initialize the EDMA controller */
	edmaInit();

    /* Clear CPU interrupt event 72 (EVTCLR2) */
	*((Uint32*)0x1800048) = 0x00000100;

    /* Enable CPU interrupt event 72 (EVTMASK2) */
	*((Uint32*)0x1800088) = 0x00000100;

	/* Re-enable global interrupts */
    HWI_enable();

}
