/**
 * \file  uartEcho.c
 *
 * \brief This is a sample application file which invokes some APIs
 *        from the UART device abstraction library to perform configuration,
 *        transmission and reception operations.
 */

/*
* Copyright (C) 2012 Texas Instruments Incorporated - http://www.ti.com/ 
*
*  Redistribution and use in source and binary forms, with or without 
*  modification, are permitted provided that the following conditions 
*  are met:
*
*    Redistributions of source code must retain the above copyright 
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the 
*    documentation and/or other materials provided with the   
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/


#include "hw_psc_C6748.h"
#include "soc_C6748.h"
#include "interrupt.h"
#include "evmC6748.h"
#include "hw_types.h"
#include "uart.h"
#include "psc.h"

#include <c6x.h>
#include <stdio.h>
#include <cslr/soc_C6748.h>
#include <cslr/cslr_dspintc.h>
#include <cslr/cslr_uart.h>
#include <cslr/cslr_psc_C6748.h>
#include <cslr/cslr_syscfg0_C6748.h>

/****************************************************************************/
/*                      LOCAL FUNCTION PROTOTYPES                           */
/****************************************************************************/
static void ConfigureIntUART(void);
static void SetupInt(void);
static void UARTIsr(void);

/****************************************************************************/
/*                      GLOBAL VARIABLES                                    */
/****************************************************************************/
char txArray[] = "StarterWare UART echo application\n\r";

CSL_PscRegsOvly psc1_register = (CSL_PscRegsOvly)(0x01E27000u);  // PSC:Power and Sleep Controller

CSL_SyscfgRegsOvly system_configuration_registers0 = (CSL_SyscfgRegsOvly)(0x01C14000u);

CSL_UartRegsOvly uart1_register = (CSL_UartRegsOvly)(0x01D0C000u);

CSL_DspintcRegsOvly intc_registers = (CSL_DspintcRegsOvly)(0x01800000u);

/****************************************************************************/
/*                      LOCAL FUNCTION DEFINITIONS                          */
/****************************************************************************/

int main(void)
{
    unsigned int intFlags = 0;
    unsigned int config = 0;

    printf("uart1_register->IIR = %x.\n", uart1_register->IIR);

    /* Enabling the PSC for UART1.*/
    PSCModuleControl(SOC_PSC_1_REGS, HW_PSC_UART1, PSC_POWERDOMAIN_ALWAYS_ON,
		     PSC_MDCTL_NEXT_ENABLE);

    printf("psc1_register->MDCTL[12] = %x.\n", psc1_register->MDCTL[12]);
    printf("psc1_register->PTSTAT = %x.\n", psc1_register->PTSTAT);
    printf("psc1_register->MDSTAT[12] = %x.\n", psc1_register->MDSTAT[12]);

    printf("uart1_register->PWREMU_MGMT = %x.\n", uart1_register->PWREMU_MGMT);

    printf("uart1_register->MDR = %x.\n", uart1_register->MDR);
    printf("uart1_register->DLL = %x.\n", uart1_register->DLL);
    printf("uart1_register->DLH = %x.\n", uart1_register->DLH);
    printf("uart1_register->LCR = %x.\n", uart1_register->LCR);

    printf("uart1_register->IIR = %x.\n", uart1_register->IIR);
    printf("uart1_register->IER = %x.\n", uart1_register->IER);
    printf("uart1_register->LSR = %x.\n", uart1_register->LSR);

    printf("IER = %x.\n", IER);
    printf("IFR = %x.\n", IFR);
    printf("intc_registers->INTMUX1 = %x.\n", intc_registers->INTMUX1);
    printf("intc_registers->INTMUX2 = %x.\n", intc_registers->INTMUX2);
    printf("intc_registers->INTMUX3 = %x.\n", intc_registers->INTMUX3);

    /* Setup PINMUX */
    // ܽŸãUARTܽ
    // һβΪں
    // ڶβδǷUART_RTSUART_CTS
    UARTPinMuxSetup(1, FALSE);
    printf("system_configuration_registers0->PINMUX4 = %x.\n", system_configuration_registers0->PINMUX4);
    

    /* Enabling the transmitter and receiver*/
    UARTEnable(SOC_UART_1_REGS);  // SOC_UART_2_REGS = 0x01D0C000
    // printf("uart1_register->PWREMU_MGMT = %x.\n", uart1_register->PWREMU_MGMT);
    // printf("uart1_register->IIR = %x.\n", uart1_register->IIR);

    /* 1 stopbit, 8-bit character, no parity */
    config = UART_WORDL_8BITS;

    /* Configuring the UART parameters*/
    // UARTConfigSetExpClk(SOC_UART_1_REGS, SOC_UART_1_MODULE_FREQ,
    //                    BAUD_115200, config,
    //                    UART_OVER_SAMP_RATE_16);

    UARTConfigSetExpClk(SOC_UART_1_REGS, SOC_UART_1_MODULE_FREQ,
                        9600, config,
                        UART_OVER_SAMP_RATE_16);

    /*
    printf("uart1_register->PWREMU_MGMT = %x.\n", uart1_register->PWREMU_MGMT);

    printf("uart1_register->MDR = %x.\n", uart1_register->MDR);
    printf("uart1_register->DLL = %x.\n", uart1_register->DLL);
    printf("uart1_register->DLH = %x.\n", uart1_register->DLH);
    printf("uart1_register->LCR = %x.\n", uart1_register->LCR);  // ֳΪ8

    printf("uart1_register->IIR = %x.\n", uart1_register->IIR);
    printf("uart1_register->IER = %x.\n", uart1_register->IER);
    printf("uart1_register->LSR = %x.\n", uart1_register->LSR);

    printf("IER = %x.\n", IER);
    printf("IFR = %x.\n", IFR);
    printf("intc_registers->INTMUX1 = %x.\n", intc_registers->INTMUX1);
    printf("intc_registers->INTMUX2 = %x.\n", intc_registers->INTMUX2);
    printf("intc_registers->INTMUX3 = %x.\n", intc_registers->INTMUX3);
    */

    // printf("uart1_register->IIR = %x.\n", uart1_register->IIR);

    /* Enabling the FIFO and flushing the Tx and Rx FIFOs.*/
    UARTFIFOEnable(SOC_UART_1_REGS);
    // printf("uart1_register->IIR = %x.\n", uart1_register->IIR);

    /* Setting the UART Receiver Trigger Level*/
    UARTFIFOLevelSet(SOC_UART_1_REGS, UART_RX_TRIG_LEVEL_1);
    // printf("uart1_register->IIR = %x.\n", uart1_register->IIR);  // FIFOˮƽ 1ֽ
    
    /*
    ** Enable AINTC to handle interrupts. Also enable IRQ interrupt in ARM 
    ** processor.
    */
    SetupInt();
    // printf("IER = %x.\n", IER);
    // printf("IFR = %x.\n", IFR);

    /* Configure AINTC to receive and handle UART interrupts. */
    ConfigureIntUART();

    /*
    printf("IER = %x.\n", IER);
    printf("IFR = %x.\n", IFR);
    printf("intc_registers->INTMUX1 = %x.\n", intc_registers->INTMUX1);
    */

    /* Preparing the 'intFlags' variable to be passed as an argument.*/
    intFlags |= (UART_INT_LINE_STAT  |  \
                 UART_INT_TX_EMPTY |    \
                 UART_INT_RXDATA_CTI);  // intFlas = 0xC0014A88

    uart1_register->IIR |= (CSL_UART_FCR_RXCLR_MASK | CSL_UART_FCR_FIFOEN_MASK);
    // printf("uart1_register->RBR = %x.\n", uart1_register->RBR);

    /* Enable the Interrupts in UART.*/
    UARTIntEnable(SOC_UART_1_REGS, intFlags);

    // Write into FCR

    // printf("uart1_register->IER = %x.\n", uart1_register->IER);
    // printf("uart1_register->IIR = %x.\n", uart1_register->IIR);

    while(1);
}

/*
** \brief   Interrupt Service Routine(ISR) to be executed on UART interrupts.
**          Depending on the source of interrupt, this 
**          1> writes to the serial communication console, or 
**          2> reads from the serial communication console, or 
**          3> reads the byte in RBR if receiver line error has occured.
*/

static void UARTIsr()
{
    static unsigned int length = sizeof(txArray);
    static unsigned int count = 0;
    unsigned char rxData = 0;
    unsigned int int_id = 0;



    // printf("uart1_register->RBR = %x.\n", uart1_register->RBR);

    /* This determines the cause of UART1 interrupt.*/
    int_id = UARTIntStatus(SOC_UART_1_REGS);
    // printf("uart1_register->IIR = %x.\n", uart1_register->IIR);
    printf("int_id = %x.\n", int_id);

#ifdef _TMS320C6X
    // Clear UART1 system interrupt in DSPINTC
    IntEventClear(SYS_INT_UART1_INT);
#else
    /* Clears the system interupt status of UART2 in AINTC. */
    IntSystemStatusClear(SYS_INT_UARTINT2);
#endif
  
    // printf("uart1_register->LSR = %x.\n", uart1_register->LSR);

    /* Checked if the cause is transmitter empty condition.*/
    if(UART_INTID_TX_EMPTY == int_id)
    {
        if(0 < length)
        {
            /* Write a byte into the THR if THR is free. */
            UARTCharPutNonBlocking(SOC_UART_1_REGS, txArray[count]);
            length--;
            count++;
        }
        if(0 == length)
        {
            /* Disable the Transmitter interrupt in UART.*/
            UARTIntDisable(SOC_UART_1_REGS, UART_INT_TX_EMPTY);
            // printf("uart1_register->IER = %x.\n", uart1_register->IER);
        }
     }

    /* Check if the cause is receiver data condition.*/
    if(UART_INTID_RX_DATA == int_id)
    {
        rxData = UARTCharGetNonBlocking(SOC_UART_1_REGS);  // RBRĴֵ
        // printf("rxData = %x.\n", rxData);

        UARTCharPutNonBlocking(SOC_UART_1_REGS, rxData);

        // printf("uart1_register->RBR = %x.\n", uart1_register->RBR);
    }


    /* Check if the cause is receiver line error condition.*/
    if(UART_INTID_RX_LINE_STAT == int_id)
    {
        while(UARTRxErrorGet(SOC_UART_1_REGS))
        {
            /* Read a byte from the RBR if RBR has data.*/
            UARTCharGetNonBlocking(SOC_UART_1_REGS);
            // printf("uart1_register->RBR = %x.\n", uart1_register->RBR);
        }
    }

    return;
}

/*
** \brief   This function invokes necessary functions to configure the ARM
**          processor and ARM Interrupt Controller(AINTC) to receive and
**          handle interrupts.
*/


static void SetupInt(void)
{
#ifdef _TMS320C6X
	// Initialize the DSP INTC
	IntDSPINTCInit();

	// Enable DSP interrupts globally
	IntGlobalEnable();
#else
    /* Initialize the ARM Interrupt Controller(AINTC). */
    IntAINTCInit();

    /* Enable IRQ in CPSR.*/     
    IntMasterIRQEnable();

    /* Enable the interrupts in GER of AINTC.*/
    IntGlobalEnable();

    /* Enable the interrupts in HIER of AINTC.*/
    IntIRQEnable();
#endif
}

/*
** \brief  This function confiugres the AINTC to receive UART interrupts.
*/ 
static void ConfigureIntUART(void)
{
#ifdef _TMS320C6X
	IntRegister(C674X_MASK_INT4, UARTIsr);
	IntEventMap(C674X_MASK_INT4, SYS_INT_UART1_INT);
	IntEnable(C674X_MASK_INT4);
#else
    /* Registers the UARTIsr in the Interrupt Vector Table of AINTC. */
    IntRegister(SYS_INT_UARTINT2, UARTIsr);

    /* Map the channel number 2 of AINTC to UART2 system interrupt. */
    IntChannelSet(SYS_INT_UARTINT2, 2);

    IntSystemEnable(SYS_INT_UARTINT2);
#endif
}

/****************************END OF FILE*************************************/
