//--------------------------------------------------------------------------
// OS Demonstration Software
//--------------------------------------------------------------------------
// OSIF.H
//
// OS Interface Functions
//
// Author: Michael A. Denio
// Copyright 1999, 2003 by Texas Instruments Inc.
//-------------------------------------------------------------------------

#ifndef _C_OSIF_H
#define _C_OSIF_H

#include <std.h>
#include <sys.h>
#include <tsk.h>
#include <sem.h>

//---------------------------------------------
//---------------------------------------------
// OS Environment Globals
//---------------------------------------------
//---------------------------------------------

// Printf Buffer Size
// Change requires a recompile of the OS library
#define OS_PRINTFBUFFER         512

// Configuration Structure
typedef struct _osenvcfg {
    uint     DbgPrintLevel;     // Debug msg print threshhold
    uint     DbgAbortLevel;     // Debug msg sys abort theshhold
    int      TaskPriLow;        // Lowest priority for stack task
    int      TaskPriNorm;       // Normal priority for stack task
    int      TaskPriHigh;       // High priority for stack task
    int      TaskPriKern;       // Kernel-level priority (highest)
    int      TaskStkLow;        // Minimum stack size
    int      TaskStkNorm;       // Normal stack size
    int      TaskStkHigh;       // Stack size for high volume tasks
    } OSENVCFG;

// Configuration
extern OSENVCFG _oscfg;

// Equates used in code
#define DBG_PRINT_LEVEL         (_oscfg.DbgPrintLevel)
#define DBG_ABORT_LEVEL         (_oscfg.DbgAbortLevel)
#define OS_TASKPRILOW           (_oscfg.TaskPriLow)
#define OS_TASKPRINORM          (_oscfg.TaskPriNorm)
#define OS_TASKPRIHIGH          (_oscfg.TaskPriHigh)
#define OS_TASKPRIKERN          (_oscfg.TaskPriKern)
#define OS_TASKSTKLOW           (_oscfg.TaskStkLow)
#define OS_TASKSTKNORM          (_oscfg.TaskStkNorm)
#define OS_TASKSTKHIGH          (_oscfg.TaskStkHigh)

// Default values
#define DEF_DBG_PRINT_LEVEL     DBG_INFO
#define DEF_DBG_ABORT_LEVEL     DBG_ERROR
#define OS_TASKPRILOW_DEF       3
#define OS_TASKPRINORM_DEF      5
#define OS_TASKPRIHIGH_DEF      7
#define OS_TASKPRIKERN_DEF      (OS_TASKPRIHIGH_DEF+1)
#define OS_TASKSTKLOW_DEF       3072
#define OS_TASKSTKNORM_DEF      4096
#define OS_TASKSTKHIGH_DEF      5120

extern int stricmp( char *s1, char *s2);

//-----------------------------------------------------------------------
//----[ PACKET BUFFER MANAGER ]------------------------------------------
//-----------------------------------------------------------------------

// Packet Buffer Object
typedef struct _PBM_Pkt {
    UINT32          Type;         // Identifier (Read Only)
    struct _PBM_Pkt *pPrev;       // Previous record
    struct _PBM_Pkt *pNext;       // Next record
    UINT8           *pDataBuffer; // Pointer to Data Buffer (Read Only)
    uint            BufferLen;    // Physical Length of buffer (Read Only)
    uint            Flags;        // Packet Flags
    uint            ValidLen;     // Length of valid data in buffer
    uint            DataOffset;   // Byte offset to valid data
    uint            EtherType;    // Ether Type Code
    uint            L2HdrLen;     // Length of L2 Hdr (on 'L3' Rx pkts)
    uint            IpHdrLen;     // Length of Ip Hdr
    HANDLE          hIFRx;        // Rx Interface
    HANDLE          hIFTx;        // Tx Interface
    HANDLE          hRoute;       // Handle to Route
    UINT32          Aux1;         // Aux1 Data
    UINT32          Aux2;         // Aux2 Data
    } PBM_Pkt;

// PBM Handle
typedef void * PBM_Handle;

// Packet Buffer Manager Initialization Functions
_extern uint        PBM_open();
_extern void        PBM_close();

// Packet Buffer Functions (re-entrant and "kernel mode" agnostic)
_extern PBM_Handle  PBM_alloc( uint MaxSize );
_extern PBM_Handle  PBM_copy( PBM_Handle hPkt );
_extern void        PBM_free( PBM_Handle hPkt );

//
// The following field Functions can be used by device drivers
// All other PBM_Pkt fields are reserved
//
#define PBM_getBufferLen(hPkt)      (((PBM_Pkt*)hPkt)->BufferLen)
#define PBM_getDataBuffer(hPkt)     (((PBM_Pkt*)hPkt)->pDataBuffer)
#define PBM_getValidLen(hPkt)       (((PBM_Pkt*)hPkt)->ValidLen)
#define PBM_getDataOffset(hPkt)     (((PBM_Pkt*)hPkt)->DataOffset)
#define PBM_getIFRx(hPkt)           (((PBM_Pkt*)hPkt)->hIFRx)

#define PBM_setValidLen(hPkt,x)     (((PBM_Pkt*)hPkt)->ValidLen=(x))
#define PBM_setDataOffset(hPkt,x)   (((PBM_Pkt*)hPkt)->DataOffset=(x))
#define PBM_setIFRx(hPkt,x)         (((PBM_Pkt*)hPkt)->hIFRx=(x))

//-----------------------------------------------------------------------
//----[ PACKET BUFFER QUEUE ]--------------------------------------------
//-----------------------------------------------------------------------

// Packet Buffer Queue Object
typedef struct _PBMQ {
  uint              Count;      // Number of packets in queue
  PBM_Pkt           *pHead;     // Pointer to first packet
  PBM_Pkt           *pTail;     // Pointer to last packet
} PBMQ;

// Packet Queue Functions (re-entrant and "kernel mode" agnostic)
#define            PBMQ_init(pQ)   mmZeroInit( pQ, sizeof(PBMQ) )
#define            PBMQ_count(pQ) ((pQ)->Count)
_extern void       PBMQ_enq( PBMQ *pQ, PBM_Handle hPkt );
_extern PBM_Handle PBMQ_deq( PBMQ *pQ );

//-----------------------------------------------------------------------
//----[ STACK EVENT OBJECT ]---------------------------------------------
//-----------------------------------------------------------------------

// Stack Event Object
typedef struct _stkevent {
    SEM_Handle  hSemEvent;
    uint        EventCodes[3];
} STKEVENT;


#define STKEVENT_TIMER        0
#define STKEVENT_ETHERNET     1
#define STKEVENT_SERIAL       2

// STKEVENT Handle
typedef void * STKEVENT_Handle;


// Packet Queue Functions (kernel mode agnostic

//
// void STKEVENT_signal( STKEVENT_Handle hEventRec, uint Event, uint Ext );
//          hEventRec       Handle to Event Record
//          Event           Event Code to Signal
//          Ext             Set to 1 if triggered by an external event (isr)
//                          Set to 0 if detected by polling function
#define STKEVENT_signal(h,event,ext) { \
                                ((STKEVENT *)h)->EventCodes[event]=1; \
                                if( ext && ((STKEVENT *)h)->hSemEvent ) \
                                SEM_post( ((STKEVENT *)h)->hSemEvent ); }

#define STKEVENT_init(h,hSem) { \
                                mmZeroInit( h, sizeof(STKEVENT) ); \
                                ((STKEVENT *)h)->hSemEvent = hSem; }


//---------------------------------------------
//---------------------------------------------
// TASK
//---------------------------------------------
//---------------------------------------------

//
// These functions may need to be hooked or ported
//
_extern HANDLE TaskCreate( void(*pFun)(), char *Name,
                          int Priority, uint StackSize,
                          UINT32 Arg1, UINT32 Arg2, UINT32 Arg3 );
_extern void   TaskDestroy( HANDLE h );
_extern void   TaskSetEnv( HANDLE h, int Slot, HANDLE hEnv );
_extern HANDLE TaskGetEnv( HANDLE h, int Slot );
_extern void   TaskExit();

//
// These functions map directly to SPOX
//
#define TaskBlock( h )          TSK_setpri( (TSK_Handle)h, -1 )
#define TaskSelf()              ((HANDLE)TSK_self())
#define TaskSetPri( h, p )      TSK_setpri( (TSK_Handle)h, p )
#define TaskGetPri( h )         TSK_getpri( (TSK_Handle)h )
#define TaskSleep( t )          TSK_sleep( t )
#define TaskYield()             TSK_yield()

//
// Kernel Level Gateway Functions
//
_extern void  llEnter();
_extern void  llExit();

// Low Resource Flag (for use by Kernel functions)
extern uint   _TaskFlagLowResource;

//---------------------------------------------
//---------------------------------------------
// SEM
//---------------------------------------------
//---------------------------------------------

//
// These functions map directly to SPOX
//
#define SEM_FOREVER SYS_FOREVER
#define SemCreate( c )          ((HANDLE *)SEM_create( c, 0 ))
#define SemDelete( h )          SEM_delete( (SEM_Handle)h )
#define SemCount( h )           SEM_count( (SEM_Handle)h ) )
#define SemPend( h, t )         SEM_pend( (SEM_Handle)h, t )
#define SemPost( h )            SEM_post( (SEM_Handle)h )
#define SemReset( h, c )        SEM_reset( (SEM_Handle)h, c )

//---------------------------------------------
//---------------------------------------------
// MEMORY
//---------------------------------------------
//---------------------------------------------

_extern void   *mmAlloc( uint Size );
_extern void   mmFree( void* pv );
_extern void   mmCopy( void* pDst, void* pSrc, uint Size );
_extern void   mmZeroInit( void* pDst, uint Size );
_extern void   *mmBulkAlloc( INT32 Size );
_extern void   mmBulkFree( void *pMemory );

//---------------------------------------------
//---------------------------------------------
// DEBUG
//---------------------------------------------
//---------------------------------------------
// Debug Log
#define LL_DEBUG_LOG_MAX        1024
extern int   DebugCritError;                // Set on critical error
extern int   DebugLogSize;                  // Size of data in DebugLog
extern char far DebugLog[LL_DEBUG_LOG_MAX]; // Buffer holding debug messages

_extern void    DbgPrintf(uint Level, char *fmt, ... );
#define DBG_INFO        1
#define DBG_WARN        2
#define DBG_ERROR       3
#define DBG_NONE        4

//---------------------------------------------
//---------------------------------------------
// Fast Crit Sections
//---------------------------------------------
//---------------------------------------------
_extern uint OEMSysCritOn();
_extern void OEMSysCritOff( uint enable );

//---------------------------------------------
//---------------------------------------------
// Cache
//---------------------------------------------
//---------------------------------------------
_extern void OEMCacheClean( void *pAddr, uint Length );
_extern void OEMCacheCleanSynch();

//---------------------------------------------
//---------------------------------------------
// PRINTF
//---------------------------------------------
//---------------------------------------------

_extern int  printf(const char *format, ...);
_extern int  sprintf(char *s, const char *format, ...);
_extern int  vprintf(const char *format, va_list arg);
_extern int  vsprintf(char *s, const char *format, va_list arg);

//---------------------------------------------
//---------------------------------------------
// Embedded File System
//---------------------------------------------
//---------------------------------------------

// File system equates
#define EFS_FILENAME_MAX    80

// File type equates
#define EFS_SEEK_SET        0
#define EFS_SEEK_CUR        1
#define EFS_SEEK_END        2
#define EFS_FILE            void

// Executable function
typedef void (*EFSFUN)();

// Functions
_extern void      efs_createfile( char *name, INT32 length, UINT8 *pData );
_extern void      efs_createfilecb( char *name, INT32 length, UINT8 *pData,
                                    EFSFUN pllDestroyFun, UINT32 MemMgrArg );

_extern void      efs_destroyfile( char *name );
_extern EFSFUN    efs_loadfunction( char *name );

_extern EFS_FILE *efs_fopen( char *name, char *mode );
_extern int       efs_fclose( EFS_FILE *stream );
_extern int       efs_feof( EFS_FILE *stream );
_extern size_t    efs_fread( void *ptr, size_t size, size_t nobj,
                            EFS_FILE *stream );
_extern size_t    efs_fwrite( void *ptr, size_t size, size_t nobj,
                             EFS_FILE *stream );
_extern INT32     efs_fseek( EFS_FILE *stream, INT32 offset, int origin );
_extern INT32     efs_ftell( EFS_FILE *stream );
_extern void      efs_rewind( EFS_FILE *stream );
_extern INT32     efs_getfilesize( EFS_FILE *f );


#endif
