/*-----------------------------------------------------------------------------
 * Project: TIMER_interrupt_armL138
 * File: main.c
 *
 * This file contains the test / demo code to demonstrate basic TIMER 
 * operations using the Regsiter CSL macros.
 *
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 *-----------------------------------------------------------------------------
 * 
 * TIMER_interrupt_armL138
 * -----------------------
 * 	Purpose: Demonstrate usage of TIMER rCSL
 * 
 *	Example Description:
 * 		This example initializes the Timer peripheral in the 32 Bit Unchain
 * 		mode by using the rCSL macros and demonstrates the Timer module as
 * 		a general purpose Timer. After initialization, the Timer runs
 * 		continuously counting the number of interrupts that occur. After
 * 		five interrupts, the example exits.
 *
 *	-> SEE README IN PROJECT FOLDER FOR DETAILS AND STEPS TO RUN EXAMPLE <-
 *
 *---------------------------------------------------------------------------*/
#include <TIMER_interrupt_armL138.h>
#include "OMAPL138_init.h"

void EDMA3_1_ISR();

/*-----------------------------------------------------------------------------
 *
 * 				BEGIN EXAMPLE: TIMER_interrupt_armL138
 * 
 *---------------------------------------------------------------------------*/
void main (void)
{
	// Initialize OMAPL138 EVM (Config Timer w/ARM)
	init_OMAPL138();
	
	// Configure the 64 Bit Timer0 as 32 Bit Unchain
	setup_Timer0();

	EDMA3_init();

	// Map system interrupts to the ARM interrupt controller
	setup_ARM_INTC();

	_enable_IRQ();

	// Run Example
	TIMER_interrupt_ARMexample();
	while(1);
}/* End of main */


/*-----------------------------------------------------------------------------
 *
 * 						Internal Functions Defined
 * 
 *---------------------------------------------------------------------------*/
static void init_OMAPL138 (void)
{
	// Intrinsic Function; Gain 'supervisor' privileges
	_call_swi(ARM_PRIV_MODE_KEY);
	
	// Open Permissions to SYSCFG Registers
	CSL_FINS(sysRegs->KICK0R, SYSCFG_KICK0R_KICK0, KICK0_KEY);
	CSL_FINS(sysRegs->KICK1R, SYSCFG_KICK1R_KICK1, KICK1_KEY);
	
	// Configure Timer0 with the ARM CPU
	CSL_FINST(sysRegs->SUSPSRC, SYSCFG_SUSPSRC_TIMER64P_0SRC, ARM);
	
    GPIO_init(); // add by Tany

	// Close Permissions to SYSCFG Registers
	CSL_FINS(sysRegs->KICK0R, SYSCFG_KICK0R_KICK0, KICK_LOCK);
}/* init_OMAPL138 */

/*---------------------------------------------------------------------------*/

static void setup_Timer0 (void)
{
	printf("Configure timer 0 register.\n");
	// Set Timer0 as 32 Bit Unchain
	CSL_FINST(tmr0Regs->TGCR, TMR_TGCR_TIMMODE, 32BIT_UNCHAIN);
	
	// Remove Timer0:12 from Reset
	CSL_FINST(tmr0Regs->TGCR, TMR_TGCR_TIM12RS, NO_RESET);
	
	// Reset the Counter for Timer0:12
	CSL_FINST(tmr0Regs->TIM12, TMR_TIM12_TIM12, RESETVAL);
	
	// Select Internal Clock for Timer0:12 (24 MHz)
	CSL_FINST(tmr0Regs->TCR, TMR_TCR_CLKSRC12, INTERNAL);
	
	// Set Timer0:12 Period (1 Second)
	CSL_FINS(tmr0Regs->PRD12, TMR_PRD12_PRD12, (CSL_ASYNC_2_FREQ));
	
	// Disable the New Timer Features
	CSL_FINST(tmr0Regs->TGCR, TMR_TGCR_PLUSEN, DISABLE);
}/* setup_Timer0 */

/*---------------------------------------------------------------------------*/

static void setup_ARM_INTC (void)
{
	// Reset AINTC
	aintcRegs->ECR1 = AINTC_SYSINT_0_31_ALL;
	aintcRegs->ECR2 = AINTC_SYSINT_32_63_ALL;
	aintcRegs->ECR3 = AINTC_SYSINT_64_95_ALL;
	aintcRegs->SECR1 = AINTC_SYSINT_0_31_ALL;
	aintcRegs->SECR2 = AINTC_SYSINT_32_63_ALL;
	aintcRegs->SECR3 = AINTC_SYSINT_64_95_ALL;
	
	// Assign ISR Functions to System Interrupts
	sysISRtbl[AINTC_EVENTID_T64P0_TINT12] = TIMER0_12_isr;
	// Register EDMA ISR Functions to System Interrupts Tony Tang
	printf("Register EDMA completion ISR\n");
	sysISRtbl[AINTC_EVENTID_EDMA3_1_CC0_INT0] = EDMA3_1_ISR;
	
	// Assign the ISR Table Address to VBR
	aintcRegs->VBR = (unsigned int) sysISRtbl;
	
	// Declare ISR Size (Function Pointer = 4 bytes)
	aintcRegs->VSR = 0;
	
	// Map Timer System Interrupts to Channel 2
	CSL_FINS(aintcRegs->CMR5, AINTC_CMR5_CHNL_NPLUS1, 2);

	//  Map EDMA1 System Interrupts to Channel 3  Tony Tang
	CSL_FINS(aintcRegs->CMR24, AINTC_CMR5_CHNL_NPLUS1, 3);

	// Enable Timer Interrupts
	CSL_FINS(aintcRegs->EISR, AINTC_EISR_INDEX, AINTC_EVENTID_T64P0_TINT12);
	// Enable EDMA1 Interrupts
	CSL_FINS(aintcRegs->EISR, AINTC_EISR_INDEX, AINTC_EVENTID_EDMA3_1_CC0_INT0);
	
	/* Enable IRQ Interrupts */
	CSL_FINS(aintcRegs->HIEISR, AINTC_HIEISR_INDEX, 1);
	
	/* Enable Interrupts */
	CSL_FINS(aintcRegs->GER, AINTC_GER_ENABLE, 1);
}/* setup_ARM_INTC */

/*---------------------------------------------------------------------------*/

static void TIMER_interrupt_ARMexample (void)
{
	printf("Example: TIMER_interrupt_ARMexample\n"
		   "Scope: Timer0:12 runs continuously causing an "
		   "interrupt every second.\n"
		   "Enable Timer...\n");

//	runExample = 1, counter = 0;

	// Intrinsic Function to Enable Interrupts
//	_enable_IRQ();
	
	// Enable Timer0:12 Continuously
	CSL_FINST(tmr0Regs->TCR, TMR_TCR_ENAMODE12, EN_CONT);
	
	// Exit After Five Interrupts Occur
//	while(runExample);
	
	// Disable Timer0:12
//	CSL_FINST(tmr0Regs->TCR, TMR_TCR_ENAMODE12, DISABLE);
	
	// Intrinsic Function to Disable Interrupts

//	_disable_IRQ();
	
//	printf("End of example.\n\n");
}/* TIMER_interrupt_ARMexample */


/*-----------------------------------------------------------------------------
 *
 * 							Interrupt Functions
 * 
 *---------------------------------------------------------------------------*/
void TIMER0_12_isr (void)
{
	counter++;
	
	printf("\tInterrupt Count: %d\n", counter);
	
	if(counter == 100)
		runExample = 0;
	
	// Acknowledge Timer0:12 Interrupt
	CSL_FINS(aintcRegs->SICR, AINTC_SICR_INDEX, AINTC_EVENTID_T64P0_TINT12);
}/* TIMER0_12_isr */





