/*  ===========================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 *   ==========================================================================
 */

/** @mainpage CHIP CSL 3.x
 *
 * @section Introduction
 *
 * @subsection xxx Purpose and Scope
 * The purpose of this document is to identify a set of common CSL APIs for
 * the CHIP module. The CSL developer is expected to refer to this document  
 * while designing APIs for the modules which use CHIP module. Some of the
 * listed APIs may not be applicable to a given module. While other cases
 * this list of APIs may not be sufficient to cover all the features required
 * for a particular Module. The CSL developer should use his discretion 
 * designing new APIs or extending the existing ones to cover these.
 *
 * @subsection aaa Terms and Abbreviations
 *   -# CSL:  Chip Support Library
 *   -# API:  Application Programmer Interface
 * 
 *
 */
 
/** @file  csl_chip.h 
 *
 *  @brief CHIP functional layer API header file
 *
 * Description
 *    - The different enumerations, structure definitions
 *      and function declarations
 *
 * Modification 
 *    - modified on: 23/11/2004
 *    - reason: Created the sources
 *
 * 
 * @author Ramitha Mathew.
 * 
 */ 

#ifndef _CSL_CHIP_H_
#define _CSL_CHIP_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <csl_types.h>
#include <cslr_chip.h>
#include <soc.h>
#include <cslr.h>
#include <csl_error.h>
#include <csl.h>

/**
 * @brief   Enumeration of the C672x System registers
 *
 * This enumeration contains the list of registers that can be manipulated
 * using the CSL_chipReadReg (..) and CSL_chipWriteReg (..) APIs
 */
 
typedef enum  {

	/** memory mapped registers with their offset from Base Address*/

	/** L1P Invalidate Start Address Register */ 
	CSL_CHIP_REG_L1PISAR          = 0x00000000u,
	/** L1P Invalidate Control Register */
	CSL_CHIP_REG_L1PICR           = 0x00000004u,
	/** Memory Control and Status Register */
	CSL_CHIP_REG_MEMCSR           = 0x00000008u,
	/** Program Counter Export Register */
	CSL_CHIP_REG_PCER             = 0x00000010u,
	/** CFGPIN0 register*/
	CSL_CHIP_REG_CFGPIN0          = 0x20000000u,
	/** CFGPIN1 register*/
	CSL_CHIP_REG_CFGPIN1          = 0x20000004u,
	/** CGFHPI register*/
	CSL_CHIP_REG_CFGHPI           = 0x20000008u,
	/** CFGHPIAMSB register*/
	CSL_CHIP_REG_CFGHPIAMSB       = 0x2000000Cu,
	/** CFGHPIAUMB register*/
	CSL_CHIP_REG_CFGHPIAUMB       = 0x20000010u,
	/** CGFRTI register*/
	CSL_CHIP_REG_CFGRTI           = 0x20000014u,
	/** CGFMCASP0 register*/
	CSL_CHIP_REG_CFGMCASP0        = 0x20000018u,
	/** CGFMCASP1 register*/
	CSL_CHIP_REG_CFGMCASP1        = 0x2000001Cu,
	/** CGFMCASP2 register*/
	CSL_CHIP_REG_CFGMCASP2        = 0x20000020u,
	/** CGFBRIDGE register*/
	CSL_CHIP_REG_CFGBRIDGE        = 0x20000024u, 
	/** IDREG register*/
	CSL_CHIP_REG_IDREG            = 0x20000080u,
	/** DFT_READ_WRITE register*/
	CSL_CHIP_REG_DFT_READ_WRITE   = 0x20000084u,

	/*non memory mapped registers */

	/** Addressing Mode Register */
	CSL_CHIP_REG_AMR              = 0x01u,
	/** Control Status Register */
	CSL_CHIP_REG_CSR              = 0x03u,
	/** Floating Point Adder Configuration register */
	CSL_CHIP_REG_FADCR            = 0x05u,
	/** Floating Point Auxiliary Configuration register */
	CSL_CHIP_REG_FAUCR            = 0x07u,
	/** Floating Point Multiplier Configuration register */
	CSL_CHIP_REG_FMCR             = 0x09u

} CSL_ChipReg;


/**
 * @brief CHIP Module Context. 
 */
typedef struct CSL_ChipContext {
	/**Context information of CHIP.
	* The following is only a  place-holder for future implementation.
	 */
	Uint32 contextInfo;
} CSL_ChipContext;

/**
 * @brief   CHIP registers base address
 */
typedef struct CSL_ChipBaseAddress {
	/** Base-address of the memory mapped CHIP registers 
	 */
	CSL_ChipRegsOvly	regs;
} CSL_ChipBaseAddress;

/******************************************************************************
 * CHIP API function declarations 
 *****************************************************************************/

/** ===========================================================================
 *   @n@b CSL_chipInit
 *
 *   @b Description
 *   @n This is the initialization function for the CHIP CSL. The function
 *      must be called before calling any other API from this CSL. This
 *      function does not modify any registers or check status. It returns 
 *      status CSL_OK. It has been kept for the future use.
 *
 *   @b Arguments
 *   @verbatim
        pContext    Pointer to module-context. As CHIP doesn't have
                    any context based information user is expected to pass NULL
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Always returns
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The CSL for CHIP is initialized
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
            CSL_chipInit (NULL);
     @endverbatim
 * ============================================================================
 */
extern CSL_Status   CSL_chipInit (   
    CSL_ChipContext 	*pContext  
);


/**============================================================================
 *   @n@b CSL_chipWriteReg
 * 
 *   @b Description 
 *   @n This function could be used to write to the CHIP registers. The
 *      register that can be specified could be one of those enumerated
 *      in CSL_ChipReg.
 *
 *   @b Arguments
     @verbatim
        regId - This is the register id specified for the register 
                through the enum
        val   - Value to be written into the register
     @endverbatim
 *
 *   <b> Return Value </b>  
 *   @li Uint32 - The value in the register before the new value being written
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_chipInit () must be called
 *
 *   <b> Post Condition </b>
 *   @n  The specified register will be modified
 *
 *   @b Modifies
 *   @n The specified register will be modified
 *
 *   @b Example:
     @verbatim
  
       Uint32 oldRegVal;
       oldRegVal = CSL_chipWriteReg ( CSL_CHIP_REG_AMR, newRegVal);
  
     @endverbatim 
 * ============================================================================
 */
extern Uint32  CSL_chipWriteReg (
    CSL_ChipReg     regId,  
    Uint32          val   
);


/** ============================================================================
 *   @n@b CSL_chipReadReg
 * 
 *   @b Description 
 *   @n This function could be used to read the CHIP registers. The
 *      register that can be specified could be one of those enumerated
 *      in CSL_ChipReg.
 *
 *   @b Arguments
     @verbatim
       regId - This is the register id specified for the register 
               through the enum       
     @endverbatim
 *
 *   <b> Return Value </b>  
 *   @li Uint32 - The value read from the register
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_chipInit () must be called
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example:
     @verbatim
  
     Uint32 readValue;
     readValue = CSL_chipReadReg (CSL_CHIP_REG_CSR);
  
     @endverbatim 
 * ============================================================================
 */
extern Uint32  CSL_chipReadReg (
    CSL_ChipReg     regId 
);

#ifdef __cplusplus
}
#endif

#endif /* _CSL_CHIP_H_ */

