/*  ===========================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 *   ==========================================================================
 */
 
/** @mainpage DMAX CSL 3.x
 *
 * @section Introduction
 *
 * @subsection xxx Purpose and Scope
 * The purpose of this document is to identify a set of common CSL APIs for
 * the DMAX module across various devices. The CSL developer is expected to
 * refer to this document while designing APIs for these modules. Some of the
 * listed APIs may not be applicable to a given DMAX module. While other cases
 * this list of APIs may not be sufficient to cover all the features of a
 * particular DMAX Module. The CSL developer should use his discretion 
 * designing new APIs or extending the existing ones to cover these.
 *
 * @subsection aaa Terms and Abbreviations
 *   -# CSL:  Chip Support Library
 *   -# API:  Application Programmer Interface
 *   -# DMAX: Data Movement Accelerator
 *
 * @subsection References
 *    -# CSL-001-DES, CSL 3.x Design Specification DocumentVersion 1.02
 *
 */

 /** @file csl_dmax.h
 *
 * @brief    Header file for functional layer of CSL
 *
 * Description
 *    - The different enumerations, structure definitions
 *      and function declarations
 *
 * Modification 
 *    - modified on: 10/Mar/2005
 *    - reason: Created the sources
 *
 * 
 * @author asr.
 *
 */

#ifndef _CSL_DMAX_H_
#define _CSL_DMAX_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <cslr.h>
#include <csl_error.h>
#include <soc.h>
#include <csl_types.h>
#include <cslr_dmax.h>

/**==========================================================================
* CSL Dmax Generic Pointers
*==========================================================================*/
/**
 * @brief Dmax Event Setup 
 */
typedef void * CSL_DmaxEventSetup;
/**
 * @brief DMAX Prameter Setup
 */
typedef void * CSL_DmaxParameterSetup;

/**
 * @brief   Invalid handle
 */

#define CSL_DMAX_BADHANDLE              (0)

/**************************************************************************\
* DMAX global macro declarations
\**************************************************************************/
#define CFGBRIDGE               *((volatile unsigned int *) (0x40000024))
/** DMAX Module ID */
#define CSL_DMAX_ID             (1)
/** DMAX First */
#define CSL_EDMAX_FIRST             (-(((CSL_DMAX_ID+1)<<5)+1))
/** DMAX Last */
#define CSL_EDMAX_LAST              (-((CSL_DMAX_ID+1)<<6))
/** DMAX Parameter Entry Full Error */
#define CSL_EDMAX_PARAM_ENTRY_FULL_ERROR        (CSL_EDMAX_FIRST-0)
/** DMAX Bad ESR Flag */
#define CSL_EDMAX_BAD_ESR_FLAG                  (CSL_EDMAX_FIRST-1)
/** DMAX Bad Event Type */
#define CSL_EDMAX_BAD_ETYPE                     (CSL_EDMAX_FIRST-2)
/** DMAX Polarity Rising Edge */
#define CSL_DMAX_POLARITY_RISING_EDGE           (0x00000001u)
/** DMAX polarity Falling Edge */
#define CSL_DMAX_POLARITY_FALLING_EDGE          (0x00000000u)
/** DMAX word size */
#define CSL_DMAX_WORD_SIZE                      (0x00000004u)
/** DMAX Transfer Complete split */
#define CSL_DMAX_TCC_SPLIT                      (0x000000FFu)
/** DMAX High Priority Event */
#define CSL_DMAX_HIPRIORITY_EVENT_ANY           (0x1000FFFFu)
/** DMAX Low Priority Event */
#define CSL_DMAX_LOPRIORITY_EVENT_ANY           (0x2000FFFFu)
/** DMAX High Priority Parameter entry */
#define CSL_DMAX_HIPRIORITY_PARAMETERENTRY_ANY  (0x3000FFFFu)
/** DMAX Low Priority Parameter entry */
#define CSL_DMAX_LOPRIORITY_PARAMETERENTRY_ANY  (0x4000FFFFu)
/** DMAX High Priority */
#define CSL_DMAX_HI_PRIORITY                    (0x00000001u)
/** DMAX FIFO Full */
#define CSL_DMAX_FIFO_FULL_MASK                 (0x01000000u)
/** DMAX FIFO Full shift */
#define CSL_DMAX_FIFO_FULL_SHIFT                (0x00000018u)
/** DMAX FIFO UnderFlow mask*/
#define CSL_DMAX_FIFO_UNDERFLOW_MASK            (0x00000001u)
/** DMAX FIFO Underflow shift */
#define CSL_DMAX_FIFO_UNDERFLOW_SHIFT           (0x00000000u)
/** DMAX FIFO Nonsequential Underflow mask */
#define CSL_DMAX_FIFO_NONSEQUENTIAL_UNDERFLOW_MASK  (0x00000100u)
/** DMAX FIFO Nonsequential Underflow shift */
#define CSL_DMAX_FIFO_NONSEQUENTIAL_UNDERFLOW_SHIFT (0x00000008u)
/** DMAX FIFO OverFlow mask */
#define CSL_DMAX_FIFO_OVERFLOW_MASK             (0x00010000u)
/** DMAX FIFO Overflow shift */
#define CSL_DMAX_FIFO_OVERFLOW_SHIFT            (0x00000010u)
/** DMAX Parameter Entry Size */
#define _CSL_DMAX_PARAM_ENTRY_SIZE              (11)
/** DMAX Parameter table entry Base offset */
#define CSL_DMAX_PTE_BASE_OFFSET                (0x00000028u)
/** DMAX Event High Priority */
#define CSL_DMAX_EVENT_HI_PRIORITY              (0x00000001u)
/** DMAX Event Low Priority */
#define CSL_DMAX_EVENT_LO_PRIORITY              (0x00000000u)
/** DMAX Bad ESR Flag */
#define CSL_DMAX_BAD_ESR_FLAG                   (0x000000FFu)
/** DMAX ESR Fifo Error */
#define CSL_DMAX_ESR_FIFOERR_MASK               (0x00000080u)
/** DMAX ESR Fifo Error */
#define CSL_DMAX_ESR_FIFOERR_SHIFT              (0x00000007u)
/** DMAX Get Parameter Entry offset */
#define CSL_DMAX_GET_PARAM_ENTRY_OFFSET(uid)    (((uid) & \
                        0xFFFFu)*_CSL_DMAX_PARAM_ENTRY_SIZE)
/**************************************************************************\
* DSMAX global typedef declarations
\**************************************************************************/

/** @brief This structure contains the necesary information to define the FIFO
 *         paramters for the reload address for the data transmission.
 *
*/
typedef struct {
    /** Linear Address */
    Uint32 linearAddr;
    /** Pointer to FIFO Descriptor */
    Uint32 pfd;
    /** Count Value */
    Uint32 count;
    /** Index0 Value */
    Uint32 indx0;
    /** Index1 Value */
    Uint32 indx1;
    /** Pointer to Dealy Table */
    Uint32 dtabPtr0;
    /** Poiter to Dealy Table */
    Uint32 dtabPtr1;
    /** Linear Reload0 Value */
    Uint32 linearReload0;
    /** Linear Reload1 Value */
    Uint32 linearReload1;
} CSL_DmaxFifoParam;

/** @brief This structure contains the necesary information to define the FIFO
 *        paramters for any FIFO that is used in dmax FIFO read/write operations
 *
 */
typedef struct {
    /** FIFO Base Address */
    Uint32  fifoBaseAddr;
    /** FIFO Write Pointer */
    Uint32  wPtr;
    /** FiFO Size */
    Uint32  fifoSize;
    /** FIFO Read Pointer */
    Uint32  rPtr;
    /** FIFO Full Watermark */
    Uint32  fmark;
    /** FIFO Empty Watermark */
    Uint32  emark;
    /** FIFO Error Field */
    Uint32  efield;
} CSL_DmaxFifoDesc;

/** @brief This structure contains the necesary information to define the FIFO
 *      paramters for any FIFO that is used in dmax FIFO read/write operations
 *
 */
typedef struct {
    /** FIFO Base Address */
    Uint32  fifoBaseAddr;
    /** FIFO Write Pointer */
    Uint32  wPtr;
    /** FIFO Size */
    Uint32  fifoSize;
    /** FIFO Read Pointer */
    Uint32  rPtr;
    /** FIFO Full Watermark */
    Uint32  fMark;
    /** FIFO Empty Watermark */
    Uint32  eMark;
    /** FIFO Full Mark Status */
    Uint8   fmsc;
    /** FIFO Empty Mark Status */
    Uint8   emsc;
    /** FIFO Error Field */
    Uint8   esize;
} CSL_DmaxFifoDescriptorSetup;

/** @brief This structure contains blank form for a FIFO descriptor object
 */
typedef struct {
    /** Word0 */
    Uint32  word0;
    /** Word1 */
    Uint32  word1;
    /** Word2 */
    Uint32  word2;
    /** Word3 */
    Uint32  word3;
    /** Word4 */
    Uint32  word4;
    /** Word5 */
    Uint32  word5;
    /** Word6 */
    Uint32  word6;
} CSL_DmaxFifoDescriptor;

/** @brief This structure contains information for setting the active set of
 *          parameters in a general purpose trasnfer parameter entry.   
 *
 */
typedef struct {
    /** Active Source Address */
    Uint32  activeSrc;
    /** Active Destination Address */
    Uint32  activeDst;
    /** Active Count0 Value */
    Uint16  activeCount0;
    /** Active Count1 Value */
    Uint16  activeCount1;
    /** Active Count2 Value */
    Uint16  activeCount2;
    /** Active Ping Pong Value */
    Uint8   activePP;
} CSL_DmaxActiveSet;

/** @brief This structure contains the information necessary to perform HW setup
 *         for a general purpose data transfer request for dmax.
 */
typedef struct {
    /** Source Reload Address0 
        This value will be loaded as the active 
        address value for the first transfer */
    Uint32  srcReloadAddr0;
    /** Destination Reload Address0
        This value will be loaded as the active 
        address value for the first transfer */
    Uint32  dstReloadAddr0;
    /** Source Reload Address1 
       If the reload is enabled the next active address
       is taken from this variable */
    Uint32  srcReloadAddr1;
    /** Destination Reload Address1
       If the reload is enabled the next active address
       is taken from this variable */
    Uint32  dstReloadAddr1;
    /** Source Index0 Value */
    Int16   srcIndex0;
    /** Destincation Index0 Value */
    Int16   dstIndex0;
    /** Source Index1 Value */
    Int16   srcIndex1;
    /** Destincation Index1 Value */
    Int16   dstIndex1;
    /** Source Index2 Value */
    Int16   srcIndex2;
    /** Destincation Index2 Value */
    Int16   dstIndex2;
    /** Dimension 0 Count Value */
    Uint16  count0;
    /** Dimension 1 Count Value */
    Uint16  count1;
    /** Dimension 2 Count Value */
    Uint16  count2;
} CSL_DmaxGPXFRParameterSetup;

/** @brief This structure contains the data items necessary to represent an
 *         FIFO R/W Parameter Entry in dmax Parameter RAM.
 *
 */
typedef struct {
    /** Pointer to FIFO Descriptor */
    Uint32  pfd;
    /** Count0 Value */
    Uint16  count0;
    /** Count1 Value */
    Uint16  count1;
    /** Linear Index0 Value */
    Int16   linearIndx0;
    /** Linear Index1 Value */
    Int16   linearIndx1;
    /** Pointer to Dealy Table0 */
    Uint32  delayTabPtr0;
    /** Pointer to Dealy Table1 */
    Uint32  delayTabPtr1;
    /** Linear Reload0 Value 
        This value will be loaded as the active 
        address value for the first transfer */
    Uint32  linearReload0;
    /** Linear Reload1 Value
       If the reload is enabled the next active address
       is taken from this variable */
    Uint32  linearReload1;
}CSL_DmaxFifoParameterSetup;

/** @brief This structure contains all data necessary to setup dmax for a 
 *         General Purpose data transfer request.
 *
 */
typedef struct {
    /** Event Type */
    Uint32                  etype;
    /** Pointer to parameterSetup */
    CSL_DmaxParameterSetup  paramSetup;
    /** Quantum Transfer Size Limit */ 
    Uint8                   qtsl;
    /** Transfer Synchronization */
    Uint8                   sync;
    /** Transfer Complete Code */
    Uint8                   tcc;
    /** Alternate Transfer Mode Interrupt */
    Uint8                   atcint;
    /** Transfer Completion Interrupt */
    Uint8                   tcint;
    /** Reload */
    Uint8                   rload;
    /** Counter Configuration */
    Uint8                   cc;
    /** Element Size */
    Uint8                   esize;
    /** Pointer to Transfer Entry */
    Uint8                   pte;
} CSL_DmaxGPTransferEventSetup;
 
/** @brief This structure contains all data necessary to setup dmax for a 
 *         data transfer request.
 *
 */
typedef struct {
    /** Event Type */
    Uint32                  etype;
    /** Pointer to parameterSetup */
    CSL_DmaxParameterSetup  paramSetup;
    /** Quantum Transfer Size Limit */  
    Uint8                   qtsl;
    /** Transfer Synchronization */    
    Uint8                   sync;
    /** Transfer Complete Code */    
    Uint8                   tcc;
    /** Alternate Transfer Mode Interrupt */    
    Uint8                   atcint;
    /** Transfer Completion Interrupt */    
    Uint8                   tcint;
    /** Reload */    
    Uint8                   rload;
    /** Water Mark Enable */    
    Uint8                   wmenab;
    /** Pointer to Transfer Entry */
    Uint8                   pte;
} CSL_DmaxFifoTransferEventSetup;

/** @brief This structure contains the information needed to perform HW setup
 *         for CPU interrupt requests.
 *
 */
 typedef struct {
     /** Event Type */
     Uint32     etype;
     /** CPU Interrupt Line */
    Uint8       cpuInt;             
 }CSL_DmaxCpuintEventSetup;

/** @brief This object contains the reference to the instance of DMAX opened
 *         using in the @a CSL_dmaxOpen().
 *
 */
typedef struct CSL_DmaxAlloc {
    /** Maximum Count */
    Uint16      maxCount;
    /** Allocation Map */
    Uint32      allocMap;
} CSL_DmaxAlloc;    

/** @brief This object contains the reference to the instance of DMAX opened
 *  using in the @a CSL_dmaxOpen().
 *
 */
typedef struct CSL_DmaxResourceAlloc {
    /** Dmax High Priority Event */
    CSL_DmaxAlloc   cslDmaxHpecrAlloc;
    /** Dmax Low Priority Event */
    CSL_DmaxAlloc   cslDmaxLpecrAlloc;
    /** Dmax High Priority Parameter */
    CSL_DmaxAlloc   cslDmaxHpptAlloc;
    /** Dmax Low Priority Parameter */
    CSL_DmaxAlloc   cslDmaxLpptAlloc;
} CSL_DmaxResourceAlloc;    

/** @brief This object contains the reference to the instance of DMAX opened
 *  using the @a CSL_dmaxOpen().
 *
 *  The pointer to this, is passed to all DMAX CSL APIs.
 */
typedef struct CSL_DmaxObj {
    /** Pointer to entry in High Event Table reserved for this event */
    Uint32  *hiTableEventEntryPtr;
    /** Pointer to entry in Low Event Table reserved for this event */
    Uint32  *loTableEventEntryPtr;
    /** Pointer to the dmax Registers */
    CSL_DmaxRegsOvly regs;
    /** Event Number */
    Uint32  eventUid;
    /** Parameter Entry Number */
    Uint32  paramUid;
    /** Pointer to the associated parameter entry */
    Uint8   paramPtr; 
    /** This is the instance of DMAX being referred to by this object  */
    CSL_InstNum perNum;
}CSL_DmaxObj;

/** @brief This will have the base-address information for the peripheral
 *  instance
 */
typedef struct {
    /** Base-address of the Configuration registers of DMAX.
    */
    CSL_DmaxRegsOvly    regs;
} CSL_DmaxBaseAddress;

/** @brief DMAX specific parameters. Present implementation doesn't have
 *  any specific parameters.
 */
typedef struct {
    /** Bit mask to be used for module specific parameters.
    *  The below declaration is just a place-holder for future
    *  implementation.
    */
    CSL_BitMask16   flags;
} CSL_DmaxParam;

/** @brief DMAX specific parameters. Present implementation doesn't have
 *  any specific parameters.The below declaration is just a place-holder 
 *  for future implementation.
*/
typedef struct {
    /** Context information of DMAX.
    *  The below declaration is just a place-holder for future
    *  implementation.
    */
    Uint16  contextInfo;
} CSL_DmaxContext;

/** @brief This structure defines the overlay for accessing entries in the Hi
*          Priority Parameter Table.
*
*/
typedef struct  {
    /** Parameter Table word0 */
    volatile Uint32 word0;
    /** Parameter Table word1 */    
    volatile Uint32 word1;
    /** Parameter Table word2 */    
    volatile Uint32 word2;
    /** Parameter Table word3 */    
    volatile Uint32 word3;
    /** Parameter Table word4 */    
    volatile Uint32 word4;
    /** Parameter Table word5 */    
    volatile Uint32 word5;
    /** Parameter Table word6 */    
    volatile Uint32 word6;
    /** Parameter Table word7 */   
    volatile Uint32 word7;
    /** Parameter Table word8 */    
    volatile Uint32 word8;
    /** Parameter Table word9 */    
    volatile Uint32 word9;
    /** Parameter Table word10 */    
    volatile Uint32 word10;
} CSL_DmaxParameterEntry;

/** @brief Config structure of DMAX. This is used to configure DMAX
 *  using CSL_HwSetupRaw function
*/

typedef struct  {
    /** Event Control */
    Uint32                  eventCtrl;
    /** DMAX parameter Entry Pointer */
    CSL_DmaxParameterEntry *paramEntry;
    /** Dmax Event Type */
    Uint8                   eventType;
    /** DMAX priority */
    Uint8                   priority;
    /** DMAX Polarity */
    Uint8                   polarity;
} CSL_DmaxConfig;

/** Default Values for Config structure */
#define CSL_DMAX_CONFIG_DEFAULTS {   \
    0x0,     \
    CSL_DMAX_EVENT0_ETYPE_FIFOWRITE, \
    0x0, \
    0x0, \
    0x0, \
    0x0, \
    0x0, \
    0x0, \
    0x0, \
    0x0, \
    0x0, \
    0x0  \
}

/**
 * @brief This data type is used to return the handle to an
 * instance of DMAX
 */

typedef struct CSL_DmaxObj *CSL_DmaxHandle;

/** @brief This has all the fields required to configure DMAX at Power Up
 *  (After a Hardware Reset) or a Soft Reset
 *
 *  This structure is used to setup or obtain existing setup of
 *  DMAX using @a CSL_dmaxHwSetup() & @a CSL_dmaxGetHwSetup() functions
 *  respectively.
 */
typedef struct CSL_DmaxHwSetup {
    /** DMAX Priority */
    Uint8 priority;
    /** DMAX Polarity */
    Uint8 polarity;
    /** Generic Pointer for setup */
    CSL_DmaxEventSetup eventSetup;
}CSL_DmaxHwSetup;

/** 
 * @brief This structure is used to setup the Event Type
 */
typedef struct {
    /** Event Type */
    Uint32 etype;
} CSL_DmaxEtype;

/** @brief Enumeration for queries passed to @a CSL_dmaxGetHwStatus()
 *
 * This is used to get the status of different operations or to get the
 * existing setup of DMAX.
 */
typedef enum {
    /**
    * @brief   Get status of the Event Flag.
    * @param   (Uint32*)
    */
    CSL_DMAX_QUERY_EVENTFLAG = 1,
    /**
    * @brief   Get the status of the Transfer Complete flag.
    * @param   (Uint32*)
    */
    CSL_DMAX_QUERY_GET_TCC = 2,
    /**
    * @brief   Get the status of TC.
    * @param   (Uint32*)
    */
    CSL_DMAX_QUERY_GET_TC = 3,
    /**
    * @brief   Get the status of FIFO full.
    * @param   (Uint32*)
    */
    CSL_DMAX_QUERY_GET_FMSC = 4,
    /**
    * @brief   Get the status of FIFO empty.
    * @param   (Uint32*)
    */
    CSL_DMAX_QUERY_GET_EMSC = 5,
    /**
    * @brief   Get the status of the FIFO error Flag/Interrupt.
    * @param   (Uint32*)
    */
    CSL_DMAX_QUERY_GET_FIFO_ERROR = 6,
    /**
    * @brief   Get the status of the FIFO error Code.
    * @param   (Uint32*)
    */
    CSL_DMAX_QUERY_GET_FIFO_ERROR_CODE = 7,
    /**
    * @brief   Get the status of the Event entry of the specified Event.
    * @param   (Uint32*)
    */
    CSL_DMAX_QUERY_GET_EVENT_ENTRY = 8,
    /**
    * @brief   Get the status of the parameter entry of the specified Event.
    * @param   (Uint32*)
    */
    CSL_DMAX_QUERY_GET_PARAMETER_ENTRY = 9,
    /**
    * @brief   Get the status of the High dmax.
    * @param   (Uint32*)
    */
    CSL_DMAX_QUERY_GET_HBC = 10,
    /**
    * @brief   Get the status of the Low dmax.
    * @param   (Uint32*)
    */
    CSL_DMAX_QUERY_GET_LBC = 11,
    /**
    * @brief   Get the status of the FIFO Address.
    * @param   (Uint32*)
    */
    CSL_DMAX_QUERY_GET_FIFO_ADDR = 12,
    /**
    * @brief   Get the Event entry Address.
    * @param   (Uint32*)
    */
    CSL_DMAX_QUERY_GET_EVENT_ENTRY_ADDR = 13,
    /**
    * @brief   Get the Parameter entry Address.
    * @param   (CSL_DmaxParameterEntry*)
    */
    CSL_DMAX_QUERY_GET_PARAMETER_ENTRY_ADDR = 14,
    /**
    * @brief   Get the FIFO full status.
    * @param   (Uint32*)
    */
    CSL_DMAX_QUERY_GET_FIFO_FULL = 15

} CSL_DmaxHwStatusQuery;


/** @brief Enumeration for queries passed to @a CSL_dmaxHwControl()
 *
 * This is used to select the commands to control the operations
 * existing setup of DMAX. The arguments to be passed with each
 * enumeration if any are specified next to the enumeration.
 */
 typedef enum {
    /**
    * @brief   Set Priority of Event.
    * @param   (None)
    */
    CSL_DMAX_CMD_SETPRIORITY = 1,
    /**
    * @brief   Sets Polarity of Event.
    * @param   (None)
    */
    CSL_DMAX_CMD_SETPOLARITY = 2,
    /**
    * @brief   Enables Event.
    * @param   (None)
    */
    CSL_DMAX_CMD_EVENTENABLE = 3,

    /**
    * @brief   Clears TCC Event flag.
    * @param   (None)
    */
    CSL_DMAX_CMD_CLEARTCC = 4,

    /**
    * @brief   Disables Event.
    * @param   (None)
    */
    CSL_DMAX_CMD_EVENTDISABLE = 5,

    /**
    * @brief   Clears the Event Entry.
    * @param   (Uint16 *)
    */
    CSL_DMAX_CMD_CLEAREVENTENTRY = 6,

    /**
    * @brief   Clears Parameter Entry.
    * @param   (Uint16 *)
    */
    CSL_DMAX_CMD_CLEARPARAMENTRY = 7,

    /**
    * @brief   Set the start Sync Transfer.
    * @param   (None)
    */
    CSL_DMAX_CMD_STARTASYNCTRANSFER = 8,

    /**
    * @brief   Set the FIFO status clear bit.
    * @param   (None)
    */
    CSL_DMAX_CMD_CLEARFIFOSTATUS = 9,

    /**
    * @brief   Set the Water Mark Enable.
    * @param   (None)
    */
    CSL_DMAX_CMD_WATERMARK_ENABLE = 10,

    /**
    * @brief   Set the Water Mark Disable.
    * @param   (None)
    */
    CSL_DMAX_CMD_WATERMARK_DISABLE = 11

} CSL_DmaxHwControlCmd;

/**************************************************************************\
* DMAX global function declarations
\**************************************************************************/

/*
 * =============================================================================
 *   @func CSL_dmaxInit
 *
 *   @desc
 *   @n This is the initialization function for the DMAX CSL. This function 
 *      needs to be called before any other DMAX CSL functions are called. 
 *      This function does not modify any registers or check status. It returns 
 *      status CSL_SOK. It has been kept for the future use.
 *
 *   @arg pContext
 *        Context information for DMAX
 *
 *   @ret CSL_Status
 *        CSL_SOK - Always returns
 *
 *   @eg
 *     CSL_dmaxInit (NULL);
 * =============================================================================
 */
extern CSL_Status  CSL_dmaxInit(
    CSL_DmaxContext * pContext
);


/*
 *============================================================================
 *   @func CSL_dmaxOpen
 *
 *   @desc
 *        This function populates the peripheral data object for the instance
 *        and returns a handle to the instance.
 *        The open call sets up the data structures for the particular instance
 *        of DMAX device. The device can be re-opened anytime after it has been
 *        normally closed if so required. DMAX Hardware setup will be performed
 *        at the end of the open call only if the HwSetup Pointer supplied was
 *        non- NULL. The handle returned by this call is input as an essential
 *        argument for rest of the APIs described for this module.
 *
 *   @arg pDmaxObj
 *        Pointer to the DMAX instance object
 *
 *   @arg dmaxNum
 *        Instance of the DMAX to be opened.
 *
 *   @arg pDmaxParam
 *        Pointer to module specific parameters
 *
 *   @arg pStatus
 *        pointer for returning status of the function call
 *
 *   @ret CSL_DmaxHandle
 *        Valid DMAX instance handle will be returned if status value is
 *        equal to CSL_SOK.
 *
 *   @eg
 *        CSL_status        status;
 *        CSL_DmaxObj       dmaxObj;
 *        CSL_DmaxHandle     hDmax;
 *
 *        hDmax = CSL_DmaxOpen (&dmaxObj,
 *                            CSL_DMAX_PER_CNT,
 *                            NULL,
 *                            &status
 *                            );
 *
 * ===========================================================================
 */
extern CSL_DmaxHandle CSL_dmaxOpen (
    CSL_DmaxObj              *pDmaxObj,
    CSL_InstNum              dmaxNum,
    CSL_DmaxParam            *pDmaxParam,
    CSL_Status               *pStatus
);


/*
 * =============================================================================
 *   @func CSL_dmaxClose
 *
 *   @b Description
 *   @n This function closes the specified instance of DMAX.
 *
 *   @arg  hDmax
           Handle to the DMAX instance
 *
 *   @ret CSL_Status
 *         CSL_SOK            - Close successful
 *         CSL_ESYS_BADHANDLE - Invalid handle
 *
 *   @eg
 *     CSL_dmaxClose (hDmax);
 * =============================================================================
 */
extern CSL_Status  CSL_dmaxClose(
    CSL_DmaxHandle  hDmax
);


/*
 * ===============================================================================
 * This function initializes the device registers with the appropriate values
 *  provided through the HwSetup Data structure. This function needs to be called
 *  only if the HwSetup Structure was not previously passed through the Open call.
 *  After the Setup is completed, the serial device is ready for data transfer.
 *  For information passed through the HwSetup Data structure refer
 *  @a CSL_DmaxHwSetup.
 *
 *  <b> Usage Constraints: </b>
 *  Both @a CSL_dmaxInit() and @a CSL_dmaxOpen() must be called
 *  successfully in that order before this function can be called. The user
 *  has to allocate space for & fill in the main setup structure appropriately
 *  before calling this function
 *
 * @b Example:
 * @verbatim
     CSL_DmaxHandle hDmax;
     CSL_DmaxHwSetup hwSetup = CSL_DMAX_HWSETUP_DEFAULTS;
     CSL_dmaxHwSetup(hDmax, &hwSetup);
  @endverbatim
 *
 * @return Returns the status of the setup operation
 *=================================================================================
 */
extern CSL_Status  CSL_dmaxHwSetup(
    CSL_DmaxHandle                         hDmax,
    CSL_DmaxHwSetup                        *setup
);

/*
 * ================================================================================
 * This function gets the current setup of the DMAX. The status is
 *  returned through @a CSL_DmaxHwSetup. The obtaining of status
 *  is the reverse operation of @a CSL_dmaxHwSetup() function.
 *
 *  <b> Usage Constraints: </b>
 *  Both @a CSL_dmaxInit() and @a CSL_dmaxOpen() must be called successfully
 *  in that order before @a CSL_dmaxGetHwSetup() can be called.
 *
 * @b Example:
 * @verbatim
      CSL_DmaxHandle hDmax;
      CSL_Status status;
      CSL_DmaxHwSetup *mysetup;
       ...
      status = CSL_dmaxGetHwSetup(hDmax, &mysetup);
   @endverbatim
 *
 * @return returns the status of the operation (see @a CSL_Status)
 *===================================================================================
 */
extern CSL_Status  CSL_dmaxGetHwSetup(
    CSL_DmaxHandle                         hDmax,
    CSL_DmaxHwSetup                        *setup
);


/*
 * ==================================================================================
 * Control operations for the DMAX.  For a particular control operation, the
 *  pointer to the corresponding data type needs to be passed as argument HwControl
 *  function Call. All the arguments (Structure elements included) passed to  the
 *  HwControl function are inputs. For the list of commands supported and argument
 *  type that can be @a void* casted & passed with a particular command refer to
 *  @a CSL_DmaxHwControlCmd.
 *
 *  <b> Usage Constraints: </b>
 *  Both @a CSL_dmaxInit() and @a CSL_dmaxOpen() must be called successfully
 *  in that order before @a CSL_dmaxHwControl() can be called. For the
 *  argument type that can be @a void* casted & passed with a particular command
 *  refer to @a CSL_DmaxHwControlCmd
 *
 * @b Example:
 * @verbatim
       CSL_DmaxHandle hDmax;
       CSL_Status status;
       ...
       status = CSL_dmaxHwControl(hDmax,
                                    CSL_DMAX_CMD_START,
                                   &command);
   @endverbatim
 *
 *  @return returns the status of the operation (see @a CSL_Status)
 *==================================================================================
 */
extern CSL_Status  CSL_dmaxHwControl(
    CSL_DmaxHandle                        hDmax,
    CSL_DmaxHwControlCmd                  cmd,
    void                                 *arg
);

/** =============================================================================
 *   @func CSL_dmaxHwSetupRaw
 *
 *   @desc
 *       This function configures the registers of DMAX as per the values given
 *       in the Config structure.
 *
 *   @arg hDmax
 *        Handle to the DMAX instance
 *
 *   @arg config
 *        Pointer to DMAX config structure
 *
 *   @ret CSL_Status
 *         CSL_SOK             - Configuration successful
 *         CSL_ESYS_BADHANDLE  - Invalid handle
 *
 *   @eg
 *
 *        CSL_Status            status;
 *        CSL_DmaxConfig        hwConfig;
 *
 *        status = CSL_dmaxHwsetupRaw (hDmax, &hwConfig);
 *
 * ===========================================================================
 */
extern CSL_Status  CSL_dmaxHwSetupRaw (
    CSL_DmaxHandle           hDmax,
    CSL_DmaxConfig *         config
);

/*
 * ===========================================================================
 * This function is used to read the current device configuration, status
 *  flags and the value present associated registers. Following table details
 *  the various status queries supported and the associated data structure to
 *  record the response. User should allocate memory for the said data type and
 *  pass its pointer as an unadorned void* argument to the status query call.
 *  For details about the various status queries supported and the associated
 *  data structure to record the response, refer to @a CSL_DmaxHwStatusQuery
 *
 *  <b> Usage Constraints: </b>
 *  Both @a CSL_dmaxInit() and @a CSL_dmaxOpen() must be called successfully
 *  in that order before @a CSL_dmaxGetHwStatus() can be called. For the
 *  argument type that can be @a void* casted & passed with a particular cmd
 *  refer to @a CSL_DmaxHwStatusQuery
 *
 * @b Example:
 * @verbatim
      CSL_DmaxHandle hDmax;
      CSL_Status status;
      Uint16  *response;
       ...
      status = CSL_dmaxGetHwStatus(hDmax,
                           CSL_DMAX_QUERY_BUS_BUSY,
                                  &response);
   @endverbatim
 *
 * @return Returns the status of the operation (see @a CSL_Status)
 *===========================================================================
 */
extern CSL_Status  CSL_dmaxGetHwStatus(
    CSL_DmaxHandle                         hDmax,
    CSL_DmaxHwStatusQuery                  query,
    void                                  *response
);


/** ============================================================================
 *   @n@b CSL_dmaxGetBaseAddress
 *
 *   @b Description
 *   @n  Function to get the base address of the peripheral instance.
 *       This function is used for getting the base address of the peripheral
 *       instance. This function will be called inside the CSL_dmaxOpen()
 *       function call. This function is open for re-implementing if the user
 *       wants to modify the base address of the peripheral object to point to
 *       a different location and there by allow CSL initiated write/reads into
 *       peripheral MMR's go to an alternate location.
 *
 *   @b Arguments
 *   @verbatim
            dmaxNum          Specifies the instance of DMAX to be opened.

            pDmaxParam       Module specific parameters.

            pBaseAddress    Pointer to baseaddress structure containing base
                            address details.

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_OK            Open call is successful
 *   @li                    CSL_ESYS_FAIL     The instance number is invalid.
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Base Address structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *
 *         2. Base address structure is modified.
 *
 *   @b Example
 *   @verbatim
        CSL_Status              status;
        CSL_DmaxBaseAddress  baseAddress;

       ...
      status = CSL_dmaxGetBaseAddress(CSL_DMAX_PER_CNT, NULL, &baseAddress);

    @endverbatim
 * ===========================================================================
 */
extern CSL_Status   CSL_dmaxGetBaseAddress (
    CSL_InstNum,
    CSL_DmaxParam*,
    CSL_DmaxBaseAddress*
);

/** =======================================================================
 *   @n@b CSL_dmaxSetupFifoDesc
 *
 *   @b Description
 *   @n This function formats a FIFO Descriptor object.
 *   @a CSL_dmaxSetupFifoDesc.
 *
 *   @b Arguments
 *   @verbatim
 *    
 *          fifoObj     Pointer to data area in RAM where descriptor is to be stored.
 *          setup       Pointer to a dMax FIFO descriptor Object.
 *           
 *   @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Successful completion of hardware     
 *                                    setup
 *   @li                    CSL_ESYS_ INVPARAMS - Hardware structure is not properly 
 *                                    initialized
 *   @li                    CSL_ESYS_ BADHANDLE - Invalid CSL handle
 *
 *
 *   <b> Pre Condition </b>
 *   @n  Both CSL_dmaxInit() and a CSL_dmaxOpen() must be called
 *
 *   <b> Post Condition </b>
 *   @n  DMAX registers are configured 
 *
 *   @b  Modifies
 *   @n  DMAX registers
 *
 *   @b Example
 *   @verbatim  
      CSL_DmaxFifoDescriptor        fifoObj;
      CSL_DmaxFifoDescriptorSetup   setup;
            
    setup.fifoBaseAddr      = 0x20000100;
    .
    .
    setup.wPtr              = 0x10000000;
    
      CSL_dmaxSetupFifoDesc(&fifoObj, &setup);
     @endverbatim
 * 
 * ===========================================================================
 */
extern CSL_Status CSL_dmaxSetupFifoDesc(
    CSL_DmaxFifoDescriptor *, 
    CSL_DmaxFifoDescriptorSetup *
);

                                        
/** ===========================================================================
 *   @n@b CSL_dmaxStartAsyncTransferMulti
 *
 *   @b Description
 *   @n This function initiates dMax data transfer by toggling the appropriate
 *      bit in the ESR(Event Set Register) register for multiple events.
 *   @a CSL_dmaxStartAsyncTransferMulti.
 *
 *   @b Arguments
 *   @verbatim
 *    
 *          hDmax           Pointer to the dMax Handle Instance
 *           
 *   @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Successful completion
 *   @li                    CSL_EDMAX_BAD_ESR_FLAG - Bad ESR Flag
 *   @li                    CSL_ESYS_ BADHANDLE - Invalid CSL handle
 *
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  DMAX registers are configured 
 *
 *   @b  Modifies
 *   @n  DMAX registers
 *
 *   @b Example
 *   @verbatim  
 *     CSL_DmaxHandle           *hDmax;
 *           
 *     CSL_dmaxStartAsyncTransferMulti(hDmax);
 *    @endverbatim
 * 
 * ============================================================================
 */
extern CSL_Status CSL_dmaxStartAsyncTransferMulti(
    CSL_DmaxHandle *
);

/** =======================================================================
 *   @n@b CSL_dmaxGetHwSetupFifoXFRParamEntry
 *
 *   @b Description
 *   @n This function initializes the device registers with the appropriate
 *      values provided through the HwSetup Data structure. After the 
 *      Setup is completed, the device is ready for  operation.      
 *      For information passed through the HwSetup Data structure refer
 *   @a CSL_dmaxHwSetup.
 *
 *   @b Arguments
 *   @verbatim
 *    
 *          paramEntry  Pointer to Prameter Table Entry where data 
 *                          is to be stored
 *
 *          setup       Pointer to a dmax FIFO Transfer Setup Object
 *           
 *   @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Successful completion of hardware     
 *                                    setup
 *   @li                    CSL_ESYS_ INVPARAMS - Hardware structure is not 
 *                                    properly initialized
 *   @li                    CSL_ESYS_ BADHANDLE - Invalid CSL handle
 *
 *
 *   <b> Pre Condition </b>
 *   @n  Both CSL_dmaxInit() and a CSL_dmaxOpen() must be called
 *
 *   <b> Post Condition </b>
 *   @n  DMAX registers are configured 
 *
 *   @b  Modifies
 *   @n  DMAX registers
 *
 *   @b Example
 *   @verbatim  
      CSL_DmaxParameterEntry    paramEntry;
      CSL_dmaxHwSetup           hwSetup;
      
    hwSetup.linearReload0 = paramEntry[0];
    .
    .
    hwSetup.delayTabPtr1  = paramEntry[9];
    
      CSL_dmaxGetHwSetupFifoXFRParamEntry(hDmax, &hwSetup);
     @endverbatim
 * 
 * ============================================================================
 */
extern CSL_Status CSL_dmaxGetHwSetupFifoXFRParamEntry(
    CSL_DmaxParameterEntry *, 
    CSL_DmaxFifoParameterSetup *
);

/** =======================================================================
 *   @n@b CSL_dmaxGetHwSetupGenXFRParamEntry
 *
 *   @b Description
 *   @n This function formats the parameter table entry for a General Purpose
 *      transfer.
 *   @a CSL_dmaxHwSetup.
 *
 *   @b Arguments
 *   @verbatim
 *    
 *          paramEntry  Pointer to Prameter Table Entry where data 
 *                      is to be stored.
 *          cc          Counter configuration.
 *          setup       Pointer to a dmax General Purpose Transfer Object.
 *           
 *   @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Successful completion of hardware     
 *                                    setup
 *   @li                    CSL_ESYS_ INVPARAMS - Hardware structure is  
 *                                    not properly initialized
 *   @li                    CSL_ESYS_ BADHANDLE - Invalid CSL handle
 *
 *
 *   <b> Pre Condition </b>
 *   @n  Both CSL_dmaxInit() and a CSL_dmaxOpen() must be called
 *
 *   <b> Post Condition </b>
 *   @n  DMAX registers are configured 
 *
 *   @b  Modifies
 *   @n  DMAX registers
 *
 *   @b Example
 *   @verbatim  
 *    CSL_DmaxParameterEntry        paramEntry;
 *    Uint8                         cc;
 *    CSL_DmaxGPXFRParameterSetup   hwSetup;
 *     
 *  hwSetup.srcReloadAddr0  = paramEntry[0];
 *  .
 *  .
 *  hwSetup.dstReloadAddr0  = paramEntry[1];
 *  
 *    CSL_dmaxGetHwSetupGenXFRParamEntry(paramEntry, cc, &hwSetup);
 *   @endverbatim
 * 
 * ============================================================================
 */
extern CSL_Status CSL_dmaxGetHwSetupGenXFRParamEntry(
    CSL_DmaxParameterEntry *,
    Uint8 , 
    CSL_DmaxGPXFRParameterSetup *
);


/******************************************************************************
* FUNCTION NAME: _CSL_dmaxSetDetr                                            *
*                                                                            *
*   Regs Modified     : A0,B4,B5,SP                                          *
*   Regs Used         : A0,A4,B3,B4,B5,SP                                    *
*   Local Frame Size  : 0 Args + 4 Auto + 0 Save = 4 byte                    *
******************************************************************************
*/

extern Uint32 CSL_dmaxSetDetr(
    Uint32
);

/******************************************************************************
* FUNCTION NAME: _CSL_dmaxGetDesr                                            *
*                                                                            *
*   Regs Modified     : A4,B4                                                *
*   Regs Used         : A4,B3,B4                                             *
*   Local Frame Size  : 0 Args + 0 Auto + 0 Save = 0 byte                    *
******************************************************************************
*/
extern Uint32 CSL_dmaxGetDesr(
    void
);

/** =======================================================================
 *   @n@b CSL_dmaxSetupGeneralXFRParameterEntry
 *
 *   @b Description
 *   @n This function formats the parameter table entry for a General Purpose
 *      transfer.
 *   @a CSL_dmaxHwSetup.
 *
 *   @b Arguments
 *   @verbatim
 *    
 *          paramEntry  Pointer to Prameter Table Entry where data is 
 *                      to be stored.
 *          cc          Counter configuration.
 *          setup       Pointer to a dmax General Purpose Transfer Object.
 *           
 *   @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Successful completion of hardware     
 *                                    setup
 *   @li                    CSL_ESYS_ INVPARAMS - Hardware structure is not 
 *                                    properly initialized
 *   @li                    CSL_ESYS_ BADHANDLE - Invalid CSL handle
 *
 *
 *   <b> Pre Condition </b>
 *   @n  Both CSL_dmaxInit() and a CSL_dmaxOpen() must be called
 *
 *   <b> Post Condition </b>
 *   @n  DMAX registers are configured 
 *
 *   @b  Modifies
 *   @n  DMAX registers
 *
 *   @b Example
 *   @verbatim  
      CSL_DmaxParameterEntry        paramEntry;
      Uint8             cc;
      CSL_DmaxGPXFRParameterSetup   hwSetup;
      
      hwSetup.srcReloadAddr0  = 0x10000100;
      hwSetup.dstReloadAddr0  = 0x20000100;
      .
      .
      hwSetup.count0        = 3;
      hwSetup.count1        = 4;
      hwSetup.count2        = 2;
    
      CSL_dmaxHwSetupGenXFRParametryEntry (paramEntry, cc, &hwSetup);
     @endverbatim
 * 
 * ============================================================================
 */
extern CSL_Status CSL_dmaxSetupGeneralXFRParameterEntry(
    CSL_DmaxParameterEntry *,
    Uint8, 
    CSL_DmaxGPXFRParameterSetup *
);

/** =======================================================================
 *   @n@b CSL_dmaxHwSetupFifoParam
 *
 *   @b Description
 *   @n This function initializes the device registers with the appropriate
 *      values provided through the HwSetup Data structure. After the 
 *      Setup is completed, the device is ready for  operation.      
 *      For information passed through the HwSetup Data structure refer
 *   @a CSL_dmaxHwSetup.
 *
 *   @b Arguments
 *   @verbatim
 *    
 *          paramEntry  Pointer to Prameter Table Entry where data is to be 
 *                      stored
 *          setup       Pointer to a dmax FIFO Transfer Setup Object
 *           
 *   @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Successful completion of hardware     
 *                                    setup
 *   @li                    CSL_ESYS_ INVPARAMS - Hardware structure is not 
 *                                    properly initialized
 *   @li                    CSL_ESYS_ BADHANDLE - Invalid CSL handle
 *
 *
 *   <b> Pre Condition </b>
 *   @n  Both CSL_dmaxInit() and a CSL_dmaxOpen() must be called
 *
 *   <b> Post Condition </b>
 *   @n  DMAX registers are configured 
 *
 *   @b  Modifies
 *   @n  DMAX registers
 *
 *   @b Example
 *   @verbatim  
      CSL_DmaxParameterEntry    paramEntry;
      CSL_dmaxHwSetup           hwSetup;
      
      hwSetup.linearReload0  = 0x10000100;
      .
      .
      hwSetup.count0         = 3;
    
      CSL_dmaxHwSetupFifoXFRParameterEntry(hDmax, &hwSetup);
     @endverbatim
 * 
 * ============================================================================
 */
extern CSL_Status CSL_dmaxSetupFifoXFRParameterEntry(
    CSL_DmaxParameterEntry *, 
    CSL_DmaxFifoParameterSetup *
);
                                

/** ===========================================================================
* @func CSL_dmaxGetNextFreeParamEntry
 *
 *   @b Description
 *   @n Searchs for next free ParamEntry for given resource type
 *
 * 
 *       
 *   @b Arguments
 *   @verbatim
 *    
 *          uid         Pointer to Parameter Table Entry where data is stored
 *          st          Status
 *           
 *   @endverbatim
 *
 *   <b> Return Value </b> CSL_Status
 *   @li                   CSL_SOK - Found next free Parameter Entry.     
 *
 *   @li                   CSL_ESYS_OVFL - Next free Parameter Entry not found
 *
 *
 *   <b> Pre Condition </b>
 *   @n  Both CSL_dmaxInit() and a CSL_dmaxOpen() must be called
 *
 *   <b> Post Condition </b>
 *   @n  None 
 *
 *   @b  Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim  
    
      CSL_dmaxGetNextFreeParameterEntry(hDmax->paramUid, &status);
     @endverbatim
 * 
 * ============================================================================
 */
extern Uint32 CSL_dmaxGetNextFreeParamEntry(
    Uint32 , 
    CSL_Status *
);

#ifdef __cplusplus
}
#endif


#endif /* _CSL_DMAX_H_ */
