/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** @file  csl_emif.h
 *
 *  @path $(CSLPATH)\emif\src
 *
 *  @brief EMIF functional layer API header file
 *
 */

/** @mainpage EMIF CSL 3.x
 *
 * @section Introduction
 *
 * @subsection xxx Purpose and Scope
 * The purpose of this document is to identify a set of common CSL APIs for
 * the EMIF module across various devices. The CSL developer is expected to
 * refer to this document while designing APIs for these modules. Some of the
 * listed APIs may not be applicable to a given EMIF module. While other cases
 * this list of APIs may not be sufficient to cover all the features of a
 * particular EMIF Module. The CSL developer should use his discretion designing
 * new APIs or extending the existing ones to cover these.
 *
 * @subsection aaa Terms and Abbreviations
 *   -# CSL:  Chip Support Library
 *   -# API:  Application Programmer Interface
 *
 * @subsection References
 *    -# CSL-001-DES, CSL 3.x Design Specification DocumentVersion 1.02
 *
 */

/* =============================================================================
 *  Revision History
 *  ===============
 *  25-Apr-2005 GKS    Changes done for reflecting new spec(SPRU711.pdf)
 *  30-Nov-2004 Author File Created.
 * =============================================================================
 */

#ifndef _CSL_EMIF_H_
#define _CSL_EMIF_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <cslr.h>
#include <soc.h>
#include <csl_error.h>
#include <csl_types.h>
#include <csl.h>
#include <cslr_emif.h>


#define EMIF_16BIT_SUPPORT      1
#define EMIF_32BIT_SUPPORT      0

#ifdef DA710
#   define EMIF_WIDTH      EMIF_32BIT_SUPPORT
#else
#   ifdef C6727
#       define EMIF_WIDTH      EMIF_32BIT_SUPPORT
#   else
#       define EMIF_WIDTH      EMIF_16BIT_SUPPORT
#   endif
#endif

/**************************************************************************\
* EMIF global typedef declarations
\**************************************************************************/

/** \brief  This enum used to set SR bit field of SDRAM Config Register.
 *  Can set SDRAM to enter or exit self refresh mode
 *  
 */
typedef enum {
	/** SDRAM exit Self refresh mode*/
	CSL_EMIF_SELF_REFRESH_DISABLE       = CSL_EMIF_SDCR_SR_DISABLE,
	/** SDRAM enter Self refresh mode*/
	CSL_EMIF_SELF_REFRESH_ENABLE        = CSL_EMIF_SDCR_SR_ENABLE
} CSL_EmifSdramSelfRefresh;

/** \brief  This enum used to set the CAS latency to be used when accessing
 * connected SDRAM devices.
 *
 */
typedef enum {
	/** CAS latency = 2 EM_CLK cycles */
	CSL_EMIF_CAS_LATENCY_SET_TO_2       = 2,
	/** CAS latency = 3 EM_CLK cycles */
	CSL_EMIF_CAS_LATENCY_SET_TO_3       = 3
} CSL_EmifSdramCasLatency;

/** \brief  This enum ued in setting the bit field ibank of SDRAM Config  
 *  Register to set the number of SDRAM internal banks.
 *
 */
typedef enum {
	/** 1 Bank SDRAM device */
	CSL_EMIF_1_SDRAM_BANKS              = 0,
	/** 2 Banks SDRAM device */
	CSL_EMIF_2_SDRAM_BANKS              = 1,
	/** 4 Banks SDRAM device */
	CSL_EMIF_4_SDRAM_BANKS              = 2
} CSL_EmifSdramInternalBankSetUp;

/** \brief  This enum used define the command required for EMIF registers
 * setup.
 *
 */
typedef enum {
	/** Enables\Disable self refresh */
	CSL_EMIF_CMD_SELF_REFRESH		
	/** Enables\Disable self refresh */
	,CSL_EMIF_CMD_SELF_REFRESH_RATE
	/** Clears Async Timeout */
	,CSL_EMIF_CMD_CLEAR_ASYNC_TIMEOUT
	/** Clears Masked Async Timeout */
	,CSL_EMIF_CMD_CLEAR_MASKED_ASYNC_TIMEOUT
	/** Set Async Timeout Interrupt*/
	,CSL_EMIF_CMD_SET_ASYNC_TIMEOUT_INTR
	/** Clear Async Timeout Interrupt*/
	,CSL_EMIF_CMD_CLR_ASYNC_TIMEOUT_INTR
	/** Setup nand configutration*/
	,CSL_EMIF_CMD_SETUPNAND
} CSL_EmifHwControlCmd;

/** \brief  This enum used define the query required for EMIF registers
 * status.
 *
 */
typedef enum {
	/** Get status if SDRAM is in Self refresh */
	CSL_EMIF_QUERY_SELF_REFRESH
	/** Get the Refresh rate of SDRAM */
	,CSL_EMIF_QUERY_REFRESH_RATE
	/** Get timeout status */
	,CSL_EMIF_QUERY_ASYNC_TIMEOUT_STATUS
	/** Get masked timeout status */
	,CSL_EMIF_QUERY_MASKED_ASYNC_TIMEOUT_STATUS
	/** Get masked timeout Intr status */
	,CSL_EMIF_QUERY_MASKED_ASYNC_TIMEOUT_INTR_STATUS
	/** Get the Status of nand setup*/
	,CSL_EMIF_QUERY_NANDSETUP_STATUS
	/** Get the status of Nand status register */
	,CSL_EMIF_QUERY_NANDSTATUSREG
	/** Get the status of Nand ECC register */
	,CSL_EMIF_QUERY_NANDECCREG
} CSL_EmifHwStatusQuery;

/** \brief  This enum used in setting the internal page size of SDRAM  
 *
 */
typedef enum {
	/** 8 column address bits (256 elements per row) */
	CSL_EMIF_256WORD_8COL_ADDR          = 0,
	/** 9 column address bits (512 elements per row) */
	CSL_EMIF_512WORD_9COL_ADDR          = 1,
	/** 10 column address bits (1024 elements per row) */
	CSL_EMIF_1024WORD_10COL_ADDR        = 2
} CSL_EmifSdramPageSize;

/** \brief  This enum used to define data bus width usage 
 * 
 *  It can either be in 32 Bit or 16 Bit
 */
typedef enum {
	/** 32-bit SDRAM data bus is used */
	CSL_EMIF_32BIT_USAGE                = 0,
	/** 16-bit SDRAM data bus is used */
	CSL_EMIF_16BIT_USAGE                = 1
} CSL_EmifNarrowMode;


/** \brief  This enum used to define the polarity of the AWAIT pin.	 
 * 
 *  
 */
typedef enum {
	/** Insert wait cycles if AWAIT is low */
	CSL_EMIF_WAIT_POLARITY_LOW          = 0,
	/** Insert wait cycles if AWAIT is high */
	CSL_EMIF_WAIT_POLARITY_HIGH         = 1
} CSL_EmifWaitPolarity;

/** \brief  This enum used to Enable or disable NAND Setup. 	 
 * 
 *  
 */
typedef enum {
	/**  Disable NAND */
	CSL_EMIF_NAND_DISABLE          = 0,
	/** Enable NAND */
	CSL_EMIF_NAND_ENABLE         = 1
} CSL_EmifNANDStatus;

/** \brief  This enum used to Enable or disable NAND ECC Setup. 	 
 * 
 *  
 */
typedef enum {
	/** Do not start NAND ECC Calculation */
    CSL_EMIF_NAND_NO_ECC          = 0,
	/** Start NAND ECC Calculation */
	CSL_EMIF_NAND_ECC_START         = 1
} CSL_EmifNANDECCStatus;

/** \brief Module specific context information. Present implementation of 
 *  emif doesn't have any context information.
 */
typedef struct {
	/** Context information of emif CSL.
	*  The below declaration is just a place-holder for future implementation.
	*/
	Uint16  contextInfo;
} CSL_EmifContext;

/** \brief Module specific parameters. Present implementation of Emif CSL
 *  doesn't have any module specific parameters.
 */
typedef struct {
	/** Bit mask to be used for module specific parameters. The below
	*  declaration is just a place-holder for future implementation.
	*/
	CSL_BitMask16   flags;
} CSL_EmifParam;

/** \brief This structure contains the base-address information for the 
 *   peripheral instance of the EMIF
 */
typedef struct {
	/** Base-address of the configuration registers of the peripheral
	*/
	CSL_EmifRegsOvly  regs;    
} CSL_EmifBaseAddress;

/** \brief EMIF object structure
 */
typedef struct CSL_EmifObj{
	/** Pointer to the register overlay structure of the emif */
	CSL_EmifRegsOvly  regs;
	/** Instance of emif being referred by this object  */
	CSL_InstNum       emifNum;
} CSL_EmifObj;

/** @brief Used for enabling or disabling Interrupt operation for EMIF.
 * 
 */
typedef struct CSL_EmifIntrConfig{
	/** Enables or Disables AT Interrupt on EMIF */
	Bool		    atIntrEn;
}CSL_EmifIntrConfig;

/**
  * @brief Defines the SDRAM timing register bit fields for EM_CLK 
  * configurations. 
  */
typedef struct CSL_EmifSdramTimingConfig {
	/** 
	*	Min number of EM_CLK cycles from Refresh to Refersh minus 1 
	*/
	Uint8		    trfc;
	/** 
	*	Min number of EM_CLK cycles from Precharge to Activate or 
	* Refresh minus 1. 
	*/
	Uint8		    trp;
	/** Min number of EM_CLK cycles from Activate to Read / Write minus 1 */      
	Uint8		    trcd;
	/**	Min number of EM_CLK cycles from last write transfer to precharge */    
	Uint8		    twr;
	/** Min number of EM_CLK cycles from Activate to precharge minus 1 */      
	Uint8		    tras;
	/**	Min number of EM_CLK cycles from Activate to Activate minus 1 */    
	Uint8		    trc;
	/**
	* Min number of EM_CLK cycles from Activate to Activate for a 
	* different bank minus 1 
	*/    
	Uint8		    trrd;
} CSL_EmifSdramTimingConfig;

/**
  * @brief This structure contains the information for the peripheral
  * interface of NAND device. 
  */
typedef struct CSL_EmifNandSetup {
	/** 
	*	Controls the ECC calculation
	*/
	Uint8		    ecc_support;
	/** 
	*	Controls the Nand support is required or not. 
	*/
	Uint8		    nand_support;	
} CSL_EmifNandSetup;

/** @brief Defines ths EMIF Async controller configuration.
 * 
 */
typedef struct CSL_EmifAsyncConfig{
	/** Select Strobe mode */
	Bool		    ss;
	/** Extend Wait mode */
	Bool		    ew;
	/** Write Strobe Setup cycles */
	Uint8		    w_setup;
	/** Write Strobe Duration cycles */
	Uint8		    w_strobe;
	/** Write Strobe Hold cycles */
	Uint8		    w_hold;
	/** Read Strobe Setup cycles */
	Uint8		    r_setup;
	/** Read Strobe Duration cycles */
	Uint8		    r_strobe;
	/** Read Strobe Hold cycles */
	Uint8		    r_hold;
	/** Turn Around cycles */
	Uint8		    ta;
	/** Asynchronous Bank Size */
	Uint8		    asize;
}CSL_EmifAsyncConfig;

/**
  * @brief Top level h/w setup structure containing pointers to individual 
  * structures used for setup and the type of SDRAM used. 
  */
typedef struct CSL_EmifHwSetup {
	/** Self refresh enable/disable 
	*/
	Bool   sdramSelfRefreshEn;
	/** SDRAM memory page size  
	*/
	CSL_EmifSdramPageSize    sdramPageSize;
	/** SDRAM Internal bank config 
	*/
	CSL_EmifSdramInternalBankSetUp    sdramIntBankSetup ;
	/** SDRAM CAS Latency  
	*/
	CSL_EmifSdramCasLatency     sdramCasLaten ;
	/** Wait polarity for AWAIT  
	*/
	CSL_EmifWaitPolarity     waitPolarityCfg ;
	/** Maximum extended wait cycles 
	*/
	Uint8  maxExtWait   ;
	/** Data Bus width
	*/
	CSL_EmifNarrowMode narrowMode  ;
	/** Defines refresh rate for the SDRAM connected device 
	*/
	Uint16   refreshRate ;
	/** To enable or disable at interrupt on EMIF 
	*/
	CSL_EmifIntrConfig    emifIntrCfg  ;
	/** Minimum EM_CLK cycles from self refresh command to any command. 
	*/
	Uint8   txs  ;
	/** To sonfigure SDRAM timing configuration Register
	  */
	CSL_EmifSdramTimingConfig	emClkCycles;
	/** To configure Asynchronous Bank 1 configuration  
	*/
	CSL_EmifAsyncConfig asyncBankCfg1 ;
	/** NAND Flash configuration
	*/
	CSL_EmifNandSetup	nandSetup ;
} CSL_EmifHwSetup;

/**
  * @brief Top level h/w setup default values
  */

#define CSL_EMIF_HWSETUP_DEFAULTS{ \
	CSL_EMIF_SELF_REFRESH_DISABLE,\
	CSL_EMIF_256WORD_8COL_ADDR,\
	CSL_EMIF_1_SDRAM_BANKS,\
	CSL_EMIF_CAS_LATENCY_SET_TO_2,\
	CSL_EMIF_WAIT_POLARITY_HIGH,\
	0x0,\
	CSL_EMIF_16BIT_USAGE,\
	0x0,\
	0x0,\
	0x0,\
	{0,0,0x2,0x01,0x0c,0x0f,0x0},\
	{0,0,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x0B},\
	{CSL_EMIF_NAND_DISABLE, CSL_EMIF_NAND_NO_ECC}\
}

/** \brief This data type is used to return the handle to the CSL of the emif
 */
typedef struct CSL_EmifObj    *CSL_EmifHandle;

/**************************************************************************\
* EMIF global function declarations
\**************************************************************************/

/** ============================================================================
 *   @n@b CSL_emifInit
 *
 *   @b Description
 *   @n This is the initialization function for the emif CSL. The function
 *      must be called before calling any other API from this CSL. This 
 *      function does not modify any registers or check status. It returns 
 *      status CSL_SOK. It has been kept for future use.
 *
 *   @b Arguments
 *   @verbatim
        pContext    Context information to EMIF
        
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                        CSL_SOK - Always returns
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_sysInit() must be called. 
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
            status = CSL_emifInit(NULL);
     @endverbatim
 * =============================================================================
 */
extern CSL_Status CSL_emifInit(
    /** EMIF specific context information
     */
    CSL_EmifContext * pContext
);

/** ============================================================================
 *   @n@b CSL_emifOpen
 *
 *   @b Description
 *   @n This function opens the EMIF CSL. It returns a handle to the CSL for the 
 *   EMIF instance. This handle is passed to all other CSL APIs, as the reference 
 *   to the EMIF instance .
 *
 *   @b Arguments
 *   @verbatim
        pEmifObj	Pointer to EMIF object.

        emifNum	    Instance of EMIF to be opened.

        pEmifParam	Module specific parameters.

        pStatus		The status of the function call.
        
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_EmifHandle
 *   @li                         Valid EMIF handle will be returned if
 *                              status value is equal to CSL_SOK.
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  1.	The status is returned in the status variable. If status 
 *           returned is
 *	@li         CSL_OK		       - Open call is successful
 *  @li         CSL_ESYS_FAIL	   - Module instance is invalid.
 *  @li			CSL_ESYS_INVPARAMS - Invalid Parametere is passed.
 *       2.	EMIF object structure is populated
 *
 *   @b Modifies
 *   @n  1.	The status variable
 *
 *       2.	EMIF object structure 
 *
 *
 *   @b Example
 *   @verbatim
     CSL_EmifObj	    emifObj;
     CSL_EmifHandle     hEmif;
     CSL_EmifHwSetup    hwSetup;
     CSL_Status         status;

     ..
     
     hEmif = CSL_emifOpen (&emifObj, CSL_EMIF, NULL, &status);
        
     ..
     @endverbatim
 * =============================================================================
 */
extern CSL_EmifHandle CSL_emifOpen (
    /** Pointer to the object that holds reference to the
    *  instance of EMIF requested after the call
    */
    CSL_EmifObj		    *pEmifObj,
    /** Instance of EMIF to which a handle is requested
    */
    CSL_InstNum          emifNum,
    /** Module specific parameters;
    * Currently there are none; the user should pass 'NULL'
    */
    CSL_EmifParam        *pEmifParam,
    /** This returns the status (success/errors) of the call.
    * Could be 'NULL' if the user does not want status information.
    */
    CSL_Status           *pStatus
);

/** ============================================================================
 *   @n@b CSL_emifClose
 *
 *   @b Description
 *   @n This function marks that CSL for the EMIF instance is closed. CSL for 
 *		the EMIF instance need to be reopened before using any EMIF CSL API 
 *		again.
 *
 *   @b Arguments
 *   @verbatim
        hEmif 			Handle to EMIF
        
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - EMIF is closed Successfully
 *   @li                    CSL_ESYS_BADHANDLE  - The handle passed is invalid
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @li  CSL for the EMIF instance is closed
 *
 *   @b Modifies
 *   @n  System data object
 * 
 *
 *   @b Example
 *   @verbatim
     CSL_EmifHandle     hEmif;
     CSL_Status         status;     
     ..
     
     status = CSL_emifClose (hEmif);
        
     ..
     @endverbatim
 * =============================================================================
 */
extern CSL_Status  CSL_emifClose(
    /** Pointer to the object that holds reference to the
    instance of EMIF requested after the call
     */
    CSL_EmifHandle	      hEmif
);

/** ============================================================================
 *   @n@b CSL_emifHwSetup
 *
 *   @b Description
 *   @n It configures the EMIF registers as per the values passed in the 
 *		hardware setup structure
 *
 *   @b Arguments
 *   @verbatim
            hEmif            Handle to the EMIF
 
            hwSetup          Pointer to harware setup structure
 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Hardware setup successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid CSL handle
 *   @li                    CSL_ESYS_INVPARAMS  - Hardware structure is not
                                                  properly initialized
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  EMIF controller registers are configured according 
 *       to the hardware setup parameters
 *
 *   @b Modifies
 *   @n EMIF controller registers
 *
 *   @b Example
 *   @verbatim
            CSL_EmifHandle	hEmif;
		    CSL_EmifObj		emifObj;
		    CSL_EmifHwSetup	hwSetup;
     		CSL_status		status;
			
			hEmif = CSL_emifOpen (&emifObj, CSL_EMIF, NULL, &status);
      		. .
			status = CSL_emifHwSetup(hEmif, &hwSetup);
			. .
			
     @endverbatim
 * ===========================================================================
 */
CSL_Status  CSL_emifHwSetup(
    /** Pointer to the object that holds reference to the
    *  instance of EMIF requested after the call
     */
    CSL_EmifHandle	      hEmif,
    /** Pointer to setup structure which contains the
    *  information to program EMIF to a useful state
     */
    CSL_EmifHwSetup	      *hwSetup
);

/** ============================================================================
 *   @n@b CSL_emifGetHwSetup
 *
 *   @b Description
 *   @n It retrives the hardware setup parameters of the emif
 *      specified by the given handle.
 *
 *   @b Arguments
 *   @verbatim
            hEmif        Handle to the emif
 
            setup         Pointer to the harware setup structure
 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Retrieving the hardware setup
 *                                                parameters is successful
 *   @li                    CSL_ESYS_BADHANDLE  - The handle is passed is
 *                                                invalid
 *   @li                    CSL_ESYS_INVPARAMS  - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The hardware setup structure is populated with the hardware setup
 *       parameters
 *
 *   @b Modifies
 *   @n hwSetup variable
 *
 *   @b Example
 *   @verbatim
            CSL_EmifHandle   hEmif;
            CSL_EmifHwSetup  hwSetup;

            ...
            
            status = CSL_emifGetHwSetup(hEmif, &hwSetup);
            
            ...
            
     @endverbatim
 * ===========================================================================
 */
extern CSL_Status  CSL_emifGetHwSetup(
    /** pointer to the object that holds reference to the
    * instance of EMIF requested after the call */
    CSL_EmifHandle                         hEmif,
    /** the query to this API which indicates the status/setup
    * to be returned */
    CSL_EmifHwSetup                        *setup
);	

/** ============================================================================
 *   @n@b CSL_emifHwControl
 *
 *   @b Description
 *   @n This function performs various control operations on the EMIF, based 
 *		on the command passed
 *
 *   @b Arguments
 *   @verbatim
            hEmif	The handle to EMIF module

			cmd		Operation to be performed on EMIF

			arg 	Argument specific to the command

 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li   	CSL_SOK 		    - Successful completion of the query
 *   @li    CSL_ESYS_INVCMD 	- Command not supported
 *   @li    CSL_ESYS_BADHANDLE 	- Invalid handle
 *	 @li	CSL_ESYS_INVPARAMS  - Invalid parameter passed
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  EMIF registers are configured according to the command and command
 *		 arguments. The command determines which registers are modified.
 *
 *   @b Modifies
 *   @n EMIF registers determined by the command.
 *
 *   @b Example
 *   @verbatim
		CSL_EmifSdramSelfRefresh     srState = CSL_EMIF_SELF_REFRESH_DISABLE;
     	CSL_Status     				 status;
     	CSL_EmifHandle 				 hEmif;
            ...
                  
      status = CSL_EmifHwControl(hEmif, 
      			CSL_EMIF_CMD_SELF_REFRESH, &srState);
            ...
            
     @endverbatim
 * ===========================================================================
 */
extern CSL_Status  CSL_emifHwControl(
    /** Pointer to the object that holds reference to the
    *  instance of EMIF requested after the call
     */
    CSL_EmifHandle	      hEmif,
    /** The command to this API, which indicates the action to be taken
     */
    CSL_EmifHwControlCmd     cmd,
    /** An optional argument @a void* casted
     */
    void		      *arg
);

/** ============================================================================
 *   @n@b CSL_emifGetHwStatus
 *
 *   @b Description
 *   @n The function is used to get data about various parameters of the EMIF 
 *		block. The data returned depends on the query passed. 
 *
 *   @b Arguments
 *   @verbatim
            hEmif 			Handle to EMIF

			query 		    Type of data query to perform

			response 		Pointer to buffer to return the data requested by 
							query passed
 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li   	CSL_SOK 		    - Successful completion of the query
 *   @li    CSL_ESYS_INVQUERY   - Query command not supported
 *   @li    CSL_ESYS_BADHANDLE 	- Invalid handle
 *	 @li	CSL_ESYS_INVPARAMS  - Invalid parameter passed
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n The input argument response.
 *
 *   @b Example
 *   @verbatim
		Uint16    	   rrRate = 0;
     	CSL_Status     status;
     	CSL_EmifHandle hEmif;
     	. .
     	status = CSL_emifGetHwStatus(hEmif, CSL_EMIF_QUERY_REFRESH_RATE, 
					&rrRate);
		. .            
     @endverbatim
 * ===========================================================================
 */
extern CSL_Status  CSL_emifGetHwStatus(
    /** Pointer to the object that holds reference to the
    *  instance of EMIF requested after the call
     */
    CSL_EmifHandle	      hEmif,
    /** The query to this API, which indicates the status
    *  to be returned
     */
    CSL_EmifHwStatusQuery    query,
    /** Placeholder to return the status; @a void* casted
     */
    void		      *response
);

/** ============================================================================
 *   @n@b CSL_emifGetBaseAddress
 *
 *   @b Description
 *   @n This function to get the base address of the EMIF instance. 
 *
 *   @b Arguments
 *   @verbatim
            emifNum			Instance number of the EMIF

			pEmifParam		Module specific parameters.

			pBaseAddress	Pointer to base address structure

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li   	CSL_SOK 		    - Successfully retrieved base address.
 *   @li    CSL_ESYS_FAIL     	- Failed to get the base address
 *	 @li 	CSL_ESYS_INVPARAMS  - If the input parameter is not valid
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Gets the base address of EMIF
 *
 *   @b Modifies
 *   @n The base address is populated in the base address structure
 *
 *   @b Example
 *   @verbatim
		CSL_status			status;
		CSL_EmifParam 		emifParam;
		CSL_EmifBaseAddress baseAddress;
		CSL_InstNum         emifNum=CSL_EMIF;	
		. .
		status = CSL_emifGetBaseAddress (emifNum, &emifParam, &baseAddress);
		. .     
     @endverbatim
 * ===========================================================================
 */
extern CSL_Status CSL_emifGetBaseAddress(
    /** Instance number
     */
    CSL_InstNum  		emifNum,
    /** Module specific parameters
     */
    CSL_EmifParam *		pEmifParam,
    /** Base address details
     */
    CSL_EmifBaseAddress *	pBaseAddress
);

#ifdef __cplusplus
}
#endif


#endif /* _CSL_EMIF_H_ */
