/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 *  ============================================================================
 */
 
/** @mainpage I2C CSL 3.x
 *
 * @section Introduction
 *
 * @subsection xxx Purpose and Scope
 * The purpose of this document is to identify a set of common CSL APIs for
 * the I2C module across various devices. The CSL developer is expected to
 * refer to this document while designing APIs for these modules. Some of the
 * listed APIs may not be applicable to a given I2C module. While other cases
 * this list of APIs may not be sufficient to cover all the features of a
 * particular I2C Module. The CSL developer should use his discretion designing
 * new APIs or extending the existing ones to cover these.
 *
 * @subsection aaa Terms and Abbreviations
 *   -# CSL:  Chip Support Library
 *   -# API:  Application Programmer Interface
 *
 * @subsection References
 *    -# CSL-001-DES, CSL 3.x Design Specification DocumentVersion 1.02
 *
 */

/** @file csl_i2c.h
 *
 * @brief    Header file for functional layer of CSL
 *
 * Description
 *    - The different enumerations, structure definitions
 *      and function declarations
 *
 * Modification 
 *    - modified on: 14/1/2005
 *    - reason: Created the sources
 *	  - Modified on: 25/10/2005
 *	  - Added command for setting I2C as master/slave
 *
 * 
 * @author RMathew.
 *
 */



#ifndef _CSL_I2C_H_
#define _CSL_I2C_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <cslr.h>
#include <csl_error.h>
#include <soc.h>
#include <csl_types.h>
#include <cslr_i2c.h>

/**************************************************************************\
* I2C global macro declarations
\**************************************************************************/


/**
 * @brief   Invalid handle
 */

#define CSL_I2C_BADHANDLE        (0)


/** Constants for passing parameters to the I2C Hardware Setup function  */
/** For setting the SLAVE Mode for I2C           */
#define CSL_I2C_MODE_SLAVE                            (0)
/** For setting the MASTER Mode for I2C          */
#define CSL_I2C_MODE_MASTER                           (1)
/** For setting the RECEIVER Mode for I2C        */
#define CSL_I2C_DIR_RECEIVE                           (0)
/** For setting the TRANSMITTER Mode for I2C     */
#define CSL_I2C_DIR_TRANSMIT                          (1)
/** For setting the 7-bit Addressing Mode for I2C*/
#define CSL_I2C_ADDRSZ_SEVEN                          (0)
/** For setting the 10-bit Addressing Mode       */
#define CSL_I2C_ADDRSZ_TEN                            (1)
/** For Disabling the Start Byte Mode for I2C(Normal Mode) */
#define CSL_I2C_STB_DISABLE                           (0)
/** For Enabling the Start Byte Mode for I2C     */
#define CSL_I2C_STB_ENABLE                            (1)
/** For enabling the tx of a ACK to the TX-ER, while in the RECEIVER mode */
#define CSL_I2C_ACK_ENABLE                            (0)
/** For enabling the tx of a NACK to the TX-ER, while in the RECEIVER mode */
#define CSL_I2C_ACK_DISABLE                           (1)
/** For enabling the free run mode of the I2C */
#define CSL_I2C_FREE_MODE_ENABLE                      (1)
/** For disabling the free run mode of the I2C */
#define CSL_I2C_FREE_MODE_DISABLE                     (0)
/** For enabling the Repeat Mode of the I2C */
#define CSL_I2C_REPEAT_MODE_ENABLE                    (1)
/** For disabling the Repeat Mode of the I2C */
#define CSL_I2C_REPEAT_MODE_DISABLE                   (0)
/** For enabling DLB mode of I2C (applicable only in case of MASTER TX-ER) */
#define CSL_I2C_DLB_ENABLE                            (1)
/** For disabling DLB mode of I2C (applicable only in case of MASTER TX-ER) */
#define CSL_I2C_DLB_DISABLE                           (0)
/** For putting the I2C in Reset */
#define CSL_I2C_IRS_ENABLE                            (0)
/** For taking the I2C out of Reset */
#define CSL_I2C_IRS_DISABLE                           (1)
/** For enabling the Free Data Format of I2C */
#define CSL_I2C_FDF_ENABLE                            (1)
/** For disabling the Free Data Format of I2C */
#define CSL_I2C_FDF_DISABLE                           (0)
/** For enabling the Backward Compatibility mode of I2C */
#define CSL_I2C_BCM_ENABLE                            (1)
/** For disabling the Backward Compatibility mode of I2C */
#define CSL_I2C_BCM_DISABLE                           (0)

/** Constants for passing parameters to the I2C Status Query function */
/** For indicating the non-completion of Reset    */
#define CSL_I2C_RESET_NOT_DONE                        (0)
/** For indicating the completion of Reset    */
#define CSL_I2C_RESET_DONE                            (1)
/** For indicating that the bus is not busy    */
#define CSL_I2C_BUS_NOT_BUSY                          (0)
/** For indicating that the bus is busy    */
#define CSL_I2C_BUS_BUSY                              (1)
/** For indicating that the Receive ready signal is low    */
#define CSL_I2C_RX_NOT_READY                          (0)
/** For indicating that the Receive ready signal is high    */
#define CSL_I2C_RX_READY                              (1)
/** For indicating that the Transmit ready signal is low    */
#define CSL_I2C_TX_NOT_READY                          (0)
/** For indicating that the Transmit ready signal is high    */
#define CSL_I2C_TX_READY                              (1)
/** For indicating that the Access ready signal is low    */
#define CSL_I2C_ACS_NOT_READY                         (0)
/** For indicating that the Access ready signal is high    */
#define CSL_I2C_ACS_READY                             (1)
/** For indicating Single Byte Data signal is set */
#define CSL_I2C_SINGLE_BYTE_DATA                      (1)
/** For indicating Receive overflow signal is set */
#define CSL_I2C_RECEIVE_OVERFLOW                      (1)
/** For indicating Transmit underflow signal is set */
#define CSL_I2C_TRANSMIT_UNDERFLOW                    (1)
/** For indicating Arbitration Lost signal is set */
#define CSL_I2C_ARBITRATION_LOST                      (1)

/** Constants for status bit clear  */
/** Clear the Arbitration Lost status bit        */
#define CSL_I2C_CLEAR_AL 0x1
/** Clear the No acknowledge status bit          */
#define CSL_I2C_CLEAR_NACK 0x2
/** Clear the Register access ready status bit   */
#define CSL_I2C_CLEAR_ARDY 0x4
/** Clear the Receive ready status bit           */
#define CSL_I2C_CLEAR_RRDY 0x8
/** Clear the Transmit ready status bit          */
#define CSL_I2C_CLEAR_XRDY 0x10
/** Clear the Stop Condition Detect  status bit  */
#define CSL_I2C_CLEAR_SCD 0x20

/**************************************************************************\
* I2C global typedef declarations
\**************************************************************************/

/** @brief This object contains the reference to the instance of I2C opened
 *  using the @a CSL_i2cOpen().
 *
 *  The pointer to this, is passed to all I2C CSL APIs.
 */
typedef struct CSL_I2cObj {
    /** Base-address of the Configuration registers of I2C. */
    CSL_I2cRegsOvly regs;
    /** This is the instance of I2C being referred to by this object  */
    CSL_InstNum perNum;
}CSL_I2cObj;

/** @brief This will have the base-address information for the peripheral
 *  instance
 */
typedef struct {
    /** Base-address of the Configuration registers of I2C. */
    CSL_I2cRegsOvly regs;
} CSL_I2cBaseAddress;

/** @brief I2C specific parameters. Present implementation doesn't have
 *  any specific parameters.
 */
typedef struct {
   /** 
    *   Bit mask to be used for module specific parameters.
    *   The below declaration is just a place-holder for future
    *   implementation.
    */
    CSL_BitMask16   flags;
} CSL_I2cParam;

/** 
 *  @brief I2C specific context information. Present implementation doesn't
 *  have any Context information.
 */

typedef struct {
   /** 
    *  Context information of I2C.
    *  The below declaration is just a place-holder for future
    *  implementation.
    */
    Uint16  contextInfo;
} CSL_I2cContext;

typedef struct  {
   /** Config structure of I2C. This is used to configure I2C
    *  using CSL_HwSetupRaw function
    */
    /* I2C Own Address Register */
    volatile Uint16 ICOAR;
    /* I2C Interrupt Mask Register */
    volatile Uint16 ICIMR;
    /* I2C Status Register */
    volatile Uint16 ICSTR;
    /* I2C Clock Divider Low Register */
    volatile Uint16 ICCLKL;
    /* I2C Clock Divider High Register */
    volatile Uint16 ICCLKH;
    /* I2C Data Count Register */
    volatile Uint16 ICCNT;
    /* I2C Slave Address Register */
    volatile Uint16 ICSAR;
    /* I2C Data Transmit Register */
    volatile Uint16 ICDXR;
    /* I2C Mode Register */
    volatile Uint16 ICMDR;
    /* I2C Interrupt Vector Register */
    volatile Uint16 ICIVR;
    /* I2C Extended Mode Register */
    volatile Uint16 ICEMDR;
    /* I2C Prescaler Register */
    volatile Uint16 ICPSC;
    /* I2C Pin Function Register */
    volatile Uint32 ICPFUNC;
    /* I2C Pin Direction Register */
    volatile Uint32 ICPDIR;
    /* I2C Pin Data Out Register */
    volatile Uint32 ICPDOUT;
    /* I2C Pin Data Set Register */
    volatile Uint32 ICPDSET;
    /* I2C Pin Data Clear Register */
    volatile Uint32 ICPDCLR;    
} CSL_I2cConfig;

/** Default Values for Config structure */
#define CSL_I2C_CONFIG_DEFAULTS {   \
        CSL_I2C_ICOAR_RESETVAL,     \
        CSL_I2C_ICIMR_RESETVAL,     \
        CSL_I2C_ICSTR_RESETVAL,     \
        CSL_I2C_ICCLKL_RESETVAL,    \
        CSL_I2C_ICCLKH_RESETVAL,    \
        CSL_I2C_ICCNT_RESETVAL,     \
        CSL_I2C_ICSAR_RESETVAL,     \
        CSL_I2C_ICDXR_RESETVAL,     \
        CSL_I2C_ICMDR_RESETVAL,     \
        CSL_I2C_ICIVR_RESETVAL,     \
        CSL_I2C_ICEMDR_RESETVAL,    \
        CSL_I2C_ICPSC_RESETVAL,     \
        CSL_I2C_ICPFUNC_RESETVAL,   \
        CSL_I2C_ICPDIR_RESETVAL,    \
        CSL_I2C_ICPDOUT_RESETVAL,   \
        CSL_I2C_ICPDSET_RESETVAL,   \
        CSL_I2C_ICPDCLR_RESETVAL    \
}

typedef struct CSL_I2cObj *CSL_I2cHandle;

/** 
 *  @brief This has all the fields required to configure the
 *  I2C clock
 */
typedef struct CSL_I2cClkSetup_{
    /** Prescalar to the input clock     */
    Uint16 prescalar;
    /** Low time period of the clock     */
    Uint16 clklowdiv;
    /** High time period of the clock    */
    Uint16 clkhighdiv;
}CSL_I2cClkSetup;


/** 
 *  @brief This has all the fields required to configure I2C at Power Up
 *  (After a Hardware Reset) or a Soft Reset
 *
 *  This structure is used to setup or obtain existing setup of
 *  I2C using @a CSL_i2cHwSetup() & @a CSL_i2cGetHwSetup() functions
 *  respectively.
 */
typedef struct CSL_I2cHwSetup_ {
    /** Master or Slave Mode : 1==> Master Mode, 0==> Slave Mode  */
    Uint16 mode;
    /** Transmitter Mode or Receiver Mode: 1==> Transmitter Mode,
     *  0 ==> Receiver Mode
     */
    Uint16 dir;
    /** Addressing Mode :0==> 7-bit Mode, 1==> 10-bit Mode       */
    Uint16 addrMode;
    /** Start Byte Mode : 1 ==> Start Byte Mode, 0 ==> Normal Mode */
    Uint16 sttbyteen;
    /** Address of the own device    */
    Uint16 ownaddr;
    /** ACK mode while receiver: 0==> ACK Mode, 1==> NACK Mode */
    Uint16 ackMode;
    /** Run mode of I2C: 0==> No Free Run, 1==> Free Run mode */
    Uint16 runMode;
    /** Repeat Mode of I2C: 0==> No repeat mode 1==> Repeat mode */
    Uint16 repeatMode;
    /** DLBack mode of I2C (master tx-er only):
     *                  0==> No loopback,
     *                  1==> Loopback Mode
     */
    Uint16 loopBackMode;
    /** Free Data Format of I2C:
     *                  0==>Free data format disable,
     *                  1==> Free data format enable
     */
    Uint16 freeDataFormat;
    /** I2C Reset Mode: 0==> Reset, 1==> Out of reset */
    Uint16 resetMode;
    /** I2C Backward Compatibility Mode :
     *                  0==> Not compatible,
     *                  1==> Compatible
     */
    Uint16 bcm;
    /** Interrupt Enable mask The mask can be for one interrupt or
     *  OR of multiple interrupts.
     */
    Uint16 inten;
    /** Prescalar, Clock Low Time and Clock High Time for Clock Setup  */
    CSL_I2cClkSetup *clksetup;
}CSL_I2cHwSetup;


/** @brief Enumeration for queries passed to @a CSL_i2cGetHwStatus()
 *
 * This is used to get the status of different operations or to get the
 * existing setup of I2C.
 */
typedef enum {
    /**
    * @brief   Get current clock setup parameters   
    * @param   (CSL_I2cClkSetup*)
    */
    CSL_I2C_QUERY_CLOCK_SETUP   = 1,
    /**
    * @brief   Get the Bus Busy status information  
    * @param   (Uint16*)
    */
    CSL_I2C_QUERY_BUS_BUSY      = 2,
    /**
    * @brief   Get the Receive Ready status information
    * @param   (Uint16*)
    */
    CSL_I2C_QUERY_RX_RDY        = 3,
    /**
    * @brief   Get the Transmit Ready status information   
    * @param   (Uint16*) 
    */
    CSL_I2C_QUERY_TX_RDY        = 4,
    /**
    * @brief   Get the Register Ready status information 
    * @param   (Uint16*)    
    */
    CSL_I2C_QUERY_ACS_RDY       = 5,
    /**
    * @brief   Get the Stop Condition Data bit information  
    * @param   (Uint16*) 
    */
    CSL_I2C_QUERY_SCD           = 6,
    /**
    * @brief   Get the Address Zero Status (General Call) detection status 
    * @param   (Uint16*)  
    */
    CSL_I2C_QUERY_AD0           = 7,
    /**
    * @brief   Get the Receive overflow status information 
    * @param   (Uint16*)  
    */
    CSL_I2C_QUERY_RSFULL        = 8,
    /**
    * @brief   Get the Transmit underflow status information
    * @param   (Uint16*)
    */
    CSL_I2C_QUERY_XSMT          = 9,
    /**
    * @brief   Get the Address as Slave bit information
    * @param   (Uint16*)
    */
    CSL_I2C_QUERY_AAS           = 10,
    /**
    * @brief   Get the Arbitration Lost status information
    * @param   (Uint16*)
    */
    CSL_I2C_QUERY_AL            = 11,
    /**
    * @brief   Get the Reset Done status bit information
    * @param   (Uint16*)
    */
    CSL_I2C_QUERY_RDONE         = 12,
    /**
    * @brief   Get no of bits of next byte to be received or transmitted
    * @param   (Uint16*)
    */
    CSL_I2C_QUERY_BITCOUNT      = 13,
    /**
    * @brief   Get the interrupt code for the interrupt that occured
    * @param   (Uint16*)
    */
    CSL_I2C_QUERY_INTCODE       = 14,
    /**
    * @brief   Get the revision level of the I2C
    * @param   (Uint16*)
    */
    CSL_I2C_QUERY_REV           = 15,
    /**
    * @brief   Get the class of the peripheral
    * @param   (Uint16*)
    */
    CSL_I2C_QUERY_CLASS         = 16,
    /**
    * @brief   Get the type of the peripheral
    * @param   (Uint16*)
    */
    CSL_I2C_QUERY_TYPE          = 17,
    /**
    * @brief   Get the slave direction
    * @param   (Uint16*)
    */
    CSL_I2C_QUERY_SDIR          = 18,
    /**
    * @brief   Get the acknowledgement status
    * @param   (Uint16*)
    */
    CSL_I2C_QUERY_NACKSNT       = 19


} CSL_I2cHwStatusQuery;


/** @brief Enumeration for queries passed to @a CSL_i2cHwControl()
 *
 * This is used to select the commands to control the operations
 * existing setup of I2C. The arguments to be passed with each
 * enumeration if any are specified next to the enumeration.
 */
typedef enum {
    /**
    * @brief   Enable the I2C
    * @param   (None)
    */
    CSL_I2C_CMD_ENABLE              = 1,
    /**
    * @brief   Configure the I2C pins as GPIO
    * @param   (None)
    */
    CSL_I2C_CMD_CONFIG_AS_GIO       = 2,
    /**
    * @brief   Reset command to the I2C
    * @param   (None)
    */
    CSL_I2C_CMD_RESET               = 3,

    /**
    * @brief   Bring the I2C out of reset
    * @param   (None)
    */
    CSL_I2C_CMD_OUTOFRESET          = 4,

    /**
    * @brief   Clear the status bits. The argument next to the command
    *          specifies the status bit to be cleared. The status bit
    *          can be :
    *            CSL_I2C_CLEAR_AL,
    *            CSL_I2C_CLEAR_NACK,
    *            CSL_I2C_CLEAR_ARDY,
    *            CSL_I2C_CLEAR_RRDY,
    *            CSL_I2C_CLEAR_XRDY,
    *            CSL_I2C_CLEAR_GC
    * @param   (None)
    */
    CSL_I2C_CMD_CLEAR_STATUS        = 5,

    /**
    * @brief   Set the address of the Slave device
    * @param   (Uint16 *)
    */
    CSL_I2C_CMD_SET_SLAVE_ADDR      = 6,

    /**
    * @brief   Set the Data Count
    * @param   (Uint16 *)
    */
    CSL_I2C_CMD_SET_DATA_COUNT       = 7,

    /**
    * @brief   Set the start condition
    * @param   (None)
    */
    CSL_I2C_CMD_START                = 8,

    /**
    * @brief   Set the stop condition
    * @param   (None)
    */
    CSL_I2C_CMD_STOP                = 9,

    /**
    * @brief   Set the transmission mode
    * @param   (None)
    */
    CSL_I2C_CMD_DIR_TRANSMIT        = 10,

    /**
    * @brief   Set the receiver mode
    * @param   (None)
    */
    CSL_I2C_CMD_DIR_RECEIVE         = 11,

    /**
    * @brief   Set the Repeat Mode
    * @param   (None)
    */
    CSL_I2C_CMD_RM_ENABLE           = 12,

    /**
    * @brief   Disable the Repeat Mode
    * @param   (None)
    */
    CSL_I2C_CMD_RM_DISABLE          = 13,

    /**
    * @brief   Set the loop back mode
    * @param   (None)
    */
    CSL_I2C_CMD_DLB_ENABLE          = 14,

    /**
    * @brief   Set the loop back mode
    * @param   (None)
    */
    CSL_I2C_CMD_DLB_DISABLE         = 15,

    /**
    * @brief   Unmask all interrupts
    * @param   (None)
    */ 
    CSL_I2C_CMD_INTR_ENABLE         = 16,

    /**
    * @brief   Mask all interrupts
    * @param   (None)
    */ 

    CSL_I2C_CMD_INTR_DISABLE        = 17,

    /**
    * @brief   Set the own address 
    * @param   (Uint16 *)
    */
    CSL_I2C_CMD_SET_OWN_ADDR        = 18,

    /**
    * @brief   Set the Clock
    * @param   (CSL_I2cClkSetup *)
    */
    CSL_I2C_CMD_SET_CLOCK           = 19,

    /**
    * @brief   Set the mode(Master/Slave)
    * @param   (Uint16 *)
    */
    CSL_I2C_CMD_SET_MODE			= 20  


} CSL_I2cHwControlCmd;



/**************************************************************************\
* I2C global function declarations
\**************************************************************************/

/** ============================================================================
 *   @n@b CSL_i2cInit
 *
 *   @b Description
 *   @n This is the initialization function for the I2C. This function is
 *      idempotent in that calling it many times is same as calling it once.
 *      This function doesn't affect the H/W.      
 *
 *   @b Arguments
     @verbatim
            pContext   Context information for the instance. Should be NULL
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Always returns
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_sysInit() must be called. 
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
      ...      
         CSL_i2cInit();
      
 *   @endverbatim
 *  ===========================================================================
 */
extern CSL_Status  CSL_i2cInit(
    CSL_I2cContext  *pContext
);


/** ===========================================================================
 *   @n@b CSL_i2cOpen
 *
 *   @b Description
 *   @n This function populates the peripheral data object for the instance
 *      and returns a handle to the instance.
 *      The open call sets up the data structures for the particular instance
 *      of I2C device. The device can be re-opened anytime after it has been
 *      normally closed if so required.
 *      The handle returned by this call is input as an essential argument for 
 *      rest of the APIs described for this module. 
 *        
 *   @b Arguments
     @verbatim

        pI2cObj     Pointer to the I2C instance object
        i2cNum      Instance of the I2C to be opened. 
        pI2cParam   Pointer to module specific parameters
        pStatus     Pointer for returning status of the function call
        
 *   @endverbatim
 *
 *   <b> Return Value </b> CSL_I2cHandle
 *
 *       - Valid I2C instance handle will be returned if status value is
 *        equal to CSL_SOK else CSL_I2C_BADHANDLE
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_i2cInit() has to be called
 *
 *   <b> Post Condition </b>
 *   @n  1. The status is returned in the status variable. If status returned is
 *   @li            CSL_OK             - Open call is successful
 *   @li            CSL_ESYS_INVPARAMS - If invalid parameter passed
 *   @li            CSL_ESYS_FAIL      - If invalid instance number
 *       2. I2C object structure is populated
 *
 *   @b Modifies
 *   @n@li      The status variable
 *     @li      I2C object structure 
 *
 *   @b Example
 *   @verbatim
 *        CSL_status        status;
 *        CSL_I2cObj        i2cObj;
 *        CSL_I2cHandle     hI2c;
 *
 *        hI2c = CSL_I2cOpen (&i2cObj,
 *                            CSL_I2C_0,
 *                            NULL,
 *                            &status
 *                            );
     @endverbatim
 * ===========================================================================
 */ 
extern CSL_I2cHandle CSL_i2cOpen (    
    CSL_I2cObj              *hI2cObj,    
    CSL_InstNum              i2cNum,    
    CSL_I2cParam            *pI2cParam,    
    CSL_Status              *status
);


/** ============================================================================
 *   @n@b CSL_i2cClose
 *
 *   @b Description
 *   @n This function closes the specified instance of I2C.
 *
 *   @b Arguments
     @verbatim
            hI2c  Handle to the I2C instance
     @endverbatim  
 *     
 *   <b> Return Value </b> CSL_Status 
 *   @li               CSL_SOK  - I2C close successful 
 *   @li               CSL_ESYS_BADHANDLE - Invalid handle
 *     
 *      
 *   <b> Pre Condition </b>
 *   @n  i2c must have opened properly
 *
 *   <b> Post Condition </b>
 *   @n  CSL for the I2C instance is closed
 *
 *   @b Modifies
 *   @n CSL_I2cObj structure values for the instance
 * 
 *   @b Example
     @verbatim
     
        CSL_i2cClose (hI2c);
       
     @endverbatim  
 * =============================================================================
 */
extern CSL_Status  CSL_i2cClose(    
    CSL_I2cHandle                         hI2c
);


/** =======================================================================
 *   @n@b CSL_i2cHwSetup
 *
 *   @b Description
 *   @n This function initializes the device registers with the appropriate
 *      values provided through the HwSetup Data structure. After the 
 *      Setup is completed, the device is ready for  operation.      
 *      For information passed through the HwSetup Data structure refer
 *   @a CSL_i2cHwSetup.
 *
 *   @b Arguments
     @verbatim
     
            hI2c   handle to the I2c instance
            setup  Pointer to setup structure that programs I2C
            
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Successful completion of hardware     
 *                                    setup
 *   @li                    CSL_ESYS_ INVPARAMS - Hardware structure is not properly 
 *                                    initialized
 *   @li                    CSL_ESYS_ BADHANDLE - Invalid CSL handle
 *
 *
 *   <b> Pre Condition </b>
 *   @n  Both CSL_i2cInit() and a CSL_i2cOpen() must be called
 *
 *   <b> Post Condition </b>
 *   @n  I2C registers are configured 
 *
 *   @b  Modifies
 *   @n  I2C registers
 *
 *   @b Example
 *   @verbatim  
        CSL_i2cHandle hI2c;
        CSL_i2cHwSetup hwSetup;
      
        hwSetup.mode           = CSL_I2C_MODE_MASTER;
        hwSetup.dir            = CSL_I2C_DIR_TRANSMIT;
        hwSetup.addrMode       = CSL_I2C_ADDRSZ_SEVEN;
        .
        .
        hwSetup.clksetup       = &clksetup;
    
        CSL_i2cHwSetup (hI2c, &hwSetup);
     @endverbatim
 * 
 * ============================================================================
 */ 
extern CSL_Status  CSL_i2cHwSetup(    
    CSL_I2cHandle                         hI2c,    
    CSL_I2cHwSetup                        *setup
);

/** ============================================================================
 *   @n@b CSL_i2cGetHwSetup
 *
 *   @b Description
 *   @n It retrives the hardware setup parameters of the I2C
 *      specified by the given handle.
 *
 *   @b Arguments
 *   @verbatim
 
            hI2c        Handle to the I2C  
            setup       Pointer to the harware setup structure

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Retrieving the hardware setup
 *                                                parameters is successful
 *   @li                    CSL_ESYS_BADHANDLE  - The handle is passed is
 *                                                invalid
 *   @li                    CSL_ESYS_INVPARAMS  - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  i2c must have opened properly
 *
 *   <b> Post Condition </b>
 *   @n  The hardware setup structure is populated with the hardware setup
 *       parameters
 *
 *   @b Modifies
 *   @n setup variable
 *
 *   @b Example
 *   @verbatim
            CSL_I2cHandle   hI2c;
            CSL_I2cHwSetup  hwSetup;

            ...

            status = CSL_i2cGetHwSetup (hI2c, &hwSetup);

            ...

     @endverbatim
 * ===========================================================================
 */
extern CSL_Status  CSL_i2cGetHwSetup(    
    CSL_I2cHandle                         hI2c,   
    CSL_I2cHwSetup                        *setup
);


/** ============================================================================
 *   @n@b CSL_i2cHwControl
 *
 *   @b Description
 *   @n Takes a command of I2C with an optional argument & implements it.
 *
 *   @b Arguments
 *   @verbatim
 
            hI2c        Handle to the I2C instance 
            cmd         The command to this API indicates the action to be
                        taken on I2C. 
            arg         An optional argument.

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li         CSL_SOK               - Status info return successful.
 *   @li         CSL_ESYS_BADHANDLE    - Invalid handle
 *   @li         CSL_ESYS_INVCMD       - Invalid command
 *   @li         CSL_ESYS_INVPARAMS    - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  i2c opened properly
 *
 *   <b> Post Condition </b>
 *   @n  device register programmed accordingly
 *
 *   @b Modifies
 *   @n The hardware registers of I2C.
 *
 *   @b Example
 *   @verbatim
        CSL_I2cHandle           hI2c;
        CSL_I2cHwControlCmd     cmd;
        void                    *arg;
        
        cmd = CSL_I2C_CMD_RESET;

        status = CSL_i2cHwControl (hI2c, cmd, &arg);

     @endverbatim
 * =============================================================================
 */ 
extern CSL_Status  CSL_i2cHwControl(    
    CSL_I2cHandle                        hI2c,    
    CSL_I2cHwControlCmd                  cmd,    
    void                                 *arg
);

/** ============================================================================
 *   @n@b CSL_i2cRead
 *
 *   @b Description
 *   @n Reads the received data from the I2C Data Receive register
 *
 *   @b Arguments
     @verbatim
            hI2c   Handle of already opened peripheral
            buf    Pointer to memory where data will be read and stored
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Data successfully extracted
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS  - Invalid parameter
 *                  
 *   <b> Pre Condition </b>
 *   @n  Both @a CSL_i2cInit() and @a CSL_i2cOpen() must be called successfully
 *
 *   <b> Post Condition </b>
 *   @n  RSFULL flag bit in ICSTR register is cleared
 *       ICRRDY flag bit in ICSTR register is cleared
 *
 *   @b Modifies
 *   @n buf variable holds the received data
 *
 *   @b Example
 *   @verbatim 
 
        Uint16        inData;
        CSL_Status    status;
        CSL_I2cHandle hI2c;
        ...
        // I2C object defined and HwSetup structure defined and initialized
        ...

        // Init, Open, HwSetup successfully done in that order
        ...

        status = CSL_i2cRead(hI2c, &inData);
     @endverbatim
 *
 *  ============================================================================
 */
extern CSL_Status CSL_i2cRead(    
    CSL_I2cHandle   hI2c,    
    void            *buf
);

/** ============================================================================
 *   @n@b CSL_i2cWrite
 *
 *   @b Description
 *   @n Writes the specified data into I2C Data Transmit register
 *
 *   @b Arguments
     @verbatim
            hI2c   Handle of already opened peripheral
            buf    Pointer to data to be written
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Data successfully written
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS  - Invalid parameter
 *                  
 *   <b> Pre Condition </b>
 *   @n  Both @a CSL_i2cInit() and @a CSL_i2cOpen() must be called successfully
 *
 *   <b> Post Condition </b>
 *   @n  XSMT flag bit in ICSTR register is set
 *       ICXRDY flag bit in ICSTR register is cleared
 *
 *   @b Modifies
 *   @n ICDXR register 
 *
 *
 *  @b Example:
 *  @verbatim

        Uint16          inData;
        CSL_Status      status;
        CSL_I2cHandle   hI2c;
        ...
        // I2C object defined and HwSetup structure defined and initialized
        ...

        // Init, Open, HwSetup successfully done in that order
        ...

        status = CSL_i2cWtrite (hi2c, &inData);
    @endverbatim
 *
 *  ============================================================================
 */
extern CSL_Status CSL_i2cWrite(    
    CSL_I2cHandle   hI2c,    
    void            *buf
);

/** ============================================================================
 *   @n@b CSL_i2cHwSetupRaw
 *
 *   @b Description
 *   @n This function initializes the device registers with the register-values
 *      provided through the Config Data structure.
 *
 *   @b Arguments
 *   @verbatim
            hI2c        Handle to the I2C  
            config      Pointer to config structure
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Configuration successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS  - Configuration is not
 *                                                properly initialized
 *
 *   <b> Pre Condition </b>
 *   @n  Both CSL_i2cInit() and a CSL_i2cOpen() must be called
 *
 *
 *   <b> Post Condition </b>
 *   @n  The registers of the specified I2C instance will be setup
 *       according to value passed.
 *
 *   @b Modifies
 *   @n Hardware registers of the specified I2C instance.
 *
 *   @b Example
 *   @verbatim
        CSL_I2cHandle       hI2c;
        CSL_I2cConfig       config = CSL_I2C_CONFIG_DEFAULTS;
        CSL_Status          status;

        status = CSL_i2cHwSetupRaw (hI2c, &config);

     @endverbatim
 * ===========================================================================
 */
extern CSL_Status  CSL_i2cHwSetupRaw (
    CSL_I2cHandle           hI2c,
    CSL_I2cConfig           *config
);

/** ============================================================================
 *   @n@b CSL_i2cGetHwStatus
 *
 *   @b Description
 *   @n Gets the status of the different operations of I2C.
 *
 *   @b Arguments
 *   @verbatim
            hI2c        Handle to the I2C instance 
            query       The query to this API of I2C which indicates the
                        status to be returned.
            response    Placeholder to return the status.

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Status info return successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVQUERY   - Invalid query command
 *   @li                    CSL_ESYS_INVPARAMS  - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  i2c opened properly
 *
 *   <b> Post Condition </b>
 *   @n  none
 *
 *   @b  Modifies
 *   @n  response variable holds the queried value
 *
 *   @b Example
 *   @verbatim
        CSL_I2cHandle           hI2c;
        CSL_I2cHwStatusQuery    query;
        void                    *reponse;

        status = CSL_Geti2cHwStatus (hI2c, query, &response);

     @endverbatim
 * =============================================================================
 */ 
extern CSL_Status  CSL_i2cGetHwStatus(
    /** Pointer to the object that holds reference to the
    *  instance of I2C requested after the call
    */
    CSL_I2cHandle                         hI2c,
    /** The query to this API which indicates the status
    *  to be returned
    */
    CSL_I2cHwStatusQuery                  query,
    /** Placeholder to return the status. @a void* casted */
    void                                  *response
);


/** ============================================================================
 *   @n@b CSL_i2cGetBaseAddress
 *
 *   @b Description
 *   @n  Function to get the base address of the peripheral instance.
 *       This function is used for getting the base address of the peripheral
 *       instance. This function will be called inside the CSL_i2cOpen()
 *       function call. 
 *
 *   @b Arguments
 *   @verbatim
 
            i2cNum        Specifies the instance of the I2C to be opened.
            pI2cParam     Module specific parameters.   
            pBaseAddress  Pointer to baseaddress structure containing base
                          address details.

     @endverbatim
 *
 *   <b> Return Value </b> CSL_Status
 *   @li                   CSL_OK              Open call is successful
 *   @li                   CSL_ESYS_FAIL       The instance number is invalid.
 *   @li                   CSL_ESYS_INVPARAMS  Invalid parameter 
 *                     
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Base Address structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable  
 *         2. Base address structure is modified.
 *
 *   @b Example
 *   @verbatim
        CSL_Status          status;
        CSL_I2cBaseAddress  baseAddress;

       ...
      status = CSL_i2cGetBaseAddress (CSL_I2C_0, NULL, &baseAddress);

    @endverbatim
 * ===========================================================================
 */
extern CSL_Status   CSL_i2cGetBaseAddress (
    CSL_InstNum             i2cNum,
    CSL_I2cParam            *pI2cParam,
    CSL_I2cBaseAddress      *pBaseAddress
);

#ifdef __cplusplus
}
#endif


#endif /* _CSL_I2C_H_ */
