/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 *  ============================================================================
 */
 
/** @mainpage INTC CSL 3.x
 *
 * @section Introduction
 *
 * @subsection xxx Purpose and Scope
 * The purpose of this document is to identify a set of common CSL APIs for
 * the INTC module. The CSL developer is expected to refer to this document  
 * while designing APIs for the modules which use INTC module. Some of the
 * listed APIs may not be applicable to a given module. While other cases
 * this list of APIs may not be sufficient to cover all the features required
 * for a particular Module. The CSL developer should use his discretion designing
 * new APIs or extending the existing ones to cover these.
 *
 * @subsection aaa Terms and Abbreviations
 *   -# CSL:  Chip Support Library
 *   -# API:  Application Programmer Interface
 * 
 *
 */
 

/** ============================================================================
 * @file csl_intc.h
 *
 * @path $(CSLPATH)\soc\c67\src
 *
 * @brief CSL functional-layer header file for the INTC
 *
 * Description
 *   -  Comprises the function declarations for the various APIs and the
 *      associated enumerations, type and symbol definitions for the INTC
 *      module.
 * Modification 
 *    - modified on: 14/12/2004
 *    - reason: Created the sources
 *
 * 
 * @author Ramitha Mathew.
 * 
 * =============================================================================
 */

#ifndef _CSL_INTC_H_
#define _CSL_INTC_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <csl.h>
#include <cslr.h>
#include <cslr_intc.h>


/**
 * @brief   Count of the number of interrupt-events
 */
#define CSL_INTC_EVENT_CNT                          (16)

/**
 * @brief   Event Enable
 */
#define CSL_INTC_EVENTMASK_ENABLE                   (1)

/**
 * @brief   Flag to disable modification of the Interrupt Vector Area 
 * on the core specified using 'CSL_IntcContext' via `CSL_intcInit(..)`
 */
#define CSL_INTC_CONTEXT_DISABLECOREVECTORWRITES    (1)

/**
 * @brief   Invalid EventId
 */
#define CSL_INTC_EVENTID_INVALID                    (-1)    
                             
/**
 * @brief   Indicates there is no associated event-handler
 */
#define CSL_INTC_EVTHANDLER_NONE        ((CSL_IntcEventHandlerRecord *) 0)

/**
 * @brief   Invalid handle
 */
#define CSL_INTC_BADHANDLE                          (0)

/**
 * @brief 	
 *		Indicates that the interruptHandler passed as the argument for the
 *  	function CSL_intcPlugEventHandler was not plugged 
 * 	 	properly in the vector table(The DMAX uid used for INTC ie
 *		CSL_DMAX_LOPRIORITY_EVENT0_UID might be in use by some other application)
 *			
 */
#define CSL_INTC_EVENTHANDLER_PLUG_ERROR			((CSL_IntcEventHandler)-2)
/**
 * @brief   Interrupt Enable/Disable
 **/
typedef enum {
    /** Enables event */
    CSL_INTC_EVENT_ENABLE      =   1,
    /** Disables event */
    CSL_INTC_EVENT_DISABLE     =   0
} CSL_IntcEvent;

/**
 * @brief   Global Interrupt Enable/Disable 
 **/
typedef enum {
    /** Global Enable */
    CSL_INTC_GLOBAL_ENABLE     =  1,
    /** Global Disable */
    CSL_INTC_GLOBAL_DISABLE    =  0
} CSL_IntcGlobal;

/**
 * @brief   Hardware status response
 **/
typedef enum{
    /** Event pending */
    CSL_INTC_EVENTPENDING = 1,
    /** Event not pending */
    CSL_INTC_EVENTNOTPENDING = 0 
}CSL_EventStatus;
    
/**
 * @brief   Enumeration of the control commands
 *
 * These are the control commands that could be used with
 * CSL_intcHwControl(..). Some of the commands expect an
 * argument as documented along-side the description of
 * the command.
 */
typedef enum {
    CSL_INTC_CMD_EVTENABLE,
    /**
     * @brief   Enables the event
     * @param   (None)
     */
    CSL_INTC_CMD_EVTDISABLE,
    /**
     * @brief   Disables the event
     * @param   (None)
     */
    CSL_INTC_CMD_EVTCLEAR,
    /**
     * @brief   Clears the event 
     * @param   (None)
     */
    CSL_INTC_CMD_EVTSET
    /**
     * @brief   Sets the event (software triggering)
     * @param   (None)
     */    

} CSL_IntcHwControlCmd;

/**
 * @brief   Enumeration of the queries
 *
 * These are the queries that could be used with CSL_intcGetHwStatus(..).
 * The queries return a value through the object pointed to by the pointer
 * that it takes as an argument. The argument supported by the query is
 * documented along-side the description of the query.
 */
typedef enum {
    
    CSL_INTC_QUERY_ISEVENTPENDING
    /**
     * @brief   Checks if event is pending
     * @param   (Bool *)
     */
} CSL_IntcHwStatusQuery;

/**
 * @brief   Interrupt Event IDs
 */
typedef Int     CSL_IntcEventId;

/**
 * @brief   INTC module parameters for open
 *
 * None as of now
 */
typedef void CSL_IntcParams;

/**
 * @brief   Event Handler pointer
 *
 * Event handlers ought to conform to this type
 */
typedef void (*CSL_IntcEventHandler)(void *);

/**
 * @brief   Event enable state
 */
typedef Uint8   CSL_IntcEventEnableState;

/**
 * @brief   Global Interrupt enable state
 */
typedef Uint8   CSL_IntcGlobalEnableState;

/**
 * @brief   Hardware Status
 */
typedef Uint8 CSL_IntcEventPendingState;

/**
 * @brief   Event Handler Record
 *
 * Used to set-up the event-handler using CSL_intcPlugEventHandler(..)
 */
typedef struct {
   /**
    *  Pointer to the event handler
    */
    CSL_IntcEventHandler    handler;    
   /**
    *  The argument to be passed to the handler when it is invoked
    *  This should be NULL for an isr.
    */
    void                    *arg;        
} CSL_IntcEventHandlerRecord;

/**
 * @brief   Context relevant to the dispatcher
 */
typedef struct {
   /**
    * Holds the isr records of the events
    */
    CSL_IntcEventHandlerRecord  eventHandlerRecord[CSL_INTC_EVENT_CNT];
} CSL_IntcDispatcherContext;

/**
 * @brief   INTC Module Context
 */
typedef struct {    
   /**
    * Flags to indicate presence or absence of OS
    */    
    CSL_BitMask16   flags; 
   /**
    * Masks which indicates the allocation of events 
    */                   
    CSL_BitMask16   eventAllocMask[CSL_INTC_EVENT_CNT];
} CSL_IntcContext;

/**
 * @brief   The interrupt handle object
 *
 * This object is referenced by the handle to identify the event.
 */
typedef struct CSL_IntcObj {
    /** The event-id */   
    CSL_IntcEventId eventId;  
    /** Reserved for the future */   
    void    *reserved;   
} CSL_IntcObj;

/**
 * @brief   The interrupt handle
 *
 * This is returned by the CSL_intcOpen(..) API. The handle is used
 * to identify the event of interest in all INTC calls.
 */
typedef struct CSL_IntcObj  *CSL_IntcHandle;

/**
 * @brief   Hardware settings of INTC.
 */
typedef struct CSL_IntcHwSetup 
{
    /** reserved for future */
    void    *reserved ;   
} CSL_IntcHwSetup ;
   

/*******************************************************************************
 * Interrupt Controller function declarations
 ******************************************************************************/

/** ============================================================================
 *   @n@b CSL_intcInit
 *
 *   @b Description
 *   This API performs any module-specific initialization. 
 *   CSL_intcInit (..) must be invoked before calling any other 
 *   API in the INTC module.
 *
 *   @b Arguments
 *   @n pContext - pointer to the intc context memory allocated
 *      by the user
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - returns on success
 *                          CSL_ESYS_ALREADY_INITIALIZED - if initialized
 *   <b> Pre Condition </b>
 *   @n  Memory has to be allocated and filled for the context variable
 *
 *   <b> Post Condition </b>
 *   @n  Intc module will be initialized 
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example:
 *   @verbatim
      if ((CSL_intcInit ( ) != CSL_SOK) && 
          (CSL_intcInit () != CSL_ESYS_ALREADY_INITIALIZED)
         ) 
         { 
         //module initialization failed! //
         }  
     @endverbatim
 * =============================================================================
 */
extern CSL_Status CSL_intcInit (
    CSL_IntcContext     *pContext
);

/** ============================================================================
 *   @n@b CSL_intcOpen
 *
 *   @b Description
 *   @n The API would reserve a cpu interrupt for use by the event specified 
 *      through the eventId. It returns a valid handle to the event only if the
 *      cpu interrupt is not currently allocated. The user could release the 
 *      interrupt after use by calling CSL_intcClose(..). The CSL-object 
 *      ('intcObj') that the user passes would be used to store information 
 *      pertaining handle.
 *
 *   @b Arguments
 *   @verbatim
              pIntcObj     pointer to the CSL-object allocated by the user              
              eventId      the event-id of the interrupt
              params       Module specific parameter(as of now pass NULL)            
              pStatus      pointer for returning status of the
                           function call(pass NULL if not required)                                      
                           

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_IntcHandle
 *   @n                     Valid INTC handle identifying the event
 *
 *   <b> Pre Condition </b>
 *   @n  1.INTC module must be initialized properly 
 *   @n  2.Memory allocated for obj structure 
 *
 *   <b> Post Condition </b>
 *   @n   1.    INTC object structure is populated
 *   @n   2.    The status is returned in the status variable. If status
 *              returned is
 *   @li            CSL_SOK             -Valid intc handle is returned
 *   @li            CSL_ESYS_FAIL       -The open command failed
 *
 *   @b Modifies
 *   @n    1. The status variable
 *   @n    2. INTC object structure
 *
 *   @b Example:
 *   @verbatim
     CSL_IntcObj     intcObj;
     CSL_IntcHandle  hIntc;
     CSL_Status      openStatus;
  
     hIntc = CSL_intcOpen(&intcObj, CSL_INTC_EVENTID_I2C, NULL, NULL);
     if (openStatus != CSL_SOK)
     {
     // open failed //
     }
    @endverbatim
 * =============================================================================
 */
extern CSL_IntcHandle CSL_intcOpen (   
    CSL_IntcObj         *intcObj,    
    CSL_IntcEventId     eventId,   
    CSL_IntcParams      *params,    
    CSL_Status          *status     
);   

/** =============================================================================
 *   @n@b CSL_intcClose
 *
 *   @b Description
 *   @n Releases an allocated cpu interrupt. CSL_intcClose() must be called to
 *      release a cpu interrupt that has been previously allocated with a call
 *      to CSL_intcOpen ().
 *
 *   @b Arguments
 *   @verbatim
            hIntc            Handle identifying the event
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Close successful
 *   @li                    CSL_ESYS_FAIL       - Close failed
 *
 *   @b Pre Condition 
 *   @n 1.INTC module must have been initialized properly 
 *   @n 2.Event must have been opened
 *
 *   @b Post Condition 
 *   @n 1. CPU interrupt could be used again 
 *   @n 2. The status is returned in the status variable 
 *   @b Modifies 
 *   @n The status variable 
 *
 *   @b Example
 *   @verbatim
            CSL_IntcHandle       hIntc;
            CSL_status           status;
            ...

            status = CSL_intcClose(hIntc);
            if (status != CSL_SOK) {
                // close failed! //
            }
     @endverbatim
 * =============================================================================
 */
extern CSL_Status CSL_intcClose (
    CSL_IntcHandle      hIntc   
);

/** ============================================================================
 *   @n@b CSL_intcHwSetup
 *
 *   @b Description
 *   @n CSL_intcHwSetup (..) API is used to configure the interrupt controller
 *      for the event identified by the handle. The user must instantiate
 *      and initialize a setup-structure with appropriate configuration
 *      parameters before passing it to the function. As no setup parameters 
 *      are identified for C672x\DA7xx INTC, this is a dummy API.
 *   
 *   @b Arguments
 *   @verbatim
       hIntc        Handle to the INTC instance          
       hwSetup      Pointer to harware setup structure
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK -  always return
 *
 *   @b IMPORTANT NOTE
 *      ==============
 *   @n In C672x\DA7xx CSL_intcHwSetup() is a function which does nothing as no h/w
 *     configuration is to be done
 *
 *   @b Pre Condition 
 *   @n 1.INTC module must have been initialized properly 
 *   @n 2.Event must have been opened
 *
 *   @b Post Condition 
 *   @n None
 *
 *   @b Modifies 
 *   @n None
 *
 *
 *   @b Example:
 *   @verbatim
        Here just pass the CSL_IntcHwSetup* as NULL        
        CSL_intcHwSetup(hIntc, &setup);  
     @endverbatim
 * =============================================================================
 */
extern CSL_Status CSL_intcHwSetup (
    CSL_IntcHandle      hIntc,  
    CSL_IntcHwSetup     *hwSetup 
);



/** ============================================================================
 *   @n@b CSL_intcHwControl
 *
 *   @b Description
 *   @n This API is used to invoke any of the control-operations supported
 *      by the module. Note: Refer to the control-command documentation
 *      for details on the parameter (if any) that a specific command accepts.
 *
 *   @b Arguments
 *   @verbatim
            hIntc           Handle identifying the event
            command         The command to this API indicates the action to be
                            taken on INTC.
            commandArg      An optional argument.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - HwControl successful.
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVCMD     - Invalid command
 *
 *   <b> Pre Condition </b>
 *   @n  INTC should be initialized and opened properly
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n The hardware registers of INTC.
 *
 *   @b Example:
 *   @verbatim
        CSL_intcHwControl (hIntc, CSL_INTC_CMD_EVTSET, NULL);
     @endverbatim
 * =============================================================================
 */
extern CSL_Status CSL_intcHwControl (
    CSL_IntcHandle          hIntc,       
    CSL_IntcHwControlCmd    command,     
    void                    *commandArg  
);


/** ============================================================================
 *   @n@b CSL_intcGetHwStatus
 *
 *   @b Description
 *   @n The CSL_intcGetHwStatus(..) API could be used to retrieve status
 *      or configuration information from the registers. The user must
 *      allocate an object that would hold the retrieved information
 *      and pass a pointer to it to the function. The type of the object
 *      is specific to the query-command.
 *
 *   @b Arguments
 *   @verbatim
            hIntc           Handle identifying the event
            query           The query to this API of INTC which indicates the
                            status to be returned.
            response        Placeholder to return the status.
     @endverbatim
 *
 *   <b> Return Value </b> CSL_Status
 *   @li                   CSL_SOK               - Status info return successful
 *   @li                   CSL_ESYS_BADHANDLE    - Invalid handle
 *   @li                   CSL_ESYS_INVQUERY     - Invalid query 
 *
 *   <b> Pre Condition </b>
 *   @n  INTC should be initialized and opened properly.
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example:
 *   @verbatim
     Bool    evtPending = FALSE;
     while (evtPending == FALSE)
      {
      CSL_intcGetHwStatus(hIntc,CSL_INTC_QUERY_ISEVENTPENDING,&evtPending);
      }
  
     @endverbatim
 *
 * =============================================================================
 */
extern CSL_Status CSL_intcGetHwStatus (
    CSL_IntcHandle          hIntc,      
    CSL_IntcHwStatusQuery   query,      
    void                    *response   
);

/** ============================================================================
 *   @n@b CSL_intcEventEnable
 *
 *   @b Description
 *   @n The API enables the specified event in the INTC. If the user wishes 
 *      to restore the enable-state of the event at a later point of time, 
 *      they may store the current state using the parameter, which could 
 *      be used with CSL_intcEventRestore (..).
 *      Note: The function directly works on the event and hence it is
 *      not necessary to "open" the event to invoke the API.
 * 
 *   @b Arguments
 *   @verbatim
         eventId     Event-ID of interest         
         prevState   (Optional) Pointer to object that would store current state
     @endverbatim
 *
 *   <b> Return Value </b> CSL_Status
 *   @li                   CSL_SOK on success
 *   @li                   CSL_INTC_EVENTID_INVALID 
 *
 *   @b Example:
 *   @verbatim
        CSL_intcEventEnable(CSL_INTC_EVENTID_SPI, NULL);
     @endverbatim
 *
 * ============================================================================
 */
extern CSL_Status CSL_intcEventEnable (
    CSL_IntcEventId             eventId,   
    CSL_IntcEventEnableState    *prevState   
);

/** ============================================================================
 *  @n@b CSL_intcEventDisable
 *
 *  @b Description
 *  @n The API disables the specified event in the INTC. If the user wishes to 
 *     restore the enable-state of the event at a later point of time, they may
 *     store the current state using the parameter, which could be used
 *     with CSL_intcEventRestore (..).
 *     Note: The function directly works on the event and hence it is
 *     not necessary to "open" the event to invoke the API.
 *
 *  @b Arguments
 *  @verbatim
        eventId     Event-ID of interest        
        prevState   (Optional) Pointer to object that would store current state
    @endverbatim
 *
 *  <b> Return Value </b> CSL_Status
 *  @li                   CSL_SOK on success
 *  @li                   CSL_INTC_EVENTID_INVALID 
 *
 * @b Example:
 * @verbatim
     CSL_IntcEventEnableState    oldState;
     CSL_intcEventDisable(CSL_INTC_EVENTID_SPI, &oldState);
   @endverbatim
 *
 * =============================================================================
 */
extern CSL_Status CSL_intcEventDisable (
    CSL_IntcEventId             eventId,    
    CSL_IntcEventEnableState    *prevState   
);

/** ============================================================================
 *  @n@b CSL_intcEventRestore
 *
 *  @b Description
 *  @n The API restores the specified event to a previous enable-state
 *     as recorded by the event-enable state passed as an argument.
 *     Note: The function directly works on the event and hence it is
 *     not necessary to "open" the event to invoke the API.
 *
 *  @b Arguments
 *  @verbatim
        eventId     Event-ID of interest        
        prevState   Object that contains information about previous state
    @endverbatim
 *
 *  <b> Return Value </b> CSL_Status
 *  @li                   CSL_SOK on success
 *  @li                   CSL_INTC_EVENTID_INVALID 
 *
 * @b Example:
 * @verbatim
        CSL_intcEventRestore(CSL_INTC_EVENTID_SPI, oldState);
   @endverbatim
 *
 * =============================================================================
 */
extern CSL_Status CSL_intcEventRestore (
    CSL_IntcEventId  eventId,          
    CSL_IntcEventEnableState    prevState   
);

/** ============================================================================
 *  @n@b CSL_intcGlobalEnable
 *
 *  @b Description
 *  @n The API enables the global interrupt by manipulating the processor's
 *     global interrupt enable/disable flag. If the user wishes to restore
 *     the enable-state at a later point, they may store the current state
 *     using the parameter, which could be used with CSL_intcGlobalRestore (..).
 *
 *  @b Arguments
 *  @verbatim
        prevState   (Optional) Pointer to object that would store current
                    stateObject that contains information about previous state
    @endverbatim
 *
 *  <b> Return Value </b> CSL_Status
 *  @li                   CSL_SOK -on success
 *
 *  @b Example:
 *  @verbatim
    CSL_IntcGlobalEnableState   prevGie;
    CSL_Status status;
    
    status = CSL_intcGlobalEnable (&prevGie);
    @endverbatim
 *
 * =============================================================================
 */
extern CSL_Status CSL_intcGlobalEnable (
    CSL_IntcGlobalEnableState   *prevState   
);

/** ============================================================================
 *  @n@b CSL_intcGlobalDisable
 *
 *  @b Description
 *  @n The API disables the global interrupt by manipulating the processor's
 *     global interrupt enable/disable flag. If the user wishes to restore
 *     the enable-state at a later point, they may store the current state
 *     using the parameter, which could be used with CSL_intcGlobalRestore (..).
 *
 *  @b Arguments
 *  @verbatim
        prevState   (Optional) Pointer to object that would store current
                    state of Global Interrupt bit
    @endverbatim
 *
 *  <b> Return Value </b> CSL_Status
 *  @li                   CSL_SOK -on success
 *
 *  @b Example:
 *  @verbatim
     CSL_IntcGlobalEnableState   gieState;
  
     CSL_intcGlobalDisable(&gieState);
       // critical-section code //
     CSL_intcGlobalRestore(gieState);
    @endverbatim
 *
 * =============================================================================
 */
extern CSL_Status CSL_intcGlobalDisable (
    CSL_IntcGlobalEnableState   *prevState   
);




/** ============================================================================
 *  @n@b CSL_intcGlobalRestore
 *
 *  @b Description
 *  @n The API restores the global interrupt enable/disable state to a previous
 *    state as recorded by the global-event-enable state passed as an argument.
 *
 *  @b Arguments
 *  @verbatim
        prevState   Object containing information about previous state
    @endverbatim
 *
 *  <b> Return Value </b> CSL_Status
 *  @li                   CSL_SOK on success
 *
 *  @b Example:
 *  @verbatim
     CSL_IntcGlobalEnableState   gieState;
  
     CSL_intcGlobalDisable(&gieState);
     // critical-section code //
     CSL_intcGlobalRestore(gieState);
    @endverbatim
 *
 * =============================================================================
 */
extern CSL_Status CSL_intcGlobalRestore (
    CSL_IntcGlobalEnableState       prevState   
);


/** ============================================================================
 *  @n@b CSL_intcDispatcherInit
 *
 *  @b Description
 *  @n Sets up the CSL interrupt dispatcher. 
 *     The user should call CSL_intcDispatcherInit (..) if they
 *     wish to make use of the dispatcher built into the CSL INTC
 *     module.
 *
 *  Note: This API must be called before using
 *  CSL_intcPlugEventHandler (..).
 *
 *  @b Arguments
 *  @n pContext - context relevant to Dispatcher
 *
 *  <b> Return Value </b> CSL_Status
 *  @li                   CSL_SOK on success
 *  @li                   CSL_ESYS_ALREADY_INITIALIZED 
 *
 *  @b Pre Condition 
 *  @n Memory has to be allocated for the context variable
 *
 *  @b Post Condition 
 *  @n None 
 *
 *  @b Modifies 
 *  @n None
 *
 * @b Example:
 * @verbatim
      if ((CSL_intcInit( ) != CSL_SOK) || 
           (CSL_intcInit() != CSL_ESYS_ALREADY_INITIALIZED)
         )
         {
         // module initialization failed! //
         }
      if ((CSL_intcDispatcherInit ( ) != CSL_SOK) && 
           (CSL_intcDispatcherInit ( ) != CSL_ESYS_ALREADY_INITIALIZED)
         )
         {
         // CSL dispatcher setting up failed! //
         }
   @endverbatim
 *
 * =============================================================================
 */
extern CSL_Status CSL_intcDispatcherInit (
    CSL_IntcDispatcherContext   *pContext
);


/** ============================================================================
 *  @n@b CSL_intcPlugEventHandler
 *
 *  @b Description
 *  @n CSL_intcPlugEventHandler (..) ties an event-handler to an event; so
 *     that the occurence of the event, would result in the event-handler
 *     being invoked.
 *
 *  @b Arguments
 *  @verbatim
        hIntc                Handle identifying the interrupt-event        
        eventHandlerRecord   Provides the details of the event-handler
    @endverbatim
 *
 *  <b> Return Value </b>
 *  @n  Returns the address of the previous handler
 *
 *  @b Pre Condition 
 *  @n 1. INTC has to be initialized and  opened for the event
 *  @n 2. DispatcherInit has to called
 *  @n 3. EventHandlerRecord has to be filled 
 
 *  @b Post Condition 
 *  @n Handler plugged

 *  @b Modifies 
 *  @n None
 *
 *  <b> Return Value </b>
 *  @n  Returns the address of the previous handler or
 *      CSL_INTC_EVENTHANDLER_PLUG_ERROR if the isr was not properly plugged  
 *
 *  @b Example:
 *  @verbatim
        CSL_IntcEventHandlerRecord  evtHandlerRecord;
  
        evtHandlerRecord.handler = myIsr;
        evtHandlerRecord.arg     = (void *)hI2c;
        CSL_intcPlugEventHandler(hIntc, &evtHandlerRecord);
    @endverbatim
 *
 * =============================================================================
 */
extern CSL_IntcEventHandler CSL_intcPlugEventHandler (
    CSL_IntcHandle                  hIntc,              
    CSL_IntcEventHandlerRecord  *eventHandlerRecord               
);

/** ============================================================================
 *  @n@b CSL_intcHookIsr
 *
 *  @b Description
 *  @n The CSL_intcHookIsr (..) hooks up the Interrupt Service Routine(ISR) to 
 *     the specified event. Note: Unlike CSL_intcPlugEventHandler (..)  , this is 
 *     done without opening the module (INTC) for the event. Hence the 
 *     CSL dispatcher is also not made use.
 *
 *  @b Arguments
 *  @verbatim
        eventId       Event Identifier        
        isrAddr      Pointer to the handler
    @endverbatim
 *    
 *  @b Pre Condition 
 *  @n INTC has to be initialized properly
 *
 *  @b Post Condition 
 *  @n Handler plugged
 *
 *  @b Modifies 
 *  @n None
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK         - Isr plugged properly
 *   @li                    CSL_ESYS_FAIL   - Isr plug failed.(The DMAX 
 *	    									  uid used in INTC ie 
 *											  CSL_DMAX_LOPRIORITY_EVENT0_UID 
 *											  might be in use by some other
 *											  application)

 *
 * @b Example:
 * @verbatim
        CSL_intcHookIsr(CSL_INTC_EVENTID_RTI_INT_REQ0 , (void *)rtiIsr);
   @endverbatim 
 * =============================================================================
 */
extern CSL_Status  CSL_intcHookIsr (
    CSL_IntcEventId       evtId,   
    Uint32                isrAddr 
);

/** ============================================================================
 *  @n@b CSL_intcSetVectorPtr
 *
 *  @b Description
 *  @n The API changes the base address of the Interrupt Vector table to the 
 *     value passed.
 *
 *  @b Arguments
 *  @verbatim
        vecPtr   New base address of the Interrupt Vector Table. Care should be 
                 taken that first ten lower bits are all zeroes
    @endverbatim
 *
 *  <b> Return Value </b> 
 *  @li                   Previously loaded ISTP value
 *
 *  @b Example:
 *  @verbatim
    #define VEC_ADDR        0x100c0000
    Uint32 prevValue;
    
    
    prevValue = CSL_intcSetVectorPtr (VEC_ADDR);
    
    @endverbatim
 *
 * =============================================================================
 */   
extern Uint32 CSL_intcSetVectorPtr(
    Uint32 vecPtr
);
         
#ifdef __cplusplus
 }
#endif

#endif /**_CSL_INTC_H_ */
