/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** @file  csl_pllc.h
 *
 *  @path $(CSLPATH)\pllc\src
 *
 *  @brief PLLC functional layer API header file
 *
 */

/** @mainpage PLLC CSL 3.x
 *
 * @section Introduction
 *
 * @subsection xxx Purpose and Scope
 * The purpose of this document is to identify a set of common CSL APIs for
 * the PLLC module across various devices. The CSL developer is expected to
 * refer to this document while designing APIs for these modules. Some of the
 * listed APIs may not be applicable to a given PLLC module. While other cases
 * this list of APIs may not be sufficient to cover all the features of a
 * particular PLLC Module. The CSL developer should use his discretion designing
 * new APIs or extending the existing ones to cover these.
 *
 * @subsection aaa Terms and Abbreviations
 *   -# CSL:  Chip Support Library
 *   -# API:  Application Programmer Interface
 *
 * @subsection References
 *    -# CSL-001-DES, CSL 3.x Design Specification DocumentVersion 1.02
 *
 */

/* =============================================================================
 *  Revision History
 *  ===============
 *  22-Nov-2004 AshaCe Created.
 * =============================================================================
 */

#ifndef _CSL_PLLC_H_
#define _CSL_PLLC_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <cslr.h>
#include <soc.h>
#include <csl_error.h>
#include <csl_types.h>
#include <csl.h>
#include <cslr_pllc.h>


/**
 *  pllc global macro declarations
 */
 
#define DELAY_200MS		6000
#define DELAY_		    100
/**
 *  pllc ENUM declarations
 */
/** \brief  This enum describes the modes of the pllc
 *
 *  It can either be in bypass/pll mode.
 */
typedef enum {
	/** The pllc is in bypass mode. */
	CSL_PLLC_BYPASS = CSL_PLLC_PLLCSR_PLLEN_BYPASS,
	/** The pllc is in pll mode. */
	CSL_PLLC_PLL_MODE = CSL_PLLC_PLLCSR_PLLEN_PLL
} CSL_PllcMode;
	
/** \brief  This enum describes the reset states of the pllc
 *
 *  It can be either released/asserted.
 */
typedef enum {
	/** The pllc reset is released. */
	CSL_PLLC_RELEASE = CSL_PLLC_PLLCSR_PLLRST_RELEASED,
	/** The pllc reset is asserted. */
	CSL_PLLC_ASSERT  = CSL_PLLC_PLLCSR_PLLRST_ASSERTED
} CSL_PllcResetState;
				 
/** \brief  This enum describes the states of the pllc
 *
 *  It can be either in operational/power down state.
 */
typedef enum {
	/** The pllc is in operational state. */
	CSL_PLLC_OPERATIONAL = CSL_PLLC_PLLCSR_PLLPWRDN_NO,
	/** The pllc is in power down state. */
	CSL_PLLC_PWRDN       = CSL_PLLC_PLLCSR_PLLPWRDN_YES
} CSL_PllcState;

/** \brief  This enum describes the Clock state of the PLL
 *
 *  It can be either in Stable / Not Stable.
 */
typedef enum {
	/** The pllc is in Stable state. */
	CSL_PLLC_CLK_STABLE = CSL_PLLC_PLLCSR_STABLE_STABLE,
	/** The pllc is not in stable state. */
	CSL_PLLC_CLK_NOT_STABLE = CSL_PLLC_PLLCSR_STABLE_NOT_STABLE
}CSL_PllcOscStableState;

/** \brief  This enum describes the GO Operation Enabled state
 *
 *  It can either be in Enabled or disabled.
 */
typedef enum {
	/** The GO Operation is disabled. */
	CSL_PLLC_GO_OPRN_DISABLE = CSL_PLLC_PLLSTAT_GOSTAT_NOT_PROGRESS,
	/** The GO Operation is in progress. */
	CSL_PLLC_GO_OPRN_IN_PROGRESS = CSL_PLLC_PLLSTAT_GOSTAT_INPROG
} CSL_PllcGoStatus;

/** \brief  This enum describes the various clock's enable status  
 *
 *  It can be either enabled/disabled.
 */
typedef enum {
	/** Clock is enabled. */
	CSL_PLLC_CLK_ENABLE  = CSL_PLLC_CKEN_AUXEN_ENABLE,
	/** Clock is disabled. */
	CSL_PLLC_CLK_DISABLE = CSL_PLLC_CKEN_AUXEN_DISABLE
} CSL_PllcClkState;

/** \brief  This enum describes the clock's enable status  
 *
 *  It can be either ON/Gated.
 */
typedef enum {
	/** Clock is On. */
	CSL_PLLC_CLK_ON  = CSL_PLLC_CKSTAT_AUXON_ON,
	/** Clock is Gated. */
	CSL_PLLC_CLK_GATED = CSL_PLLC_CKSTAT_AUXON_GATED
} CSL_PllcClockStatus;

/** \brief  This enum describes the pllc dividers  
 *
 */

typedef enum {
	/* Divider for PLL mode*/
	CSL_PLLC_DIV0	= 0,
	/* Divider for SYSCLK 1*/
	CSL_PLLC_DIV1	= 1,
	/* Divider for SYSCLK 2*/
	CSL_PLLC_DIV2	= 2,
	/* Divider for SYSCLK 3*/
	CSL_PLLC_DIV3	= 3
}CSL_PllcDivNum;
				  
/** \brief  This enum describes the sysclk   
 *
 */

typedef enum {
	/* PLLC DIV1 output*/
	CSL_PLLC_SYSCLK1	= 0,
	/* PLLC DIV2 output*/
	CSL_PLLC_SYSCLK2	= 1,
	/* PLLC DIV3 output*/
	CSL_PLLC_SYSCLK3	= 2
}CSL_PllcSysClk;

/** \brief  This enum describes the states of the pllc dividers  
 *
 *  It can be either enabled/disabled.
 */
typedef enum {
	/** The pllc divider  is enabled. */
	CSL_PLLC_PLLDIV_ENABLE  = CSL_PLLC_PLLDIV0_D0EN_ENABLE,
	/** The pllc divider 0 is disabled. */
	CSL_PLLC_PLLDIV_DISABLE = CSL_PLLC_PLLDIV0_D0EN_DISABLE
} CSL_PllcDivState;

/** \brief This enum describes control commands passed to CSL_pllcHwControl()
 *
 *  This is the set of commands that are passed to CSL_pllcHwControl() with
 *  an optional argument type-casted to a void* .
 *  The arguments to be passed with each enumeration (if any) are specified
 *  next to the enumeration.
 */
typedef enum {
	/**<
	* @brief   Set PLL mode: Either bypass/ pll mode.
	*/
	CSL_PLLC_CMD_SET_MODE             = 1,
	/**<
	* @brief   Set PLL state: Either power down/ operational state.
	*/
	CSL_PLLC_CMD_SET_PLL_STATE        = 2,
	/**<
	* @brief   Set Oscillator state: Either power down/ operational state.
	*/
	CSL_PLLC_CMD_SET_OSC_STATE        = 3,
	/**<
	* @brief   Assert reset to pll: Either reset released/ asserted.
	*/
	CSL_PLLC_CMD_RESET_CONTROL        = 4,
	/**<
	* @brief   Set GO Operation.
	*/
	CSL_PLLC_CMD_SET_GO_OPN           = 5,
	/**<
	* @brief   Controls ALIGNing with the corresponding SYSCLK .
	*/
	CSL_PLLC_CMD_ALIGN_CONTROL        = 6,
	/**<
	* @brief   Set PLLM multiplier factor.
	*/
	CSL_PLLC_CMD_SET_PLLM_MULFACTOR	   = 7,
	/**<
	* @brief   Controls the given divider by enabling/disabling the divider
	 * and/or setting the divider ratio of the given divider.
	*/
	 CSL_PLLC_CMD_DIV_CONTROL         = 8,
	/**<
	* @brief   Controls the AuxCLK enable/disable.
	*/ 
	CSL_PLLC_CMD_AUCXLK_CONTROL      = 9
}CSL_PllcHwControlCmd;
    

/** \brief This enum describes queries passed to CSL_PllcGetHwStatus()
 *
 *  This is used to get the status of different operations.
 *  The arguments to be passed with each enumeration if any
 *  are specified next to the enumeration
 */
typedef enum {
	/**<
	* @brief   Queries PLL Controller Mode (bypass/ pll).
	* @param   (CSL_PllcMode*)
	*/
	CSL_PLLC_QUERY_MODE                 = 1,
	/**<
	* @brief   Queries PLL Controller State (operational/ powerdown).
	* @param   (CSL_PllcState*)
	*/
	CSL_PLLC_QUERY_STATE                = 2,
	/**<
	* @brief   Queries Oscillator Powerdown State (operational/ powerdown).
	* @param   (CSL_PllcState*)
	*/
	CSL_PLLC_QUERY_OSC_STATE            = 3,
	/**<
	* @brief   Queries PLL Controller Reset (reset released/ asserted).
	* @param   (CSL_PllcResetState*)
	*/
	CSL_PLLC_QUERY_RESET                = 4,
	/**<
	* @brief   Queries PLL Controller DISABLE (Disable released/ asserted).
	* @param   (CSL_PllcDisableState*)
	*/
	CSL_PLLC_QUERY_DISABLE              = 5,
	/**<
	* @brief   Queries Oscillator Input stable state (stable/ Not stable).
	* @param   (CSL_PllcOscStableState*)
	*/ 
	CSL_PLLC_QUERY_OSC_STABLE_STATE     = 6,
	/**<
	* @brief   Queries PLL Core Lock Status (LOCKED / UNLOCKED).
	* @param   (CSL_PllcLockState*)
	*/ 
	CSL_PLLC_QUERY_LOCK_STATE     		= 7,
	/**<
	* @brief   Queries Go Operation Enabled Status (Enabled/Disabled).
	* @param   (CSL_PllcGoState*)
	*/ 
	CSL_PLLC_QUERY_GOEN_STATE     		= 8,
	/**<
	* @brief   Queries Go transition Status (Enabled/Disabled).
	* @param   (CSL_PllcGoStatus*)
	*/ 
	CSL_PLLC_QUERY_GOSTAT     			= 9,
	/**<
	* @brief   Queries PLL Controller Multiplier multiplication factor.
	* @param   (CSL_BitMask16*)
	*/
	CSL_PLLC_QUERY_PLLM_MUL_FACTOR      = 10,
	/**<
	* @brief   Queries PLL Controller Divider  state (enabled/ disabled).
	* and its divider ratio.
	 * @param   (CSL_PllcDivCntrl*)
	*/
	CSL_PLLC_QUERY_DIVIDER_STATE        = 11,
	/**<
	* @brief   Queries the given clock (AUX) state (ON/Gated)
	* @param   (CSL_PllcClockStatus*)
	*/
	CSL_PLLC_QUERY_CLK_STATE       		= 12,
	 /**<
	* @brief   Queries the given SYSCLK clock state (ON/Gated)
	* @param   (CSL_PllcSysClkStatus*)
	*/
	CSL_PLLC_QUERY_SYSCLK_STATE       	= 13
}CSL_PllcHwStatusQuery;


typedef struct {
	/** Divider State (Enabled/Disabled) */
	CSL_PllcDivState	divEnable;
	/** Divider ratio */
	Uint32        		pllDivRatio;
} CSL_PllcDivEnable;

/** \brief  This struct describes the divider, its state and divider ratio 
 *
 *  It can be either enabled/disabled.
 */
typedef struct {
	/** Divider number (DIV0..DIV3, OD1*/
	CSL_PllcDivNum		divNum;
	CSL_PllcDivEnable	divControl; 
}CSL_PllcDivCntrl;	
	
/** \brief Hardware setup structure for PLLC
 */
typedef struct {
	/** PLLC Mode BYPASS or PLL */
	CSL_PllcMode	pllcMode;

	/** divider 0 enable/disable and its divider ratio */
	CSL_PllcDivEnable div0Enable;

	/** divider 1 enable/disable and its divider ratio */
	CSL_PllcDivEnable div1Enable;

	/** divider 2 enable/disable and its divider ratio */
	CSL_PllcDivEnable div2Enable;

	/** divider 3 enable/disable and its divider ratio */
	CSL_PllcDivEnable div3Enable;

	/** PLL Multiplier factor */
	Uint32        pllM;

	/** Setup that can be used for future implementation */
	void         *extendSetup;
} CSL_PllcHwSetup;


/** \brief Config-structure
 *
 * Used to configure the pllc using CSL_wdtHwSetupRaw()
 */
typedef struct {
	/** pllc control/status register */
	volatile Uint32 PLLCSR;

	/** pllc multiplier control register */
	volatile Uint32 PLLM;

	/** pllc divider 0 register */
	volatile Uint32 PLLDIV0;

	/** pllc divider 1 register */
	volatile Uint32 PLLDIV1;

	/** pllc divider 2 register */
	volatile Uint32 PLLDIV2;

	/** pllc divider 3 register */
	volatile Uint32 PLLDIV3;

	/** PLL Command register */
	volatile Uint32 PLLCMD;

	/** PLL Align control register */
	volatile Uint32 ALNCTL;

	/** PLL Clock enable register */
	volatile Uint32 CKEN;

} CSL_PllcConfig;


/** \brief  This structure describes the sysclk and its corresponding
 *	status of the clock.
 *
 */
typedef struct {
	CSL_PllcSysClk	sysClk;			/* SYSCLK1 to SYSCLK8 */
	CSL_PllcClockStatus	pllClkState; /* Enable/Disable */	
}CSL_PllcSysClkStatus;

/** \brief Module specific context information. Present implementation of pllc CSL
 *  doesn't have any context information.
 */
typedef struct {
	/** Context information of pllc CSL.
	*  The below declaration is just a place-holder for future implementation.
	*/
	Uint16  contextInfo;
} CSL_PllcContext;

/** \brief Module specific parameters. Present implementation of pllc CSL
 *  doesn't have any module specific parameters.
 */
typedef struct {
	/** Bit mask to be used for module specific parameters. The below
	*  declaration is just a place-holder for future implementation.
	*/
	CSL_BitMask16   flags;
} CSL_PllcParam;

/** \brief This structure contains the base-address information for the peripheral
 *  instance of the PLLC
 */
typedef struct {
	/** Base-address of the configuration registers of the peripheral
	*/
	CSL_PllcRegsOvly  regs;
} CSL_PllcBaseAddress;

/** \brief PLLC object structure
 */
typedef struct CSL_PllcObj{
	/** Pointer to the register overlay structure of the pllc */
	CSL_PllcRegsOvly  regs;
	/** Instance of pllc being referred by this object  */
	CSL_InstNum       pllcNum;
} CSL_PllcObj;


/** \brief Default values for config structure
 */
#define CSL_PLLC_CONFIG_DEFAULTS {  \
	CSL_PLLC_PLLCSR_RESETVAL,   \
	CSL_PLLC_PLLM_RESETVAL,     \
	CSL_PLLC_PLLDIV0_RESETVAL,  \
	CSL_PLLC_PLLDIV1_RESETVAL,  \
	CSL_PLLC_PLLDIV2_RESETVAL,  \
	CSL_PLLC_PLLDIV3_RESETVAL,  \
	CSL_PLLC_PLLCMD_RESETVAL,   \
	CSL_PLLC_ALNCTL_RESETVAL,   \
	CSL_PLLC_CKEN_RESETVAL      \
}

/** \brief Default hardware setup parameters
 */
#define CSL_PLLC_HWSETUP_DEFAULTS{  \
	CSL_PLLC_BYPASS,\
	{CSL_PLLC_PLLDIV_ENABLE,0x0},\
	{CSL_PLLC_PLLDIV_ENABLE,0x0},\
	{CSL_PLLC_PLLDIV_ENABLE,0x1},\
	{CSL_PLLC_PLLDIV_ENABLE,0x2},\
	CSL_PLLC_PLLM_RESETVAL,     \
	NULL                        \
}

/** \brief This data type is used to return the handle to the CSL of the pllc
 */
typedef struct CSL_PllcObj    *CSL_PllcHandle;

/*******************************************************************************
 * PLL controller function declarations
 ******************************************************************************/

/** ============================================================================
 *   @n@b CSL_pllcInit
 *
 *   @b Description
 *   @n This is the initialization function for the pllc CSL. The function
 *      must be called before calling any other API from this CSL. This
 *      function is idem-potent. Currently, the function just return status
 *      CSL_SOK, without doing anything.
 *
 *   @b Arguments
 *   @verbatim
        pContext    Pointer to module-context. As pllc doesn't have
                    any context based information user is expected to pass NULL.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                        CSL_SOK - Always returns
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_sysInit() must be called. 
 *
 *   <b> Post Condition </b>
 *   @n  The CSL for pllc is initialized
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
            CSL_pllcInit(NULL);
     @endverbatim
 * =============================================================================
 */
extern CSL_Status  CSL_pllcInit (
    CSL_PllcContext * pContext
);

/** ============================================================================
 *   @n@b CSL_pllcOpen
 *
 *   @b Description
 *   @n This function returns the handle to the PLL controller
 *      instance. This handle is passed to all other CSL APIs.
 *
 *   @b Arguments
 *   @verbatim
            pllcObj     Pointer to pllc object.
 
            pllcNum     Instance of pll CSL to be opened. 
                           There is only one instance of the pllc available.
                           So, the value for this parameter will be 
                           CSL_PLLC always.
 
            pPllcParam  Module specific parameters.
 
            status         Status of the function call
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_PllcHandle
 *   @n                         Valid pllc handle will be returned if
 *                              status value is equal to CSL_SOK.
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n   1.    The status is returned in the status variable. If status
 *              returned is
 *   @li            CSL_SOK             Valid pllc handle is returned
 *   @li            CSL_ESYS_FAIL       The pllc instance is invalid
 *   @li            CSL_ESYS_INVPARAMS  Invalid parameter
 *
 *        2.    PLLC object structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *
 *         2. PLLC object structure
 *
 *   @b Example
 *   @verbatim
            CSL_status              status;
            CSL_PllcObj          pllcObj;
            CSL_PllcHandle       hPllc;

            ...
                
            hPllc = CSL_pllcOpen(&pllcObj, CSL_PLLC, NULL, &status);
            
            ...
    @endverbatim
 * =============================================================================
 */
extern CSL_PllcHandle CSL_pllcOpen (
    CSL_PllcObj         *pllcObj,
    CSL_InstNum         pllcNum,
    CSL_PllcParam       *pPllcParam,
    CSL_Status          *status
);

/** ============================================================================
 *   @n@b CSL_pllcGetBaseAddress
 *
 *   @b Description
 *   @n  Function to get the base address of the peripheral instance.
 *       This function is used for getting the base address of the peripheral
 *       instance. This function will be called inside the CSL_pllcOpen()
 *       function call. This function is open for re-implementing if the user
 *       wants to modify the base address of the peripheral object to point to
 *       a different location and there by allow CSL initiated write/reads into
 *       peripheral. MMR's go to an alternate location.
 *
 *   @b Arguments
 *   @verbatim
            pllcNum      Specifies the instance of the pllc to be opened.

            pPllcParam   Module specific parameters.

            pBaseAddress    Pointer to base address structure containing base
                            address details.

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_OK              Open call is successful
 *   @li                    CSL_ESYS_FAIL       The instance number is invalid.
 *   @li                    CSL_ESYS_INVPARAMS  Invalid parameter 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Base address structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *
 *         2. Base address structure is modified.
 *
 *   @b Example
 *   @verbatim
        CSL_Status              status;
        CSL_PllcBaseAddress  baseAddress;

        ...

        status = CSL_pllcGetBaseAddress(CSL_PLLC_PLLC_CNT, NULL,
                                           &baseAddress);
        ...

    @endverbatim
 * ===========================================================================
 */
extern CSL_Status CSL_pllcGetBaseAddress (
    CSL_InstNum              pllcNum,
    CSL_PllcParam *          pPllcParam,
    CSL_PllcBaseAddress *    pBaseAddress
);

/** =============================================================================
 *   @n@b CSL_pllcClose
 *
 *   @b Description
 *   @n This function closes the specified instance of PLLC.
 *
 *   @b Arguments
 *   @verbatim
            hPllc            Handle to the pllc

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Close successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *
 *   @b Example
 *   @verbatim
            CSL_PllcHandle   hPllc;
            CSL_status          status;

            ...

            status = CSL_pllcClose(hPllc);
     @endverbatim
 * ===========================================================================
 */
extern CSL_Status  CSL_pllcClose (
    CSL_PllcHandle    hPllc
);

/** ============================================================================
 *   @n@b CSL_pllcHwSetup
 *
 *   @b Description
 *   @n It configures the pllc registers as per the values passed
 *      in the hardware setup structure.
 *
 *   @b Arguments
 *   @verbatim
            hPllc            Handle to the pllc
 
            hwSetup             Pointer to harware setup structure
 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Hardware setup successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS  - Hardware structure is not
                                                  properly initialized
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  PLL controller registers are configured according 
 *       to the hardware setup parameters
 *
 *   @b Modifies
 *   @n PLL controller registers
 *
 *   @b Example
 *   @verbatim
            CSL_PllcHandle   hPllc;
            CSL_PllcObj      pllcObj;
            CSL_PllcHwSetup  hwSetup;
            CSL_status          status;

            ...
            
            hPllc = CSL_pllcOpen(&pllcObj, CSL_PLLC, NULL, &status);            
                      
            status = CSL_pllcHwSetup(hPllc, &hwSetup);
     @endverbatim
 * ===========================================================================
 */
extern CSL_Status  CSL_pllcHwSetup(
    CSL_PllcHandle       hPllc,
    CSL_PllcHwSetup     *hwSetup
);

/** ============================================================================
 *   @n@b CSL_pllcHwControl
 *
 *   @b Description
 *   @n Takes a command of PLLC with an optional argument & implements it.
 *
 *   @b Arguments
 *   @verbatim
            hPllc        Handle to the PLLC instance

            cmd             The command to this API indicates the action to be
                            taken on PLLC.

            arg             An optional argument.

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li         CSL_SOK               - Status info return successful.
 *   @li         CSL_ESYS_BADHANDLE    - Invalid handle
 *   @li         CSL_ESYS_INVCMD       - Invalid command
 *   @li         CSL_ESYS_INVPARAMS    - Invalid parameter
 *   @li         CSL_ESYS_NOTSUPPORTED - Action Not Supported
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n The hardware registers of PLLC.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle         hPllc;
        CSL_PllcHwControlCmd   cmd=CSL_PLLC_CMD_SET_MODE;
        CSL_PllcMode           arg=CSL_PLLC_PLL_MODE;

        status = CSL_pllcHwControl (hPllc, cmd, &arg);

     @endverbatim
 * =============================================================================
 */
extern CSL_Status  CSL_pllcHwControl(
    CSL_PllcHandle          hPllc,
    CSL_PllcHwControlCmd    cmd,
    void                    *cmdArg
);

/** ============================================================================
 *   @n@b CSL_pllcGetHwStatus
 *
 *   @b Description
 *   @n Gets the status of the different operations of PLLC.
 *
 *   @b Arguments
 *   @verbatim
            hPllc      Handle to the PLLC instance

            query      The query to this API of PLLC which indicates the
                       status to be returned.

            response   Placeholder to return the status.

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Status info return successful
 *   @li                    CSL_ESYS_BADHANDLE   - Invalid handle
 *   @li                    CSL_ESYS_INVQUERY    - Invalid query command
 *   @li                    CSL_ESYS_INVPARAMS   - Invalid parameter
 *   @li                    CSL_ESYS_NOTSUPPORTED- Action Not Supported
 *  
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle   hPllc;
        CSL_PllcState    response;

        status = CSL_pllcGetHwStatus (hPllc, CSL_PLLC_QUERY_OSC_STATE, &response);

     @endverbatim
 * =============================================================================
 */
extern CSL_Status  CSL_pllcGetHwStatus (
    CSL_PllcHandle           hPllc,
    CSL_PllcHwStatusQuery    query,
    void                       *response
);

/** ============================================================================
 *   @n@b CSL_pllcHwSetupRaw
 *
 *   @b Description
 *   @n This function initializes the device registers with the register-values
 *      provided through the Config Data structure.
 *
 *   @b Arguments
 *   @verbatim
            hPllc        Handle to the PLLC instance

            config          Pointer to config structure
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li            CSL_SOK               - Configuration successful
 *   @li            CSL_ESYS_BADHANDLE    - Invalid handle
 *   @li            CSL_ESYS_INVPARAMS    - Configuration is not
                                            properly initialized
 *   @li            CSL_ESYS_NOTSUPPORTED - Action not supported.
                                            (system not in bypass mode. if the 
                                             mode is not bypass mode & the 
                                             developer tries to write into
                                             D0 or PLLM registers or tries to 
                                             set the pllc in powerdown state
                                             then this error status is returned)
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The registers of the specified PLLC instance will be setup
 *       according to input configuration structure values.
 *
 *   @b Modifies
 *   @n Hardware registers of the specified PLLC instance.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle       hPllc;
        CSL_PllcConfig       config = CSL_PLLC_CONFIG_DEFAULTS;
        CSL_Status              status;

        status = CSL_pllcHwSetupRaw (hPllc, &config);

     @endverbatim
 * =============================================================================
 */
extern CSL_Status  CSL_pllcHwSetupRaw (
    CSL_PllcHandle           hPllc,
    CSL_PllcConfig          *config
);

/** ============================================================================
 *   @n@b CSL_pllcGetHwSetup
 *
 *   @b Description
 *   @n It retrives the hardware setup parameters of the pllc
 *      specified by the given handle.
 *
 *   @b Arguments
 *   @verbatim
            hPllc        Handle to the pllc
 
            hwSetup         Pointer to the harware setup structure
 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Retrieving the hardware setup
 *                                                parameters is successful
 *   @li                    CSL_ESYS_BADHANDLE  - The handle is passed is
 *                                                invalid
 *   @li                    CSL_ESYS_INVPARAMS  - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The hardware setup structure is populated with the hardware setup
 *       parameters
 *
 *   @b Modifies
 *   @n hwSetup variable
 *
 *   @b Example
 *   @verbatim
            CSL_PllcHandle   hPllc;
            CSL_PllcHwSetup  hwSetup;

            ...
            
            status = CSL_pllcGetHwSetup(hPllc, &hwSetup);
            
            ...
            
     @endverbatim
 * ===========================================================================
 */
extern CSL_Status  CSL_pllcGetHwSetup(
    CSL_PllcHandle       hPllc,
    CSL_PllcHwSetup     *hwSetup
);

#ifdef __cplusplus
}
#endif

#endif  /* _CSL_PLLC_H_ */
