/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */
 
/** ============================================================================
 *   @file  csl_rti.h
 *
 *   @path  $(CSLPATH)\soc\c67\src
 *
 *   @desc  API header file for Real Time Interrupt CSL
 *
 */ 
 
/* =============================================================================
 *  Revision History
 *  ===============
 * 2-Dec-2004 Sd File Created.
 * 25-Oct-2005 sd Comments added for query CSL_RTI_QUERY_CUR_UC0_CNT
 *
 * =============================================================================
 */
 
#ifndef _CSL_RTI_H_
#define _CSL_RTI_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <csl.h>
#include <cslr_rti.h>
#include <soc.h>

/* enable/disable the RTI Digital watchdog */
#define CSL_RTI_DWD_ENABLE  0xACED5312 
#define CSL_RTI_DWD_DISABLE 0x5312ACED

/* key values to set digital watchdog*/
#define CSL_RTI_WDKEY1  0xE51A
#define CSL_RTI_WDKEY2  0xA35C

/**
 *  Enumeration for Start/Continue Counters when device in debug mode
 */
typedef enum {
    /** Stop the Counters in debug mode */
    CSL_RTI_COUNTERS_STOP = 0,
    /** Continue the counters in debug mode */
    CSL_RTI_COUNTERS_RUN = 1
} CSL_RtiContOnSuspend;

/**
 *  Enumeration for Configure external interrupt source for both UC0 and FRC0
 *  and Configure external interrupt source for both UC1 and FRC1.
 */
typedef enum {
    /** Enable capture event triggered by Capture Event Source 0 */
    CSL_RTI_CAPTURE_EVENT_SOURCE0 = 0,
    /** Enable capture event triggered by Capture Event Source 1 */
    CSL_RTI_CAPTURE_EVENT_SOURCE1 = 1
} CSL_RtiExtnControl;

/**
 * Enumeration for free running counters with which
 * compare registers value is compared.
 */
typedef enum {
    /** Enable compare with FRC0 */
    CSL_RTI_FRC0_COMPARE_ENABLE = 0,
    /** enable compare with FRC1 */
    CSL_RTI_FRC1_COMPARE_ENABLE = 1
} CSL_RtiCompareCntl;

/** 
 *  This will have the base-address information for the peripheral
 *  instance
 */
typedef struct {
    /** Base-address of the Configuration registers of the peripheral */
    CSL_RtiRegsOvly regs;
} CSL_RtiBaseAddress;

/** 
 *  Module specific parameters. Present implementation doesn't have
 *  any module specific parameters.
 */
typedef struct {
   /** 
    *   Bit mask to be used for module specific parameters.
    *  The below declaration is just a place-holder for future
    *  implementation.
    */
    CSL_BitMask16   flags;
} CSL_RtiParam;

/** 
 *  Module specific context information. Present implementation doesn't
 *  have any Context information.
 */

typedef struct {
   /** 
    *  Context information of Real Time Interrupt.
    *  The below declaration is just a place-holder for future
    *  implementation.
    */
    Uint16  contextInfo;
} CSL_RtiContext;

/** 
 * Enumeration for hardware control commands
 */
typedef enum {
   /**
    * @brief    Start the Block0 Counters
    * @param   (None)
    */
    CSL_RTI_CMD_START_BLOCK0            = 0,
   /**
    * @brief    Stop the Block0 Counters
    * @param    (None)
    */
    CSL_RTI_CMD_STOP_BLOCK0             = 1,
   /**
    * @brief    Start the Block1 Counters
    * @param    (None)
    */
    CSL_RTI_CMD_START_BLOCK1            = 2,
   /**
    * @brief    Stop the Block1 Counters
    * @param    (None)
    */
    CSL_RTI_CMD_STOP_BLOCK1             = 3,
   /**
    * @brief    Clear the Compare Interrupt0
    * @param (None)
    */
    CSL_RTI_CMD_CLEAR_INT0              = 4,
   /**
    * @brief    Clear the Compare Interrupt1
    * @param (None)
    */
    CSL_RTI_CMD_CLEAR_INT1              = 5,
   /**
    * @brief    Clear the Compare Interrupt2
    * @param    (None)
    */
    CSL_RTI_CMD_CLEAR_INT2              = 6,
   /**
    * @brief    Clear the Compare Interrupt3
    * @param    (None)
    */
    CSL_RTI_CMD_CLEAR_INT3              = 7,
   /**
    * @brief    Clear the Overflow Interrupt0
    * @param    (None)
    */
    CSL_RTI_CMD_CLEAR_OVFINT0           = 8,
   /**
    * @brief    Clear the Overflow Interrupt1
    * @param    (None)
    */
    CSL_RTI_CMD_CLEAR_OVFINT1           = 9,
   /**
    * @brief    Enable Digital Watchdog 
    * @param    (None)
    */
    CSL_RTI_CMD_DWD_ENABLE              = 10,
   /**
    * @brief    Disable Digital Watchdog 
    * @param    (None)
    */
    CSL_RTI_CMD_DWD_DISABLE             = 11,
   /**
    * @brief    Clear Digital Watchdog status
    * @param    (None)
    */
    CSL_RTI_CMD_CLEAR_DWD               = 12,    
   /**
    * @brief    Set the Digital Watchdog Key
    * @param    (None)
    */
    CSL_RTI_CMD_WDKEY                   = 13
    
}   CSL_RtiHwControlCmd;

/**
 *  Enumeration for hardware status query commands
 */
typedef enum {
   /**
    * @brief   Query the current value of free running counter0
    * @param   (Uint32 *)
    */
    CSL_RTI_QUERY_CUR_FRC0_CNT          = 0,
   /**
    * @brief    Query the current value of up counter0
	* 			Reading the Free running counter before the Up counter read
	*           is taken care by this query.
	*           [NB: Up counter value is updated by a previous read of the 
	*			Free Running Counter]
    * @param    (Uint32 *)
    */
    CSL_RTI_QUERY_CUR_UC0_CNT           = 1,
   /**
    * @brief    Query the current value of free running counter1
    * @param    (Uint32 *)
    */
    CSL_RTI_QUERY_CUR_FRC1_CNT          = 2,
   /**
    * @brief    Query the current value of up counter1
	* 			Reading the Free running counter before the Up counter read
	*           is taken care by this query.
	*           [NB: Up counter value is updated by a previous read of the 
	*			Free Running Counter]
    * @param    (Uint32 *)
    */
    CSL_RTI_QUERY_CUR_UC1_CNT           = 3,
   /**
    * @brief    Query the captured value of up counter0
    * @param    (Uint32 *)
    */
    CSL_RTI_QUERY_UC0_CAPTURE_VAL       = 4,
   /**
    * @brief    Query the captured value of free running counter0
    * @param    (Uint32 *)
    */
    CSL_RTI_QUERY_FRC0_CAPTURE_VAL      = 5,
   /**
    * @brief    Query the captured value of up counter1
    * @param    (Uint32 *)
    */
    CSL_RTI_QUERY_UC1_CAPTURE_VAL       = 6,
   /**
    * @brief    Query the captured value of free running counter1
    * @param    (Uint32 *)
    */
    CSL_RTI_QUERY_FRC1_CAPTURE_VAL      = 7,
   /**
    * @brief    Query the value of compare0 register
    * @param    (Uint32 *)
    */
    CSL_RTI_QUERY_COMP0_CNT             = 8,
   /**
    * @brief    Query the value of compare1 register
    * @param    (Uint32 *)
    */
    CSL_RTI_QUERY_COMP1_CNT             = 9,
   /**
    * @brief    Query the value of compare2 register
    * @param    (Uint32 *)
    */
    CSL_RTI_QUERY_COMP2_CNT             = 10,
   /**
    * @brief    Query the value of compare3 register
    * @param    (Uint32 *)
    */  
    CSL_RTI_QUERY_COMP3_CNT             = 11,
   /**
    * @brief    Query the Status of overflow interrupt0
    * @param    (Bool *)
    */  
    CSL_RTI_QUERY_OVLINT0_STATUS        = 12,
   /**
    * @brief    Query the Status of overflow interrupt1
    * @param    (Bool *)
    */  
    CSL_RTI_QUERY_OVLINT1_STATUS        = 13,
   /**
    * @brief    Query the Status of compare interrupt0
    * @param    (Bool *)
    */  
    CSL_RTI_QUERY_INT0_STATUS           = 14,
   /**
    * @brief    Query the Status of compare interrupt1
    * @param    (Bool *)
    */
    CSL_RTI_QUERY_INT1_STATUS           = 15,
   /**
    * @brief    Query the Status of compare interrupt2
    * @param    (Bool *)
    */
    CSL_RTI_QUERY_INT2_STATUS           = 16,
   /**
    * @brief    Query the Status of compare interrupt3
    * @param    (Bool *)
    */
    CSL_RTI_QUERY_INT3_STATUS           = 17,
   /**
    * @brief    Query the Status of digital watchdog
    * @param    (Bool *)
    */
    CSL_RTI_QUERY_WATCHDOG_STATUS       = 18,
   /**
    * @brief    Query the digital watchdog Counter value
    * @param    (Uint32 *)
    */
    CSL_RTI_QUERY_WATCHDOG_DWNCTR       = 19
}   CSL_RtiHwStatusQuery;

/**
 * This structure is used to set compare up counter values.
 */
typedef struct {
    /** set Compare up counter0 value */
    Uint32 compareUpCntr0;
    /** set Compare up counter1 value */
    Uint32 compareUpCntr1;
} CSL_RtiCompUpCounter;

/** 
 * This structure is used to set Up counters and Free running counters values.
 */
typedef struct {
    /** Set Free Running Counter0 value */
    Uint32 frc0Counter;
    /** Set Up Counter0 value */
    Uint32 uc0Counter;
    /** Set Free Running Counter1 value */
    Uint32 frc1Counter;
    /** Set Up Counter1 value */
    Uint32 uc1Counter;
} CSL_RtiCounters;

/** 
 * This structure is used to set Compare register values.
 */
typedef struct {
    /** Set Compare0 value */
    Uint32 comp0Val;
    /** Set Compare1 value */
    Uint32 comp1Val;
    /** Set Compare2 value */
    Uint32 comp2Val;
    /** Set Compare3 value */
    Uint32 comp3Val;
} CSL_RtiCompareVal;

/** 
 * This structure is used to set Update Compare register values.
 */
typedef struct {
    /** Set Update Compare0 value */
    Uint32 updateComp0Val;
    /** Set Update Compare1 value */
    Uint32 updateComp1Val;
    /** Set Update Compare2 value */
    Uint32 updateComp2Val;
    /** Set Update Compare3 value */
    Uint32 updateComp3Val;
} CSL_RtiUpdateCompVal;

/** 
 * This structure is used enable/disable the DMA requests.
 */
typedef struct {
    /** Enable/Disable DMA request3 */
    Bool dmaReq3En;
    /** Enable/Disable DMA request2 */
    Bool dmaReq2En;
    /** Enable/Disable DMA request1 */
    Bool dmaReq1En;
    /** Enable/Disable DMA request0 */
    Bool dmaReq0En;
} CSL_RtiDmaReq; 

/** 
 * This structure is used enable/disable the Interrupts
 */
typedef struct {
    /** Enable/Disable Overflow Interrupt1 */
    Bool ovlIntr1En;
    /** Enable/Disable Overflow Interrupt0 */
    Bool ovlIntr0En;
    /** Enable/Disable Interrupt3 */
    Bool compIntr3En;
    /** Enable/Disable Interrupt2 */
    Bool compIntr2En;
    /** Enable/Disable Interrupt1 */
    Bool compIntr1En;
    /** Enable/Disable Interrupt0 */
    Bool compIntr0En;
} CSL_RtiIntrConfig;


/** Hardware setup structure */
typedef struct CSL_RtiHwSetup {
    /** Stop/continue counters */
    CSL_RtiContOnSuspend            contOnSuspend;
    /** Select capture event source0/capture event source1 for block0 */
    CSL_RtiExtnControl              blk0ExtnCntl;
    /** Select capture event source0/capture event source1 for block1 */
    CSL_RtiExtnControl              blk1ExtnCntl;
    /** Select compare counter FRC0/FRC1 */
    CSL_RtiCompareCntl              compare3Cntl;
    /** Select compare counter FRC0/FRC1 */
    CSL_RtiCompareCntl              compare2Cntl;
    /** Select compare counter FRC0/FRC1 */
    CSL_RtiCompareCntl              compare1Cntl;
    /** Select compare counter FRC0/FRC1 */
    CSL_RtiCompareCntl              compare0Cntl;
    /** Set value for counters */
    CSL_RtiCounters                 counters;
    /** Set values for Compare up counters */
    CSL_RtiCompUpCounter            compareUpCntrs;
    /** Set compare register values */
    CSL_RtiCompareVal               compVal;
    /** Set update compare register values */
    CSL_RtiUpdateCompVal            updateCompVal;
    /** Configuration for enable the interrupt  */
    CSL_RtiIntrConfig               intConfig;
    /** Configuration for enable dma Request */
    CSL_RtiDmaReq                   dmaReqEn;
    /** Set preload value for digital watchdog */
    Uint16                          preLoadWatchdog;
} CSL_RtiHwSetup;

/**
 * Configuration structure.This is used to configure real time interrupt
 * instance using CSL_rtiHwSetupRaw function
 */
typedef struct  {
    /** RTI Global Control Register  */
    volatile Uint32 RTIGCTRL;
    /** RTI Capture Control Register */
    volatile Uint32 RTICAPCTRL;
    /** RTI Compare Control Register */
    volatile Uint32 RTICOMPCTRL; 
    /** RTI Free Running Counter 0 Register */
    volatile Uint32 RTIFRC0;
    /** RTI Up Counter 0 Register */
    volatile Uint32 RTIUC0;
    /** RTI Compare Up Counter 0 Register */
    volatile Uint32 RTICPUC0;
    /** RTI Free Running Counter 1 Register */
    volatile Uint32 RTIFRC1;
    /** RTI Up Counter 1 Register */
    volatile Uint32 RTIUC1;
    /** RTI Compare Up Counter 1 Register */
    volatile Uint32 RTICPUC1; 
    /** RTI Compare 0 Register */
    volatile Uint32 RTICOMP0;
    /** RTI Update Compare 0 Register */
    volatile Uint32 RTIUDCP0;
    /** RTI Compare 1 Register */
    volatile Uint32 RTICOMP1;
    /** RTI Update Compare 1 Register */
    volatile Uint32 RTIUDCP1;
    /** RTI Compare 2 Register */
    volatile Uint32 RTICOMP2;
    /** RTI Update Compare 2 Register */
    volatile Uint32 RTIUDCP2;
    /** RTI Compare 3 Register */
    volatile Uint32 RTICOMP3;
    /** RTI Update Compare 3 Register */
    volatile Uint32 RTIUDCP3;
    /** RTI Set/Status Interrupt Register */
    volatile Uint32 RTISETINT;
    /** RTI Clear/Status Interrupt Register */
    volatile Uint32 RTICLEARINT;
    /** Digital Watchdog Control Register */
    volatile Uint32 RTIDWDCTRL;
    /** Digital Watchdog Preload Register */
    volatile Uint32 RTIDWDPRLD;
    /** Watchdog Key Register */
    volatile Uint32 RTIWDKEY ;
} CSL_RtiConfig;

/** Real Time Interrupt object structure */
typedef struct CSL_RtiObj {
    /** Pointer to the register overlay structure for the peripheral */
    CSL_RtiRegsOvly     regs;
    /** Specifies a particular instance of real time interrupt */
    CSL_InstNum         perNum; 
} CSL_RtiObj;

/** Default counters setup parameters   */
#define CSL_RTI_COUNTERS_SETUP_DEFAULTS { \
    0,  \
    0,  \
    0,  \
    0   \
}

/** Default compare upcounters setup parameters */
#define CSL_RTI_COMP_UPCOUNTERS_SETUP_DEFAULTS { \
    0,  \
    0   \
}

/** Default compare register setup parameters   */
#define CSL_RTI_COMPARE_SETUP_DEFAULTS { \
    0,   \
    0,   \
    0,   \
    0    \
}

/** Default update compare setup parameters */
#define CSL_RTI_UPDATE_COMPARE_SETUP_DEFAULTS { \
    0,   \
    0,   \
    0,   \
    0    \
}

/** Default interrupt config setup parameters   */
#define CSL_RTI_INT_SETUP_DEFAULTS { \
    FALSE,   \
    FALSE,   \
    FALSE,   \
    FALSE,   \
    FALSE,   \
    FALSE    \
}

/** Default dma request setup parameters    */
#define CSL_RTI_DMA_SETUP_DEFAULTS { \
    FALSE,   \
    FALSE,   \
    FALSE,   \
    FALSE    \
}

/** Default hardware setup parameters */

#define CSL_RTI_HWSETUP_DEFAULTS  {\
    CSL_RTI_COUNTERS_STOP,  \
    CSL_RTI_CAPTURE_EVENT_SOURCE0,  \
    CSL_RTI_CAPTURE_EVENT_SOURCE0,  \
    CSL_RTI_FRC0_COMPARE_ENABLE,    \
    CSL_RTI_FRC0_COMPARE_ENABLE,    \
    CSL_RTI_FRC0_COMPARE_ENABLE,    \
    CSL_RTI_FRC0_COMPARE_ENABLE,    \
    CSL_RTI_COUNTERS_SETUP_DEFAULTS, \
    CSL_RTI_COMP_UPCOUNTERS_SETUP_DEFAULTS, \
    CSL_RTI_COMPARE_SETUP_DEFAULTS, \
    CSL_RTI_UPDATE_COMPARE_SETUP_DEFAULTS,  \
    CSL_RTI_INT_SETUP_DEFAULTS, \
    CSL_RTI_DMA_SETUP_DEFAULTS, \
    0x1FFF  \
}    
   
/** Default Values for Config structure */
#define CSL_RTI_CONFIG_DEFAULTS {\
    CSL_RTI_RTIGCTRL_RESETVAL,\
    CSL_RTI_RTICAPCTRL_RESETVAL,\
    CSL_RTI_RTICOMPCTRL_RESETVAL,\
    CSL_RTI_RTIFRC0_RESETVAL,\
    CSL_RTI_RTIUC0_RESETVAL,\
    CSL_RTI_RTICPUC0_RESETVAL,\
    CSL_RTI_RTIFRC1_RESETVAL,\
    CSL_RTI_RTIUC1_RESETVAL,\
    CSL_RTI_RTICPUC1_RESETVAL,\
    CSL_RTI_RTICOMP0_RESETVAL,\
    CSL_RTI_RTIUDCP0_RESETVAL,\
    CSL_RTI_RTICOMP1_RESETVAL,\
    CSL_RTI_RTIUDCP1_RESETVAL,\
    CSL_RTI_RTICOMP2_RESETVAL,\
    CSL_RTI_RTIUDCP2_RESETVAL,\
    CSL_RTI_RTICOMP3_RESETVAL,\
    CSL_RTI_RTIUDCP3_RESETVAL,\
    CSL_RTI_RTISETINT_RESETVAL,\
    CSL_RTI_RTICLEARINT_RESETVAL,\
    CSL_RTI_RTIDWDCTRL_RESETVAL,\
    CSL_RTI_RTIDWDPRLD_RESETVAL,\
    CSL_RTI_RTIWDKEY_RESETVAL\
}

/**
 * This data type is used to return the handle to an instance of 
 * real time interrupt
 */
typedef struct CSL_RtiObj    *CSL_RtiHandle;

/*******************************************************************************
 * Real time interrupt global function declarations
 ******************************************************************************/
 
/** ============================================================================
 *   @n@b CSL_rtiInit
 *
 *   @b Description
 *   @n This is the initialization function for the real time interrupt CSL. The
 *      function must be called before calling any other API from this CSL.
 *      This function is idem-potent. Currently, the function just return status
 *      CSL_SOK, without doing anything.
 *
 *   @b Arguments
 *   @verbatim
        pContext    Pointer to module-context. As real time interrupt doesn't 
                    have any context based information user is expected to 
                    pass NULL.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Always returns
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_sysInit() must be called. 
 *
 *   <b> Post Condition </b>
 *   @n  The CSL for real time interrupt is initialized 
 *
 *   @b Modifies
 *   @n  None  
 *
 *   @b Example
 *   @verbatim
            CSL_rtiInit(NULL);
     @endverbatim
 * =============================================================================
 */
extern CSL_Status  CSL_rtiInit(
    CSL_RtiContext  *pContext 
);

/** ============================================================================
 *   @n@b CSL_rtiOpen
 *
 *   @b Description
 *   @n This function returns the handle to the Real time interrupt controller
 *      instance. This handle is passed to all other CSL APIs.
 *
 *   @b Arguments
 *   @verbatim
            rtiObj      Pointer to Real time interrupt object.
 
            rtiNum      Instance of Real time interrupt CSL to be opened. 
                        There is only one instance of the Real time 
                        interrupt available. So, the value for this  
                        parameter will be CSL_RTI always.
 
            pRtiParam   Module specific parameters.
 
            status      Status of the function call
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_RtiHandle
 *   @n                     Valid Real time interrupt handle will be returned if
 *                          status value is equal to CSL_SOK.
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n   1.    The status is returned in the status variable. If status
 *              returned is
 *   @li            CSL_SOK         Valid Real time interrupt handle is returned
 *   @li            CSL_ESYS_FAIL   The Real time interrupt instance is invalid
 *
 *        2.    Real time interrupt object structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *
 *         2. Real time interrupt object structure
 *
 *   @b Example
 *   @verbatim
            CSL_status              status;
            CSL_RtiObj              rtiObj;
            CSL_RtiHandle           hRti;

            ...
                
            hRti = CSL_rtiOpen (&rtiObj, CSL_RTI, NULL, &status);
            
            ...
    @endverbatim
 * =============================================================================
 */
extern CSL_RtiHandle CSL_rtiOpen (
    CSL_RtiObj          *pRtiObj,
    CSL_InstNum          rtiNum,
    CSL_RtiParam        *pRtiParam,
    CSL_Status          *pStatus
);

/** ============================================================================
 *   @n@b csl_rtiClose.c
 *
 *   @b Description
 *   @n This function marks that CSL for the real time interrupt instance is 
 *      closed. CSL for the real time interrupt instance need to be reopened 
 *      before using any real time interrupt CSL API.
 *      
 *   @b Arguments
 *   @verbatim
            hRti         Handle to the real time interrupt instance 
     @endverbatim
 *
 *   <b> Return Value </b> CSL_Status
 *   @li                   CSL_SOK - Real Time Interrupt is closed successfully
 *   @li                   CSL_ESYS_BADHANDLE - The handle passed is invalid
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  1. The real time interrupt CSL APIs can not be called until the real 
 *          time interrupt CSL is reopened again using CSL_rtiOpen() 
 *
 *   @b Modifies
 *   @n  None
 *            
 *   @b Example
 *   @verbatim
            CSL_RtiHandle   hRti;
            
            ...
            
            CSL_rtiClose (hRti);
            
            ...
     @endverbatim
 * =============================================================================
 */
extern CSL_Status CSL_rtiClose (
    CSL_RtiHandle hRti
);

/** ============================================================================
 *   @n@b CSL_rtiHwSetup
 *
 *   @b Description
 *   @n It configures the  Real time interrupt registers as per the values 
 *      passed in the hardware setup structure.
 *
 *   @b Arguments
 *   @verbatim
            hRti            Handle to the  Real time interrupt
 
            hwSetup         Pointer to harware setup structure
 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Hardware setup successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS  - Hardware structure is not
                                                  properly initialized
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n   Real time interrupt registers are configured according to the 
 *        hardware setup parameters
 *
 *   @b Modifies
 *   @n Real time interrupt registers
 *
 *   @b Example
 *   @verbatim
         CSL_RtiHandle              hRti;
         CSL_RtiObj                 rtiObj;
         CSL_status                 status;
         CSL_RtiHwSetup             hwSetup;
        
         ...
        
         hRti = CSL_rtiOpen (&rtiObj, CSL_RTI, NULL, &status);
        
         hwSetup.contOnSuspend                  = CSL_RTI_COUNTERS_RUN;
         hwSetup.blk0ExtnCntl                   = CSL_RTI_CAPTURE_EVENT_SOURCE0;
         hwSetup.blk1ExtnCntl                   = CSL_RTI_CAPTURE_EVENT_SOURCE0;
         hwSetup.compare0Cntl                   = CSL_RTI_FRC0_COMPARE_ENABLE;
         hwSetup.compare1Cntl                   = CSL_RTI_FRC0_COMPARE_ENABLE;
         hwSetup.compare2Cntl                   = CSL_RTI_FRC0_COMPARE_ENABLE;
         hwSetup.compare3Cntl                   = CSL_RTI_FRC0_COMPARE_ENABLE;
         hwSetup.counters.frc0Counter           = FRC0_TEST_VALUE; 
         hwSetup.counters.uc0Counter            = UC0_TEST_VALUE; 
         hwSetup.counters.frc1Counter           = FRC1_TEST_VALUE; 
         hwSetup.counters.uc1Counter            = UC1_TEST_VALUE; 
         hwSetup.compareUpCntrs.compareUpCntr0  = UC0_COMP_VALUE;
         hwSetup.compareUpCntrs.compareUpCntr1  = UC1_COMP_VALUE;
         hwSetup.compVal.comp0Val               = COMPARE0_VALUE;
         hwSetup.compVal.comp1Val               = COMPARE1_VALUE;
         hwSetup.compVal.comp2Val               = COMPARE2_VALUE;
         hwSetup.compVal.comp3Val               = COMPARE3_VALUE;   
         hwSetup.updateCompVal.updateComp0Val   = UPDATE_COMPARE0_VALUE;
         hwSetup.updateCompVal.updateComp1Val   = UPDATE_COMPARE1_VALUE;
         hwSetup.updateCompVal.updateComp2Val   = UPDATE_COMPARE2_VALUE;
         hwSetup.updateCompVal.updateComp3Val   = UPDATE_COMPARE3_VALUE;
         hwSetup.intEnable.compIntr0En          = FALSE; 
         hwSetup.intEnable.compIntr1En          = FALSE;
         hwSetup.intEnable.compIntr2En          = FALSE;
         hwSetup.intEnable.compIntr3En          = FALSE;
         hwSetup.intEnable.ovlInt0En            = FALSE;
         hwSetup.intEnable.ovlInt0En            = FALSE;
         hwSetup.dmaReq.dmareq0En               = FALSE;
         hwSetup.dmaReq.dmareq1En               = FALSE;
         hwSetup.dmaReq.dmareq2En               = FALSE;
         hwSetup.dmaReq.dmareq3En               = FALSE;
         hwSetup.preLoadWatchdog                = 0x1FFF;
 
            status = CSL_rtiHwsetup (hRti, &hwSetup);
     @endverbatim
 * =============================================================================
 */
extern CSL_Status CSL_rtiHwSetup (
    CSL_RtiHandle      hRti,
    CSL_RtiHwSetup     *hwSetup
);

/** ============================================================================
 *   @n@b CSL_rtiHwSetupRaw
 *
 *   @b Description
 *   @n This function initializes the device registers with the register-values
 *      provided through the Config Data structure.  
 *
 *   @b Arguments
 *   @verbatim
            hRti            Handle to the RTI instance
 
            config          Pointer to config structure
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Configuration successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS  - Configuration is not
 *                                                properly initialized
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The registers of the specified RTI instance will be setup 
 *       according to value passed
 *
 *   @b Modifies
 *   @n Hardware registers of the specified RTI instance
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle       hRti;
        CSL_RtiConfig       config = CSL_RTI_CONFIG_DEFAULTS;
        CSL_Status          status;

        status = CSL_rtiHwSetupRaw (hRti, &config);
            
     @endverbatim
 * =============================================================================
 */

extern CSL_Status  CSL_rtiHwSetupRaw (
    CSL_RtiHandle           hRti,
    CSL_RtiConfig           *config
);

/** ============================================================================
 *   @n@b CSL_rtiGetHwSetup
 *
 *   @b Description
 *   @n It retrives the hardware setup parameters of the real time interrupt 
 *      module specified by the given handle.
 *
 *   @b Arguments
 *   @verbatim
            hRti            Handle to the real time interrupt
 
            hwSetup         Pointer to the harware setup structure
 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK            - Retrieving the hardware setup
 *                                               parameters is successful
 *   @li                    CSL_ESYS_BADHANDLE - The handle is passed is
 *                                               invalid
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The hardware setup structure is populated with the hardware setup
 *       parameters
 *
 *   @b Modifies
 *   @n hwSetup variable
 *
 *   @b Example
 *   @verbatim
            CSL_RtiHandle   hRti;
            CSL_RtiHwSetup  hwSetup;

            ...
            
            status = CSL_rtiGetHwSetup (hRti, &hwSetup);
            
            ...
            
     @endverbatim
 * ===========================================================================
 */
extern CSL_Status CSL_rtiGetHwSetup (
    CSL_RtiHandle   hRti,
    CSL_RtiHwSetup  *hwSetup
);

/** ============================================================================
 *   @n@b CSL_rtiHwControl
 *
 *   @b Description
 *   @n This function performs various control operations on the real time 
 *      interrupt, based on the command passed.
 *
 *   @b Arguments
 *   @verbatim
            hRti         Handle to the real time interrupt
 
            cmd          Operation to be performed on the real time interrupt
 
            cmdArg       Arguement specific to the command 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK            - Command execution successful.
 *   @li                    CSL_ESYS_BADHANDLE - Invalid handle
 *   @li                    CSL_ESYS_INVCMD    - Invalid command
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Real time interrupt registers are configured according to the command 
 *       and the command arguements. The command determines which registers 
 *       are modified.
 *
 *   @b Modifies
 *   @n Real time interrupt registers determined by the command
 *
 *   @b Example
 *   @verbatim
            CSL_RtiHandle   hRti;
            Uint32          dwdCounterDis = 0x5312ACED;
            
            ...
            
            status = CSL_rtiHwControl(hRti, CSL_RTI_CMD_DWD_ENABLE, \
                            &dwdCounterDis);
            
            ...
            
     @endverbatim
 * =============================================================================
 */
extern CSL_Status CSL_rtiHwControl (
    CSL_RtiHandle           hRti,
    CSL_RtiHwControlCmd     cmd,
    void                    *arg
);

/** ============================================================================
 *   @n@b CSL_rtiGetHwStatus
 *
 *   @b Description
 *   @n This function is used to get the value of various parameters of the
 *      real time interrupt. The value returned depends on the query passed.
 *
 *   @b Arguments
 *   @verbatim
            hRti            Handle to the real time interrupt
 
            query           Query to be performed. 
 
            response        Pointer to buffer to return the data requested by
                            the query passed
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK            - Successful completion of the
 *                                               query
 *
 *   @li                    CSL_ESYS_BADHANDLE - Invalid handle
 *
 *   @li                    CSL_ESYS_INVQUERY  - Query command not supported
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *       Data requested by the query is returned through the variable "response"
 *
 *   @b Modifies
 *   @n  The input arguement "response" is modified
 *
 *   @b Example
 *   @verbatim
            Uint32       count = 0;
            CSL_Status   status;
            
            ...
            
            status = CSL_rtiGetHwStatus (hRti, CSL_RTI_QUERY_CUR_FRC0_CNT,
                                        &count);
            ...
     @endverbatim
 * =============================================================================
 */

extern CSL_Status CSL_rtiGetHwStatus (
    CSL_RtiHandle               hRti,
    CSL_RtiHwStatusQuery        query,
    void                        *response
);

/** ============================================================================
 *   @n@b CSL_rtiGetBaseAddress
 *
 *   @b Description
 *   @n  Function to get the base address of the peripheral instance.
 *       This function is used for getting the base address of the peripheral
 *       instance. This function will be called inside the CSL_rtiOpen()
 *       function call.This function is open for re-implementing if the user 
 *       wants to modify the base address of the peripheral object to point to
 *       a different location and there by allow CSL initiated write/reads into
 *       peripheral MMR's go to an alternate location. 
 *
 *   @b Arguments
 *   @verbatim      
            rtiNum          Specifies the instance of the RTI to be opened.
 
            pRtiParam       Module specific parameters.
 
            pBaseAddress    Pointer to baseaddress structure containing base 
                            address details.
            
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_OK            Open call is successful
 *   @li                    CSL_ESYS_FAIL     The instance number is invalid.
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Base Address structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *
 *         2. Base address structure is modified.
 *
 *   @b Example
 *   @verbatim
        CSL_Status              status;
        CSL_rtiBaseAddress  baseAddress;

       ...
      status = CSL_rtiGetBaseAddress(CSL_RTI, NULL, &baseAddress);

    @endverbatim
 * ===========================================================================
 */
extern CSL_Status   CSL_rtiGetBaseAddress (     
    CSL_InstNum              rtiNum,
    CSL_RtiParam            *pRtiParam,
    CSL_RtiBaseAddress      *pBaseAddress 
);

#ifdef __cplusplus
}
#endif

#endif  /* _CSL_RTI_H_ */
