/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** @file  csl_uhpi.h
 *
 *
 *  @brief UHPI functional layer API header file
 *
 */

/** @mainpage UHPI CSL 3.x
 *
 * @section Introduction
 *
 * @subsection xxx Purpose and Scope
 * The purpose of this document is to identify a set of common CSL APIs for
 * the UHPI module across various devices. The CSL developer is expected to
 * refer to this document while designing APIs for these modules. Some of the
 * listed APIs may not be applicable to a given UHPI module. While other cases
 * this list of APIs may not be sufficient to cover all the features of a
 * particular UHPI Module. The CSL developer should use his discretion designing
 * new APIs or extending the existing ones to cover these.
 *
 * @subsection Terms and Abbreviations
 *   -# CSL:  Chip Support Library
 *   -# API:  Application Programmer Interface
 *
 * @subsection References
 *    -# CSL-001-DES, CSL 3.x Design Specification DocumentVersion 1.02
 *
 */

/* =============================================================================
 *
 *  30-Jan-2005 asr File Created.
 * =============================================================================
 */

#ifndef _CSL_UHPI_H_
#define _CSL_UHPI_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <cslr.h>
#include <soc.h>
#include <csl_error.h>
#include <csl_types.h>
#include <csl.h>
#include <cslr_uhpi.h>

/** 
* Enumeration for hardware status query commands
*/

typedef enum {
	/**<
	* @brief   Query the current value of Peripheral Revision
	* @param   (Uint32 *)
	*/
	CSL_UHPI_QUERY_PID_REV      = 0,
	/**<
	* @brief   Query the current value of Peripheral Class
	* @param   (Uint32 *)
	*/
	CSL_UHPI_QUERY_PID_CLASS    = 1,
	/**<
	* @brief   Query the current value of Peripheral Type
	* @param   (Uint32 *)
	*/
	CSL_UHPI_QUERY_PID_TYPE     = 2,
	/**<
	* @brief   Query the current value of GP Interrupt Enable-0
	* @param   (Uint32 *)
	*/    
	CSL_UHPI_QUERY_GPINT_EN0    = 3,
	/**<
	* @brief   Query the current value of GP Interrupt Enable 1
	* @param   (Uint32 *)
	*/    
	CSL_UHPI_QUERY_GPINT_EN1    = 4,
	/**<
	* @brief   Query the current value of GP Interrupt Enable 2
	* @param   (Uint32 *)
	*/    
	CSL_UHPI_QUERY_GPINT_EN2    = 5,
	/**<
	* @brief   Query the current value of GPIO Direction 1
	* @param   (Uint32 *)
	*/    
	CSL_UHPI_QUERY_GPIO_DIR1    = 6,
	/**<
	* @brief   Query the current value of GPIO Direction 2
	* @param   (Uint32 *)
	*/    
	CSL_UHPI_QUERY_GPIO_DIR2    = 7,
	/**<
	* @brief   Query the current value of GPIO Direction 3
	* @param   (Uint32 *)
	*/    
	CSL_UHPI_QUERY_GPIO_DIR3    = 8,
	/**<
	* @brief   Query the current value of Host Ready
	* @param   (Uint32 *)
	*/    
	CSL_UHPI_QUERY_HRDY      	= 9,
	/**<
	* @brief   Query the current value of Host Fetch
	* @param   (Uint32 *)
	*/    
	CSL_UHPI_QUERY_FETCH    	= 10,
	/**<
	* @brief   Query the current value of HPI Reset
	* @param   (Uint32 *)
	*/    
	CSL_UHPI_QUERY_HPI_RST		= 11,
	/**<
	* @brief   Query the current value of Half-word ordering status
	* @param   (Uint32 *)
	*/    
	CSL_UHPI_QUERY_HWOB_STAT	= 12

}CSL_UhpiHwStatusQuery;

typedef enum {
	/**<
	* @brief	set the GP Interrupt
	* @param	Uint32
	*/
	CSL_UHPI_CMD_GPINT_SET           = 1, 
	/**<
	* @brief	set the GP Interrupt Invert
	* @param	Uint32
	*/    
	CSL_UHPI_CMD_SET_GPINT_INV       = 2, 
	/**<
	* @brief	set the GPIO Enable
	* @param	Uint32
	*/    
	CSL_UHPI_CMD_SET_GPIO_EN         = 3, 
	/**<
	* @brief	set the GPIO Direction-1
	* @param	Uint32
	*/    
	CSL_UHPI_CMD_SET_GPIO_DIR1       = 4, 
	/**<
	* @brief	set the GPIO Direction-2
	* @param	Uint16
	*/    
	CSL_UHPI_CMD_SET_GPIO_DIR2       = 6, 
	/**<
	* @brief	set the GPIO Direction-3
	* @param	Uint32
	*/     
	CSL_UHPI_CMD_SET_GPIO_DIR3       = 7, 
	/**<
	* @brief	Set the HPIC Host-to-DSP Interrupt.
	* @param	(None)
	*/    
	CSL_UHPI_CMD_SET_DSP_INT         = 8,
	/**<
	* @brief	Reset the HPIC Host-to-DSP Interrupt.
	* @param	(None)
	*/    
	CSL_UHPI_CMD_RESET_DSP_INT       = 9,
	/**<
	* @brief	Set the HPIC DSP-to-Host Interrupt.
	* @param	(None)
	*/    
	CSL_UHPI_CMD_SET_HINT            = 10,
	/**<
	* @brief	Reset the HPIC DSP-to-Host Interrupt.
	* @param	(None)
	*/    
	CSL_UHPI_CMD_RESET_HINT          = 11
}CSL_UhpiHwControlCmd;

/**
     @brief Structure configures Host
     Port Interface Write & Read Address 
**/ 

typedef struct {
	/** Host Port Interface Write Address **/
	Uint32     hpiaWrtAddr;
	/** Host Port Interface Read Address **/
	Uint32     hpiaReadAddr;
}CSL_UhpiAddrCfg;
 
typedef enum {
	/* Half-word Ordering Bit*/
	CSL_UHPI_HWOB       = 0x1, 
	/* Host-to-DSP Interrupt */
	CSL_UHPI_DSP_INT    = 0x2, 
	/* DSP-to-Host Interrupt */      
	CSL_UHPI_HINT       = 0x4, 
	/* Host Ready */
	CSL_UHPI_HRDY       = 0x8,   
	/* Host Fetch */
	CSL_UHPI_FETCH      = 0x10,  
	/* UHPI Extended Address */
	CSL_UHPI_XHPIA      = 0x20,  
	/* HPI Reset */
	CSL_UHPI_HPI_RST    = 0x80,
	/* Half-word ordering bit status */
	CSL_UHPI_HWOB_STAT  = 0x100,
	/* Dual HPIA mode configuration bit */
	CSL_UHPI_DUAL_HPIA  = 0x200, 
	/*HPIA register select bit */    
	CSL_UHPI_HPIA_RW_SEL= 0x800
}CSL_UhpiCtrl;            
                 
typedef enum {
	/* This DAT bit corresponds to the /HAS pin */
	CSL_UHPI_GPIO_DAT0    = 0x1,
	/* controls/statis the level of the  /HCS pin */    
	CSL_UHPI_GPIO_DAT1    = 0x2,   
	/* controls/statis the  level of the /HDS1 pin */
	CSL_UHPI_GPIO_DAT2    = 0x4,
	/* controls/status the  level of the /HDS1 pin */    
	CSL_UHPI_GPIO_DAT3    = 0x8,   
	/* Controls/status the  level of the HR/W pin */
	CSL_UHPI_GPIO_DAT4    = 0x10,  
	/* controls/status the  level of the HHWIL pin */    
	CSL_UHPI_GPIO_DAT5    = 0x20,  
	/* controls/status the  level of the HCNTLB pin */
	CSL_UHPI_GPIO_DAT6    = 0x40,  
	/* controls/status the level of the HCNTLA pin */    
	CSL_UHPI_GPIO_DAT7    = 0x80,  
	/* controls/status the level of the /HINT pin */    
	CSL_UHPI_GPIO_DAT8    = 0x100, 
	/* control/status the level of the HRDY pin */    
	CSL_UHPI_GPIO_DAT9    = 0x200, 
	/* control/status the level of the /HRDY pin */    
	CSL_UHPI_GPIO_DAT10   = 0x400, 
	/* control/status the level of the /HBE0 pin */
	CSL_UHPI_GPIO_DAT11   = 0x800, 
	/* control/status the level of the /HBE1 pin */    
	CSL_UHPI_GPIO_DAT12   = 0x1000,
	/* control/status the level of the /HBE2 pin */    
	CSL_UHPI_GPIO_DAT13   = 0x2000,
	/* control/status the level of the /HBE3 pin */    
	CSL_UHPI_GPIO_DAT14   = 0x4000
}CSL_UhpiGpioDat2;

typedef enum {
	/* control the direction of the /HAS pin */
	CSL_UHPI_GPIO_DIR0  = 0x1,    
	/* control the direction of the /HCS pin */
	CSL_UHPI_GPIO_DIR1  = 0x2,    
	/* control the direction of the /HDS1 pin */
	CSL_UHPI_GPIO_DIR2  = 0x4,    
	/* control the direction of the /HDS2 pin */
	CSL_UHPI_GPIO_DIR3  = 0x8,    
	/* control the direction of the HR/W pin */    
	CSL_UHPI_GPIO_DIR4  = 0x10,   
	/* control the direction of the HHWIL pin */    
	CSL_UHPI_GPIO_DIR5  = 0x20,   
	/* control the direction of the HCNTLB pin */    
	CSL_UHPI_GPIO_DIR6  = 0x40,   
	/* control the direction of the HCNTLA pin */    
	CSL_UHPI_GPIO_DIR7  = 0x80,   
	/* control the direction of the /HINT pin */    
	CSL_UHPI_GPIO_DIR8  = 0x100,  
	/* control the direction of the HRDY pin */    
	CSL_UHPI_GPIO_DIR9  = 0x200,  
	/* control the direction of the /HRDY pin */
	CSL_UHPI_GPIO_DIR10 = 0x400,  
	/* control the direction of the /HBE0 pin */    
	CSL_UHPI_GPIO_DIR11 = 0x800,  
	/* control the direction of the /HBE1 pin */
	CSL_UHPI_GPIO_DIR12 = 0x1000, 
	/* control the direction of the /HBE2 pin */    
	CSL_UHPI_GPIO_DIR13 = 0x2000, 
	/* control the direction of the /HBE3 pin */    
	CSL_UHPI_GPIO_DIR14 = 0x4000
}CSL_UhpiGpioDir2;

typedef struct {
	/** GPIO Direction - 1 Register **/
	Uint32              gpioData1;
	/** GPIO Direction - 2 Register **/
	CSL_UhpiGpioDat2    gpioData2;
	/** GPIO Direction - 3 Register **/
	Uint32              gpioData3;
}CSL_UhpiGpioData;
 
typedef struct {
	/** GPIO Direction - 1 Register **/
	Uint32              gpioDir1;
	/** GPIO Direction - 2 Register **/
	CSL_UhpiGpioDir2    gpioDir2;
	/** GPIO Direction - 3 Register **/
	Uint32              gpioDir3;
}CSL_UhpiGpioDir;

typedef enum {
	/* GPIO_EN0 bank includes the /HCS /HDS1 /HDS2 pins */
	CSL_UHPI_GPIO_EN0  = 0x1,
	/* GPIO_EN1 bank includes HCNTL[B:A] pins */    
	CSL_UHPI_GPIO_EN1  = 0x2,
	/* GPIO_EN2 bank includes the /HAS pin */    
	CSL_UHPI_GPIO_EN2  = 0x4,
	/* GPIO_EN3 bank includes the /HBE[3:0] */
	CSL_UHPI_GPIO_EN3  = 0x8,
	/* GPIO_EN4 bank includes the /HHWIL pin */
	CSL_UHPI_GPIO_EN4  = 0x10,
	/* GPIO_EN5 bank includes the /HRDY pin */
	CSL_UHPI_GPIO_EN5  = 0x20,
	/* GPIO_EN6 bank includes the /HINT pin */
	CSL_UHPI_GPIO_EN6  = 0x40,
	/* GPIO_EN7 bank includes the HD[7:0] pin */
	CSL_UHPI_GPIO_EN7  = 0x80,
	/* GPIO_EN8 bank includes the HD[15:8] pin */    
	CSL_UHPI_GPIO_EN8  = 0x100,
	/* GPIO_EN9 bank includes the HD[23:16] pin */    
	CSL_UHPI_GPIO_EN9  = 0x200,
	/* GPIO_EN10 bank includes the HD[31:24] pin */    
	CSL_UHPI_GPIO_EN10 = 0x400,
	/* GPIO_EN10 bank includes the HA[0:7] pin */
	CSL_UHPI_GPIO_EN11 = 0x800,
	/* GPIO_EN10 bank includes the HA[15:8] pin */    
	CSL_UHPI_GPIO_EN12 = 0x1000,
	/* GPIO_EN10 bank includes the HA[19:16] pin */    
	CSL_UHPI_GPIO_EN13 = 0x2000,
	/* GPIO_EN10 bank includes the HA[23:20] pin */    
	CSL_UHPI_GPIO_EN14 = 0x4000,
	/* GPIO_EN10 bank includes the HA[27:24] pin */    
	CSL_UHPI_GPIO_EN15 = 0x8000,
	/* GPIO_EN10 bank includes the HA[31:28] pin */    
	CSL_UHPI_GPIO_EN16 = 0x10000
}CSL_UhpiGpioEnable;    

typedef enum {
	/* GPINT_INV0 inverts the corresponding GPINT_EN0 */
	CSL_UHPI_GPINT_INV0 = 0x1,
	/* GPINT_INV1 corresponds with the HCNTLA GPINT */
	CSL_UHPI_GPINT_INV1 = 0x2,
	/* GPINT_INV2 corresponds with the /HAS GPINT */
	CSL_UHPI_GPINT_INV2 = 0x4
}CSL_UhpiGpIntInvEnable;

typedef enum {
	/* GPINT_EN0 enables HCNTLB as a GPINT sourcing DMA_WRITE_EVENT */
	CSL_UHPI_GPINT_EN0 = 0x1,
	/* GPINT_EN1 enables HCNTLA as a GPINT sourcing DMA_READ_EVENT */    
	CSL_UHPI_GPINT_EN1 = 0x2,
	/* GPINT_EN2 enables /HAS as a GPINT sourcing the CPU Interrupt */    
	CSL_UHPI_GPINT_EN2 = 0x4
}CSL_UhpiGpIntEnable;

typedef struct {
	/* GPINT control Interrupt Enable */
	CSL_UhpiGpIntEnable     gpIntEnable;
	/* GPINT control Interrupt Invert Enable */    
	CSL_UhpiGpIntInvEnable  gpIntInvEnable;
}CSL_UhpiGpIntCtrl; 
 
typedef struct {
	/** General Purpose Interrrupt Control Register **/
	CSL_UhpiGpIntCtrl   gpIntCtrl;
	/** General Purpose I/O Enable Register **/
	CSL_UhpiGpioEnable  gpioEnable;
	/** General Purpose I/O Direction Register **/
	CSL_UhpiGpioDir     gpioDir;
	/** General Purpose I/O Data Register **/
	CSL_UhpiGpioData    gpioData;
	/** Host port Interface control Register **/
	CSL_UhpiCtrl        hpiCtrl;
	/** Host port Interface Read & Write Address Register **/
	CSL_UhpiAddrCfg     hpiAddr;
	/** Emulation Mode parameter*/
	Uint32              emu;
}CSL_UhpiHwSetup;


/**************************************************************************\
* Configuration Structure 
\**************************************************************************/
typedef struct  {
	/* Power and Emulation Management Register */
	volatile Uint32 PWREMU_MGMT;
	/* General Purpose Interrupt Control Register */    
	volatile Uint32 GPINT_CTRL;
	/* GP I/O Enable Register */
	volatile Uint32 GPIO_EN;
	/* GPIO Direction 1 Register */    
	volatile Uint32 GPIO_DIR1;
	/* GPIO Data 2 Register */    
	volatile Uint32 GPIO_DAT1;
	/* GPIO Direction Register */    
	volatile Uint32 GPIO_DIR2;
	/* GPIO Data2 Register */
	volatile Uint32 GPIO_DAT2;
	/* GPIO Direction3 Register */    
	volatile Uint32 GPIO_DIR3;
	/* GPIO Data3 Register */    
	volatile Uint32 GPIO_DAT3;
	/* Host Port Interface Control Register */    
	volatile Uint32 HPIC;
	/* Host Port Interface Write Address Register */
	volatile Uint32 HPIAW;
	/* Host Port Interface Read Address Register */
	volatile Uint32 HPIAR;
} CSL_UhpiConfig;


/** Default Values for Config structure */
#define CSL_UHPI_CONFIG_DEFAULTS {\
	CSL_HPI_PWREMU_MGMT_RESETVAL,\
	CSL_HPI_GPIO_EN_RESETVAL,\
	CSL_RTI_RTICOMPCTRL_RESETVAL,\
	CSL_HPI_GPIO_DIR1_RESETVAL,\
	CSL_HPI_GPIO_DAT1_RESETVAL,\
	CSL_HPI_GPIO_DIR2_RESETVAL,\
	CSL_HPI_GPIO_DAT2_RESETVAL,\
	CSL_HPI_GPIO_DIR3_RESETVAL,\
	CSL_HPI_GPIO_DAT3_RESETVAL,\
	CSL_HPI_HPIC_RESETVAL,\
	CSL_HPI_HPIAW_RESETVAL,\
	CSL_HPI_HPIAR_RESETVAL,\
}

/** \brief Module specific context information. Present implementation of 
 *  uhpi CSL doesn't have any context information.
 */
typedef struct {
	/** Context information of uhpi CSL.
	*  The below declaration is just a place-holder for future implementation.
	*/
	Uint32  contextInfo;
} CSL_UhpiContext;

/** \brief Module specific parameters. Present implementation of uhpi CSL
 *  doesn't have any module specific parameters.
 */
typedef struct {
	/** Bit mask to be used for module specific parameters. The below
	*  declaration is just a place-holder for future implementation.
	*/
	CSL_BitMask32   flags;
} CSL_UhpiParam;

/** \brief This structure contains the base-address information for the peripheral
 *  instance of the UHPI
 */
typedef struct {
	/** Base-address of the configuration registers of the peripheral
	*/
	CSL_UhpiRegsOvly  regs;
} CSL_UhpiBaseAddress;

/** \brief UHPI object structure
 */
typedef struct CSL_UhpiObj{
	/** Pointer to the register overlay structure of the uhpi */
	CSL_UhpiRegsOvly    regs;
	/** Instance of uhpi being referred by this object  */
	CSL_InstNum         uhpiNum;
} CSL_UhpiObj;

/** \brief This data type is used to return the handle to the CSL of the UHPI
 */
typedef struct CSL_UhpiObj    *CSL_UhpiHandle;

/*******************************************************************************
 * DSP UHPI controller function declarations
 ******************************************************************************/

/** ============================================================================
 *   @n@b CSL_uhpiInit
 *
 *   @b Description
 *   @n This is the initialization function for the uhpi CSL. The function
 *      must be called before calling any other API from this CSL.  
 *      This function does not modify any registers or check status. It returns 
 *      status CSL_SOK. It has been kept for the future use.

 *
 *   @b Arguments
 *   @verbatim
        pContext    Pointer to module-context. As uhpi doesn't have
                    any context based information user is expected to pass NULL.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                        CSL_SOK - Always returns
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_sysInit() must be called. 
 *
 *   <b> Post Condition </b>
 *   @n  The CSL for uhpi is initialized
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
            CSL_uhpiInit(NULL);
     @endverbatim
 * =============================================================================
 */
extern CSL_Status  CSL_uhpiInit (
    CSL_UhpiContext * pContext
);

/** ============================================================================
 *   @n@b CSL_uhpiOpen
 *
 *   @b Description
 *   @n This function returns the handle to the DSP UHPI controller
 *      instance. This handle is passed to all other CSL APIs.
 *
 *   @b Arguments
 *   @verbatim
            pUhpiObj     Pointer to UHPI object.
 
            uhpiNum     Instance of DSP UHPI CSL to be opened. 
                        There is only one instance of the uhpi available.
                        So, the value for this parameter will be 
                        CSL_UHPI always.
 
            pUhpiParam  Module specific parameters.
 
            pStatus      Status of the function call
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_UhpiHandle
 *   @n                         Valid uhpi handle will be returned if
 *                              status value is equal to CSL_SOK.
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n   1.    The status is returned in the status variable. If status
 *              returned is
 *   @li            CSL_SOK             Valid uhpi handle is returned
 *   @li            CSL_ESYS_FAIL       The uhpi instance is invalid
 *   @li            CSL_ESYS_INVPARAMS  Invalid parameter
 *
 *        2.    UHPI object structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *
 *         2. UHPI object structure
 *
 *   @b Example
 *   @verbatim
            CSL_status           status;
            CSL_UhpiObj          uhpiObj;
            CSL_UhpiHandle       hUhpi;

            ...
                
            hUhpi = CSL_uhpiOpen(&uhpiObj, CSL_UHPI, NULL, &status);
            
            ...
    @endverbatim
 * =============================================================================
 */
extern CSL_UhpiHandle CSL_uhpiOpen (
    CSL_UhpiObj         *pUhpiObj,
    CSL_InstNum         uhpiNum,
    CSL_UhpiParam       *pUhpiParam,
    CSL_Status          *pStatus
);

/** ============================================================================
 *   @n@b CSL_uhpiGetBaseAddress
 *
 *   @b Description
 *   @n  Function to get the base address of the peripheral instance.
 *       This function is used for getting the base address of the peripheral
 *       instance. This function will be called inside the CSL_uhpiOpen()
 *       function call. This function is open for re-implementing if the user
 *       wants to modify the base address of the peripheral object to point to
 *       a different location and there by allow CSL initiated write/reads into
 *       peripheral. MMR's go to an alternate location.
 *
 *   @b Arguments
 *   @verbatim
            uhpiNum         Specifies the instance of the uhpi to be opened.

            pUhpiParam      Module specific parameters.

            pBaseAddress    Pointer to base address structure containing base
                            address details.

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_OK              Open call is successful
 *   @li                    CSL_ESYS_FAIL       The instance number is invalid.
 *   @li                    CSL_ESYS_INVPARAMS  Invalid parameter 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Base address structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *
 *         2. Base address structure is modified.
 *
 *   @b Example
 *   @verbatim
        CSL_Status              status;
        CSL_UhpiBaseAddress  baseAddress;

        ...

        status = CSL_uhpiGetBaseAddress(CSL_UHPI_PER_CNT, NULL,
                                           &baseAddress);
        ...

    @endverbatim
 * ===========================================================================
 */
extern CSL_Status CSL_uhpiGetBaseAddress (
    CSL_InstNum              uhpiNum,
    CSL_UhpiParam *          pUhpiParam,
    CSL_UhpiBaseAddress *    pBaseAddress
);

/** =============================================================================
 *   @n@b CSL_uhpiClose
 *
 *   @b Description
 *   @n This function closes the specified instance of UHPI.
 *
 *   @b Arguments
 *   @verbatim
            hUhpi            Handle to the uhpi

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Close successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *
 *   @b Example
 *   @verbatim
            CSL_UhpiHandle   hUhpi;
            CSL_status          status;

            ...

            status = CSL_uhpiClose(hUhpi, &hwSetup);
     @endverbatim
 * ===========================================================================
 */
extern CSL_Status  CSL_uhpiClose (
    CSL_UhpiHandle    hUhpi
);

/** ============================================================================
 *   @n@b CSL_uhpiHwSetup
 *
 *   @b Description
 *   @n It configures the uhpi registers as uhpi the values passed
 *      in the hardware setup structure.
 *
 *   @b Arguments
 *   @verbatim
            hUhpi            Handle to the uhpi
 
            hwSetup             Pointer to harware setup structure
 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Hardware setup successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS  - Hardware structure is not
                                                  properly initialized
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  DSP UHPI controller registers are configured according 
 *       to the hardware setup parameters
 *
 *   @b Modifies
 *   @n ARM PLL controller registers
 *
 *   @b Example
 *   @verbatim
            CSL_UhpiHandle   hUhpi;
            CSL_UhpiObj      uhpiObj;
            CSL_UhpiHwSetup  hwSetup;
            CSL_status          status;

            ...
            
            hUhpi = CSL_uhpiOpen(&uhpiObj, CSL_UHPI, NULL, &status);            
                      
            status = CSL_uhpiHwSetup(hUhpi, &hwSetup);
     @endverbatim
 * ===========================================================================
 */
extern CSL_Status  CSL_uhpiHwSetup(
    CSL_UhpiHandle       hUhpi,
    CSL_UhpiHwSetup     *hwSetup
);

/** ============================================================================
 *   @n@b CSL_uhpiHwControl
 *
 *   @b Description
 *   @n Takes a command of UHPI with an optional argument & implements it.
 *
 *   @b Arguments
 *   @verbatim
            hUhpi        Handle to the UHPI instance

            cmd             The command to this API indicates the action to be
                            taken on UHPI.

            arg             An optional argument.

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li            CSL_SOK               - Status info return successful.
 *   @li            CSL_ESYS_BADHANDLE    - Invalid handle
 *   @li            CSL_ESYS_INVCMD       - Invalid command
 *   @li            CSL_ESYS_NOTSUPPORTED - Action not supported.
                                            (system not in bypass mode. if the 
                                             mode is not bypass mode & the 
                                             developer tries to write into
                                             D0 or PLLM registers or tries to 
                                             set the uhpi in powerdown state
                                             then this error status is returned)
 *   @li            CSL_ESYS_INVPARAMS    - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n The hardware registers of UHPI.
 *
 *   @b Example
 *   @verbatim
        CSL_UhpiHandle         huhpi;
        CSL_UhpiHwControlCmd   cmd;
        void                      arg;

        status = CSL_uhpiHwControl (hUhpi, cmd, &arg);

     @endverbatim
 * =============================================================================
 */
extern CSL_Status  CSL_uhpiHwControl(
    CSL_UhpiHandle          hUhpi,
    CSL_UhpiHwControlCmd    cmd,
    void                    *cmdArg
);

/** ============================================================================
 *   @n@b CSL_uhpiGetHwStatus
 *
 *   @b Description
 *   @n Gets the status of the different operations of UHPI.
 *
 *   @b Arguments
 *   @verbatim
            hUhpi      Handle to the UHPI instance

            query         The query to this API of UHPI which indicates the
                          status to be returned.

            response      Placeholder to return the status.

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Status info return successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVQUERY   - Invalid query command
 *   @li                    CSL_ESYS_INVPARAMS  - Invalid parameter
 *  
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_UhpiHandle          hUhpi;
        CSL_uhpiHwStatusQuery   query;
        void                       reponse;

        status = CSL_uhpiGetHwStatus (hUhpi, query, &response);

     @endverbatim
 * =============================================================================
 */
extern CSL_Status  CSL_uhpiGetHwStatus (
    CSL_UhpiHandle           hUhpi,
    CSL_UhpiHwStatusQuery    query,
    void                     *response
);

/** ============================================================================
 *   @n@b CSL_uhpiHwSetupRaw
 *
 *   @b Description
 *   @n This function initializes the device registers with the register-values
 *      provided through the Config Data structure.
 *
 *   @b Arguments
 *   @verbatim
            hUhpi        Handle to the UHPI instance

            config          Pointer to config structure
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li            CSL_SOK               - Configuration successful
 *   @li            CSL_ESYS_BADHANDLE    - Invalid handle
 *   @li            CSL_ESYS_INVPARAMS    - Configuration is not
                                            properly initialized
 *   @li            CSL_ESYS_NOTSUPPORTED - Action not supported.
                                            (system not in bypass mode. if the 
                                             mode is not bypass mode & the 
                                             developer tries to write into
                                             D0 or PLLM registers or tries to 
                                             set the uhpi in powerdown state
                                             then this error status is returned)
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The registers of the specified UHPI instance will be setup
 *       according to input configuration structure values.
 *
 *   @b Modifies
 *   @n Hardware registers of the specified UHPI instance.
 *
 *   @b Example
 *   @verbatim
        CSL_UhpiHandle       hUhpi;
        CSL_UhpiConfig       config = CSL_UHPI_CONFIG_DEFAULTS;
        CSL_Status              status;

        status = CSL_uhpiHwSetupRaw (hUhpi, &config);

     @endverbatim
 * =============================================================================
 */
extern CSL_Status  CSL_uhpiHwSetupRaw (
    CSL_UhpiHandle           hUhpi,
    CSL_UhpiConfig          *config
);

/** ============================================================================
 *   @n@b CSL_uhpiGetHwSetup
 *
 *   @b Description
 *   @n It retrives the hardware setup parameters of the uhpi
 *      specified by the given handle.
 *
 *   @b Arguments
 *   @verbatim
            hUhpi        Handle to the uhpi
 
            hwSetup         Pointer to the harware setup structure
 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Retrieving the hardware setup
 *                                                parameters is successful
 *   @li                    CSL_ESYS_BADHANDLE  - The handle is passed is
 *                                                invalid
 *   @li                    CSL_ESYS_INVPARAMS  - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The hardware setup structure is populated with the hardware setup
 *       parameters
 *
 *   @b Modifies
 *   @n hwSetup variable
 *
 *   @b Example
 *   @verbatim
            CSL_UhpiHandle   hUhpi;
            CSL_UhpiHwSetup  hwSetup;

            ...
            
            status = CSL_uhpiGetHwSetup(hUhpi, &hwSetup);
            
            ...
            
     @endverbatim
 * ===========================================================================
 */
extern CSL_Status  CSL_uhpiGetHwSetup(
    CSL_UhpiHandle       hUhpi,
    CSL_UhpiHwSetup     *hwSetup
);

#ifdef __cplusplus
}
#endif

#endif  /* _CSL_UHPI_H_ */
