/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2005
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** @file csl_dmaxAux.h
 *
 * @brief   Header file for functional layer of CSL
 *
 * Description
 *    - The defines inline function definitions
 *
 *  Modification 1
 *    - Modified on: 11/Mar/2005
 *    - Reason: created the sources
 * 
 *  @author asr.
 */
 


#ifndef _CSL_DMAXAUX_H_
#define _CSL_DMAXAUX_H_

#include <csl_dmax.h>

#ifdef __cplusplus
extern "C" {
#endif

extern far cregister volatile unsigned int IFR;
extern far cregister volatile unsigned int OUT;

#define CSL_DMAX_CPUINT_REG      IFR

/**
 *  Control Functions of dmax.
 */
/** ============================================================================
 *   @n@b CSL_dmaxSetPriority
 *
 *   @b Description
 *   @n Sets the appropriate bit in the EHPR or ELPR registers to set priority
 *      for the event.
 *   @b Arguments
 *   @verbatim

            hDmax            Handle to DMAX instance

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxEnable (hDmax);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxSetPriority (
    CSL_DmaxHandle        hDmax,
    void                 *arg
)
{
	Uint16 	flag;
	Uint32 	mask;

	flag = hDmax->eventUid & 0xFFFF;
	mask = 1u << flag;

	if ((*(Int *)arg) == CSL_DMAX_EVENT_HI_PRIORITY) {
		hDmax->regs->DEHPR = mask;
	} else {
		hDmax->regs->DELPR = mask;
	}
}

/** ============================================================================
 *   @n@b CSL_dmaxSetPolarity
 *
 *   @b Description
 *   @n Sets the appropriate bit in the EHPR or ELPR registers to set priority
 *      for the event.
 *   @b Arguments
 *   @verbatim

            hDmax            Handle to DMAX instance

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxSetPolarity (hDmax);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxSetPolarity (
    CSL_DmaxHandle        hDmax,
    void                 *arg
)
{
	Uint16 flag;
	Uint32 mask;

	flag = hDmax->eventUid & 0xFFFF;
	mask = ((*(Uint32 *)arg) == CSL_DMAX_POLARITY_RISING_EDGE) << flag;
	hDmax->regs->DEPR = mask;
}

/** ============================================================================
 *   @n@b CSL_dmaxEventEnable
 *
 *   @b Description
 *   @n Enables the event..
 *
 *   @b Arguments
 *   @verbatim

            hDmax            Handle to DMAX instance

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxEventEnable (hDmax);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxEventEnable (
    CSL_DmaxHandle        hDmax
)
{
	Uint16 	flag;
	Uint32 	mask;

	flag = hDmax->eventUid & 0xFFFF;
	mask = 1u << flag;
	hDmax->regs->DEER = mask;
}

/** ============================================================================
 *   @n@b CSL_dmaxClearTcc
 *
 *   @b Description
 *   @n Clears TCC bit for this event.
 *
 *   @b Arguments
 *   @verbatim

            hDmax            Handle to DMAX instance

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxClearTcc (hDmax);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxClearTcc (
    CSL_DmaxHandle        hDmax
)
{
	Uint16 	flag;
	Uint32 	tmpReg;

	if (*(hDmax->hiTableEventEntryPtr) != 0) {
		tmpReg = *(hDmax->hiTableEventEntryPtr);
		flag = CSL_FEXT(tmpReg, DMAX_EVENT0_TCC);
		flag = 1u << flag;
		if (flag <= CSL_DMAX_TCC_SPLIT) {
			hDmax->regs->DTCR0=flag;
		} else {
			flag = flag >> 8;
			hDmax->regs->DTCR1=flag;
		}
	} else if (*(hDmax->loTableEventEntryPtr) != 0){
		tmpReg = *(hDmax->loTableEventEntryPtr);
		flag = CSL_FEXT(tmpReg, DMAX_EVENT0_TCC);		
		flag = 1u << flag;
		if (flag <= CSL_DMAX_TCC_SPLIT) {
			hDmax->regs->DTCR0=flag;
		} else {
			flag = flag >> 8;
			hDmax->regs->DTCR1=flag;
		}
	}
}

/** ============================================================================
 *   @n@b CSL_dmaxEventDisable
 *
 *   @b Description
 *   @n Disables the event.
 *
 *   @b Arguments
 *   @verbatim

            hDmax            Handle to DMAX instance

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxEventDisable (hDmax);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxEventDisable (
    CSL_DmaxHandle        hDmax
)
{
	Uint16 	flag;
	Uint32 	mask;

	flag = hDmax->eventUid & 0xFFFF;
	mask = 1u << flag;
	hDmax->regs->DEDR = mask;
}

/** ============================================================================
 *   @n@b CSL_dmaxClearEventEntry
 *
 *   @b Description
 *   @n Clears event entry for this event.
 *
 *   @b Arguments
 *   @verbatim
 *
 *         hDmax           Handle to DMAX instance
 *
 *   @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxClearEventEntry (hDmax);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxClearEventEntry (
    CSL_DmaxHandle        hDmax
)
{
	hDmax->hiTableEventEntryPtr = 0;
	hDmax->loTableEventEntryPtr = 0;
}

/** ============================================================================
 *   @n@b CSL_dmaxClearParameterEntry
 *
 *   @b Description
 *   @n Clears parameter table entry for this event.
 *
 *   @b Arguments
 *   @verbatim

            hDmax            Handle to DMAX instance

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxClearParameterEntry (hDmax);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxClearParameterEntry (
    CSL_DmaxHandle        hDmax
)
{
	CSL_DmaxParameterEntry *tmpPtr;
	
        if (*(hDmax->hiTableEventEntryPtr) != 0) {
	        tmpPtr = (CSL_DmaxParameterEntry*)
	                 (&(hDmax->regs->HiMaxParam));
        } else {
	        tmpPtr = (CSL_DmaxParameterEntry*)
	                 (&(hDmax->regs->LoMaxParam));
        }
	/*
	* :KLUDGE: KSB Nov 13,2004
	* For dmax Event Entries the parameter table offset is expressed
	* as the number of dmax word offsets from the top of the Parameter
	* Table RAM. So address caluclation is done in bytes first and
	* then cast to entry type.
	*/
	tmpPtr = (CSL_DmaxParameterEntry *)(((Uint8 *)tmpPtr) 
	     + ((hDmax->paramPtr) * CSL_DMAX_WORD_SIZE));
	tmpPtr->word0 = 0;
	tmpPtr->word1 = 0;
	tmpPtr->word2 = 0;
	tmpPtr->word3 = 0;
	tmpPtr->word4 = 0;
	tmpPtr->word5 = 0;
	tmpPtr->word6 = 0;
	tmpPtr->word7 = 0;
	tmpPtr->word8 = 0;
	tmpPtr->word9 = 0;
	tmpPtr->word10 = 0;
}

/** ============================================================================
 *   @n@b CSL_dmaxStartAsyncTransfer
 *
 *   @b Description
 *   @n This function initiates a dmax data transfer by toggling the
 *      appropriate bit in the ESR (Event Set Register) register.
 *
 *   @b Arguments
 *   @verbatim

            hDmax           Handle to Dmax instance

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxStartAsyncTransfer (hDmax);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxStartAsyncTransfer (
    CSL_DmaxHandle        hDmax
)
{
	Uint16 		flag;
	Uint32		mask;
	Uint32 		polarity;

	flag = ((hDmax->eventUid) >> 16) & 0x00FF;
	if (flag != CSL_DMAX_BAD_ESR_FLAG) {
		mask = 1u << flag;
		polarity = (hDmax->regs->DEPR) & (1u << (hDmax->eventUid & 0xFF));

		if (polarity != 0) {
			CSL_dmaxSetDetr(0);
			CSL_dmaxSetDetr(mask);
		} else {
			CSL_dmaxSetDetr(mask);
			CSL_dmaxSetDetr(0);
		}                
	}
}

/** ============================================================================
 *   @n@b CSL_dmaxClearFifoStatus
 *
 *   @b Description
 *   @n Clears FIFO Error code (EFIELD) in FIFO descriptor.
 *
 *   @b Arguments
 *   @verbatim

            hDmax            Handle to DMAX instance

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxClearFifoStatus (hDmax);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxClearFifoStatus (
    CSL_DmaxHandle        hDmax
)
{
	Uint16 				etype;
	Uint32 				eventCtl;
	Uint32 				*tmpPtr;
	CSL_DmaxFifoDesc 	*fifoObj;

	if (*(hDmax->hiTableEventEntryPtr) != 0) {
		eventCtl = *(hDmax->hiTableEventEntryPtr);
		tmpPtr = (Uint32 *)(hDmax->regs->HiMaxParam);
	} else {
		eventCtl = *(hDmax->loTableEventEntryPtr);
		tmpPtr = (Uint32 *)(hDmax->regs->LoMaxParam);            
	}
	etype = CSL_FEXT(eventCtl, DMAX_EVENT0_ETYPE);
	/* Make sure we have a valid FIFO event */          
	if ((etype == CSL_DMAX_EVENT0_ETYPE_FIFOREAD) 
		|| (etype == CSL_DMAX_EVENT0_ETYPE_FIFOWRITE)){
		tmpPtr += (hDmax->paramPtr) * CSL_DMAX_WORD_SIZE;
		fifoObj = (CSL_DmaxFifoDesc *)(((CSL_DmaxFifoParam *)tmpPtr)->pfd);
		fifoObj->efield = 0x00000000u;
	}
}

/** ============================================================================
 *   @n@b CSL_dmaxWatermarkEnable
 *
 *   @b Description
 *   @n Sets the the Watermark enable bit in the event entry associated
 *      with this event.
 *   @b Arguments
 *   @verbatim

            hDmax            Handle to DMAX instance

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxWatermarkEnable (hDmax);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxWatermarkEnable (
    CSL_DmaxHandle        hDmax
)
{
	Uint32 	tmpEv;

	if (*(hDmax->hiTableEventEntryPtr) != 0) {
		tmpEv = *(hDmax->hiTableEventEntryPtr);
		tmpEv |= CSL_DMAX_EVENT0_WMENAB_MASK;
		*(hDmax->hiTableEventEntryPtr) = tmpEv;
	} else {
		tmpEv = *(hDmax->loTableEventEntryPtr);
		tmpEv |= CSL_DMAX_EVENT0_WMENAB_MASK;
		*(hDmax->loTableEventEntryPtr) = tmpEv;
	}
}

/** ============================================================================
 *   @n@b CSL_dmaxWatermarkDisable
 *
 *   @b Description
 *   @n Clears the the Watermark enable bit in the event entry associated
 *      with this event.
 *   @b Arguments
 *   @verbatim

            hDmax            Handle to DMAX instance

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxWatermarkDisable (hDmax);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxWatermarkDisable (
    CSL_DmaxHandle        hDmax
)
{
	Uint32 	tmpEv;

	if (*(hDmax->hiTableEventEntryPtr) != 0) {
		tmpEv = *(hDmax->hiTableEventEntryPtr);
		tmpEv &= (~CSL_DMAX_EVENT0_WMENAB_MASK);
		*(hDmax->hiTableEventEntryPtr) = tmpEv;
	} else {
		tmpEv = *(hDmax->loTableEventEntryPtr);
		tmpEv &= (~CSL_DMAX_EVENT0_WMENAB_MASK);
		*(hDmax->loTableEventEntryPtr) = tmpEv;
	}
}

/** ============================================================================
 *   @n@b CSL_dmaxGetEventFlag
 *
 *   @b Description
 *   @n Returns status of event flag in EFR register.
 *
 *   @b Arguments
 *   @verbatim

            hDmax           Handle to DMAX instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxGetEventFlag (hDMAX, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxGetEventFlag (
    CSL_DmaxHandle        hDmax,
    void                 *arg
)
{
	Uint8 flag;
	Uint32 mask;
	
	flag = (hDmax->eventUid) & 0xFFFFu;
	mask = 1u << flag;
	*((Uint32 *)arg) = (((hDmax->regs->DEFR) & mask) >> flag) & 0x1u;
}

/** ============================================================================
 *   @n@b CSL_dmaxGetTcc
 *
 *   @b Description
 *   @n Returns the right-shifted value of the TCC bit associated with this
 *      event.
 *
 *   @b Arguments
 *   @verbatim

            hDmax           Handle to DMAX instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxGetTcc (hDmax, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxGetTcc (
    CSL_DmaxHandle        hDmax,
    void                 *response
)
{
	Uint16 		flag;
	Uint32 		tmpReg;

	if (*(hDmax->hiTableEventEntryPtr) != 0) {
		tmpReg = *(hDmax->hiTableEventEntryPtr);
	} else if(*(hDmax->loTableEventEntryPtr) != 0){
		tmpReg = *(hDmax->loTableEventEntryPtr);
	}
	flag = CSL_FEXT(tmpReg, DMAX_EVENT0_TCC);
	flag = 1u << flag;
	if (flag <= CSL_DMAX_TCC_SPLIT) {
		tmpReg = hDmax->regs->DTCR0;
	} else {
		flag = flag >> 8;
		tmpReg = hDmax->regs->DTCR1;
	}
	tmpReg &= flag;
	*((Uint32 *)response) =(Uint32) tmpReg;
}

/** ============================================================================
 *   @n@b CSL_dmaxGetTc
 *
 *   @b Description
 *   @n Returns right-shifted value of transfer complete flag.
 *
 *   @b Arguments
 *   @verbatim

            hDmax           Handle to DMAX instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxGetTc (hDmax, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxGetTc (
    CSL_DmaxHandle        hDmax,
    void                 *response
)
{
	Uint32 tmpReg;

	hDmax = hDmax;
	tmpReg = CSL_DMAX_CPUINT_REG;
	*((Uint32 *)response) = CSL_FEXT(tmpReg , DMAX_CPUINT8);
}

/** ============================================================================
 *   @n@b CSL_dmaxGetFmsc
 *
 *   @b Description
 *   @n Returns the value of FIFO Full mark condition flag.
 *
 *   @b Arguments
 *   @verbatim

            hDmax           Handle to DMAX instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxGetFmsc (hDmax, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxGetFmsc (
    CSL_DmaxHandle        hDmax,
    void                 *response
)
{
	Uint16 				flag;
	Uint16 				etype;
	Uint32 				eventCtl;
	Uint32 				mask;
	Uint32 				tmpReg;
	Uint32 				*tmpPtr;
	CSL_DmaxFifoDesc 	*fifoObj;


	if (*(hDmax->hiTableEventEntryPtr) != 0) {
		eventCtl = *(hDmax->hiTableEventEntryPtr);
		tmpPtr = (Uint32 *)(hDmax->regs->HiMaxParam);
	} else {
		eventCtl = *(hDmax->loTableEventEntryPtr);
		tmpPtr = (Uint32 *)(hDmax->regs->LoMaxParam);            
	}
	etype = CSL_FEXT(eventCtl, DMAX_EVENT0_ETYPE);
	/* Make sure we have a valid FIFO event */          
	if ((etype == CSL_DMAX_EVENT0_ETYPE_FIFOREAD) 
		|| (etype == CSL_DMAX_EVENT0_ETYPE_FIFOWRITE)){
		tmpPtr += (hDmax->paramPtr) * CSL_DMAX_WORD_SIZE;
		fifoObj = (CSL_DmaxFifoDesc *)(((CSL_DmaxFifoParam *)tmpPtr)->pfd);
		flag = CSL_FEXT(fifoObj->fmark, DMAX_FIFODESC_FMSC);
		mask = 1u << flag;
		if (mask <= CSL_DMAX_TCC_SPLIT) {
		       tmpReg = hDmax->regs->DFSR0;
		} else {
		       mask = mask >> 8;
		       flag = flag - 8;
		       tmpReg = hDmax->regs->DFSR1;
		}
		*((Uint32 *)response) = ((tmpReg & mask) >> flag) & 0x1u;  
	}
}


/** ============================================================================
 *   @n@b CSL_dmaxGetEmsc
 *
 *   @b Description
 *   @n Returns the value of FIFO Empty mark condition flag.
 *
 *   @b Arguments
 *   @verbatim

            hDmax           Handle to DMAX instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxGetEmsc (hDmax, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxGetEmsc (
    CSL_DmaxHandle        hDmax,
    void                 *response
)
{
	Uint16 flag;
	Uint16 etype;
	Uint32 mask;
	Uint32 eventCtl;
	Uint32 tmpReg;
	Uint32 *tmpPtr;
	CSL_DmaxFifoDesc *fifoObj;
	
        if (*(hDmax->hiTableEventEntryPtr) != 0) {
		eventCtl = *(hDmax->hiTableEventEntryPtr);
		tmpPtr = (Uint32 *)(hDmax->regs->HiMaxParam);
        } else {
		eventCtl = *(hDmax->loTableEventEntryPtr);
		tmpPtr = (Uint32 *)(hDmax->regs->LoMaxParam);            
        }
        etype = CSL_FEXT(eventCtl, DMAX_EVENT0_ETYPE);
        /* Make sure we have a valid FIFO event */          
        if ((etype == CSL_DMAX_EVENT0_ETYPE_FIFOREAD) 
		|| (etype == CSL_DMAX_EVENT0_ETYPE_FIFOWRITE)){
		tmpPtr += (hDmax->paramPtr) * CSL_DMAX_WORD_SIZE;
		fifoObj = (CSL_DmaxFifoDesc *)(((CSL_DmaxFifoParam *)tmpPtr)->pfd);
		flag = CSL_FEXT(fifoObj->emark, DMAX_FIFODESC_EMSC);
		mask = 1u << flag;
		if (mask <= CSL_DMAX_TCC_SPLIT) {
			tmpReg = hDmax->regs->DFSR0;
		} else {
			mask = mask >> 8;
			flag = flag - 8;
			tmpReg = hDmax->regs->DFSR1;
		}
		*((Uint32 *)response) = ((tmpReg & mask) >> flag) & 0x1u;  
       }
}


/** ============================================================================
 *   @n@b CSL_dmaxGetFifoError
 *
 *   @b Description
 *   @n Checks INT7 for FIFO Error.
 *
 *   @b Arguments
 *   @verbatim

            hDmax           Handle to DMAX instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxGetFifoError (hDmax, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxGetFifoError (
    CSL_DmaxHandle        hDmax,
    void                 *response
)
{
	Uint32 tmpReg;
	hDmax = hDmax;
	tmpReg = CSL_DMAX_CPUINT_REG;
	*((Uint32 *)response) = CSL_FEXT(tmpReg, DMAX_ESR_FIFOERR);
}

/** ============================================================================
 *   @n@b CSL_dmaxGetFifoErrorCode
 *
 *   @b Description
 *   @n Returns the value of the FIFO Error code (EFIELD) in FIFO descriptor.
 *
 *   @b Arguments
 *   @verbatim

            hDmax           Handle to DMAX instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxGetFifoErrorCode (hDmax, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxGetFifoErrorCode (
    CSL_DmaxHandle        hDmax,
    void                 *response
)
{
	Uint16 etype;
	Uint32 eventCtl;
	Uint32 *tmpPtr;
	CSL_DmaxFifoDesc *fifoObj;

	if (*(hDmax->hiTableEventEntryPtr) != 0) {
		eventCtl = *(hDmax->hiTableEventEntryPtr);
		tmpPtr = (Uint32 *)(hDmax->regs->HiMaxParam);
	} else {
		eventCtl = *(hDmax->loTableEventEntryPtr);
		tmpPtr = (Uint32 *)(hDmax->regs->LoMaxParam);            
	}
	etype = CSL_FEXT(eventCtl,DMAX_EVENT0_ETYPE);
	/* Make sure we have a valid FIFO event */          
	if ((etype == CSL_DMAX_EVENT0_ETYPE_FIFOREAD) 
		|| (etype == CSL_DMAX_EVENT0_ETYPE_FIFOWRITE)){
		tmpPtr += (hDmax->paramPtr) * CSL_DMAX_WORD_SIZE;
		fifoObj = (CSL_DmaxFifoDesc *)(((CSL_DmaxFifoParam *)tmpPtr)->pfd);
		*((Uint32 *)response) = CSL_FEXT(fifoObj->efield, DMAX_FIFODESC_EFIELD) \
													 & 0x000FFFFFu;
	}
}

/** ============================================================================
 *   @n@b CSL_dmaxGetEventEntry
 *
 *   @b Description
 *   @n Returns current contents of Event Entry for this event.
 *
 *   @b Arguments
 *   @verbatim

            hDmax           Handle to DMAX instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxGetEventEntry (hDmax, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxGetEventEntry (
    CSL_DmaxHandle        hDmax,
    void                 *response
)
{
	if (*(hDmax->hiTableEventEntryPtr) != 0) {
		*((Uint32 *)response) = *(Uint32 *)(hDmax->hiTableEventEntryPtr);
	} else {
		*((Uint32 *)response) = *(Uint32 *)(hDmax->loTableEventEntryPtr);
	}          
}

/** ============================================================================
 *   @n@b CSL_dmaxGetParameterEntry
 *
 *   @b Description
 *   @n Clears parameter table entry for this event.
 *
 *   @b Arguments
 *   @verbatim

            hDmax           Handle to DMAX instance

            paramObj        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxGetParameterEntry (hDmax, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxGetParameterEntry (
    CSL_DmaxHandle        		hDmax,
    CSL_DmaxParameterEntry   *paramObj
)
{
	CSL_DmaxParameterEntry *tmpPtr;

	if (*(hDmax->hiTableEventEntryPtr) != 0) {
		tmpPtr = (CSL_DmaxParameterEntry*)
			(&(hDmax->regs->HiMaxParam));
	} else {
		tmpPtr = (CSL_DmaxParameterEntry*)
			(&(hDmax->regs->HiMaxParam));
	}
	/*
	* :KLUDGE: KSB Nov 13,2004
	* For dmax Event Entries the parameter table offset is expressed
	* as the number of dmax word offsets from the top of the Parameter
	* Table RAM. So address caluclation is done in bytes first and
	* then cast to entry type.
	*/
	tmpPtr = (CSL_DmaxParameterEntry *)(((Uint8 *)tmpPtr) 
		+ ((hDmax->paramPtr) * CSL_DMAX_WORD_SIZE));
	paramObj->word0 = tmpPtr->word0;
	paramObj->word1 = tmpPtr->word1;
	paramObj->word2 = tmpPtr->word2;
	paramObj->word3 = tmpPtr->word3;
	paramObj->word4 = tmpPtr->word4;
	paramObj->word5 = tmpPtr->word5;
	paramObj->word6 = tmpPtr->word6;
	paramObj->word7 = tmpPtr->word7;
	paramObj->word8 = tmpPtr->word8;
	paramObj->word9 = tmpPtr->word9;
	paramObj->word10 = tmpPtr->word10;
}

/** ============================================================================
 *   @n@b CSL_dmaxGetHbc
 *
 *   @b Description
 *   @n Returns the current status bits for Hi Priority dMax
 *
 *   @b Arguments
 *   @verbatim

            hDmax           Handle to Dmax instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxGetHbc (hDmax, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxGetHbc (
    CSL_DmaxHandle        hDmax,
    void                 *response
)
{
	hDmax = hDmax;
	*((Uint32 *)response) = CSL_FEXT(CSL_dmaxGetDesr(), DMAX_DESR_HBC);	
}

/** ============================================================================
 *   @n@b CSL_dmaxGetLbc
 *
 *   @b Description
 *   @n Returns the current status bits for Lo Priority dmax.
 *
 *   @b Arguments
 *   @verbatim

            hDmax           Handle to DMAX instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxGetLbc (hDmax, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxGetLbc (
    CSL_DmaxHandle        hDmax,
    void                 *response
)
{
	hDmax = hDmax;
	*((Uint32 *)response) = CSL_FEXT(CSL_dmaxGetDesr(), DMAX_DESR_LBC);	       
}

/** ============================================================================
 *   @n@b CSL_dmaxGetFifoAddr
 *
 *   @b Description
 *   @n Returns the address of the Fifo Descriptor associated with this event.
 *
 *   @b Arguments
 *   @verbatim

            hDmax           Handle to DMAX instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxGetFifoAddr (hDmax, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxGetFifoAddr (
    CSL_DmaxHandle        hDmax,
    void                 *response
)
{
	Uint32 eventCtl;
	Uint16 etype;
	Uint32 *tmpPtr;
	CSL_DmaxFifoDesc *fifoObj;

	if (*(hDmax->hiTableEventEntryPtr) != 0) {
		eventCtl = *(hDmax->hiTableEventEntryPtr);
		tmpPtr = (Uint32 *)(hDmax->regs->HiMaxParam);
	} else {
		eventCtl = *(hDmax->loTableEventEntryPtr);
		tmpPtr = (Uint32 *)(hDmax->regs->LoMaxParam);            
	}
	etype = CSL_FEXT(eventCtl,DMAX_EVENT0_ETYPE);
	/* Make sure we have a valid FIFO event */          
	if ((etype == CSL_DMAX_EVENT0_ETYPE_FIFOREAD) 
		|| (etype == CSL_DMAX_EVENT0_ETYPE_FIFOWRITE)){
		tmpPtr += (hDmax->paramPtr) * CSL_DMAX_WORD_SIZE;
		fifoObj = (CSL_DmaxFifoDesc *)(((CSL_DmaxFifoParam *)tmpPtr)->pfd);
		*((Uint32 *)response) = (Int)fifoObj;
	}
}

/** ============================================================================
 *   @n@b CSL_dmaxGetEventEntryAddr
 *
 *   @b Description
 *   @n Returns address of the Event entry for this event.
 *
 *   @b Arguments
 *   @verbatim

            hDmax           Handle to DMAX instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxGetEventEntryAddr (hDmax, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxGetEventEntryAddr (
    CSL_DmaxHandle        hDmax,
    void                 *response
)
{
	Uint32 *tmpPtr;

	if (*(hDmax->hiTableEventEntryPtr) != 0) {
		tmpPtr = hDmax->hiTableEventEntryPtr;
	} else {
		tmpPtr = hDmax->loTableEventEntryPtr;
	}
	*((Uint32 *)response) = (Uint32)tmpPtr;
}

/** ============================================================================
 *   @n@b CSL_dmaxGetParameterEntryAddr
 *
 *   @b Description
 *   @n This function gets the interrupt code of the DMAX module.
 *
 *   @b Arguments
 *   @verbatim

            hDmax           Handle to Dmax instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxGetParameterEntryAddr (hDmax, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxGetParameterEntryAddr (
    CSL_DmaxHandle        hDmax,
    void                 *response
)
{
	CSL_DmaxParameterEntry *tmpPtr;

	if (*(hDmax->hiTableEventEntryPtr) != 0) {
		tmpPtr = (CSL_DmaxParameterEntry*)
			(&(hDmax->regs->HiMaxParam));
	} else {
		tmpPtr = (CSL_DmaxParameterEntry*)
			(&(hDmax->regs->HiMaxParam));
	}
	/*
	* :KLUDGE: KSB Nov 13,2004
	* For dmax Event Entries the parameter table offset is expressed
	* as the number of dmax word offsets from the top of the Parameter
	* Table RAM. So address caluclation is done in bytes first and
	* then cast to entry type.
	*/
	tmpPtr = (CSL_DmaxParameterEntry *)(((Uint8 *)tmpPtr) 
		+ ((hDmax->paramPtr) * CSL_DMAX_WORD_SIZE));
	*((Uint32 *)response) = (Uint32)tmpPtr;
}

/** ============================================================================
 *   @n@b CSL_dmaxGetFifoFull
 *
 *   @b Description
 *   @n Returns the value of the FIFO Full bit in EFIELD of fifo descriptor.
 *
 *   @b Arguments
 *   @verbatim

            hDmax           Handle to DMAX instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_dmaxGetFifoFull (hDmax, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_dmaxGetFifoFull (
    CSL_DmaxHandle        hDmax,
    void                 *response
)
{
	Uint16 				etype;
	Uint32 				eventCtl;	
	Uint32 				*tmpPtr;
	CSL_DmaxFifoDesc 	*fifoObj;

	if (*(hDmax->hiTableEventEntryPtr) != 0) {
		eventCtl = *(hDmax->hiTableEventEntryPtr);
		tmpPtr = (Uint32 *)(hDmax->regs->HiMaxParam);
	} else {
		eventCtl = *(hDmax->loTableEventEntryPtr);
		tmpPtr = (Uint32 *)(hDmax->regs->LoMaxParam);            
	}
	etype = CSL_FEXT(eventCtl,DMAX_EVENT0_ETYPE);
	/* Make sure we have a valid FIFO event */          
	if ((etype == CSL_DMAX_EVENT0_ETYPE_FIFOREAD) 
		|| (etype == CSL_DMAX_EVENT0_ETYPE_FIFOWRITE)){
		tmpPtr += (hDmax->paramPtr) * CSL_DMAX_WORD_SIZE;
		fifoObj = (CSL_DmaxFifoDesc *)(((CSL_DmaxFifoParam *)tmpPtr)->pfd);
		*((Uint32 *)response) = CSL_FEXT(fifoObj->efield,DMAX_FIFO_FULL);
	}
}

#ifdef __cplusplus
}
#endif

#endif /* CSL_DMAXAUX_H_ */
