/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** @file  csl_emifAux.h
 *
 *  @path $(CSLPATH)\emif\src
 *
 *  @brief emif controller CSL implementation. 
 *
 */

/* =============================================================================
 *  Revision History
 *  ===============
 *  23-Nov-2004 GaneshK  File Created.
 * =============================================================================
 */

#ifndef _CSL_EMIFAUX_H_
#define _CSL_EMIFAUX_H_

#include <csl_emif.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 *	Status query functions of the EMIF
 */
/** ============================================================================
 *   @n@b CSL_emifGetSelfRefreshStatus
 *
 *   @b Description		Get the self refersh status of SDRAM
 *   @n 
 *
 *   @b Arguments
 *   @verbatim
            hEmif           Handle to the EMIF instance
     @endverbatim
 *
 *   <b> Return Value </b>  
 *			CSL_EmifSdramSelfRefresh	Self referesh status
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_EmifHandle          hEmif;
        CSL_EmifSdramSelfRefresh      response;

        response = CSL_emifGetSelfRefreshStatus (hEmif);

     @endverbatim
 * =============================================================================
 */
static inline
CSL_EmifSdramSelfRefresh CSL_emifGetSelfRefreshStatus (
    CSL_EmifHandle hEmif
)
{
	CSL_EmifSdramSelfRefresh response;
	response = (CSL_EmifSdramSelfRefresh)CSL_FEXT(hEmif->regs->SDCR,
			EMIF_SDCR_SR);
	return(response);
}

/** ============================================================================
 *   @n@b CSL_emifGetRefreshRate
 *
 *   @b Description		Get the self refersh rate of SDRAM
 *   @n 
 *
 *   @b Arguments
 *   @verbatim
            hEmif           Handle to the EMIF instance
     @endverbatim
 *
 *   <b> Return Value </b>  
 *			Uint16			Self referesh rate
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_EmifHandle          hEmif;
        Uint16      response;

        response = CSL_emifGetRefreshRate (hEmif);

     @endverbatim
 * =============================================================================
 */
static inline
Uint16 CSL_emifGetRefreshRate (
    CSL_EmifHandle hEmif
)
{
	Uint16 response;
	response = (Uint16)CSL_FEXT(hEmif->regs->SDRCR,EMIF_SDRCR_RR);
	return(response);
}

#if (EMIF_WIDTH== EMIF_32BIT_SUPPORT)        	
/** ============================================================================
 *   @n@b CSL_emifGetAsyncTimeoutStatus
 *
 *   @b Description
 *   @n 
 *
 *   @b Arguments
 *   @verbatim
            hEmif           Handle to the EMIF instance
     @endverbatim
 *
 *   <b> Return Value </b>  
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_EmifHandle          hEmif;
        Bool      response;

        response = CSL_emifGetAsyncTimeoutStatus (hEmif);

     @endverbatim
 * =============================================================================
 */
static inline
Bool CSL_emifGetAsyncTimeoutStatus (
    CSL_EmifHandle hEmif
)
{
	Bool response;
	response = (Bool)CSL_FEXT(hEmif->regs->EIRR,EMIF_EIRR_AT);
	return(response);
}

/** ============================================================================
 *   @n@b CSL_emifGetMaskedAsyncTimeoutStatus
 *
 *   @b Description
 *   @n 
 *
 *   @b Arguments
 *   @verbatim
            hEmif           Handle to the EMIF instance
     @endverbatim
 *
 *   <b> Return Value </b>  
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_EmifHandle          hEmif;
        Bool      response;

        response = CSL_emifGetMaskedAsyncTimeoutStatus (hEmif);

     @endverbatim
 * =============================================================================
 */
static inline
Bool CSL_emifGetMaskedAsyncTimeoutStatus (
    CSL_EmifHandle hEmif
)
{
	Bool response;
	response = (Bool)CSL_FEXT(hEmif->regs->EIMR, EMIF_EIMR_ATM);
	return(response);
}

/** ============================================================================
 *   @n@b CSL_emifGetMaskedAsyncTimeoutIntrStatus
 *
 *   @b Description
 *      Will read the Interrupt set register to give the status 
 *
 *   @b Arguments
 *   @verbatim
            hEmif           Handle to the EMIF instance
     @endverbatim
 *
 *   <b> Return Value </b>  
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_EmifHandle          hEmif;
        Bool      response;

        response = CSL_emifGetMaskedAsyncTimeoutIntrStatus (hEmif);

     @endverbatim
 * =============================================================================
 */
static inline
Bool CSL_emifGetMaskedAsyncTimeoutIntrStatus (
    CSL_EmifHandle hEmif
)
{
	Bool response;
	response = (Bool)CSL_FEXT(hEmif->regs->EIMR, EMIF_EIMSR_ATMSET);
	return(response);
}
#endif

/** ============================================================================
 *   @n@b CSL_emifGetNandStatus
 *
 *   @b Description
 *      Will read NAND flash control registers  
 *
 *   @b Arguments
 *   @verbatim
            hEmif           Handle to the EMIF instance
     @endverbatim
 *
 *   <b> Return Value </b>  
 *			CSL_EmifNandSetup Nand setup data structure
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_EmifHandle          hEmif;
        CSL_EmifNandSetup      response;

        response = CSL_emifGetNandStatus (hEmif);

     @endverbatim
 * =============================================================================
 */
static inline
CSL_EmifNandSetup CSL_emifGetNandStatus (
    CSL_EmifHandle hEmif
)
{
	CSL_EmifNandSetup response;
	response.nand_support = (Uint8)CSL_FEXT(hEmif->regs->NANDFCR, 
			EMIF_NANDFCR_CS2NAND);
	response.ecc_support = (Uint8)CSL_FEXT(hEmif->regs->NANDFCR, 
		EMIF_NANDFCR_CS2ECC);

	return(response);
}

/** ============================================================================
 *   @n@b CSL_emifGetNandWaitStatus
 *
 *   @b Description
 *      Will read NAND flash Status registers  
 *
 *   @b Arguments
 *   @verbatim
            hEmif           Handle to the EMIF instance
     @endverbatim
 *
 *   <b> Return Value </b>  
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_EmifHandle          hEmif;
        Bool			      response;

        response = CSL_emifGetNandWaitStatus (hEmif);

     @endverbatim
 * =============================================================================
 */
static inline
Bool CSL_emifGetNandWaitStatus (
    CSL_EmifHandle hEmif
)
{
	Bool response;
	response = (Bool)CSL_FEXT(hEmif->regs->NANDFSR,EMIF_NANDFSR_WAITST);
	return(response);
}

/** ============================================================================
 *   @n@b CSL_emifGetNandEccStatus
 *
 *   @b Description
 *      Will read NAND flash 1 ECC registers  
 *
 *   @b Arguments
 *   @verbatim
            hEmif           Handle to the EMIF instance
     @endverbatim
 *
 *   <b> Return Value </b>  
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_EmifHandle          hEmif;
        Uint32			      response;

        response = CSL_emifGetNandEccStatus (hEmif);

     @endverbatim
 * =============================================================================
 */
static inline
Bool CSL_emifGetNandEccStatus (
    CSL_EmifHandle hEmif
)
{
	Uint32 response;
	response = (Uint32)hEmif->regs->NANDF1ECC;
	return(response);
}

/**
 *	Control command functions of the EMIF
 */
/** ============================================================================
 *   @n@b CSL_emifSetSelfRefresh
 *
 *   @b Description
 *      Set the self refresh mode for SDRAM.
 * 
 *   @b Arguments
 *   @verbatim
        hEmif     	 Handle to the EMIF instance
        loadVal      Value to be loaded to emif control/status register
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  SDRAM will enter to or exit from Self refresh mode 
 *
 *   @b Modifies
 *   @n SDRAM configuration Register (SDCR)
 *
 *   @b Example
 *   @verbatim
        CSL_EmifHandle      hEmif;
        CSL_EmifSdramSelfRefresh  		loadVal = CSL_EMIF_SELF_REFRESH_ENABLE;
        ...
        CSL_emifSetSelfRefresh (hEmif, loadVal);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_emifSetSelfRefresh (
    CSL_EmifHandle    		hEmif,
    CSL_EmifSdramSelfRefresh 	loadVal
)
{
	/* Getting the address of SDCR register. Add 3 to point to Most 
	 *  significant Byte for self refresh mode
	 */
	Uint8 *regAddr = ((Uint8 *)&hEmif->regs->SDCR)+3;
	/* Since other bits of MSB are reserved make it zero. This just
	 *  as a safety measure
	 */
	loadVal &= 0x01;
	/* right shift to load value into bit 31*/
	loadVal <<= 7;   
	*regAddr = loadVal;	
		
}

/** ============================================================================
 *   @n@b CSL_emifSetSelfRefreshRate
 *
 *   @b Description
 *      Set the self refresh rate field with given value.
 * 
 *   @b Arguments
 *   @verbatim
        hEmif     	 Handle to the EMIF instance
        loadVal      Value to be loaded to emif control/status register
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Self refresh rate of SDRAM is modified
 *
 *   @b Modifies
 *   @n SDRAM Refresh Control Register (SDRCR)
 *
 *   @b Example
 *   @verbatim
        CSL_EmifHandle    	hEmif;
        Uint16 				loadVal=390;
        ...
        CSL_emifSetSelfRefreshRate (hEmif, loadVal);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_emifSetSelfRefreshRate (
    CSL_EmifHandle    	hEmif,
    Uint16 				loadVal
)
{   
	CSL_FINS(hEmif->regs->SDRCR,EMIF_SDRCR_RR,loadVal);   
}

#if (EMIF_WIDTH== EMIF_32BIT_SUPPORT)        	
/** ============================================================================
 *   @n@b CSL_emifSetAsyncTimeOutIntr
 *
 *   @b Description
 *      Enables the emif Async Time out Interrupt.
 * 
 *   @b Arguments
 *   @verbatim
        hEmif     	 Handle to the EMIF instance        
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Enables the emif Async Time out Interrupt.
 *
 *   @b Modifies
 *   @n Emif Interrupt Mask Set Register (EIMSR)
 *
 *   @b Example
 *   @verbatim
        CSL_EmifHandle    	hEmif;
        ...
        CSL_emifSetAsyncTimeOutIntr (hEmif);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_emifSetAsyncTimeOutIntr (
    CSL_EmifHandle    	hEmif	
)
{   
	CSL_FINS(hEmif->regs->EIMSR,EMIF_EIMSR_ATMSET,1);    
}

/** ============================================================================
 *   @n@b CSL_emifClrAsyncTimeOutIntr
 *
 *   @b Description
 *      Disables the emif Async Time out Interrupt.
 * 
 *   @b Arguments
 *   @verbatim
        hEmif     	 Handle to the EMIF instance        
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Disables the emif Async Time out Interrupt.
 *
 *   @b Modifies
 *   @n Emif Interrupt Mask Clear Register (EIMCR)
 *
 *   @b Example
 *   @verbatim
        CSL_EmifHandle    	hEmif;
        ...
        CSL_emifClrAsyncTimeOutIntr (hEmif);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_emifClrAsyncTimeOutIntr (
    CSL_EmifHandle    	hEmif	
)
{   
	CSL_FINS(hEmif->regs->EIMCR,EMIF_EIMCR_ATMCLR,1);    
}

/** ============================================================================
 *   @n@b CSL_emifClearAsyncTimeOut
 *
 *   @b Description
 *      Clears the Asynchronous Time out bit set by Hardware
 * 
 *   @b Arguments
 *   @verbatim
        hEmif     	 Handle to the EMIF instance
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  AT field of EIRR and ATM bit of EIMR will be cleared
 *
 *   @b Modifies
 *   @n EMIF interrupt Raw Register (EIRR)
 *   @n EMIF interrupt Masked Register (EIMR)	
 *
 *   @b Example
 *   @verbatim
        CSL_EmifHandle      hEmif;
         ...
        CSL_emifClearAsyncTimeOut (hEmif);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_emifClearAsyncTimeOut (
    CSL_EmifHandle    	hEmif
)
{   
	CSL_FINS(hEmif->regs->EIRR,EMIF_EIRR_AT,1);    
}

/** ============================================================================
 *   @n@b CSL_emifClearMaskedAsyncTimeOut
 *
 *   @b Description
 *      Clears the Masked Asynchronous Time out bit set by Hardware
 * 
 *   @b Arguments
 *   @verbatim
        hEmif     	 Handle to the EMIF instance        
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  Masked Async Time out INTR should have been enabled in EIMSR
 *
 *   <b> Post Condition </b>
 *   @n  AT field of EIRR and ATM bit of EIMR will be cleared
 *
 *   @b Modifies
 *   @n EMIF interrupt Masked  Register (EIMR) 
 *   @n EMIF interrupt Raw Register (EIRR)
 *
 *   @b Example
 *   @verbatim
        CSL_EmifHandle      hEmif;
         ...
        CSL_emifClearMaskedAsyncTimeOut (hEmif);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_emifClearMaskedAsyncTimeOut (
    CSL_EmifHandle    	hEmif
)
{   
	CSL_FINS(hEmif->regs->EIMR,EMIF_EIMR_ATM,1);    
}
#endif //32bit usage check

/** ============================================================================
 *   @n@b CSL_emifSetNandSetup
 *
 *   @b Description
 *      Sets the NAND flash control register
 * 
 *   @b Arguments
 *   @verbatim
        hEmif     	 Handle to the EMIF instance        
		arg			 DataStructure containing the NAND setup values	
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Nand flash control register is set to required values
 *
 *   @b Modifies
 *   @n EMIF Nand Flash Control Register (NANDFCR) 
 *
 *   @b Example
 *   @verbatim
        CSL_EmifHandle      hEmif;
		Uint8			   nandSetup
         ...
		 nandSetup = CSL_EMIF_NAND_ENABLE;

        CSL_emifSetNandSetup (hEmif, nandSetup);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_emifSetNandSetup (
    CSL_EmifHandle    	hEmif,
	Uint8	arg
)
{   
    CSL_FINS (hEmif->regs->NANDFCR, EMIF_NANDFCR_CS2NAND, arg);    
}

/** ============================================================================
 *   @n@b CSL_emifSetNandEccSetup
 *
 *   @b Description
 *      Sets the NAND flash control register
 * 
 *   @b Arguments
 *   @verbatim
        hEmif     	 Handle to the EMIF instance        
		arg			 DataStructure containing the NAND setup values	
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Nand flash control register is set to required values
 *
 *   @b Modifies
 *   @n EMIF Nand Flash Control Register (NANDFCR) 
 *
 *   @b Example
 *   @verbatim
        CSL_EmifHandle      hEmif;
		Uint8   			nandEccSetup
         ...
		 nandEccSetup = CSL_EMIF_NAND_ECC_ENABLE;

        CSL_emifSetNandEccSetup (hEmif, nandEccSetup);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_emifSetNandEccSetup (
    CSL_EmifHandle    	hEmif,
	Uint8	arg
)
{   
    CSL_FINS (hEmif->regs->NANDFCR, EMIF_NANDFCR_CS2ECC, arg);    
}

#ifdef __cplusplus
}
#endif

#endif /* _CSL_EMIFAUX_H_ */
