/*  ===========================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found in 
 *   the license agreement under which this software has been supplied.
 *   ==========================================================================
 */  
 
/** ===========================================================================
 *   @file  csl_icache.h
 *
 *   @path  $(CSLPATH)\soc\c67\src
 *
 *   @desc  API header file for L1P ICACHE CSL
 *
 */  
 


/* ===========================================================================
 *  Revision History
 *  ===============   
 *  28-Jan-2005 Ramitha Mathew File Created. 
 *
 * ============================================================================
 */

#ifndef _CSL_ICACHE_H_
#define _CSL_ICACHE_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <csl.h>
#include <cslr_icache.h> 
#include <soc.h>

 /**
  * @brief No: of bytes/words to be invalidated
  */
typedef Uint32 CSL_IcacheCount;

 /**
  * @brief memory location address to be invalidated
  */
typedef Uint32 CSL_IcacheAddr; 



/***************************************************************************\
*                 ENUMERATIONS                            
\***************************************************************************/

/**
 *  This enum is used to convey the information of the operating mode
 *  to be set using hardware control commands or the value returned
 *  using status query commands 
 */
typedef enum {

   /** 
    *  For the hardware control commands, this enumeration value is
    *  used to set the mode parameter associated with the control 
    *  command.For status query commands this enumeration value 
    *  tells that queried field is set cache enabled.
    */
    CSL_ICACHE_MODE_CACHE_ENABLE        =  2,

   /** 
    *  For the hardware control commands, this enumeration value is
    *  used to set the mode parameter associated with the control 
    *  command.For status query commands this enumeration value 
    *  tells that queried field is set cache freezed.
    */
    CSL_ICACHE_MODE_CACHE_FREEZE         =  3,

   /** 
    *  For the hardware control commands, this enumeration value is
    *  used to set the mode parameter associated with the control 
    *  command.For status query commands this enumeration value 
    *  tells that queried field is set cache bypassed.
    */

    CSL_ICACHE_MODE_CACHE_BYPASS         =  4

} CSL_IcacheMode;

/**
 * This enum is used to convey the information regarding RAM access
 * priority to be set using hardware control commands or the value returned 
 * using status query commands
 */
typedef enum {

    /** RAM access priority for I-Cache */
    CSL_ICACHE_PRIO_CACHE                = 0,

    /** RAM access priority for CSP slave port */
    CSL_ICACHE_PRIO_CSP                  = 1
    
} CSL_IcachePrio;


/**
 * This enum is used to issue a wait till the cache operation is complete 
 * in accordance with the hardware control command
 */
typedef enum {

    /** Return immediately without any wait */
    CSL_ICACHE_NO_WAIT                  = 0,

    /** Wait till the cache operation is done */
    CSL_ICACHE_WAIT                     = 1

} CSL_IcacheWait;


/**
 *  Status query commands that are supported by the L1P I-cache CSL.
 */
typedef enum {

   /** 
    * @brief Gets the current configured value PCC field of CSR register. 
    * @param (CSL_IcacheMode *)
    */
    CSL_ICACHE_QUERY_MODE               = 0,

   /** 
    * @brief Gets the current value of the Invalidate Word Count 
    *        field in the L1PICR register.  
    * @param (CSL_IcacheCount*) 
    */
    CSL_ICACHE_QUERY_INV_WORDCNT        = 1,

   /** 
    * @brief Gets the current value of the Invalidate Start Address
    *        field in the L1PSAR register.
    * @param (CSL_IcacheAddr*)
    */
    CSL_ICACHE_QUERY_INV_STARTADDR      = 2,

   /** 
    * @brief Gets the current value of the IP field in the L1PICR register. 
    * @param (Uint8 *)
    */
    CSL_ICACHE_QUERY_INV_CACHE          = 3,

   /**
    * @brief Get the priority setting in P bit of MEMCSR
    * @param (CSL_IcachePrio *)
    */
    CSL_ICACHE_QUERY_PRIO              = 4

} CSL_IcacheHwStatusQuery;


/**
 * Enumeration for L1P ICACHE Control commands
 */
typedef enum {

   /** 
    * @brief Command for enabling L1P I-cache
    * @param (None)      
    */
    CSL_ICACHE_CMD_CACHE_ENABLE       = 0,
 
   /** 
    * @brief Command for freezing I-cache  
    * @param (None)     
    */
    CSL_ICACHE_CMD_CACHE_FREEZE       = 1,

   /** 
    * @brief Command for bypassing I-cache 
    * @param (None) 
    */
    CSL_ICACHE_CMD_CACHE_BYPASS       = 2,

   /** 
    * @brief Command for setting the start address for the
    *        cache region to be invalidated 
    * @param (CSL_IcacheAddr) 
    */
    CSL_ICACHE_CMD_INV_STARTADDR      = 3,

   /** 
    * @brief Command for setting the number of words to be invalidated
    *         starting from the Start Address. User has to to make a delay
    *         for the invalidation; if required
    * @param (CSL_IcacheCount*)
    *         
    */
    CSL_ICACHE_CMD_INV_BYTECNT        = 4,

   /** 
    * @brief Command for invalidating a specific I-cache line in 
    *        L1P I-cache. User has to to make a delay
    *         for the invalidation; if required 
    * @param (CSL_IcacheAddr) 
    */
    CSL_ICACHE_CMD_INV_CACHELINE      = 5,

   /** 
    * @brief Command for invalidating the entire L1P I-cache
    *        User can make a request to wait till the Invalidation 
    *        completes along with the control command
    * @param (CSL_IcacheWait*) 
    */
    CSL_ICACHE_CMD_INV_CACHE          = 6    


} CSL_IcacheHwControlCmd;


/*********************************************************************\
*                 STRUCTURES                            
\*********************************************************************/


/** 
 *  The default value for CSL_IcacheHwSetup structure 
 *  Cache Mode - Cache enable
 */
#define CSL_ICACHE_HWSETUP_DEFAULTS { \
        CSL_ICACHE_MODE_CACHE_ENABLE, \
        CSL_ICACHE_PRIO_CSP          \
 }  


/**
 *  This structure is used to provide I-cache configuration parameters 
 *  to the function CSL_icacheHwSetup
 */
typedef struct CSL_IcacheHwSetup {
    
   /** The operating modes of the cache are CACHE ENABLE,
    *  CACHE FREEZE and CACHE BYPASS 
    */
    CSL_IcacheMode mode;

   /**
    * Determines the priority for RAM access between I-Cache and CSP Slave port
    */
    CSL_IcachePrio ramAccessPrio;
     
} CSL_IcacheHwSetup;


/**
 *   L1P ICACHE data object
 */
typedef struct CSL_IcacheObj {

    /** icache instance */
    CSL_InstNum             icacheNum;

    /** Pointer to the I-CACHE Register Overlay structure */                                            
    CSL_IcacheRegsOvly      regs;

} CSL_IcacheObj;


/** 
 *   Pointer to the L1P ICACHE Object structure
 */
typedef struct CSL_IcacheObj*    CSL_IcacheHandle;

/** 
 * This will have the base-address information for the peripheral instance 
 */
typedef struct {

    /** Base-address of the Configuration registers of the peripheral */
    CSL_IcacheRegsOvly   regs;
    
} CSL_IcacheBaseAddress;

/**
 * Icache Module specific parameters. Present implementation doesn't have
 * any module specific parameters.
 */
typedef struct {

   /** Bit mask to be used for module specific parameters.
    *  The below declaration is just a place-holder for future
    *  implementation.
    */
    CSL_BitMask16   flags;

} CSL_IcacheParam;

/** 
 * Icache module specific context information. Present implementation doesn't
 * have any Context information.
 */
typedef struct {

   /** Context information of Icache.
    *  The below declaration is just a place-holder for future
    *  implementation.
    */
    Uint16  contextInfo;
    
} CSL_IcacheContext; 

/** 
 * Icache Config structure
 */
typedef struct CSL_IcacheConfig {
    /* Variable which holds the value for Control and Status register. 
     * Here it is used to read and write the PCC field that corresponds
     * to the icache mode.
     */
    volatile Uint32 CSR;
    /* Variable which holds the value for Memory Control and Status
     * register. Here it is used to read and write the P field that 
     * determines the priority between CSP Slave port and L1P cache 
     * for RAM access
     */
    volatile Uint32 MEMCSR;
         
} CSL_IcacheConfig;



/**************************************************************************\
*  global function declarations
\**************************************************************************/

/** ============================================================================
 *   @n@b CSL_icacheInit
 *
 *   @b Description
 *   @n This is the initialization function for the icache CSL. This function 
 *      needs to be called before any other icache CSL functions are called. 
 *      This function is idem-potent. Currently, this function does not perform 
 *      anything.
 *
 *   @b Arguments
     @verbatim
            pContext    Pointer to module-context. As ICACHE doesn't have any 
                        context based information user is expected to pass NULL.

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Always returns
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_sysInit() must be called. 
 *
 *   <b> Post Condition </b>
 *   @n  None 
 *
 *   @b Modifies
 *   @n  None  
 *
 *   @b Example
 *   @verbatim
            CSL_Status      status;
            
            status = CSL_icacheInit(NULL);
     @endverbatim
 * =============================================================================
 */
extern CSL_Status CSL_icacheInit(
    CSL_IcacheContext    *pContext
);

/** ============================================================================
 *   @n@b CSL_icacheOpen
 *
 *   @b Description
 *   @n This Function populates the peripheral data object for the instance and
 *      returns handle to it
 *
 *   @b Arguments
 *   @verbatim
            pIcacheObj       Pointer to the data object for icache instance 
            icacheNum        Specifies the instance of the icache to be opened 
            pIcacheParam     Module specific parameter for icache instance 
            status           Status of the function call
      @endverbatim
 *
 *   <b> Return Value </b>  CSL_IcacheHandle
 *   @n                     Valid icache handle will be returned if
 *                          status value is equal to CSL_SOK.
 *
 *   <b> Pre Condition </b>
 *   @n  I-cache module has to be initialized
 *
 *   <b> Post Condition </b>
 *   @n     1.    The status is returned in the status variable. If status
 *                returned is
 *   @li            CSL_OK          Open call is successful
 *   @li            CSL_ESYS_FAIL   Open call failed
 *
 *          2.    Icache object structure is populated
 *
 *   @b Modifies
 *   @n     1. The status variable
 *          2. The data object for the instance
 *
 *   @b Example
 *   @verbatim
            CSL_status            status;
            CSL_IcacheObj         icacheObj;
            CSL_IcacheHandle      hIcache;

            hIcache = CSL_IcacheOpen (&icacheObj, CSL_ICACHE, NULL, &status);
     @endverbatim
 * =============================================================================
 */
extern CSL_IcacheHandle  CSL_icacheOpen(
    CSL_IcacheObj               *icacheObj,
    CSL_InstNum                 icacheNum,
    CSL_IcacheParam             *param,
    CSL_Status                  *status
);

/** ============================================================================
 *   @n@b CSL_icacheHwSetupRaw
 *
 *   @b Description
 *   @n This function configures the icache using the register-values passed 
 *      through the config-structure.
 *
 *   @b Arguments
 *   @verbatim

            hIcache          Pointer to the data object for icache instance 
            icacheConfig     Pointer to setup structure which contains the
                             information to program the icache to a useful state 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Configuration successful
 *   @li                    CSL_ESYS_BADHANDLE - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS - Invalid configuration 
 *                                               parameters    
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n     The registers of the specified icache instance will be setup 
 *          according to value passed
 *
 *   @b Modifies
 *   @n     Hardware registers of the specified icache instance
 *
 *   @b Example
 *   @verbatim         
            
            status = CSL_icacheHwSetupRaw (hIcache, &icacheConfig);
     @endverbatim
 * =============================================================================
 */
extern CSL_Status CSL_icacheHwSetupRaw (
    CSL_IcacheHandle    hIcache,
    CSL_IcacheConfig    *icacheConfig
);


/** ============================================================================
 *   @n@b CSL_icacheHwSetup
 *
 *   @b Description
 *   @n This function initializes the L1P I-cache hardware to a known state. 
 *
 *   @b Arguments
 *   @verbatim

            hIcache         Handle to the L1P I-cache instance 
            hwSetup         Pointer to hardware setup structure
 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Hardware setup successful
 *   @li                    CSL_ESYS_BADHANDLE - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS - Invalid parameter 
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_icacheInit () and CSL_icacheOpen () must be called
 *
 *   <b> Post Condition </b>
 *   @n  The hardware set up structure will be populated with values for the 
 *       setup parameters

 *   @b Modifies
 *   @n Hardware registers
 *
 *   @b Example
 *   @verbatim
            CSL_IcacheHandle    hIcache;
            CSL_IcacheHwSetup   icacheHwsetup;
                        
            icacheHwsetup = CSL_ICACHE_HWSETUP_DEFAULTS;                               
            CSL_icacheHwSetup (hIcache, &icacheHwsetup);
     @endverbatim
 * ===========================================================================
 */
extern CSL_Status  CSL_icacheHwSetup(
    CSL_IcacheHandle         hIcache,
    CSL_IcacheHwSetup       *hwSetup
);

/** ============================================================================
 *   @n@b CSL_icacheHwControl
 *
 *   @b Description
 *   @n This function is used for controlling the L1P I-cache instance. It 
 *      accepts the commands and data to configure/control the hardware. 
 *      Some commands does not need any data.
 *
 *   @b Arguments
 *   @verbatim

            hIcache         Handle to the L1P I-cache instance 
            cmd             Command to be executed 
            cmdArg          Data pointer corresponding to command 
 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Command execution successful.
 *   @li                    CSL_ESYS_BADHANDLE - Invalid handle
 *   @li                    CSL_ESYS_INVCMD    - Invalid command 
 *
 *   <b> Pre Condition </b>
 *   @n  icache would have opened properly
 *
 *   <b> Post Condition </b>
 *   @n   1.    The status is returned. If status
 *              returned is
 *   @li            CSL_OK                Command successful
 *   @li            CSL_ESYS_BADHANDLE    Invalid handle
 *   @li            CSL_ESYS_INVCMD       Invalid command   
 *        2.    Appropriate hardware parameters are configured based on
 *              the command
 *
 *
 *   @b Modifies
 *   @n Hardware registers
 *
 *   @b Example
 *   @verbatim
            CSL_Status                  status;
            CSL_IcacheHandle            hIcache;

            ...
            
            status  = CSL_icacheHwControl (hIcache,
                          CSL_ICACHE_CMD_CACHE_ENABLE,
                          NULL);
            ...
            
     @endverbatim
 * ===========================================================================
 */
extern CSL_Status  CSL_icacheHwControl(
    CSL_IcacheHandle            hIcache,
    CSL_IcacheHwControlCmd      cmd,
    void                        *cmdArg
);

/** ============================================================================
 *   @n@b CSL_icacheGetHwStatus
 *
 *   @b Description
 *   @n This function is for querying the hardware status. It accepts the query 
 *      and populates the data structure passed with the corresponding hardware 
 *      status.
 *
 *   @b Arguments
 *   @verbatim

            hIcache         Handle to the DSP I-cache instance 
            myQuery         Query to be performed.  
            response        Pointer to data structure to hold the 
                            status information
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Query successful
 *   @li                    CSL_ESYS_BADHANDLE - Invalid handle
 *   @li                    CSL_ESYS_INVQUERY  - Invalid query
 *   @li                    CSL_ESYS_INVPARAMS - Invalid parameters
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n   1.    The status is returned. If status
 *              returned is
 *   @li            CSL_OK              Query is successful
 *   @li            CSL_ESYS_BADHANDLE  Invalid handle
 *   @li            CSL_ESYS_INVQUERY   Invalid query 
 *
 *        2.    The status information is copied to the corresponding
 *              data structure 
 *
 *   @b Modifies
 *   @n    Data structure pointed by response
 *
 *   @b Example
 *   @verbatim
            CSL_Status                  status;
            CSL_IcacheMode              modarg;
            CSL_IcacheHandle            hIcache;

            ...

            status = CSL_icacheGetHwStatus (hIcache, 
                         CSL_ICACHE_QUERY_MODE, 
                         (void*) &modarg);
                          
            ...
     @endverbatim
 * ===========================================================================
 */
extern CSL_Status  CSL_icacheGetHwStatus(
    CSL_IcacheHandle                hIcache,
    CSL_IcacheHwStatusQuery         myQuery,
    void                            *response
);

/** ============================================================================
 *   @n@b CSL_icacheClose
 *
 *   @b Description
 *   @n This function releases the instance of the icache.
 *      
 *   @b Arguments
 *   @verbatim
            hIcache          Handle to the icache instance 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Close successful
 *   @li                    CSL_ESYS_BADHANDLE - Invalid handle
 *
 *   <b> Pre Condition </b>
 *   @n  icache would have opened properly
 *
 *   <b> Post Condition </b>
 *   @n  Icache instance is closed and its usage is illegal until next open
 *
 *   @b Modifies
 *   @n  peripheral data object
 *            
 *   @b Example
 *   @verbatim
            status = CSL_icacheClose(hIcache);
    @endverbatim
 * =============================================================================
 */
extern CSL_Status  CSL_icacheClose (
    CSL_IcacheHandle         hIcache
);

/** ============================================================================
 *   @n@b CSL_icacheGetBaseAddress
 *
 *   @b Description
 *   @n  Function to get the base address of the peripheral instance.
 *       This function is used for getting the base address of the peripheral
 *       instance. This function will be called inside the CSL_icacheOpen()
 *       function call.This function is open for re-implementing if the user 
 *       wants to modify the base address of the peripheral object to point to
 *       a different location and there by allow CSL initiated write/reads into
 *       peripheral MMR's go to an alternate location. 
 *
 *   @b Arguments
 *   @verbatim      
            icacheNum        Specifies the instance of the Icache to be opened
 
            picacheParam     Module specific parameters
 
            pBaseAddress     Pointer to baseaddress structure containing base 
                             address details
            
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_OK            Open call is successful
 *   @li                    CSL_ESYS_FAIL     The instance number is invalid
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Base Address structure is populated
 *
 *   @b Modifies
 *   @n  Base address structure is modified
 *
 *   @b Example
 *   @verbatim
        CSL_Status              status;
        CSL_IcacheBaseAddress   baseAddress;

        ...
        status = CSL_icacheGetBaseAddress(CSL_ICACHE, NULL, &baseAddress);

    @endverbatim
 * =============================================================================
 */ 
extern CSL_Status   CSL_icacheGetBaseAddress (    
    CSL_InstNum              icacheNum,
    CSL_IcacheParam          *pIcacheParam,
    CSL_IcacheBaseAddress    *pBaseAddress 
);

/** ============================================================================
 *   @n@b CSL_icacheGetHwSetup
 *
 *   @b Description
 *   @n It retrives the hardware setup parameters of the I-cache module
 *      specified by the given handle.
 *
 *   @b Arguments
 *   @verbatim

            hIcache         Handle to the instance of I-cache 
            hwSetup         Pointer to the harware setup structure
 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Retrieving the hardware setup
 *                                                parameters is successful
 *   @li                    CSL_ESYS_BADHANDLE  - The handle is passed is
 *                                                invalid
 *   @li                    CSL_ESYS_INVPARAMS  - Invalid parameters
 *
 *   <b> Pre Condition </b>
 *   @n  icache would have opened properly
 *
 *   <b> Post Condition </b>
 *   @n  The hardware setup structure is populated with the hardware setup
 *       parameters
 *
 *   @b Modifies
 *   @n hwSetup variable
 *
 *   @b Example
 *   @verbatim
            CSL_IcacheHandle   hIcache;
            CSL_IcacheHwSetup  hwSetup;

            ...
            
            status = CSL_icacheGetHwSetup(hIcache, &hwSetup);
            
            ...
            
     @endverbatim
 * ===========================================================================
 */
extern CSL_Status  CSL_icacheGetHwSetup(
    CSL_IcacheHandle      hIcache,
    CSL_IcacheHwSetup     *hwSetup
);

/* ============================================================================
 *   @n@b _CSL_icacheConfg
 *
 *   @b Description
 *   @n This function is internal to the L1P I-cache CSL. The function 
 *      configures the I-cache operating mode and is called from the  
 *      function CSL_icacheHwSetup().
 *
 *   @b Arguments
 *   @verbatim
             
            icacheMode      Required I-cache mode
            icachePrio      RAM access priority 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Hardware setup successful
 *   @li                    CSL_ESYS_INVPARAMS - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  I-cache has to be opened properly
 *
 *   <b> Post Condition </b>
 *   @n   1.    The status is returned. If status
 *              returned is
 *   @li            CSL_OK                Hardware setup successful
 *   @li            CSL_ESYS_INVPARAMS    Invalid parameter  
 *
 *        2.    Hardware registers are initialized
 *
 *
 *   @b Modifies
 *   @n   1. Hardware registers
 *
 *
 *   @b Example
 *   @verbatim
 
            CSL_IcacheMode            icacheMode;
            CSL_IcachePrio            icachePrio;
            ...
            _CSL_icacheConfg (icacheMode,icachePrio);
            ...
     @endverbatim
 * ===========================================================================
 */
extern CSL_Status _CSL_icacheConfg (
    CSL_IcacheMode            icacheMode,
    CSL_IcachePrio            icachePrio
);


#ifdef __cplusplus
}
#endif

#endif  /* _CSL_ICACHE_H_ */
