/* =============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *  ============================================================================
 */

/** ============================================================================
 *   @file  csl_icacheAux.h
 *
 *   @path  $(CSLPATH)\soc\c67\src
 *
 *   @desc  L1P I-cache CSL implementation on L1P side
 *
 */ 


/* =============================================================================
 *  Revision History
 *  ===============   
 *  28-Jan-2005 Ramitha Mathew File Created. 
 *
 * =============================================================================
 */
 
#ifndef _CSL_ICACHEAUX_H_
#define _CSL_ICACHEAUX_H_

#include <csl_icache.h>
#include <csl_chip.h>
 
#ifdef __cplusplus
extern "C" {
#endif


/* Inline functions */

/**=============================================================================
 *   @n@b CSL_icacheSetMode
 *
 *   @b Description
 *   @n This function is used for  setting the L1P cache mode in PCC field of
 *      CSR register
 *
 *   @b Arguments
     @verbatim
             CSL_IcacheMode            icacheMode           
             
     @endverbatim
 *
 *   <b> Return Value </b>  
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None        
 *
 *   <b> Post Condition </b>
 *   @n  PCC field of CSR is modified  
 *
 *   @b Modifies
 *   @n CSRregister
 *
 *   @b Example
 *   @verbatim
             CSL_IcacheMode   icacheMode ;             
             
             CSL_icacheSetMode (icacheMode);
     @endverbatim
 * =============================================================================
 */
static inline 
void  CSL_icacheSetMode (
    CSL_IcacheMode            icacheMode
)
{
    Uint32 readValue;
    Uint32 writeValue;

    readValue = CSL_chipReadReg(CSL_CHIP_REG_CSR);   
    writeValue =  (readValue & ~(CSL_CHIP_CSR_PCC_MASK ))
                 | (icacheMode << CSL_CHIP_CSR_PCC_SHIFT);
    CSL_chipWriteReg (CSL_CHIP_REG_CSR, writeValue);
    return;
}


/**=============================================================================
 *   @n@b CSL_icacheSetInvAddr
 *
 *   @b Description
 *   @n This function is used for  setting the Start address of the region
 *      in the I-cache to be Invalidated
 *
 *   @b Arguments
     @verbatim
             hIcache         Handle to the DSP I-cache instance
 
             cmdArg          CSL_IcacheAddr 
             
     @endverbatim
 *
 *   <b> Return Value </b>  
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None        
 *
 *   <b> Post Condition </b>
 *   @n  L1PISAR register filled with Starting address of region 
 *       to be invalidated     
 *
 *   @b Modifies
 *   @n L1PISAR register
 *
 *   @b Example
 *   @verbatim
             CSL_IcacheAddr   addr;
             CSL_IcacheHandle hIcache;
             
             CSL_icacheSetInvAddr (hIcache, addr);
     @endverbatim
 * =============================================================================
 */
static inline 
void  CSL_icacheSetInvAddr (
    CSL_IcacheHandle    hIcache,         
    CSL_IcacheAddr      addr
)
{      
    hIcache->regs->L1PISAR = addr;
    return;
}     
        
/**=============================================================================
 *   @n@b CSL_icacheSetInvCnt
 *
 *   @b Description
 *   @n This function is used for  setting the word count for cache region  
 *      to be invalidated
 *
 *   @b Arguments
     @verbatim
             hIcache         Handle to the L1P I-cache instance
             count           Pointer to the variable holding the Word Count
     @endverbatim
 *
 *   <b> Return Value </b>  
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None        
 *
 *   <b> Post Condition </b>
 *   @n  WORDCNT field of L1PICR register filled with the count value
 *
 *   @b Modifies
 *   @n L1PICR register
 *
 *   @b Example
 *   @verbatim
             CSL_IcacheCount    *count;
             CSL_IcacheHandle   hIcache;

             CSL_icacheSetInvCnt (hIcache,count);
     @endverbatim
 * =============================================================================
 */
static inline 
void  CSL_icacheSetInvCnt (
    CSL_IcacheHandle        hIcache,
    CSL_IcacheCount         *count
)
{
    /* Setting word count */
    CSL_FINS(hIcache->regs->L1PICR, ICACHE_L1PICR_INV_WORDCNT, *count);

}

/**=============================================================================
 *   @n@b CSL_icacheSetIp
 *
 *   @b Description
 *   @n This function is used to invalidate the entire cache
 *
 *   @b Arguments
     @verbatim
             hIcache         Handle to the I-cache instance 
             
     @endverbatim
 *
 *   <b> Return Value </b>  
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None        
 *
 *   <b> Post Condition </b>
 *   @n  IP bit field in L1PICR is set  
 *
 *   @b Modifies
 *   @n IP bit field in L1PICR is set 
 *
 *   @b Example
 *   @verbatim
             CSL_IcacheHandle hIcache;
             
             CSL_icacheSetIp (hIcache);
     @endverbatim
 * =============================================================================
 */
static inline 
void  CSL_icacheSetIp (
    CSL_IcacheHandle    hIcache   
)
{     
    /* Setting IP in L1PICR */
    CSL_FINS(hIcache->regs->L1PICR, ICACHE_L1PICR_IP,CSL_ICACHE_L1PICR_IP_SET);
}

/**=============================================================================
 *   @n@b CSL_icacheGetInvAddr
 *
 *   @b Description
 *   @n This function is used to get the starting address(byte address) of the  
 *      cache region to be invalidated 
 *
 *   @b Arguments
     @verbatim
             hIcache         Handle to the L1P I-cache instance
     @endverbatim
 *
 *   <b> Return Value </b>  
 *   @n  CSL_IcacheAddr
 *
 *   <b> Pre Condition </b>
 *   @n  None        
 *
 *   <b> Post Condition </b>
 *   @n  response will hold the starting address     
 *
 *   @b Modifies
 *   @n response
 *
 *   @b Example
 *   @verbatim
             CSL_icacheGetInvAddr (hIcache);
     @endverbatim
 * =============================================================================
 */
static inline 
CSL_IcacheAddr  CSL_icacheGetInvAddr (
    CSL_IcacheHandle        hIcache         
)
{
    return ((CSL_IcacheAddr)(hIcache->regs->L1PISAR));
}

/**=============================================================================
 *   @n@b CSL_icacheGetInvCnt
 *
 *   @b Description
 *   @n function to  get the number of words in Cache that are to be invalidated
 *
 *   @b Arguments
     @verbatim
             hIcache         Handle to the L1P I-cache instance
     @endverbatim
 *
 *   <b> Return Value </b>  
 *   @n  CSL_IcacheCount*
 *
 *   <b> Pre Condition </b>
 *   @n  None       
 *
 *   <b> Post Condition </b>
 *   @n  response will contain Invalidate word count
 *
 *   @b Modifies
 *   @n response
 *
 *   @b Example
 *   @verbatim
             CSL_IcacheHandle     hIcache;
             
             CSL_icacheGetInvCnt (hIcache);
     @endverbatim
 * =============================================================================
 */
static inline
CSL_IcacheCount CSL_icacheGetInvCnt (
    CSL_IcacheHandle        hIcache
)
{
    return((CSL_IcacheCount)(CSL_FEXT(hIcache->regs->L1PICR, \
        ICACHE_L1PICR_INV_WORDCNT)));
}    

/** =============================================================================
 *   @n@b CSL_icacheGetIp
 *
 *   @b Description
 *   @n function to  get current value of IP field of L1PICR register
 *
 *   @b Arguments
     @verbatim
             hIcache        Handle to the L1P I-cache instance
     @endverbatim
 *
 *   <b> Return Value </b>  
 *   @n  Uint8*
 *
 *   <b> Pre Condition </b>
 *   @n  None       
 *
 *   <b> Post Condition </b>
 *   @n  response will contain value of IP field of L1PICR register 
 *
 *   @b Modifies
 *   @n response  
 *
 *   @b Example
 *   @verbatim
             CSL_icacheGetIsEnabled (hIcache);
     @endverbatim
 * =============================================================================
 */
static inline
Uint8 CSL_icacheGetIp (
    CSL_IcacheHandle hIcache
)
{ 
    return ((Uint8)(CSL_FEXT(hIcache->regs->L1PICR, \
        ICACHE_L1PICR_IP))); 
}    

#ifdef __cplusplus
}
#endif

#endif /*_CSL_ICACHEAUX_H_*/
