/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** @file  csl_pllcAux.h
 *
 *  @path $(CSLPATH)\pllc\src
 *
 *  @brief PLL controller CSL implementation. 
 *
 */

/* =============================================================================
 *  Revision History
 *  ===============
 *  23-Nov-2004 GaneshK  File Created.
 * =============================================================================
 */

#ifndef _CSL_PPLCAUX_H_
#define _CSL_PLLCAUX_H_

#include <csl_pllc.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 *	Status query functions of the pllc
 */

/** ============================================================================
 *   @n@b CSL_pllcGetResetStatus
 *
 *   @b Description
 *   @n Gets the reset status of the pllc. (reset released/ asserted)
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_PllcResetState
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle          hPllc;
        CSL_PllcResetState      response;

        response = CSL_pllcGetResetStatus (hPllc);

     @endverbatim
 * =============================================================================
 */
static inline
CSL_PllcResetState CSL_pllcGetResetStatus (
    CSL_PllcHandle hPllc
)
{
	CSL_PllcResetState response = (CSL_PllcResetState) 0;

	response = (CSL_PllcResetState) CSL_FEXT (hPllc->regs->PLLCSR,
		    PLLC_PLLCSR_PLLRST);
	return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetPllMode
 *
 *   @b Description
 *   @n Gets the mode of the pllc. (bypass mode/ pll mode)
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_PllcMode
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle          hPllc;
        CSL_PllcMode      response;

        response = CSL_pllcGetMode (hPllc);

     @endverbatim
 * =============================================================================
 */
static inline
CSL_PllcMode CSL_pllcGetPllMode (
    CSL_PllcHandle hPllc
)
{
	CSL_PllcMode response = (CSL_PllcMode) 0;

	response = (CSL_PllcMode) CSL_FEXT (hPllc->regs->PLLCSR,
		  PLLC_PLLCSR_PLLEN);   
	return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetPllState
 *
 *   @b Description
 *   @n Gets the state of the pllc. (Operational / Power Down)
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_PllcState
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle          hPllc;
        CSL_PllcState       response;

        response = CSL_pllcGetPllState (hPllc);

     @endverbatim
 * =============================================================================
 */
static inline
CSL_PllcState CSL_pllcGetPllState (
    CSL_PllcHandle hPllc
)
{
	CSL_PllcState response = (CSL_PllcState) 0;

	response = (CSL_PllcState) CSL_FEXT (hPllc->regs->PLLCSR,
		     PLLC_PLLCSR_PLLPWRDN);
	return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetOscState
 *
 *   @b Description
 *   @n Gets the state of the pllc Oscillator. (Operational / Power Down)
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_PllcMode
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle          hPllc;
        CSL_PllcState       response;

        response = CSL_pllcGetOscState (hPllc);

     @endverbatim
 * =============================================================================
 */
static inline
CSL_PllcState CSL_pllcGetOscState (
    CSL_PllcHandle hPllc
)
{
	CSL_PllcState response = (CSL_PllcState) 0;

	response = (CSL_PllcState) CSL_FEXT (hPllc->regs->PLLCSR,
		   PLLC_PLLCSR_OSCPWRDN);
	return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetOscStableState
 *
 *   @b Description
 *   @n Gets the State of the Oscillator. (Stable / Not Stable)
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_PllcOscStableState
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle          hPllc;
        CSL_PllcOscStableState      response;

        response = CSL_pllcGetMode (hPllc);

     @endverbatim
 * =============================================================================
 */
static inline
CSL_PllcOscStableState CSL_pllcGetOscStableState (
    CSL_PllcHandle hPllc
)
{
	CSL_PllcOscStableState response = (CSL_PllcOscStableState) 0;

	response = (CSL_PllcOscStableState) CSL_FEXT (hPllc->regs->PLLCSR,
		    PLLC_PLLCSR_STABLE);
	return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetGoStatus
     *
 *   @b Description
 *   @n Gets the Status of the pllc Go Operation. ( Disabled / In Operation)
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_PllcGoStatus
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle          hPllc;
        CSL_PllcGoStatus      response;

        response = CSL_pllcGetGoStatus(hPllc);

     @endverbatim
 * =============================================================================
 */
static inline
CSL_PllcGoStatus CSL_pllcGetGoStatus(
    CSL_PllcHandle hPllc
)
{
	CSL_PllcGoStatus response = (CSL_PllcGoStatus) 0;

	response = (CSL_PllcGoStatus) CSL_FEXT(hPllc->regs->PLLCMD,
		    PLLC_PLLSTAT_GOSTAT);
	return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetPllmMulFactor
     *
 *   @b Description
 *   @n Gets the Multiplier factor of PLLM. 
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>  Uint16
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle          hPllc;
        Uint16                response;

        response = CSL_pllcGetPllmMulFactor (hPllc);

     @endverbatim
 * =============================================================================
 */
static inline
Uint16 CSL_pllcGetPllmMulFactor (
    CSL_PllcHandle hPllc
)
{
	Uint16 response ;

	response = (Uint16) CSL_FEXT(hPllc->regs->PLLM,PLLC_PLLM_PLLM);
	return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetDiv0Ctrl
     *
 *   @b Description
 *   @n Gets the divider control 0 status and its ratio. 
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_PllcDivCntrl
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle          hPllc;
        CSL_PllcDivCntrl    response;

        response = CSL_pllcGetDiv0Ctrl (hPllc);

     @endverbatim
 * =============================================================================
 */
static inline
CSL_PllcDivCntrl CSL_pllcGetDiv0Ctrl (
    CSL_PllcHandle hPllc
)
{
	CSL_PllcDivCntrl response ;

	response.divNum = CSL_PLLC_DIV0;
	response.divControl.divEnable = (CSL_PllcDivState)
			CSL_FEXT(hPllc->regs->PLLDIV0, PLLC_PLLDIV0_D0EN);
	response.divControl.pllDivRatio =  CSL_FEXT(hPllc->regs->PLLDIV0,
			PLLC_PLLDIV0_RATIO);
	return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetDiv1Ctrl
     *
 *   @b Description
 *   @n Gets the divider control 1 status and its ratio. 
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_PllcDivCntrl
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle          hPllc;
        CSL_PllcDivCntrl    response;

        response = CSL_pllcGetDiv1Ctrl (hPllc);

     @endverbatim
 * =============================================================================
 */
static inline
CSL_PllcDivCntrl CSL_pllcGetDiv1Ctrl (
    CSL_PllcHandle hPllc
)
{
	CSL_PllcDivCntrl response ;

	response.divNum = CSL_PLLC_DIV1;
	response.divControl.divEnable = (CSL_PllcDivState)
		CSL_FEXT(hPllc->regs->PLLDIV1, PLLC_PLLDIV1_D1EN);
	response.divControl.pllDivRatio = CSL_FEXT(hPllc->regs->PLLDIV1,
		PLLC_PLLDIV1_RATIO);
	return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetDiv2Ctrl
     *
 *   @b Description
 *   @n Gets the divider control 2 status and its ratio. 
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_PllcDivCntrl
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle          hPllc;
        CSL_PllcDivCntrl    response;

        response = CSL_pllcGetDiv2Ctrl (hPllc);

     @endverbatim
 * =============================================================================
 */
static inline
CSL_PllcDivCntrl CSL_pllcGetDiv2Ctrl (
    CSL_PllcHandle hPllc
)
{
	CSL_PllcDivCntrl response ;

	response.divNum = CSL_PLLC_DIV2;
	response.divControl.divEnable = (CSL_PllcDivState)
		CSL_FEXT(hPllc->regs->PLLDIV2, PLLC_PLLDIV2_D2EN);
	response.divControl.pllDivRatio =  CSL_FEXT(hPllc->regs->PLLDIV2,
		PLLC_PLLDIV2_RATIO);
	return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetDiv3Ctrl
     *
 *   @b Description
 *   @n Gets the divider control 3 status and its ratio. 
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_PllcDivCntrl
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle          hPllc;
        CSL_PllcDivCntrl    response;

        response = CSL_pllcGetDiv3Ctrl (hPllc);

     @endverbatim
 * =============================================================================
 */
static inline
CSL_PllcDivCntrl CSL_pllcGetDiv3Ctrl (
    CSL_PllcHandle hPllc
)
{
	CSL_PllcDivCntrl response ;

	response.divNum = CSL_PLLC_DIV3;
	response.divControl.divEnable = (CSL_PllcDivState)
		CSL_FEXT(hPllc->regs->PLLDIV3, PLLC_PLLDIV3_D3EN);
	response.divControl.pllDivRatio = CSL_FEXT(hPllc->regs->PLLDIV3,
		PLLC_PLLDIV3_RATIO);
	return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetAuxClkStatus
     *
 *   @b Description
 *   @n Gets the AUXCLK Status. ( Enable / Disable )
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_PllcClockStatus
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle          hPllc;
        CSL_PllcClockStatus    response;

        response = CSL_pllcGetClkStatus (hPllc);

     @endverbatim
 * =============================================================================
 */
static inline
CSL_PllcClockStatus CSL_pllcGetAuxClkStatus (
    CSL_PllcHandle hPllc
)
{
	CSL_PllcClockStatus response  ;

	response = (CSL_PllcClockStatus)
		CSL_FEXT(hPllc->regs->CKEN,PLLC_CKEN_AUXEN);
	return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetSysClk1Status
     *
 *   @b Description
 *   @n Gets the SYSCLK1 Status. ( Gated / On )
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_PllcSysClkStatus
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle          hPllc;
        CSL_PllcSysClkStatus    response;

        response = CSL_pllcGetSysClk1Status (hPllc);

     @endverbatim
 * =============================================================================
 */
static inline
CSL_PllcSysClkStatus CSL_pllcGetSysClk1Status (
    CSL_PllcHandle hPllc
)
{
	CSL_PllcSysClkStatus response  ;

	response.sysClk = CSL_PLLC_SYSCLK1;
	response.pllClkState = (CSL_PllcClockStatus)
		CSL_FEXT(hPllc->regs->SYSTAT,PLLC_SYSTAT_SYS1ON);
	return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetSysClk2Status
     *
 *   @b Description
 *   @n Gets the SYSCLK2 Status. ( Gated / On )
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_PllcSysClkStatus
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle          hPllc;
        CSL_PllcSysClkStatus    response;

        response = CSL_pllcGetSysClk2Status (hPllc);

     @endverbatim
 * =============================================================================
 */
static inline
CSL_PllcSysClkStatus CSL_pllcGetSysClk2Status (
    CSL_PllcHandle hPllc
)
{
	CSL_PllcSysClkStatus response  ;

	response.sysClk = CSL_PLLC_SYSCLK2;
	response.pllClkState = (CSL_PllcClockStatus)
		CSL_FEXT(hPllc->regs->SYSTAT,PLLC_SYSTAT_SYS2ON);
	return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetSysClk3Status
     *
 *   @b Description
 *   @n Gets the SYSCLK3 Status. ( Gated / On )
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_PllcSysClkStatus
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle          hPllc;
        CSL_PllcSysClkStatus    response;

        response = CSL_pllcGetSysClk3Status (hPllc);

     @endverbatim
 * =============================================================================
 */
static inline
CSL_PllcSysClkStatus CSL_pllcGetSysClk3Status (
    CSL_PllcHandle hPllc
)
{
	CSL_PllcSysClkStatus response ;

	response.sysClk = CSL_PLLC_SYSCLK3;
	response.pllClkState = (CSL_PllcClockStatus)
		CSL_FEXT(hPllc->regs->SYSTAT,PLLC_SYSTAT_SYS3ON);
	return response;
}

/**
 *	Control command functions of the pllc
 */

/** ============================================================================
 *   @n@b CSL_pllcSetMode
 *
 *   @b Description
 *      Controls the mode of the PLL.
 * 
 *   @b Arguments
 *   @verbatim
        hPllc     	 Handle to the PLLC instance
        loadVal      Value to be loaded to pllc control/status register
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc mode will be PLL Mode / ByPass.
 *
 *   @b Modifies
 *   @n pllc PLLCSR register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_PllcMode        loadVal = CSL_PLLC_PLLCSR_PLLEN_PLL_MODE;
        ...
        CSL_pllcSetMode (hPllc, loadVal);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_pllcSetMode (
    CSL_PllcHandle         hPllc,
    CSL_PllcMode           loadVal
)
{   
	CSL_FINS (hPllc->regs->PLLCSR,PLLC_PLLCSR_PLLEN,loadVal);            
}

/** ============================================================================
 *   @n@b CSL_pllcSetPllState
 *
 *   @b Description
 *      Controls the State of the PLL.
 * 
 *   @b Arguments
 *   @verbatim
        hPllc     	 Handle to the PLLC instance
        loadVal      Value to be loaded to pllc control/status register
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc state is operational / Powerdown.
 *
 *   @b Modifies
 *   @n pllc PLLCSR register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_PllcState  loadVal = CSL_PLLC_PLLCSR_PLLPWRDN_PLLSEL_OPRN;
        ...
        CSL_pllcSetPllState (hPllc, loadVal);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_pllcSetPllState (
    CSL_PllcHandle         hPllc,
    CSL_PllcState           loadVal
)
{   
	CSL_FINS (hPllc->regs->PLLCSR,PLLC_PLLCSR_PLLPWRDN,loadVal);            
}

/** ============================================================================
 *   @n@b CSL_pllcSetOscState
 *
 *   @b Description
 *      Controls the State of the Oscillator.
 * 
 *   @b Arguments
 *   @verbatim
        hPllc     	 Handle to the PLLC instance
        loadVal      Value to be loaded to pllc control/status register
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc Osicllator is Operational/ Powerdown.
 *
 *   @b Modifies
 *   @n pllc PLLCSR register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_PllcState  loadVal = CSL_PLLC_PLLCSR_PLLPWRDN_PLLSEL_OPRN;
        ...
        CSL_pllcSetOscState (hPllc, loadVal);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_pllcSetOscState (
    CSL_PllcHandle         hPllc,
    CSL_PllcState           loadVal
)
{   
	CSL_FINS (hPllc->regs->PLLCSR,PLLC_PLLCSR_OSCPWRDN,loadVal);            
}

/** ============================================================================
 *   @n@b CSL_pllcResetControl
 *
 *   @b Description
 *      Controls reset of pllc.
 * 
 *   @b Arguments
 *   @verbatim
        hPllc     	 Handle to the PLLC instance
        loadVal          Value to be loaded to pllc control/status register
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc reset is asserted/ released.
 *
 *   @b Modifies
 *   @n pllc PLLCSR register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_PllcResetState  loadVal = CSL_PLLC_PLLCSR_ASSERT;
        ...
        CSL_pllcResetControl (hPllc, loadVal);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_pllcResetControl (
    CSL_PllcHandle hPllc,
    CSL_PllcResetState loadVal
)
{   
	/* author Original code:
	* volatile unsigned int loopCount;
	*/

	/* author Added the following code according to review comment no. 1
	* for file csl_pllcAux.h
	*/

	Uint32 loopCount;

	CSL_FINS (hPllc->regs->PLLCSR, PLLC_PLLCSR_PLLRST, loadVal);

	/* author Original code:
	* loopCount = 3000;
	*/

	/* author Added following code to use #define value instead
	* of directly using the number according to review comment no.2 
	* for file csl_pllcAux.h
	*/

	/* Fixed time delay - waiting for the PLL to re-lock */
	loopCount = DELAY_200MS;
	while(loopCount--)
		asm("	NOP");
}

/** ============================================================================
 *   @n@b CSL_pllcSetGoOperation
 *
 *   @b Description
 *      Enables Go operation of PLLC
 * 
 *   @b Arguments
 *   @verbatim
        hPllc     	 Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Go operation is triggerd.
 *
 *   @b Modifies
 *   @n pllc PLLCMD register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
         
         ...
        CSL_pllcSetGoOperation (hPllc);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_pllcSetGoOperation (
    CSL_PllcHandle         hPllc
)
{
	Uint32 loopCount;

	CSL_FINS (hPllc->regs->PLLCMD,PLLC_PLLCMD_GOSET,CSL_PLLC_PLLCMD_GOSET_SET);  

	/* Fixed time delay - waiting for the PLL to re-lock */
	loopCount = DELAY_200MS;
	while(loopCount--)
		asm("	NOP");
}

/** ============================================================================
 *   @n@b CSL_pllcSetSysClk1AlignControl
 *
 *   @b Description
 *      Set the Alignment of SycClk1 to align with SYSCLK of the PLL.
 * 
 *   @b Arguments
 *   @verbatim
        hPllc     	 Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc SysClk1 is aligned.
 *
 *   @b Modifies
 *   @n pllc ALNCTL register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        ...
        CSL_pllcSetSysClk1AlignControl (hPllc);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_pllcSetSysClk1AlignControl (
    CSL_PllcHandle         hPllc
)
{   
	CSL_FINS (hPllc->regs->ALNCTL,PLLC_ALNCTL_ALN1,CSL_PLLC_ALNCTL_ALN1_ALIGN);            
}

/** ============================================================================
 *   @n@b CSL_pllcSetSysClk2AlignControl
 *
 *   @b Description
 *      Set the Alignment of SycClk2 to align with SYSCLK of the PLL.
 * 
 *   @b Arguments
 *   @verbatim
        hPllc     	 Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc SysClk2 is aligned.
 *
 *   @b Modifies
 *   @n pllc ALNCTL register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        ...
        CSL_pllcSetSysClk2AlignControl (hPllc);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_pllcSetSysClk2AlignControl (
    CSL_PllcHandle         hPllc
)
{   
	CSL_FINS (hPllc->regs->ALNCTL,PLLC_ALNCTL_ALN2,CSL_PLLC_ALNCTL_ALN2_ALIGN);            
}

/** ============================================================================
 *   @n@b CSL_pllcSetSysClk3AlignControl
 *
 *   @b Description
 *      Set the Alignment of SycClk3 to align with SYSCLK of the PLL.
 * 
 *   @b Arguments
 *   @verbatim
        hPllc     	 Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc SysClk3 is aligned.
 *
 *   @b Modifies
 *   @n pllc ALNCTL register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        ...
        CSL_pllcSetSysClk3AlignControl (hPllc);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_pllcSetSysClk3AlignControl (
    CSL_PllcHandle         hPllc
)
{   
	CSL_FINS (hPllc->regs->ALNCTL,PLLC_ALNCTL_ALN3,CSL_PLLC_ALNCTL_ALN3_ALIGN);            
}

/** ============================================================================
 *   @n@b CSL_pllcSetPllmMulFactor
 *
 *   @b Description
 *      Controls the Multiplier factor of the PLL.
 * 
 *   @b Arguments
 *   @verbatim
        hPllc     	 Handle to the PLLC instance
        loadVal      Value to be loaded to pllc control/status register
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc Multiplier factor is Changed.
 *
 *   @b Modifies
 *   @n pllc PLLM register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        Uint16            loadVal = 0x08;
        ...
        CSL_pllcSetAlignControl (hPllc, loadVal);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_pllcSetPllmMulFactor (
    CSL_PllcHandle         hPllc,
    Uint16                  loadVal
)
{
	Uint32 loopCount;

	CSL_FINS (hPllc->regs->PLLM,PLLC_PLLM_PLLM,loadVal);            

	/* Fixed time delay - waiting for the PLL to re-lock */
	loopCount = DELAY_200MS;
	while(loopCount--)
		asm("	NOP");
}

/** ============================================================================
 *   @n@b CSL_pllcSetDiv0Enable
 *
 *   @b Description
 *      Controls prescalar(DIV0) value of the PLL.
 * 
 *   @b Arguments
 *   @verbatim
        hPllc     	 Handle to the PLLC instance
        loadVal      Value to be loaded to pllc control/status register
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc prescalar(DIV0) will be enabled / disabled.
 *
 *   @b Modifies
 *   @n pllc PLLDIV0 register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_PllcDivCntrl    loadVal ;
        loadVal.divControl.divEnable = CSL_PLLC_PLLDIV_ENABLE;
        ...
        CSL_pllcSetDiv0Enable (hPllc, loadVal);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_pllcSetDiv0Enable (
    CSL_PllcHandle                          hPllc,
    CSL_PllcDivState                        loadVal
)
{   
	Uint32 loopCount;

	CSL_FINS (hPllc->regs->PLLDIV0,PLLC_PLLDIV0_D0EN,
	loadVal);

	/* Fixed time delay - waiting for the PLL to re-lock */
	loopCount = DELAY_200MS;
	while(loopCount--)
		asm("	NOP");
}

/** ============================================================================
 *   @n@b CSL_pllcSetDiv0Ratio
 *
 *   @b Description
 *      Controls the Ratio of prescalar(DIV0) of the PLL.
 * 
 *   @b Arguments
 *   @verbatim
        hPllc     	 Handle to the PLLC instance
        loadVal      Value to be loaded to pllc control/status register
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc prescalar(DIV0) ratio will be changed.
 *
 *   @b Modifies
 *   @n pllc PLLDIV0 register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_PllcDivCntrl    loadVal;
        loadVal.divControl.pllRatio = 0x08;
        ...
        CSL_pllcSetDiv0Ratio (hPllc, loadVal);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_pllcSetDiv0Ratio (
    CSL_PllcHandle      hPllc,
    Uint32              loadVal
)
{   
	Uint32 loopCount;

	CSL_FINS (hPllc->regs->PLLDIV0,PLLC_PLLDIV0_RATIO,
	loadVal);

	/* Fixed time delay - waiting for the PLL to re-lock */
	loopCount = DELAY_200MS;
	while(loopCount--)
		asm("	NOP");
}

/** ============================================================================
 *   @n@b CSL_pllcSetDiv1Enable
 *
 *   @b Description
 *      Controls prescalar(DIV1) value of the PLL.
 * 
 *   @b Arguments
 *   @verbatim
        hPllc     	 Handle to the PLLC instance
        loadVal      Value to be loaded to pllc control/status register
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc prescalar(DIV1) will be enabled / disabled.
 *
 *   @b Modifies
 *   @n pllc PLLDIV1 register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_PllcDivCntrl    loadVal ;
        loadVal.divControl.divEnable = CSL_PLLC_PLLDIV_ENABLE;
        ...
        CSL_pllcSetDiv1Enable (hPllc, loadVal);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_pllcSetDiv1Enable (
    CSL_PllcHandle      hPllc,
    CSL_PllcDivState    loadVal
)
{   
	Uint32 loopCount;

	CSL_FINS (hPllc->regs->PLLDIV1,PLLC_PLLDIV1_D1EN,
	loadVal);

	/* Fixed time delay - waiting for the PLL to re-lock */
	loopCount = DELAY_200MS;
	while(loopCount--)
		asm("	NOP");
}

/** ============================================================================
 *   @n@b CSL_pllcSetDiv1Ratio
 *
 *   @b Description
 *      Controls the Ratio of prescalar(DIV1) of the PLL.
 * 
 *   @b Arguments
 *   @verbatim
        hPllc     	 Handle to the PLLC instance
        loadVal      Value to be loaded to pllc control/status register
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc prescalar(DIV1) ratio will be changed.
 *
 *   @b Modifies
 *   @n pllc PLLDIV1 register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_PllcDivCntrl   loadVal;
        loadVal.divControl.pllRatio = 0x08;
        ...
        CSL_pllcSetDiv1Ratio (hPllc, loadVal);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_pllcSetDiv1Ratio (
    CSL_PllcHandle  hPllc,
    Uint32          loadVal
)
{   
	Uint32 loopCount;

	CSL_FINS (hPllc->regs->PLLDIV1,PLLC_PLLDIV1_RATIO,loadVal);

	/* Fixed time delay - waiting for the PLL to re-lock */
	loopCount = DELAY_200MS;
	while(loopCount--)
		asm("	NOP");
}

/** ============================================================================
 *   @n@b CSL_pllcSetDiv2Enable
 *
 *   @b Description
 *      Controls prescalar(DIV2) value of the PLL.
 * 
 *   @b Arguments
 *   @verbatim
        hPllc     	 Handle to the PLLC instance
        loadVal      Value to be loaded to pllc control/status register
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc prescalar(DIV2) will be enabled / disabled.
 *
 *   @b Modifies
 *   @n pllc PLLDIV1 register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_PllcDivCntrl    loadVal ;
        loadVal.divControl.divEnable = CSL_PLLC_PLLDIV_ENABLE;
        ...
        CSL_pllcSetDiv2Enable (hPllc, loadVal);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_pllcSetDiv2Enable (
    CSL_PllcHandle      hPllc,
    CSL_PllcDivState    loadVal
)
{   
	Uint32 loopCount;

	CSL_FINS (hPllc->regs->PLLDIV2,PLLC_PLLDIV2_D2EN,loadVal);

	/* Fixed time delay - waiting for the PLL to re-lock */
	loopCount = DELAY_200MS;
	while(loopCount--)
		asm("	NOP");
}

/** ============================================================================
 *   @n@b CSL_pllcSetDiv2Ratio
 *
 *   @b Description
 *      Controls the Ratio of prescalar(DIV2) of the PLL.
 * 
 *   @b Arguments
 *   @verbatim
        hPllc     	 Handle to the PLLC instance
        loadVal      Value to be loaded to pllc control/status register
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc prescalar(DIV2) ratio will be changed.
 *
 *   @b Modifies
 *   @n pllc PLLDIV2 register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_PllcDivCntrl   loadVal;
        loadVal.divControl.pllRatio = 0x08;
        ...
        CSL_pllcSetDiv2Ratio (hPllc, loadVal);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_pllcSetDiv2Ratio (
    CSL_PllcHandle      hPllc,
    Uint32              loadVal
)
{   
	Uint32 loopCount;

	CSL_FINS (hPllc->regs->PLLDIV2,PLLC_PLLDIV2_RATIO,loadVal);

	/* Fixed time delay - waiting for the PLL to re-lock */
	loopCount = DELAY_200MS;
	while(loopCount--)
		asm("	NOP");
}

/** ============================================================================
 *   @n@b CSL_pllcSetDiv3Enable
 *
 *   @b Description
 *      Controls prescalar(DIV3) value of the PLL.
 * 
 *   @b Arguments
 *   @verbatim
        hPllc     	 Handle to the PLLC instance
        loadVal      Value to be loaded to pllc control/status register
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc prescalar(DIV3) will be enabled / disabled.
 *
 *   @b Modifies
 *   @n pllc PLLDIV3 register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_PllcDivCntrl    loadVal ;
        loadVal.divControl.divEnable = CSL_PLLC_PLLDIV_ENABLE;
        ...
        CSL_pllcSetDiv3Enable (hPllc, loadVal);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_pllcSetDiv3Enable (
    CSL_PllcHandle      hPllc,
    CSL_PllcDivState    loadVal
)
{   
	Uint32 loopCount;

	CSL_FINS (hPllc->regs->PLLDIV3,PLLC_PLLDIV3_D3EN,loadVal);

	/* Fixed time delay - waiting for the PLL to re-lock */
	loopCount = DELAY_200MS;
	while(loopCount--)
		asm("	NOP");
}

/** ============================================================================
 *   @n@b CSL_pllcSetDiv3Ratio
 *
 *   @b Description
 *      Controls the Ratio of prescalar(DIV3) of the PLL.
 * 
 *   @b Arguments
 *   @verbatim
        hPllc     	 Handle to the PLLC instance
        loadVal      Value to be loaded to pllc control/status register
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc prescalar(DIV3) ratio will be changed.
 *
 *   @b Modifies
 *   @n pllc PLLDIV3 register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_PllcDivCntrl   loadVal;
        loadVal.divControl.pllRatio = 0x08;
        ...
        CSL_pllcSetDiv3Ratio (hPllc, loadVal);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_pllcSetDiv3Ratio (
    CSL_PllcHandle  hPllc,
    Uint32          loadVal
)
{   
	Uint32 loopCount;

	CSL_FINS (hPllc->regs->PLLDIV3,PLLC_PLLDIV3_RATIO,loadVal);

	/* Fixed time delay - waiting for the PLL to re-lock */
	loopCount = DELAY_200MS;
	while(loopCount--)
		asm("	NOP");
}

/** ============================================================================
 *   @n@b CSL_pllcSetAuxClkEnable
 *
 *   @b Description
 *      Controls the State of Aux Clock (Disable / Enable) of the PLL.
 * 
 *   @b Arguments
 *   @verbatim
        hPllc     	 Handle to the PLLC instance
        loadVal      Value to be loaded to pllc control/status register
     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc AUXClk is Enabled / Disabled.
 *
 *   @b Modifies
 *   @n pllc CKEN register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_PllcClkControl  loadVal;
        loadVal.pllClkState = CSL_PLLC_CLK_ENABLE;
        ...
        CSL_pllcSetAuxClkEnable (hPllc, loadVal);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_pllcSetAuxClkEnable (
    CSL_PllcHandle          hPllc,
    CSL_PllcClkState        loadVal
)
{   
	Uint32 loopCount;

	CSL_FINS (hPllc->regs->CKEN,PLLC_CKEN_AUXEN,loadVal);            

	/* Fixed time delay - waiting for the PLL to re-lock */
	loopCount = DELAY_200MS;
	while(loopCount--)
		asm("	NOP");
}

#ifdef __cplusplus
}
#endif

#endif /* _CSL_PLLCAUX_H_ */
