/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */
 
/** ============================================================================
 *   @file  csl_rtiAux.h
 *
 *   @path  $(CSLPATH)\soc\c67\src
 *
 *   @desc  API header file for the real time interrupt CSL
 */

/* =============================================================================
 *  Revision History
 *  ===============
 *  3-Dec-2004 sd 	File Created.
 *	25-Oct-2005 sd 	Changed the APIs CSL_rtiGetUc0Cnt and CSL_rtiGetUc1Cnt
 *
 * =============================================================================
 */
 
#ifndef _CSL_RTIAUX_H_
#define _CSL_RTIAUX_H_

#include <csl_rti.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 *  Status query functions of the real time interrupt
 */

/** ============================================================================
 *   @n@b CSL_rtiGetFrc0Cnt
 *
 *   @b Description
 *   @n Gets the current count value from the FRC0 register
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Uint32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns current value of the FRC0 register
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Uint32            count;

        ...
        count = CSL_rtiGetFrc0Cnt (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline 
Uint32 CSL_rtiGetFrc0Cnt (
    CSL_RtiHandle hRti
)
{
    Uint32  count = 0;

    /* Read the current count value from the FRC0 register */
    count = (Uint32)hRti->regs->RTIFRC0;

    return count;
}

/** ============================================================================
 *   @n@b CSL_rtiGetUc0Cnt
 *
 *   @b Description
 *   @n Gets the current count value from the UC0 register.As the Up counter
 *      value is updated by a previous read of the Free Running Counter, Free 
 * 		Running Counter is read before the Up Counter read.
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Uint32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns current value of the UC0 register
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Uint32            count;

        ...
        count = CSL_rtiGetUc0Cnt (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Uint32 CSL_rtiGetUc0Cnt (
    CSL_RtiHandle hRti
)
{
    Uint32  count = 0;

    /* Read the current count value from the FRC0 register */
    count = (Uint32)hRti->regs->RTIFRC0;

    /* Read the current count value from the UC0 register */
    count = (Uint32)hRti->regs->RTIUC0;

    return count;
}

/** ============================================================================
 *   @n@b CSL_rtiGetFrc1Cnt
 *
 *   @b Description
 *   @n Gets the current count value from the FRC1 register
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Uint32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns current value of the FRC1 register
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Uint32            count;

        ...
        count = CSL_rtiGetFrc1Cnt (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Uint32 CSL_rtiGetFrc1Cnt (
    CSL_RtiHandle hRti
)
{
    Uint32  count = 0;

    /* Read the current count value from the FRC1 register */
    count = (Uint32)hRti->regs->RTIFRC1;

    return count;
}

/** ============================================================================
 *   @n@b CSL_rtiGetUc1Cnt
 *
 *   @b Description
 *   @n Gets the current count value from the UC1 register.As the Up counter
 *      value is updated by a previous read of the Free Running Counter, Free 
 * 		Running Counter is read before the Up Counter read.
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Uint32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns current value of the UC1 register
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Uint32            count;

        ...
        count = CSL_rtiGetUc1Cnt (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Uint32 CSL_rtiGetUc1Cnt (
    CSL_RtiHandle hRti
)
{
    Uint32  count = 0;
    
	/* Read the current count value from the FRC1 register */
    count = (Uint32)hRti->regs->RTIFRC1;

    /* Read the current count value from the UC1 register */
    count = (Uint32)hRti->regs->RTIUC1;

    return count;
}

/** ============================================================================
 *   @n@b CSL_rtiGetUc0CapVal
 *
 *   @b Description
 *   @n Get the captured value of Up counter0
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Uint32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns captured value of Up counter0
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Uint32            captValue;

        ...
        captVal = CSL_rtiGetUc0CapVal (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Uint32 CSL_rtiGetUc0CapVal (
    CSL_RtiHandle hRti
)
{
    Uint32  captVal = 0;

    /* Read the captured value of Up counter0 */
    captVal = (Uint32)hRti->regs->RTICAUC0;

    return captVal;
}


/** ============================================================================
 *   @n@b CSL_rtiGetFrc0CapVal
 *
 *   @b Description
 *   @n Get the captured value of Free running counter0
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Uint32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns captured value of Free running counter0
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Uint32            captValue;

        ...
        captVal = CSL_rtiGetFrc0CapVal (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Uint32 CSL_rtiGetFrc0CapVal (
    CSL_RtiHandle hRti
)
{
    Uint32  captVal = 0;

    /* Read the captured value of Free running counter0 */
    captVal = (Uint32)hRti->regs->RTICAFRC0;

    return captVal;
}

/** ============================================================================
 *   @n@b CSL_rtiGetUc1CapVal
 *
 *   @b Description
 *   @n Get the captured value of Up counter1
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Uint32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns captured value of Up counter1
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Uint32            captValue;

        ...
        captVal = CSL_rtiGetUc1CapVal (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Uint32 CSL_rtiGetUc1CapVal (
    CSL_RtiHandle hRti
)
{
    Uint32  captVal = 0;

    /* Read the captured value of Up counter1 */
    captVal = (Uint32)hRti->regs->RTICAUC1;

    return captVal;
}

/** ============================================================================
 *   @n@b CSL_rtiGetFrc1CapVal
 *
 *   @b Description
 *   @n Get the captured value of Free running counter1
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Uint32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns captured value of Free running counter1
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Uint32            captValue;

        ...
        captVal = CSL_rtiGetFrc1CapVal (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Uint32 CSL_rtiGetFrc1CapVal (
    CSL_RtiHandle hRti
)
{
    Uint32  captVal = 0;

    /* Read the captured value of Free running counter1 */
    captVal = (Uint32)hRti->regs->RTICAFRC1;

    return captVal;
}

/** ============================================================================
 *   @n@b CSL_rtiGetComp0Cnt
 *
 *   @b Description
 *   @n Get the compare0 register value
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Uint32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns compare0 register value
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Uint32            compCnt;

        ...
        compCnt = CSL_rtiGetComp0Cnt (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Uint32 CSL_rtiGetComp0Cnt (
    CSL_RtiHandle hRti
)
{
    Uint32  compCnt = 0;

    /* Read the compare0 register value */
    compCnt = (Uint32)hRti->regs->RTICOMP0;

    return compCnt;
}

/** ============================================================================
 *   @n@b CSL_rtiGetComp1Cnt
 *
 *   @b Description
 *   @n Get the compare1 register value
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Uint32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns compare1 register value
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Uint32            compCnt;

        ...
        compCnt = CSL_rtiGetComp1Cnt (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Uint32 CSL_rtiGetComp1Cnt (
    CSL_RtiHandle hRti
)
{
    Uint32  compCnt = 0;

    /* Read the compare1 register value */
    compCnt = (Uint32)hRti->regs->RTICOMP1;

    return compCnt;
}

/** ============================================================================
 *   @n@b CSL_rtiGetComp2Cnt
 *
 *   @b Description
 *   @n Get the compare2 register value
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Uint32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns compare2 register value
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Uint32            compCnt;

        ...
        compCnt = CSL_rtiGetComp2Cnt (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Uint32 CSL_rtiGetComp2Cnt (
    CSL_RtiHandle hRti
)
{
    Uint32  compCnt = 0;

    /* Read the compare2 register value */
    compCnt = (Uint32)hRti->regs->RTICOMP2;

    return compCnt;
}

/** ============================================================================
 *   @n@b CSL_rtiGetComp3Cnt
 *
 *   @b Description
 *   @n Get the compare3 register value
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Uint32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns compare3 register value
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Uint32            compCnt;

        ...
        compCnt = CSL_rtiGetComp3Cnt (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Uint32 CSL_rtiGetComp3Cnt (
    CSL_RtiHandle hRti
)
{
    Uint32  compCnt = 0;

    /* Read the compare3 register value */
    compCnt = (Uint32)hRti->regs->RTICOMP3;

    return compCnt;
}

/** ============================================================================
 *   @n@b CSL_rtiGetOvlInt0Stat
 *
 *   @b Description
 *   @n Get the status overflow interrupt0
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Bool
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns status overflow interrupt0
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Bool              pend;

        ...
        pend = CSL_rtiGetOvlInt0Stat (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Bool CSL_rtiGetOvlInt0Stat (
    CSL_RtiHandle hRti
)
{
    Bool pend;

    /* Read status overflow interrupt0 */
    pend = (Bool) CSL_FEXT(hRti->regs->RTIINTFLAG, RTI_RTIINTFLAG_OVL0INT); 

    return pend;
}

/** ============================================================================
 *   @n@b CSL_rtiGetOvlInt1Stat
 *
 *   @b Description
 *   @n Get the status overflow interrupt1
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Bool
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns status overflow interrupt1
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Bool              pend;

        ...
        pend = CSL_rtiGetOvlInt1Stat (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Bool CSL_rtiGetOvlInt1Stat (
    CSL_RtiHandle hRti
)
{
    Bool pend;

    /* Read status overflow interrupt1 */
    pend = (Bool) CSL_FEXT(hRti->regs->RTIINTFLAG, RTI_RTIINTFLAG_OVL1INT); 

    return pend;
}

/** ============================================================================
 *   @n@b CSL_rtiGetInt0Stat
 *
 *   @b Description
 *   @n Get the status compare interrupt0
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Bool
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns status compare interrupt0
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Bool              pend;

        ...
        pend = CSL_rtiGetInt0Stat (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Bool CSL_rtiGetInt0Stat (
    CSL_RtiHandle hRti
)
{
    Bool pend;

    /* Read status compare interrupt0 */
    pend = (Bool) CSL_FEXT(hRti->regs->RTIINTFLAG, RTI_RTIINTFLAG_INT0); 

    return pend;
}

/** ============================================================================
 *   @n@b CSL_rtiGetInt1Stat
 *
 *   @b Description
 *   @n Get the status compare interrupt1
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Bool
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns status compare interrupt1
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Bool              pend;

        ...
        pend = CSL_rtiGetInt1Stat (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Bool CSL_rtiGetInt1Stat (
    CSL_RtiHandle hRti
)
{
    Bool pend;

    /* Read status compare interrupt1 */
    pend = (Bool) CSL_FEXT(hRti->regs->RTIINTFLAG, RTI_RTIINTFLAG_INT1); 

    return pend;
}

/** ============================================================================
 *   @n@b CSL_rtiGetInt2Stat
 *
 *   @b Description
 *   @n Get the status compare interrupt2
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Bool
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns status compare interrupt2
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Bool              pend;

        ...
        pend = CSL_rtiGetInt2Stat (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Bool CSL_rtiGetInt2Stat (
    CSL_RtiHandle hRti
)
{
    Bool pend;

    /* Read status compare interrupt2 */
    pend = (Bool) CSL_FEXT(hRti->regs->RTIINTFLAG, RTI_RTIINTFLAG_INT2); 

    return pend;
}

/** ============================================================================
 *   @n@b CSL_rtiGetInt3Stat
 *
 *   @b Description
 *   @n Get the status compare interrupt3
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Bool
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns status compare interrupt3
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Bool              pend;

        ...
        pend = CSL_rtiGetInt3Stat (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Bool CSL_rtiGetInt3Stat (
    CSL_RtiHandle hRti
)
{
    Bool pend;

    /* Read status compare interrupt3 */
    pend = (Bool) CSL_FEXT(hRti->regs->RTIINTFLAG, RTI_RTIINTFLAG_INT3); 

    return pend;
}

/** ============================================================================
 *   @n@b CSL_rtiGetDwdStatus
 *
 *   @b Description
 *   @n Get the status digital watchdog
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Bool
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns status digital watchdog
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Bool              status;

        ...
        status = CSL_rtiGetDwdStatus (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Bool CSL_rtiGetDwdStatus (
    CSL_RtiHandle hRti
)
{
    Bool status;

    /* Read status digital watchdog */
    status = (Bool) CSL_FEXT(hRti->regs->RTIWDSTATUS, RTI_RTIWDSTATUS_DWDST); 

    return status;
}

/** ============================================================================
 *   @n@b CSL_rtiGetDwdDwnCtrStatus
 *
 *   @b Description
 *   @n Get the status of digital watchdog down counter
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance
 *   @endverbatim
 *
 *   <b> Return Value </b>  Uint32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns status digital watchdog down counter
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        Uint32            status;

        ...
        status = CSL_rtiGetDwdDwnCtrStatus (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
Uint32 CSL_rtiGetDwdDwnCtrStatus (
    CSL_RtiHandle hRti
)
{
    Uint32 status;

    /* Read status digital watchdog */
    status = (Uint32) CSL_FEXT(hRti->regs->RTIDWDCNTR, RTI_RTIDWDCNTR_DWDCNTR); 

    return status;
}

/**
 *  Control command functions of the Real time interrupt
 */
 
/** ============================================================================
 *   @n@b CSL_rtiBlock0Start
 *
 *   @b Description
 *       Start the Block0 Counters
 *
 *   @b Arguments
 *   @verbatim
        hRti    Handle to the real time interrupt instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  real time interrupt block0 counters started
 *
 *   @b  Modifies
 *   @n  RTI Global Control register
 *
 *   @b  Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        ...
        CSL_rtiBlock0Start (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
void CSL_rtiBlock0Start (
    CSL_RtiHandle   hRti
)
{
    CSL_FINST (hRti->regs->RTIGCTRL, RTI_RTIGCTRL_CNT0EN, STARTBLK0);
}

/** ============================================================================
 *   @n@b CSL_rtiBlock0Stop
 *
 *   @b Description
 *       Stop the Block0 Counters
 *
 *   @b Arguments
 *   @verbatim
        hRti    Handle to the real time interrupt instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  real time interrupt block0 counters stoped
 *
 *   @b  Modifies
 *   @n  RTI Global Control register
 *
 *   @b  Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        ...
        CSL_rtiBlock0Stop (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
void CSL_rtiBlock0Stop (
    CSL_RtiHandle   hRti
)
{
    CSL_FINST (hRti->regs->RTIGCTRL, RTI_RTIGCTRL_CNT0EN, STOPBLK0);
}

/** ============================================================================
 *   @n@b CSL_rtiBlock1Start
 *
 *   @b Description
 *       Start the Block1 Counters
 *
 *   @b Arguments
 *   @verbatim
        hRti    Handle to the real time interrupt instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  real time interrupt block1 counters started
 *
 *   @b  Modifies
 *   @n  RTI Global Control register
 *
 *   @b  Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        ...
        CSL_rtiBlock1Start (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
void CSL_rtiBlock1Start (
    CSL_RtiHandle   hRti
)
{
    CSL_FINST (hRti->regs->RTIGCTRL, RTI_RTIGCTRL_CNT1EN, STARTBLK1);
}

/** ============================================================================
 *   @n@b CSL_rtiBlock1Stop
 *
 *   @b Description
 *       Stop the Block1 Counters
 *
 *   @b Arguments
 *   @verbatim
        hRti    Handle to the real time interrupt instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  real time interrupt block1 counters stoped
 *
 *   @b  Modifies
 *   @n  RTI Global Control register
 *
 *   @b  Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        ...
        CSL_rtiBlock1Stop (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
void CSL_rtiBlock1Stop (
    CSL_RtiHandle   hRti
)
{
    CSL_FINST (hRti->regs->RTIGCTRL, RTI_RTIGCTRL_CNT1EN, STOPBLK1);
}

/** ============================================================================
 *   @n@b CSL_rtiClearInt0
 *
 *   @b Description
 *       clear the compare interrupt0
 *
 *   @b Arguments
 *   @verbatim
        hRti    Handle to the real time interrupt instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  real time interrupt compare interrupt0 cleared
 *
 *   @b  Modifies
 *   @n  RTI Interrupt Flag Register
 *
 *   @b  Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        ...
        CSL_rtiClearInt0 (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
void CSL_rtiClearInt0 (
    CSL_RtiHandle   hRti
)
{
    CSL_FINST (hRti->regs->RTIINTFLAG, RTI_RTIINTFLAG_INT0, CLEARINT0);
}

/** ============================================================================
 *   @n@b CSL_rtiClearInt1
 *
 *   @b Description
 *       clear the compare interrupt1
 *
 *   @b Arguments
 *   @verbatim
        hRti    Handle to the real time interrupt instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  real time interrupt compare interrupt1 cleared
 *
 *   @b  Modifies
 *   @n  RTI Interrupt Flag Register
 *
 *   @b  Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        ...
        CSL_rtiClearInt1 (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
void CSL_rtiClearInt1 (
    CSL_RtiHandle   hRti
)
{
    CSL_FINST (hRti->regs->RTIINTFLAG, RTI_RTIINTFLAG_INT1, CLEARINT1);
}

/** ============================================================================
 *   @n@b CSL_rtiClearInt2
 *
 *   @b Description
 *       clear the compare interrupt2
 *
 *   @b Arguments
 *   @verbatim
        hRti    Handle to the real time interrupt instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  real time interrupt compare interrupt2 cleared
 *
 *   @b  Modifies
 *   @n  RTI Interrupt Flag Register
 *
 *   @b  Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        ...
        CSL_rtiClearInt2 (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
void CSL_rtiClearInt2 (
    CSL_RtiHandle   hRti
)
{
    CSL_FINST (hRti->regs->RTIINTFLAG, RTI_RTIINTFLAG_INT2, CLEARINT2);
}

/** ============================================================================
 *   @n@b CSL_rtiClearInt3
 *
 *   @b Description
 *       clear the compare interrupt3
 *
 *   @b Arguments
 *   @verbatim
        hRti    Handle to the real time interrupt instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  real time interrupt compare interrupt3 cleared
 *
 *   @b  Modifies
 *   @n  RTI Interrupt Flag Register
 *
 *   @b  Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        ...
        CSL_rtiClearInt3 (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
void CSL_rtiClearInt3 (
    CSL_RtiHandle   hRti
)
{
    CSL_FINST (hRti->regs->RTIINTFLAG, RTI_RTIINTFLAG_INT3, CLEARINT3);
}

/** ============================================================================
 *   @n@b CSL_rtiClearOvfInt0
 *
 *   @b Description
 *       clear the overflow interrupt0
 *
 *   @b Arguments
 *   @verbatim
        hRti    Handle to the real time interrupt instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  real time interrupt overflow interrupt0 cleared
 *
 *   @b  Modifies
 *   @n  RTI Interrupt Flag Register
 *
 *   @b  Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        ...
        CSL_rtiClearOvfInt0 (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
void CSL_rtiClearOvfInt0 (
    CSL_RtiHandle   hRti
)
{
    CSL_FINST (hRti->regs->RTIINTFLAG, RTI_RTIINTFLAG_OVL0INT, CLEAROVFINT0);
}

/** ============================================================================
 *   @n@b CSL_rtiClearOvfInt1
 *
 *   @b Description
 *       clear the overflow interrupt1
 *
 *   @b Arguments
 *   @verbatim
        hRti    Handle to the real time interrupt instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  real time interrupt overflow interrupt1 cleared
 *
 *   @b  Modifies
 *   @n  RTI Interrupt Flag Register
 *
 *   @b  Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        ...
        CSL_rtiClearOvfInt0 (hRti);
        ...
     @endverbatim
 * =============================================================================
 */ 
static inline
void CSL_rtiClearOvfInt1 (
    CSL_RtiHandle   hRti
)
{
    CSL_FINST (hRti->regs->RTIINTFLAG, RTI_RTIINTFLAG_OVL1INT, CLEAROVFINT1);
}

/** ============================================================================
 *   @n@b CSL_rtiDwdEnable
 *
 *   @b Description
 *      Enable the Digital Watchdog
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance

     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  RTI Digital Watchdog Control register is loaded with 
         the  0xACED5312 value and enable the digital watchdog
 *
 *   @b Modifies
 *   @n RTI Digital Watchdog Control register
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        ...
        CSL_rtiDwdEnable (hRti);
        ...
     @endverbatim
 * =============================================================================
 */
static inline
void CSL_rtiDwdEnable (
    CSL_RtiHandle   hRti
)
{
    hRti->regs->RTIDWDCTRL = CSL_RTI_DWD_ENABLE;
}

/** ============================================================================
 *   @n@b CSL_rtiDwdDisable
 *
 *   @b Description
 *      Disable the Digital Watchdog
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance

     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  RTI Digital Watchdog Control register is loaded with 
         the 0x5312ACED  value and disable the digital watchdog
 *
 *   @b Modifies
 *   @n RTI Digital Watchdog Control register
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        ...
        CSL_rtiDwdDisable (hRti);
        ...
     @endverbatim
 * =============================================================================
 */
static inline
void CSL_rtiDwdDisable (
    CSL_RtiHandle   hRti
)
{
    hRti->regs->RTIDWDCTRL = CSL_RTI_DWD_DISABLE;
}

/** ============================================================================
 *   @n@b CSL_rtiClearDwd
 *
 *   @b Description
 *      Clear the Digital Watchdog
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance

     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *   @b Modifies
 *   @n RTI Watchdog Status Register
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        ...
        CSL_rtiClearDwd (hRti);
        ...
     @endverbatim
 * =============================================================================
 */
static inline
void CSL_rtiClearDwd (
    CSL_RtiHandle hRti
)
{
    CSL_FINST (hRti->regs->RTIWDSTATUS, RTI_RTIWDSTATUS_DWDST, CLEARDWD);
}
    
/** ============================================================================
 *   @n@b CSL_rtiWdKey
 *
 *   @b Description
 *      Set the Digital Watchdog key
 *
 *   @b Arguments
 *   @verbatim
        hRti            Handle to the real time interrupt instance

     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  RTI Watchdog Key Register  is loaded with 
         0xE51A followed by 0xA35C, the 25-bit DWD Down Counter is reloaded with 
         the 13- bit preload value stored in RTIDWDPRLD. 
 *   @b Modifies
 *   @n RTI Watchdog Key Register
 *
 *   @b Example
 *   @verbatim
        CSL_RtiHandle     hRti;
        ...
        CSL_rtiWdKey (hRti);
        ...
     @endverbatim
 * =============================================================================
 */
static inline
void CSL_rtiWdKey (
    CSL_RtiHandle   hRti
)
{
    CSL_FINS (hRti->regs->RTIWDKEY, RTI_RTIWDKEY_WDKEY, CSL_RTI_WDKEY1);
    CSL_FINS (hRti->regs->RTIWDKEY, RTI_RTIWDKEY_WDKEY, CSL_RTI_WDKEY2);
}

#ifdef __cplusplus
}
#endif

#endif /* _CSL_RTIAUX_H_ */







