/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** @file csl_spiAux.h
 *
 * @brief   Header file for functional layer of CSL
 *
 * Description
 *    - The defines inline function definitions
 *
 */

/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */

#ifndef _CSL_SPIAUX_H_
#define _CSL_SPIAUX_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <csl_spi.h>


/** ============================================================================
 *   @n@b CSL_spiGetHwSetupPriFmt
 *
 *   @b Description
 *   @n This function gets the hardware setup of data format regs of the SPI.
 *
 *   @b Arguments
 *   @verbatim
            hSpi                    Handle to SPI instance
            fmtNum                  format number,
            *confFmt                pointer to config format

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiGetHwSetupPriFmt (hSpi, fmtNum, &confFmt);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiGetHwSetupPriFmt(
    CSL_SpiHandle                   hSpi,
    Uint8                           fmtNum,
    CSL_SpiHwSetupPriFmt            *confFmt
)
{

	Uint32 temp = hSpi->regs->SPIFMT[fmtNum];

	confFmt->wDelay   = CSL_FEXT(temp, SPI_SPIFMT_WDELAY);
	confFmt->preScale = CSL_FEXT(temp, SPI_SPIFMT_PRESCALE);
	confFmt->charLen  = CSL_FEXT(temp, SPI_SPIFMT_CHARLEN);
	confFmt->waitEna  = (CSL_SpiWaitEn)CSL_FEXT(temp, SPI_SPIFMT_WAITENA);
	confFmt->phase    = (CSL_SpiClkPhase)CSL_FEXT(temp, SPI_SPIFMT_PHASE);
	confFmt->polarity = (CSL_SpiClkPolarity) CSL_FEXT(temp, SPI_SPIFMT_POLARITY);
	confFmt->shiftDir = (CSL_SpiShDir) CSL_FEXT(temp, SPI_SPIFMT_SHIFTDIR);
}


/** ============================================================================
*   @n@b CSL_spiGetHwSetupCptInt
*
*   @b Description
*   @n This function gets the hardware setup of interrupt levels of the SPI
*
*   @b Arguments
*   @verbatim
    hSpi            Handle to SPI instance
    *intVal         pointer to the interrupt level

    @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiGetHwSetupCptInt (hSpi, &intVal);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiGetHwSetupCptInt(
    CSL_SpiHandle                   hSpi,
    Uint32                          *intVal
)
{
	*intVal = hSpi->regs->SPILVL;
}


/** ============================================================================
 *   @n@b CSL_spiGetHwSetupGen
 *
 *   @b Description
 *   @n This function gets the general hardware setup of the SPI module.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            *genPtr         pointer to the structure

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiGetHwSetupGen (hSpi, &genPtr);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiGetHwSetupGen(
    CSL_SpiHandle                   hSpi,
    CSL_SpiHwSetupGen               *genPtr
)
{
	CSL_SpiRegsOvly regsPtr = hSpi->regs;
	Uint32 temp;

	genPtr->opMode   = (CSL_SpiOpMod)CSL_FEXTR(regsPtr->SPIGCR1, 1, 0);
	genPtr->enaHiZ   = (CSL_SpiEnaHiZ)CSL_FEXT(regsPtr->SPIINT0,
				SPI_SPIINT0_ENABLEHIGHZ);

	temp = regsPtr->SPIDELAY;
	genPtr->c2tDelay = CSL_FEXT(temp, SPI_SPIDELAY_C2TDELAY);
	genPtr->t2cDelay = CSL_FEXT(temp, SPI_SPIDELAY_T2CDELAY);
	genPtr->t2eTmout = CSL_FEXT(temp, SPI_SPIDELAY_T2EDELAY);
	genPtr->c2eTmout = CSL_FEXT(temp, SPI_SPIDELAY_C2EDELAY);
}

/** ============================================================================
 *   @n@b CSL_spiGetHwSetupGpio
 *
 *   @b Description
 *   @n This function gets the hardware setup of Global Pin config regs of SPI.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            gpioType        GPIO type
            *value          return value

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiGetHwSetupGpio (hSpi, gpioType, &value);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiGetHwSetupGpio(
    CSL_SpiHandle               hSpi,
    CSL_SpiGpioType             gpioType,
    Uint16                      *value
)
{
	if(gpioType == CSL_SPI_GPIOTYPE_FUNC) {
		*value = hSpi->regs->SPIPC0;
	}
	if(gpioType == CSL_SPI_GPIOTYPE_DIR) {
		*value = hSpi->regs->SPIPC1;
	}
}

/** ============================================================================
 *   @n@b CSL_spiGetHwSetupCptFmtCtrl
 *
 *   @b Description
 *   @n This function gets the hardwar setup of the DAT1 reg of the SPI module.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            *fmtCtrlPtr     pointer to the format controller structure

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiGetHwSetupCptFmtCtrl (hSpi, &fmtCtrlPtr);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiGetHwSetupCptFmtCtrl(
    CSL_SpiHandle                   hSpi,
    CSL_SpiHwSetupFmtCtrl           *fmtCtrlPtr
)
{
	Uint32 temp = hSpi->regs->SPIDAT1;

	fmtCtrlPtr->csHold = (CSL_SpiCsHold)CSL_FEXT(temp, SPI_SPIDAT1_CSHOLD);
	fmtCtrlPtr->wDel   = (CSL_SpiWDelayEn)CSL_FEXT(temp, SPI_SPIDAT1_WDEL);
	fmtCtrlPtr->fmtSel = (CSL_SpiFmtSel)CSL_FEXT(temp, SPI_SPIDAT1_DFSEL);
	fmtCtrlPtr->cSel   = CSL_FEXT(temp, SPI_SPIDAT1_CSNR);
}

/** ============================================================================
 *   @n@b CSL_spiPinVal
 *
 *   @b Description
 *   @n This function returns the value of pin control reg 2 of the SPI module.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            *pinVal         pointer to the SPI PC2 register content value

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiPinVal (hSpi, &pinVal);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiPinVal(
    /** pointer to the object that holds reference to the
    * instance of SPI requested after the call */
    CSL_SpiHandle                   hSpi,
    Uint32                          *pinVal
)
{
	*pinVal = hSpi->regs->SPIPC2;
}


/** ============================================================================
 *   @n@b CSL_spiIntEnabled
 *
 *   @b Description
 *   @n This function returns the interrupt register value of the SPI module.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            *intVal         pointer to the interrupt reg content value

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiIntEnabled (hSpi, &intVal);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiIntEnabled(
    /** pointer to the object that holds reference to the
    * instance of SPI requested after the call */
    CSL_SpiHandle                   hSpi,
    Uint32                          *intVal
)
{
	*intVal = CSL_FEXTR(hSpi->regs->SPIINT0, 8, 0);
}


/** ============================================================================
 *   @n@b CSL_spiEvtStatus
 *
 *   @b Description
 *   @n This function sets the flag status register of the SPI module.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            *intVal         pointer to the flag register contents value

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiEvtStatus (hSpi, &intval);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiEvtStatus(
    /** pointer to the object that holds reference to the
    * instance of SPI requested after the call */
    CSL_SpiHandle                   hSpi,
    Uint8                           *intVal
)
{
	*intVal = CSL_FEXTR(hSpi->regs->SPIFLG, 8, 1);
	CSL_FINSR(hSpi->regs->SPIFLG, 8, 1, *intVal);
}


/** ============================================================================
 *   @n@b CSL_spiIntVector
 *
 *   @b Description
 *   @n This function gets the status of Interruipt Vector reg of SPI module.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            intType         type of interrupt
            *intVec         pointer to int vector structure

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiIntVector (hSpi, intType, &intVec);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiIntVector(
    /** pointer to the object that holds reference to the
    * instance of SPI requested after the call */
    CSL_SpiHandle                   hSpi,
    CSL_SpiHwStatusQuery            intType,
    CSL_SpiIntVec                   *intVec
)
{
	Uint32 temp;

	if (CSL_SPI_QUERY_INT_VECTOR0 == intType) {
		temp = hSpi->regs->TGINTVECT[0];
	}
	else if (CSL_SPI_QUERY_INT_VECTOR1 == intType) {
		temp = hSpi->regs->TGINTVECT[1];
	}

	intVec->intVal  = CSL_FEXT(temp, SPI_TGINTVECT_INTVECT);
	
}

/** ============================================================================
 *   @n@b CSL_spiCptDmaEnabled
 *
 *   @b Description
 *   @n This function gets the DMA Request mode from int reg of SPI module.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            *dmaval         pointer to the value got from int register
     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiCptDmaEnabled (hSpi, &dmaVal);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiCptDmaEnabled(
    /** pointer to the object that holds reference to the
    * instance of SPI requested after the call */
    CSL_SpiHandle                   hSpi,
    Uint16                          *dmaVal
)
{
	*dmaVal = CSL_FEXT(hSpi->regs->SPIINT0, SPI_SPIINT0_DMAREQEN);
}

/** ============================================================================
 *   @n@b CSL_spiPriResetOut
 *
 *   @b Description
 *   @n This function does a soft reset of the SPI.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiPriResetOut (hSpi);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiPriResetOut(
    /** pointer to the object that holds reference to the
    * instance of SPI requested after the call */
    CSL_SpiHandle                   hSpi
)
{
	CSL_FINST(hSpi->regs->SPIGCR0, SPI_SPIGCR0_RESET, NO);
}

/** ============================================================================
 *   @n@b CSL_spiCptDma
 *
 *   @b Description
 *   @n This function handles the DMA request of the SPI module.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            dmaEn           Enable/disable DMA

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiCptDma (hSpi, dmaEn);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiCptDma(
    /** pointer to the object that holds reference to the
    * instance of SPI requested after the call */
    CSL_SpiHandle                  hSpi,
    CSL_SpiCptDma                  dmaEn
)
{
	CSL_FINS(hSpi->regs->SPIINT0, SPI_SPIINT0_DMAREQEN, dmaEn);
}

/** ============================================================================
 *   @n@b CSL_spiCptWrite0
 *
 *   @b Description
 *   @n This function writes data on the SPI.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            val             data to be written

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiCptWrite0 (hSpi, val);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiCptWrite0(
    /** pointer to the object that holds reference to the
    * instance of SPI requested after the call */
    CSL_SpiHandle                  hSpi,
    Uint16                         val
)
{
	CSL_FINS(hSpi->regs->SPIDAT0, SPI_SPIDAT0_SPIDAT0, val);
}


/** ============================================================================
 *   @n@b CSL_spiCptWrite1
 *
 *   @b Description
 *   @n This function writes data on the SPI.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            val             value to be written

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiCptWrite1 (hSpi, val);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiCptWrite1(
    /** pointer to the object that holds reference to the
    * instance of SPI requested after the call */
    CSL_SpiHandle                  hSpi,
    Uint16                         val
)
{
	CSL_FINS(hSpi->regs->SPIDAT1, SPI_SPIDAT1_SPIDAT1, val);
}


/** ============================================================================
 *   @n@b CSL_spiCptRead
 *
 *   @b Description
 *   @n This function reads the SPI data buffer.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            *data           pointer to the buffer that receives data
            *bufStat        pointer to the buffer status register

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiCptRead (hSpi, &data, &bufStat);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiCptRead(
    /** pointer to the object that holds reference to the
    * instance of SPI requested after the call */
    CSL_SpiHandle                  hSpi,
    Uint16                         *data,
    CSL_SpiBufStat                 *bufStat
)
{
	Uint32 temp = hSpi->regs->SPIBUF;
	*data = CSL_FEXT(temp, SPI_SPIBUF_SPIBUF);
	if (bufStat) {
		bufStat->status = CSL_FEXTR(temp, 31, 25);
	}
}


/** ============================================================================
 *   @n@b CSL_spiInt
 *
 *   @b Description
 *   @n This function initializes the SPI module.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            cmd             command to enable or disable interrupts
            intVal          interrupt value

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiInt (hSpi, cmd, intVal);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiInt(
    /** pointer to the object that holds reference to the
    * instance of SPI requested after the call */
    CSL_SpiHandle                  hSpi,
    CSL_SpiHwControlCmd            cmd,
    Uint16                         intVal
)
{
	if (CSL_SPI_CMD_INT_ENABLE == cmd) {
		hSpi->regs->SPIINT0 |= intVal;
	}
	else if (CSL_SPI_CMD_INT_DISABLE == cmd) {
		hSpi->regs->SPIINT0 &= ~intVal;
	}
}

/** ============================================================================
 *   @n@b CSL_spiPins
 *
 *   @b Description
 *   @n This function sets/clears the PC3/4/5 registers of the SPI module.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            cmd             command to set the PC3/4/5 registers
            pinVal          value to be set

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiPins (hSpi, cmd, pinVal);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiPins(
    /** pointer to the object that holds reference to the
    * instance of SPI requested after the call */
    CSL_SpiHandle                  hSpi,
    CSL_SpiHwControlCmd            cmd,
    Uint32                         pinVal
)
{
	if (CSL_SPI_CMD_PINS_VALUE == cmd) {
		hSpi->regs->SPIPC3 = pinVal;
	}
	else if (CSL_SPI_CMD_PINS_SET == cmd) {
		hSpi->regs->SPIPC4 = pinVal;
	}
	else if (CSL_SPI_CMD_PINS_CLEAR == cmd) {
		hSpi->regs->SPIPC5 = pinVal;
	}
}

/** ============================================================================
 *   @n@b CSL_spiXferCtrl
 *
 *   @b Description
 *   @n This function configures the transmission params for the SPI module.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            xferEn          Enable/hiold the SPI in reset

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiXferCtrl (hSpi, xferEn);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiXferCtrl(
    /** pointer to the object that holds reference to the
    * instance of SPI requested after the call */
    CSL_SpiHandle                  hSpi,
    CSL_SpiXferEn                  xferEn
)
{
	CSL_FINS(hSpi->regs->SPIGCR1, SPI_SPIGCR1_ENABLE, xferEn);
}

/** ============================================================================
 *   @n@b CSL_spiHwSetupPriFmt
 *
 *   @b Description
 *   @n This function sets up the SPI a/c to Hwsetup structure of SPI.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            fmtNum          Format register
            *confFmt        pointer to h/w setup fmt structure

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiHwSetupPriFmt (hSpi, fmtNum, &conffmt);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiHwSetupPriFmt(
    /** pointer to the object that holds reference to the
    * instance of SPI requested after the call */
    CSL_SpiHandle                  hSpi,
    /** instance number of the format register to setup */
    Uint8                          fmtNum,
    /** setup structure for the format register */
    CSL_SpiHwSetupPriFmt           *confFmt
)
{
	hSpi->regs->SPIFMT[fmtNum] = CSL_FMK(SPI_SPIFMT_WDELAY,
		confFmt->wDelay) |
		CSL_FMK(SPI_SPIFMT_PRESCALE,
		confFmt->preScale) |
		CSL_FMK(SPI_SPIFMT_CHARLEN,
		confFmt->charLen) |
		CSL_FMK(SPI_SPIFMT_WAITENA,
		confFmt->waitEna) |
		CSL_FMK(SPI_SPIFMT_POLARITY,
		confFmt->polarity) |
		CSL_FMK(SPI_SPIFMT_PHASE,
		confFmt->phase) |
		CSL_FMK(SPI_SPIFMT_SHIFTDIR,
		confFmt->shiftDir);
}

/** ============================================================================
 *   @n@b CSL_spiHwSetupCptInt
 *
 *   @b Description
 *   @n This function sets the interrupt level of the SPI module.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            intVal          bit-vector for interrupts

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiHwSetupCptInt (hSpi, intVal);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiHwSetupCptInt(
    /** pointer to the object that holds reference to the
    * instance of SPI requested after the call */
    CSL_SpiHandle                  hSpi,
    /** bit-vector for interrupts */
    Uint32                         intVal
)
{
	hSpi->regs->SPILVL = intVal;
}


/** ============================================================================
 *   @n@b CSL_spiHwSetupGen
 *
 *   @b Description
 *   @n This function does the hardware setup of the SPI module.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            *genPtr         pointer to the HW setup gen structure

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiHwSetupGen (hSpi, &genPtr);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiHwSetupGen(
    /** pointer to the object that holds reference to the
    * instance of SPI requested after the call */
    CSL_SpiHandle                  hSpi,
    CSL_SpiHwSetupGen              *genPtr
)
{
	CSL_SpiRegsOvly regsPtr = hSpi->regs;
	Uint32 temp;

	temp = regsPtr->SPIGCR1;
	CSL_FINS(temp, SPI_SPIGCR1_CLKMOD, CSL_FEXTR(genPtr->opMode, 1, 1));
	CSL_FINS(temp, SPI_SPIGCR1_MASTER, CSL_FEXTR(genPtr->opMode, 0, 0));
	regsPtr->SPIGCR1 = temp;

	CSL_FINS(regsPtr->SPIINT0, SPI_SPIINT0_ENABLEHIGHZ, genPtr->enaHiZ);

	temp = CSL_FMK(SPI_SPIDELAY_C2TDELAY, genPtr->c2tDelay) |
		CSL_FMK(SPI_SPIDELAY_T2CDELAY, genPtr->t2cDelay) |
		CSL_FMK(SPI_SPIDELAY_T2EDELAY, genPtr->t2eTmout) |
		CSL_FMK(SPI_SPIDELAY_C2EDELAY, genPtr->c2eTmout);
	regsPtr->SPIDELAY = temp;
}

/** ============================================================================
 *   @n@b CSL_spiHwSetupGpio
 *
 *   @b Description
 *   @n This function sets up the PC0/1/6/7 registers of the SPI module.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            gpioType        gpiotype
            value           value to be set

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiHwSetupGpio (hSpi, gpioType, value);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiHwSetupGpio(
    /** pointer to the object that holds reference to the
    * instance of SPI requested after the call */
    CSL_SpiHandle                  hSpi,
    CSL_SpiGpioType               gpioType,
    Uint32                         value
)
{
	if(gpioType == CSL_SPI_GPIOTYPE_FUNC){
		hSpi->regs->SPIPC0 = value;
	}
	if(gpioType ==  CSL_SPI_GPIOTYPE_DIR){
		hSpi->regs->SPIPC1 = value;       
	}
}

/** ============================================================================
*   @n@b CSL_spiHwSetupCptFmtCtrl
*
*   @b Description
*   @n This function sets up the SPI a/c to fmtCtrl structure.
 *
 *   @b Arguments
 *   @verbatim
            hSpi            Handle to SPI instance
            *fmtCtrlPtr     pointer to the fmtCtrl Structure

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_spiHwSetupCptFmtCtrl (hSpi, &fmtCtrlPtr);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_spiHwSetupCptFmtCtrl(
    /** pointer to the object that holds reference to the
    * instance of SPI requested after the call */
    CSL_SpiHandle                  hSpi,
    CSL_SpiHwSetupFmtCtrl          *fmtCtrlPtr
)
{
	Uint32 temp;
	temp = CSL_FMK(SPI_SPIDAT1_CSHOLD, fmtCtrlPtr->csHold) |
		CSL_FMK(SPI_SPIDAT1_WDEL, fmtCtrlPtr->wDel) |
		CSL_FMK(SPI_SPIDAT1_DFSEL, fmtCtrlPtr->fmtSel) |
		CSL_FMK(SPI_SPIDAT1_CSNR, fmtCtrlPtr->cSel);
	CSL_FINSR(hSpi->regs->SPIDAT1, 32, 16, CSL_FEXTR(temp, 32, 16));
}

#ifdef __cplusplus
}
#endif

#endif /* _CSLR_SPI_H_ */
