/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** @file  csl_uhpiAux.h
 *
 *
 *  @brief PLL controller CSL implementation on DSP side. 
 *
 */

/* =============================================================================
 *  Revision History
 *  ===============
 *  04-FEB-2005 asr  File Created.
 * =============================================================================
 */

#ifndef _CSL_UHPIAUX_H_
#define _CSL_UHPIAUX_H_

#include <csl_uhpi.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 *  Status query functions of the uhpi
 */

/** ============================================================================
 *   @n@b CSL_uhpiSetGpint
 *
 *   @b Description
 *   @n Gets the GPINT_ENx control status of the uhpi.
 *
 *   @b Arguments
 *   @verbatim
            hUhpi           Handle to the UHPI instance
     @endverbatim
 *
 *   <b> Return Value </b>  
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_UhpiHandle          hUhpi;

        CSL_uhpiSetGpint (hUhpi , 0x4);

     @endverbatim
 * =============================================================================
 */
static inline
void CSL_uhpiSetGpint(
    CSL_UhpiHandle  hUhpi,
    Uint32          loadVal
)
{
	CSL_FINSR (hUhpi->regs->GPINT_CTRL,2,0,loadVal);
}       

/** ============================================================================
 *   @n@b CSL_uhpiSetGpintInv
 *
 *   @b Description
 *   @n Gets the GPINT_INVx Invert status of the uhpi.
 *
 *   @b Arguments
 *   @verbatim
            hUhpi           Handle to the UHPI instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_UhpiHandle          hUhpi;

        CSL_uhpiSetGpint (hUhpi , 0x4);

     @endverbatim
 * =============================================================================
 */
static inline
void CSL_uhpiSetGpintInv(
    CSL_UhpiHandle  hUhpi,
    Uint32          loadVal
)
{
	CSL_FINSR (hUhpi->regs->GPINT_CTRL,18,16,loadVal);
}

/** ============================================================================
 *   @n@b CSL_uhpiGpioEnable
 *
 *   @b Description
 *   @n Gets the GPIO Enable status of the uhpi.
 *
 *   @b Arguments
 *   @verbatim
            hUhpi           Handle to the UHPI instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_UhpiHandle          hUhpi;

        CSL_uhpiGpioEnable (hUhpi , 0x4);

     @endverbatim
 * =============================================================================
 */

static inline
void CSL_uhpiGpioEnable(
    CSL_UhpiHandle  hUhpi,
    Uint32          loadVal
)
{
	hUhpi->regs->GPIO_EN = loadVal;
}

/** ============================================================================
 *   @n@b CSL_uhpiSetGpioDir1
 *
 *   @b Description
 *   @n Gets the GPIO Direction status of the uhpi.
 *
 *   @b Arguments
 *   @verbatim
            hUhpi           Handle to the UHPI instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_UhpiHandle          hUhpi;

        CSL_uhpiSetGpioDir1 (hUhpi , 0xFF);

     @endverbatim
 * =============================================================================
 */
static inline
void CSL_uhpiSetGpioDir1(
    CSL_UhpiHandle  hUhpi,
    Uint32          dirVal
)
{
	hUhpi->regs->GPIO_DIR1 = dirVal;
}

/** ============================================================================
 *   @n@b CSL_uhpiSetGpioDir2
 *
 *   @b Description
 *   @n Gets the GPIO Direction2 status of the uhpi.
 *
 *   @b Arguments
 *   @verbatim
            hUhpi           Handle to the UHPI instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_UhpiHandle          hUhpi;

        CSL_uhpiSetGpioDir2 (hUhpi , 0xFF);

     @endverbatim
 * =============================================================================
 */
 
static inline
void CSL_uhpiSetGpioDir2(
    CSL_UhpiHandle  hUhpi,
    Uint16          dirVal
)
{
	hUhpi->regs->GPIO_DIR2 = dirVal;
}

/** ============================================================================
 *   @n@b CSL_uhpiSetGpioDir3
 *
 *   @b Description
 *   @n Gets the GPIO Direction3 status of the uhpi.
 *
 *   @b Arguments
 *   @verbatim
            hUhpi           Handle to the UHPI instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_UhpiHandle          hUhpi;

        CSL_uhpiSetGpioDir3 (hUhpi , 0xFF);

     @endverbatim
 * =============================================================================
 */
static inline
void CSL_uhpiSetGpioDir3(
    CSL_UhpiHandle  hUhpi,
    Uint32          dirVal
)
{
	hUhpi->regs->GPIO_DIR3 = dirVal;
}

/** ============================================================================
 *   @n@b CSL_uhpiSetDspInt
 *
 *   @b Description
 *   @n Gets the HPIC Host-to-DSP Interrupt status of the uhpi.
 *
 *   @b Arguments
 *   @verbatim
            hUhpi           Handle to the UHPI instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_UhpiHandle          hUhpi;

        CSL_uhpiSetDspInt (hUhpi , 1);

     @endverbatim
 * =============================================================================
 */
static inline
void CSL_uhpiSetDspInt(
    CSL_UhpiHandle  hUhpi
)
{
	CSL_FINS (hUhpi->regs->HPIC,HPI_HPIC_DSP_INT,TRUE); 
}       

/** ============================================================================
 *   @n@b CSL_uhpiResetDspInt
 *
 *   @b Description
 *   @n Gets the Reset HPIC Host-to-DSP Interrupt status of the uhpi.
 *
 *   @b Arguments
 *   @verbatim
            hUhpi           Handle to the UHPI instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_UhpiHandle          hUhpi;

        CSL_uhpiResetDspInt (hUhpi , 0);

     @endverbatim
 * =============================================================================
 */

static inline
void CSL_uhpiResetDspInt(
    CSL_UhpiHandle  hUhpi
)
{
	CSL_FINS (hUhpi->regs->HPIC,HPI_HPIC_DSP_INT,FALSE);
}       

/** ============================================================================
 *   @n@b CSL_uhpiSetHint
 *
 *   @b Description
 *   @n Gets the HPIC DSP-to-Host Interrupt status of the uhpi.
 *
 *   @b Arguments
 *   @verbatim
            hUhpi           Handle to the UHPI instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_UhpiHandle          hUhpi;

        CSL_uhpiSetHint (hUhpi , 1);

     @endverbatim
 * =============================================================================
 */

static inline
void CSL_uhpiSetHint(
    CSL_UhpiHandle  hUhpi
)
{
	CSL_FINS (hUhpi->regs->HPIC,HPI_HPIC_HINT,TRUE);    
}       

/** ============================================================================
 *   @n@b CSL_uhpiResetHint
 *
 *   @b Description
 *   @n Gets the Reset HPIC DSP-to-Host Interrupt status of the uhpi.
 *
 *   @b Arguments
 *   @verbatim
            hUhpi           Handle to the UHPI instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_UhpiHandle          hUhpi;

        CSL_uhpiSetHint (hUhpi , 0);

     @endverbatim
 * =============================================================================
 */
 
static inline
void CSL_uhpiResetHint(
    CSL_UhpiHandle  hUhpi
)
{
	CSL_FINS (hUhpi->regs->HPIC,HPI_HPIC_HINT,FALSE);   
}       

/** ===========================================================================
 *   @n@b CSL_uhpiGetPidRev
 *
 *   @b Description
 *   @n This function  gets the Peripheral ID Revision.
 *
 *   @b Arguments
 *   @verbatim

            hUhpi           Handle to UHPI instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_uhpiGetPidRev (hUhpi, &response);

     @endverbatim
 * ===========================================================================
 */

static inline
void CSL_uhpiGetPidRev(
    CSL_UhpiHandle  hUhpi,
    void *          response
)
{
	/*Read the peripheral revision from PID */
	*((Uint32 *)response) = CSL_FEXT(hUhpi->regs->PID, HPI_PID_REV);
}

/** ===========================================================================
 *   @n@b CSL_uhpiGetPidClass
 *
 *   @b Description
 *   @n This function  gets the Peripheral ID Class.
 *
 *   @b Arguments
 *   @verbatim

            hUhpi           Handle to UHPI instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_uhpiGetPidClass (hUhpi, &response);

     @endverbatim
 * ===========================================================================
 */

static inline
void CSL_uhpiGetPidClass(
    CSL_UhpiHandle  hUhpi,
    void *          response
)
{
	*((Uint32 *)response) = CSL_FEXT(hUhpi->regs->PID, HPI_PID_CLASS);
}

/** ===========================================================================
 *   @n@b CSL_uhpiGetPidType
 *
 *   @b Description
 *   @n This function  gets the Peripheral ID Tupe.
 *
 *   @b Arguments
 *   @verbatim

            hUhpi           Handle to UHPI instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_uhpiGetPidType (hUhpi, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_uhpiGetPidType(
    CSL_UhpiHandle  hUhpi,
    void *          response
)
{
	*((Uint32 *)response) = CSL_FEXT(hUhpi->regs->PID, HPI_PID_TYPE);
}

/** ===========================================================================
 *   @n@b CSL_uhpiGetGpintEn0
 *
 *   @b Description
 *   @n This function  gets GPINT Enable 0.
 *
 *   @b Arguments
 *   @verbatim

            hUhpi           Handle to UHPI instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_uhpiGetGpintEn0 (hUhpi, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_uhpiGetGpintEn0(
    CSL_UhpiHandle  hUhpi,
    void *          response
)
{
	*((Uint32 *)response) = CSL_FEXT(hUhpi->regs->GPINT_CTRL, 
												HPI_GPINT_CTRL_GPINT_EN0);
}

/** ===========================================================================
 *   @n@b CSL_uhpiGetGpintEn1
 *
 *   @b Description
 *   @n This function  gets GPINT Enable 1.
 *
 *   @b Arguments
 *   @verbatim

            hUhpi           Handle to UHPI instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_uhpiGetGpintEn1 (hUhpi, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_uhpiGetGpintEn1(
    CSL_UhpiHandle  hUhpi,
    void *          response
)
{
	*((Uint32 *)response) = CSL_FEXT(hUhpi->regs->GPINT_CTRL, 
										HPI_GPINT_CTRL_GPINT_EN1);
}

/** ===========================================================================
 *   @n@b CSL_uhpiGetGpintEn2
 *
 *   @b Description
 *   @n This function  gets GPINT Enable 2.
 *
 *   @b Arguments
 *   @verbatim

            hUhpi           Handle to UHPI instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_uhpiGetGpintEn2 (hUhpi, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_uhpiGetGpintEn2(
    CSL_UhpiHandle  hUhpi,
    void *          response
)
{
	*((Uint32 *)response) = CSL_FEXT(hUhpi->regs->GPINT_CTRL, \
										HPI_GPINT_CTRL_GPINT_EN2);
}
       
/** ===========================================================================
 *   @n@b CSL_uhpiGetGpioDir1
 *
 *   @b Description
 *   @n This function  gets GPIO Direction-1.
 *
 *   @b Arguments
 *   @verbatim

            hUhpi           Handle to UHPI instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_uhpiGetGpioDir1 (hUhpi, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_uhpiGetGpioDir1(
    CSL_UhpiHandle  hUhpi,
    void *          response
)
{
	*((Uint32 *)response) = hUhpi->regs->GPIO_DIR1;
}               

/** ===========================================================================
 *   @n@b CSL_uhpiGetGpioDir2
 *
 *   @b Description
 *   @n This function  gets GPIO Direction-2.
 *
 *   @b Arguments
 *   @verbatim

            hUhpi           Handle to UHPI instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_uhpiGetGpioDir2 (hUhpi, &response);

     @endverbatim
 * ===========================================================================
 */        
static inline
void CSL_uhpiGetGpioDir2(
    CSL_UhpiHandle  hUhpi,
    void *          response
)
{
	*((Uint32 *)response) = hUhpi->regs->GPIO_DIR2;
}  
             
/** ===========================================================================
 *   @n@b CSL_uhpiGetGpioDir3
 *
 *   @b Description
 *   @n This function  gets GPIO Direction-3.
 *
 *   @b Arguments
 *   @verbatim

            hUhpi           Handle to UHPI instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_uhpiGetGpioDir3 (hUhpi, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_uhpiGetGpioDir3(
    CSL_UhpiHandle  hUhpi,
    void *          response
)
{
	*((Uint32 *)response) = hUhpi->regs->GPIO_DIR3;
}               

/** ===========================================================================
 *   @n@b CSL_uhpiGetHpicHrdy
 *
 *   @b Description
 *   @n This function  gets HRDY bit of the HPIC.
 *
 *   @b Arguments
 *   @verbatim

            hUhpi           Handle to UHPI instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_uhpiGetHpicHrdy (hUhpi, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_uhpiGetHpicHrdy(
    CSL_UhpiHandle  hUhpi,
    void *          response
)
{
	*((Uint32 *)response) = CSL_FEXT(hUhpi->regs->HPIC, HPI_HPIC_HRDY);
}               
        
/** ===========================================================================
 *   @n@b CSL_uhpiGetHpicFetch
 *
 *   @b Description
 *   @n This function  gets FETCH bit of the HPIC Register.
 *
 *   @b Arguments
 *   @verbatim

            hUhpi           Handle to UHPI instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_uhpiGetHpicFetch (hUhpi, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_uhpiGetHpicFetch(
    CSL_UhpiHandle  hUhpi,
    void *          response
)
{
	*((Uint32 *)response) = CSL_FEXT(hUhpi->regs->HPIC, HPI_HPIC_FETCH);
}               

/** ===========================================================================
 *   @n@b CSL_uhpiGetHpicHpiRst
 *
 *   @b Description
 *   @n This function  gets HPI_RST bit of the HPIC Register.
 *
 *   @b Arguments
 *   @verbatim

            hUhpi           Handle to UHPI instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_uhpiGetHpicHpiRst (hUhpi, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_uhpiGetHpicHpiRst(
    CSL_UhpiHandle  hUhpi,
    void *          response
)
{
	*((Uint32 *)response) = CSL_FEXT(hUhpi->regs->HPIC, HPI_HPIC_HPI_RST);
}

/** ===========================================================================
 *   @n@b CSL_uhpiGetHpicHwobStat
 *
 *   @b Description
 *   @n This function  gets HWOB_STAT bit of the HPIC Register.
 *
 *   @b Arguments
 *   @verbatim

            hUhpi           Handle to UHPI instance

            response        Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

        CSL_uhpiGetHpicHwobStat (hUhpi, &response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_uhpiGetHpicHwobStat(
    CSL_UhpiHandle  hUhpi,
    void *          response
)
{
	*((Uint32 *)response) = CSL_FEXT(hUhpi->regs->HPIC, HPI_HPIC_HWOB_STAT);
}


#ifdef __cplusplus
}
#endif

#endif /* _CSL_UHPIAUX_H_ */
