#ifndef _GGMEMMAP_H
#define _GGMEMMAP_H
/******************************************************************************
 * FILE PURPOSE: Memory map for DSP software
 ******************************************************************************
 * FILE NAME:  ggmemmap.h
 *
  * DESCRIPTION: Contains memory map for DSP software.  This file is comprised
 *              of three distinct sections:
 *
 *              (1) Heap-allocated ECO buffer definitions.
 *                  Here, all ECOs that are entirely allocated from the heap
 *                  or not entirely in the build are specified by commenting out
 *                  their relevant #define.
 *                  This section also contains START and LENGTH definitions
 *                  for ECO buffers that are allocated from heap for those
 *                  ECOs that do *not* have all buffers taken from heap.
 *
 *              (2) Memory block sections.
 *                  These are the physical sections of the DSP's memory,
 *                  broken into sections conducive for placement of the
 *                  build's ECO requirements.
 *
 *              (3) Hand placed ECO buffers.
 *                  Here, all ECO buffers not placed in step (1) above (i.e.
 *                  those not allocated from heap) are placed into the memory
 *                  block sections defined in step (2).  This includes both
 *                  per-channel buffers as well as overlayable scratch buffers
 *                  that may be used across multiple channels/applications.
 *
 * TABS: NONE
 *
 * (C) Copyright 2003, Texas Instruments Incorporated.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

#include "types.h"
#include "ggCodecTestCfg.h"
#include "ggswcfg.h"
#include "ggmbuf.h"
/*############################################################################*/
/* STEP 1: Define all build specific buffer sizes not be defined in ggport.h  */
/*############################################################################*/
#ifdef GG_NEU_B2_L
  #undef GG_NEU_B2_L
  #define GG_NEU_B2_L                                  1972
#endif

#ifdef GG_NEU_B4_L
  #undef GG_NEU_B4_L
  #define GG_NEU_B4_L                                  0
#endif

/*############################################################################*/
/* STEP 1: Heap-allocated ECO Buffer definitions                              */
/*############################################################################*/

/*******************************************************************************
 * Section for defining which ECOs are to be completely allocated from heap.   *
 * If ECO's buffers are not to be completely allocated from heap, then must    *
 * uncomment out those ECOs and specify their placement farther below.         *
 * If an ECO is not to be present in the build,it should also be specified as  *
 * heap allocated                                                              *
 ******************************************************************************/
/*The following modules cannot be allocated from heap:
  a)All Codecs
  b)ECU
  c)TDU
  d)RCU
  e)ACU
*/
/*----------------------------------------------------------------------------*/
/* Expanding $ch = 1 */
/*----------------------------------------------------------------------------*/
//#define  GG_NEU_C1_HEAP_ALLOC
//#define  GG_TFTP_C1_HEAP_ALLOC

/*******************************************************************************
 * Section for defining which ECO buffers are to be allocated from heap, for.
 * those ECOs in which not all buffers are allocated from heap. No codec
 * buffers are allowed to be allocated through heap because of switchover
 * Hand placed buffers occur later in this file, in Step 3.
 ******************************************************************************/

#define GG_NEU_B0_C1_START                                           0x0000
#define GG_NEU_B0_C1_L                                                    0
#define GG_NEU_B1_C1_START                                           0x0000
#define GG_NEU_B1_C1_L                                                    0
#define GG_NEU_B3_C1_START                                           0x0000
#define GG_NEU_B3_C1_L                                                    0

/*----------------------------------------------------------------------------*/
/* TFTP                                                                       */
/*----------------------------------------------------------------------------*/

#define GG_TFTP_B0_C1_START                                          0x0000
#define GG_TFTP_B0_C1_L                                                   0

/*----------------------------------------------------------------------------*/
/* End expansion $ch = 1 */
/* Expanding $ch = 2 */
/*----------------------------------------------------------------------------*/
//#define  GG_NEU_C2_HEAP_ALLOC
//#define  GG_TFTP_C2_HEAP_ALLOC

/*******************************************************************************
 * Section for defining which ECO buffers are to be allocated from heap, for.
 * those ECOs in which not all buffers are allocated from heap. No codec
 * buffers are allowed to be allocated through heap because of switchover
 * Hand placed buffers occur later in this file, in Step 3.
 ******************************************************************************/

#define GG_NEU_B0_C2_START                                           0x0000
#define GG_NEU_B0_C2_L                                                    0
#define GG_NEU_B1_C2_START                                           0x0000
#define GG_NEU_B1_C2_L                                                    0
#define GG_NEU_B3_C2_START                                           0x0000
#define GG_NEU_B3_C2_L                                                    0

/*----------------------------------------------------------------------------*/
/* TFTP                                                                       */
/*----------------------------------------------------------------------------*/

#define GG_TFTP_B0_C2_START                                          0x0000
#define GG_TFTP_B0_C2_L                                                   0

/*----------------------------------------------------------------------------*/
/* End expansion $ch = 2 */

/*----------------------------------------------------------------------------*/

/*############################################################################*/
/* STEP 2: Memory section and block definitions                               */
/*############################################################################*/

/*----------------------------------------------------------------------------*/
/* Voice, Fax and other heap sizes                                            */
/*----------------------------------------------------------------------------*/
#define GG_VIDEO_HEAP_L         0x5B0       
#define GG_VIDEO_HEAP_A         4
#define GG_VIDEO_HEAP_EXT_A     4

#define GG_VIDEO_AUX_HEAP_L     0

/*******************************************/
/* HAL heap Size - Build Specific          */
/*******************************************/
#define GG_HAL_HEAP_L           0x11C0
#define GG_HAL_HEAP_A           4
#define GG_HEAP_HAL_START                                       0x86B950
#define GG_HEAP_HAL_END                                         0x86CB10

#define GG_GMP_AREAS            1
/************************************************************************************/
/* GMP Size Calculation:                                                            */
/* If VPU FIFO depth is 100/110 ms, or 20 packets, each of 22 words,                */
/* GMP size should be 20 times 22 words times number of channels times 1.05 margin. */
/* For 32 channel build it should be 440 times 32 times 1.05 = 14784 words          */
/* GMP_AREA_SIZE = Total GMP Size/Number of Areas                                   */
/* Make sure that each GMP Area Size is an integral multiple of a GMP_GMC_SIZE(22)  */
/* 400*34*1.05 = 14280 =3570*4 = 3586 ~3592 8byte aligned = E08                     */
/*************************************************************************************/

#define    GG_CH_LENGTH       0x00CC0
/*******************************************************/
/* Start of memory map definition                      */
/*******************************************************/

/*----------------------------------------------------------------------------*/
#define GG_COMMON_SCRATCH_START                                      0x86B000
#define GG_COMMON_SCRATCH_END                                        0x86B620

/* ----------------------------- MSG scratch -------------------------------- */
#define GG_COMPOUND_MSG_SCRATCH_START                                GG_COMMON_SCRATCH_START

#define GG_DATA_GMP0_START                                           0x86B630
#define GG_DATA_GMP0_END                                             0x86B950
#define GG_GMNDU_NEU_BUFS_START                                      0x86CB10
#define GG_GMNDU_NEU_BUFS_END                                        0x86D010

#define GG_GMNDU_MSG_FIFOS_START                                     0x86B620
#define GG_GMNDU_MSG_FIFOS_END                                       0x86B630

/* using area alloted for tx network as it is not used */

#define GG_NEU_SCRATCH_START                                        (0x86D010)
#define GG_NEU_SCRATCH_END                                          (0x86D810)

/*----------------------------------------------------------------------------*/
/* Expanding $ch = 1 */
/*----------------------------------------------------------------------------*/
#define GG_CH1_START                                             (0x86D810 + (1 - 1) * GG_CH_LENGTH)
#define GG_CH1_END                                               (GG_CH1_START + GG_CH_LENGTH)
/*----------------------------------------------------------------------------*/
/* End expansion $ch = 1 */
/* Expanding $ch = 2 */
/*----------------------------------------------------------------------------*/
#define GG_CH2_START                                             (0x86E4D0 + (2 - 1) * GG_CH_LENGTH)
#define GG_CH2_END                                               (GG_CH2_START + GG_CH_LENGTH)
/*----------------------------------------------------------------------------*/
/* End expansion $ch = 2 */

/*----------------------------------------------------------------------------*/

/* NEU SCRATCH AREA */
#define GG_NEU_TX_B_START   align (GG_NEU_SCRATCH_START, GG_NEU_B2_A)
#define GG_NEU_TX_B_END     (GG_NEU_TX_B_START + GG_NEU_B2_L)

/* NEU RX */
#define GG_NEU_RX_B_START          align(GG_NEU_TX_B_END, GG_NEU_B4_A)
#define GG_NEU_RX_B_END            (GG_NEU_RX_B_START + GG_NEU_B4_L)

#define GG_NEU_SCRATCH_USED (GG_NEU_RX_B_END - GG_NEU_SCRATCH_START)

#define GG_COMMON_SCRATCH_LAST               GG_COMMON_SCRATCH_START + 1
#define GG_COMMON_SCRATCH_USED               (GG_COMMON_SCRATCH_LAST - GG_COMMON_SCRATCH_START)

#if ((GG_COMMON_SCRATCH_START+GG_COMMON_SCRATCH_USED) > GG_COMMON_SCRATCH_END)
# error GG_COMMON_SCRATCH size exceeded
#endif

/*----------------------------------------------------------------------------*/
/* HAL Heap                                                                   */
/*----------------------------------------------------------------------------*/
#define GG_HAL_HEAP_START        align(GG_HEAP_HAL_START, GG_HAL_HEAP_A)
#define GG_HAL_HEAP_END          (GG_HAL_HEAP_START + GG_HAL_HEAP_L)

/*----------------------------------------------------------------------------*/
/*------------------------- GMAC-NDU-NEU  BUFFERS -------------------------*/
/*----------------------------------------------------------------------------*/

#define GG_GMNDU_NEU_MSG_INST_START	 align(GG_GMNDU_NEU_BUFS_START,4)
#define GG_GMNDU_NEU_MSG_INST_END       (GG_GMNDU_NEU_MSG_INST_START+\
											  GG_GMNDU_NEU_INST_L)

#define GG_GMNDU_NEU_MSG_TX_SCRATCH_START  align(GG_GMNDU_NEU_MSG_INST_END,4)
#define GG_GMNDU_NEU_MSG_TX_SCRATCH_END    (GG_GMNDU_NEU_MSG_TX_SCRATCH_START+\
												 GG_GMNDU_NEU_TX_MSG_SCRATCH_L)

#define GG_GMNDU_NEU_HLT_TX_SCRATCH_START    GG_GMNDU_NEU_MSG_TX_SCRATCH_START

#define GG_GMNDU_NEU_MSG_RX_PKTASSEM_START align(GG_GMNDU_NEU_MSG_TX_SCRATCH_END,4)
#define GG_GMNDU_NEU_MSG_RX_PKTASSEM_END  (GG_GMNDU_NEU_MSG_RX_PKTASSEM_START+\
											    GG_GMNDU_NEU_RX_PKTASSEM_L)

#define GG_GMNDU_NEU_HLT_RX_PKTASSEM_START  GG_GMNDU_NEU_MSG_RX_PKTASSEM_START

#define GG_GMNDU_NEU_HLT_INST_START  align(GG_GMNDU_NEU_MSG_RX_PKTASSEM_END,4)
#define GG_GMNDU_NEU_HLT_INST_END	  (GG_GMNDU_NEU_HLT_INST_START+\
											    GG_GMNDU_NEU_INST_L)

#define GG_GMNDU_NEU_MSG_TX_HEADER_START  align(GG_GMNDU_NEU_HLT_INST_END, 4)
#define GG_GMNDU_NEU_MSG_TX_HEADER_END    (GG_GMNDU_NEU_MSG_TX_HEADER_START+\
												GG_GMNDU_NEU_TX_HEADER_L)

#define GG_GMNDU_NEU_HLT_TX_HEADER_START   align (GG_GMNDU_NEU_MSG_TX_HEADER_END,4)
#define GG_GMNDU_NEU_HLT_TX_HEADER_END	   (GG_GMNDU_NEU_HLT_TX_HEADER_START+\
										    GG_GMNDU_NEU_HLT_HEADER_L)

#define GG_GMNDU_NEU_START         GG_GMNDU_NEU_MSG_INST_START
#define GG_GMNDU_NEU_LAST          GG_GMNDU_NEU_HLT_TX_HEADER_END
#define GG_GMNDU_NEU_USED          (GG_GMNDU_NEU_LAST-GG_GMNDU_NEU_START)

#if (GG_GMNDU_NEU_LAST > GG_GMNDU_NEU_BUFS_END)
# error GG_GMNDU_NEU Bufs size exceeded
#endif

/*----------------------------------------------------------------------------*/
/*----------------------- GMAC NDU MSG FIFOs ---------------------------------*/
/*----------------------------------------------------------------------------*/
/* Attention! Please make sure that the normal,high,low fifos are in the same
   order as below. This order is assumed in ndu/ndugmac.c                     */
#define GG_GMNDU_MSG_NORMAL_START     GG_GMNDU_MSG_FIFOS_START
#define GG_GMNDU_MSG_NORMAL_END       (GG_GMNDU_MSG_NORMAL_START+\
                                      (GG_NDU_MSG_NORMAL_FIFO_L*2))

#define GG_GMNDU_MSG_HIGH_START       GG_GMNDU_MSG_NORMAL_END
#define GG_GMNDU_MSG_HIGH_END         (GG_GMNDU_MSG_HIGH_START+\
                                      (GG_NDU_MSG_HIGH_FIFO_L*2))

#define GG_GMNDU_MSG_LOW_START    align(GG_GMNDU_MSG_HIGH_END, 2)
#define GG_GMNDU_MSG_LOW_END      (GG_GMNDU_MSG_LOW_START + \
                                   (GG_NDU_MSG_LOW_FIFO_L*2))

#define GG_GMNDU_MSG_FIFOS_LAST   GG_GMNDU_MSG_LOW_END
#define GG_GMNDU_MSG_FIFOS_USED    (GG_GMNDU_MSG_FIFOS_LAST-GG_GMNDU_MSG_FIFOS_START)


#if (GG_GMNDU_MSG_FIFOS_LAST > GG_GMNDU_MSG_FIFOS_END)
# error GMNDU MSG FIFOS size exceeded
#endif

/*----------------------------------------------------------------------------*/
/* PER CHANNEL BUFFER ALLOCATION BEGINS                                       */
/*----------------------------------------------------------------------------*/
/* Expanding $ch = 1 */
/*----------------------------------------------------------------------------*/

#define GG_VIDEO_HEAP_C1_START    align(GG_CH1_START, GG_VIDEO_HEAP_A)
#define GG_VIDEO_HEAP_C1_END      (GG_VIDEO_HEAP_C1_START + GG_VIDEO_HEAP_L)

/* Per channel SIU data stored with channel memory */
#define GG_SIUHEAP_INST_EXT_C1_START align(GG_VIDEO_HEAP_C1_END, GG_SIUHEAP_INST_EXT_A)
#define GG_SIUHEAP_INST_EXT_C1_END   (GG_SIUHEAP_INST_EXT_C1_START + GG_SIUHEAP_INST_EXT_L)
#define GG_SIUHEAP_INST_C1_START     align(GG_SIUHEAP_INST_EXT_C1_END, GG_SIUHEAP_INST_A)
#define GG_SIUHEAP_INST_C1_END       (GG_SIUHEAP_INST_C1_START + GG_SIUHEAP_INST_L)

#define GG_CH1_LAST                 GG_SIUHEAP_INST_C1_END
#define GG_CH1_USED                 (GG_CH1_LAST  - GG_CH1_START)

#if (GG_CH1_USED > GG_CH_LENGTH)
#error GG_CH_1_END exceeds the channel length
#endif

/*******************************************************************************
 * -------------------------------- NEU Buffers -------------------------------*
 ******************************************************************************/

#define GG_NEU_B2_C1_START          align(GG_NEU_TX_B_START, GG_NEU_B2_A)
#define GG_NEU_B2_C1_END            (GG_NEU_B2_C1_START + GG_NEU_B2_L)
#define GG_NEU_B2_C1_L              GG_NEU_B2_L

#define GG_NEU_B4_C1_START          align(GG_NEU_RX_B_START, GG_NEU_B4_A)
#define GG_NEU_B4_C1_END            (GG_NEU_B4_C1_START + GG_NEU_B4_L)
#define GG_NEU_B4_C1_L              GG_NEU_B4_L

/*----------------------------------------------------------------------------*/
/* End expansion $ch = 1 */
/* Expanding $ch = 2 */
/*----------------------------------------------------------------------------*/

#define GG_VIDEO_HEAP_C2_START    align(GG_CH2_START, GG_VIDEO_HEAP_A)
#define GG_VIDEO_HEAP_C2_END      (GG_VIDEO_HEAP_C2_START + GG_VIDEO_HEAP_L)

/* Per channel SIU data stored with channel memory */
#define GG_SIUHEAP_INST_EXT_C2_START align(GG_VIDEO_HEAP_C2_END, GG_SIUHEAP_INST_EXT_A)
#define GG_SIUHEAP_INST_EXT_C2_END   (GG_SIUHEAP_INST_EXT_C2_START + GG_SIUHEAP_INST_EXT_L)
#define GG_SIUHEAP_INST_C2_START     align(GG_SIUHEAP_INST_EXT_C2_END, GG_SIUHEAP_INST_A)
#define GG_SIUHEAP_INST_C2_END       (GG_SIUHEAP_INST_C2_START + GG_SIUHEAP_INST_L)

#define GG_CH2_LAST                 GG_SIUHEAP_INST_C2_END
#define GG_CH2_USED                 (GG_CH2_LAST  - GG_CH2_START)

#if (GG_CH2_USED > GG_CH_LENGTH)
#error GG_CH_2_END exceeds the channel length
#endif

/*******************************************************************************
 * -------------------------------- NEU Buffers -------------------------------*
 ******************************************************************************/

#define GG_NEU_B2_C2_START          align(GG_NEU_TX_B_START, GG_NEU_B2_A)
#define GG_NEU_B2_C2_END            (GG_NEU_B2_C2_START + GG_NEU_B2_L)
#define GG_NEU_B2_C2_L              GG_NEU_B2_L

#define GG_NEU_B4_C2_START          align(GG_NEU_RX_B_START, GG_NEU_B4_A)
#define GG_NEU_B4_C2_END            (GG_NEU_B4_C2_START + GG_NEU_B4_L)
#define GG_NEU_B4_C2_L              GG_NEU_B4_L

/*----------------------------------------------------------------------------*/
/* End expansion $ch = 2 */

/*----------------------------------------------------------------------------*/


#endif /* GGMEMMAP_H */
/* nothing past this point */

