/*H***************************************************************************
*
* $Archive:: /boards/dsk6455_v1/examples/test/test.c                          $
* $Revision:: 1                                                               $
* $Date:: 4/11/06 9:43a                                                       $
* $Author:: Shilpab                                                           $
*
* DESCRIPTION:
*
*   
* (C) Copyright 2005 by Spectrum Digital Incorporated
* All rights reserved
*
*H***************************************************************************/

/*------------------------------------------------------------------------*
 *  This is a test program that uses DSK6455A Board Support Library to
 *  perform a confidence test on a DSK6455A.  While it is running, it
 *  will display the current test index in binary on the LEDs.  If it
 *  fails a test, the test index will blink indefinitely to alert you to
 *  the failure.  If all tests pass, all of the LEDs will blink 3 times
 *  then stay on.
 *
 *  The following tests are performed:
 *
 *  Index    Description
 *    1      Internal memory Version
 *    2      External memory(SDRAM)
 *    3      Flash
 *    4      Codec
 *    5      UserIO
 *    6      MemPattern(Fill pattern)
 *    7      MemPattern(Address Pattern)
 *    8      DSP
 *    9      LED/Swt
 *    10     Codec LoopBack
 *
 *  Please see the DSK6455A BSL Examples for more detailed information.

 *  DSP/BIOS is configured using the DSP/BIOS configuration tool.  Settings
 *  for this example are stored in a configuration file called test.cdb.  At
 *  compile time, Code Composer will auto-generate DSP/BIOS related files
 *  based on these settings.  A header file called testcfg.h contains the
 *  results of the autogeneration and must be included for proper operation.
 *  The name of the file is taken from test.cdb and adding cfg.h.
 *------------------------------------------------------------------------*/

#define test_c


/*****************************************************************************
* INCLUDE FILES
*****************************************************************************/
/*---- system and platform files -------------------------------------------*/

#include "testcfg.h"
/*
 *  The 6455 DSK Board Support Library is divided into several modules, each
 *  of which has its own include file.  The file dsk6455.h must be included
 *  in every program that uses the BSL.  This example also includes
 *  dsk6455_aic23.h, dsk6455_led.h and dsk6455_flash.h because it uses
 *  their respective BSL modules.
 */
#include "dsk6455.h"
#include "dsk6455_led.h"
#include "dsk6455_aic23.h"
#include "dsk6455_flash.h"
#include "dsk6455_eeprom.h"


/* Length of sine wave table */
#define SINE_TABLE_SIZE  48

Uint16 buffer[256];

/* Pre-generated sine wave data, 16-bit signed samples */
int sinetable[SINE_TABLE_SIZE] = {
    0x0000, 0x10b4, 0x2120, 0x30fb, 0x3fff, 0x4dea, 0x5a81, 0x658b,
    0x6ed8, 0x763f, 0x7ba1, 0x7ee5, 0x7ffd, 0x7ee5, 0x7ba1, 0x76ef,
    0x6ed8, 0x658b, 0x5a81, 0x4dea, 0x3fff, 0x30fb, 0x2120, 0x10b4,
    0x0000, 0xef4c, 0xdee0, 0xcf06, 0xc002, 0xb216, 0xa57f, 0x9a75,
    0x9128, 0x89c1, 0x845f, 0x811b, 0x8002, 0x811b, 0x845f, 0x89c1,
    0x9128, 0x9a76, 0xa57f, 0xb216, 0xc002, 0xcf06, 0xdee0, 0xef4c
};

/* Codec configuration settings */
DSK6455_AIC23_Config config = { \
    0x0017,  /* 0 DSK6455_AIC23_LEFTINVOL  Left line input channel volume */ \
    0x0017,  /* 1 DSK6455_AIC23_RIGHTINVOL Right line input channel volume */\
    0x00d8,  /* 2 DSK6455_AIC23_LEFTHPVOL  Left channel headphone volume */  \
    0x00d8,  /* 3 DSK6455_AIC23_RIGHTHPVOL Right channel headphone volume */ \
    0x0011,  /* 4 DSK6455_AIC23_ANAPATH    Analog audio path control */      \
    0x0000,  /* 5 DSK6455_AIC23_DIGPATH    Digital audio path control */     \
    0x0000,  /* 6 DSK6455_AIC23_POWERDOWN  Power down control */             \
    0x0043,  /* 7 DSK6455_AIC23_DIGIF      Digital audio interface format */ \
    0x0081,  /* 8 DSK6455_AIC23_SAMPLERATE Sample rate control */            \
    0x0001   /* 9 DSK6455_AIC23_DIGACT     Digital interface activation */   \
};

/* Define buffers for host side communication */
#pragma DATA_SECTION(cmdbuf , "CMD_BUFFER")
Uint16  cmdbuf[512];
#pragma DATA_SECTION(databuf , "DATA_BUFFER")
Uint16  databuf[512];


/*--------------------------------ACCESSORY FUNCTIONS-------------------------------------*/
/*F***************************************************************************
* NAME:  TEST_sleep(Int16 sleeptime) 
*
* DESCRIPTION: Accessory Function
*
* NOTES:
*   
*F***************************************************************************/
void TEST_sleep(Int16 sleeptime)
{
	DSK6455_waitusec( sleeptime * 1000 );
}
/*----------------------------END ACCESSORY FUNCTIONS-------------------------------------*/
/*-----------------------------LED FUNCTIONS-----------------------------------------------*/
/*F***************************************************************************
* NAME:  LED_binary(Int16 ledmask) 
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
void LED_binary(Int16 ledmask)
{
    Int16 i, bit;

    /* Walk through the bits in num setting corresponding LEDs */
    bit = 1;
    for (i = 0; i < 4; i++)
    {
        if (ledmask & bit)
            DSK6455_LED_on(i);
        else
            DSK6455_LED_off(i);
        bit = bit << 1;
    }

}

/*F***************************************************************************
* NAME:  LED_blink(Int16 ledmask, Int16 count)  
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
void LED_blink(Int16 ledmask, Int16 count)
{
    while (count > 0)
    {
        LED_binary(ledmask);
        TEST_sleep(100);
        LED_binary(0);
        TEST_sleep(150);
        count--;
    }
}

/*F***************************************************************************
* NAME:  void LED_error(Int16 ledmask)  
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
void LED_error(Int16 ledmask)
{
    while(1)
        LED_blink(ledmask, 1);
}
/*-----------------------------END OF LED FUNCTIONS-----------------------------------------------*/
/*-----------------------------MEMORY FUNCTIONS---------------------------------------------------*/
/*F***************************************************************************
* NAME:  MEM_fill(Uint32 start, Uint32 len, Uint32 val) 
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 MEM_fill(Uint32 start, Uint32 len, Uint32 val)
{
    Uint32 i, end;

    /* Calculate end of range */
    end = start + len;

    /* Fill a range with a value */
    for (i = start; i < end; i+=sizeof(Uint32))
    {
        *((Uint32 *)i) = val;
    }

    /* Verify the data */
    for (i = start; i < end; i+=sizeof(Uint32))
    {
        if (*((Uint32 *)i) != val)
            return 1;
    }

    return 0;
}
/*F***************************************************************************
* NAME:  MEM_addr(Uint32 start, Uint32 len)  
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 MEM_addr(Uint32 start, Uint32 len)
{
    Uint32 i, end;

    /* Calculate end of range */
    end = start + len;

    /* Fill the range with its address */
    for (i = start; i < end; i+=sizeof(Uint32))
    {
        *((Uint32 *)i) = i;
    }

    /* Verify the data */
    for (i = start; i < end; i+=sizeof(Uint32))
        if (*((Uint32 *)i) != i)
            return 1;

    return 0;
}
/*F***************************************************************************
* NAME: MEM_addrInv(Uint32 start, Uint32 len)  
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 MEM_addrInv(Uint32 start, Uint32 len)
{
    Uint32 i, end;

    /* Calculate end of range */
    end = start + len;

    /* Fill the range with its address */
    for (i = start; i < end; i+=sizeof(Uint32))
    {
        *((Uint32 *)i) = ~i;
    }

    /* Verify the data */
    for (i = start; i < end; i+=sizeof(Uint32))
        if (*((Uint32 *)i) != (~i))
            return 1;

    return 0;
}

/*F***************************************************************************
* NAME:  MEM_test(Uint32 start, Uint32 len, Int16 patterntype)  
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 MEM_test(Uint32 start, Uint32 len, Int16 patterntype)
{
    Int16 status = 0;

    if (!patterntype)
    {
        /* Run the fill tests */
        status |= MEM_fill(start, len, 0x00000000);
        status |= MEM_fill(start, len, 0x55555555);
        status |= MEM_fill(start, len, 0xAAAAAAAA);
        status |= MEM_fill(start, len, 0xFFFFFFFF);
    } else
    {
        /* Run the address tests */
        status |= MEM_addr(start, len);
        status |= MEM_addrInv(start, len);
    }

    return status;
}
/*F***************************************************************************
* NAME:  MEM_walking(Uint32 add)
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 MEM_walking(Uint32 add)
{
    Int16 i;
    Uint32 mask, *pdata;

    pdata = (Uint32 *)add;

    /* Walking ones and zeros */
    mask = 1;
    for (i = 0; i < 32; i++)
    {
        /* Test one in bit position i */
        *pdata = mask;
        if (*pdata != mask)
            return 1;

        /* Test zero in bit position i */
        *pdata = ~mask;
        if (*pdata != (~mask))
            return 1;

        mask = mask << 1;
    }

    return 0;
}
/*F***************************************************************************
* NAME:  MEM_walking(Uint32 add)
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 MEM_bytestrobe(Uint32 add)
{
    Uint32 *pdata;
    Uint8 *pcheck;

    /* Write pattern */
    pdata = (Uint32 *)add;
    *pdata = 0x12345678;

    /* Do dummy write */
    pdata = (Uint32 *)DSK6455_FLASH_BASE;
    *pdata = 0x80808080;

    /* Check pattern */
    pcheck = (Uint8 *)add;
    if (*pcheck++ != 0x78)
        return 0x10;
    if (*pcheck++ != 0x56)
        return 0x20;
    if (*pcheck++ != 0x34)
        return 0x40;
    if (*pcheck++ != 0x12)
        return 0x80;

    return 0;
}

/*---------------------------END OF MEMORY FUNCTIONS------------------------------------------*/

/*---------------------------FLASH FUNCTIONS--------------------------------------------------*/
/*F***************************************************************************
* NAME:  TEST_flashID()
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 TEST_flashID()
{
    Uint8 MfgId,DevId;
    Int16 i;

    /* Reset flash */
    *((Uint8 *)DSK6455_FLASH_BASE) = 0xf0;

    /* Configure to read manufacturer ID */
    *((Uint8 *)DSK6455_FLASH_BASE) = 0xaa;
    *((Uint8 *)DSK6455_FLASH_BASE) = 0x55;
    *((Uint8 *)DSK6455_FLASH_BASE) = 0x90;

    /* Insert small delay for device to respond */
    for (i = 0; i < 10; i++);

    /* Read IDs */
    MfgId = *((Uint8 *)DSK6455_FLASH_BASE);
    DevId = *((Uint8 *)DSK6455_FLASH_BASE + 1);

    /* Reset flash */
    *((Uint8 *)DSK6455_FLASH_BASE) = 0xf0;

    /* Check IDs */
    if ((MfgId != 0x0001) || (DevId != 0xa3))
        return 1;

    /* Test passed */
    return 0;
}
/*---------------------------END OF FLASH FUNCTIONS-------------------------------------------*/
/*---------------------------DMA FUNCTIONS----------------------------------------------------*/
/*F***************************************************************************
* NAME:  TEST_dma()
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 TEST_dma()
{
    /* Test passed */
    return 0;
}
/*---------------------------END DMA FUNCTIONS------------------------------------------------*/

/*---------------------------TIMER FUNCTIONS--------------------------------------------------*/
/*F***************************************************************************
* NAME:  TEST_timer()
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 TEST_timer()
{
    /* Wait for 100 timer interrupts */
    TEST_sleep(100);
    
    return 0;
}
/*---------------------------END TIMER FUNCTIONS----------------------------------------------*/
/*---------------------------McBSP FUNCTIONS--------------------------------------------------*/
/*F***************************************************************************
* NAME:  TEST_mcbsp(Int16 devid, Int16 delayed)
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 TEST_mcbsp(Int16 devid, Int16 delayed)
{

    /* Test passed */
    return 0;
}
/*F***************************************************************************
* NAME:  TEST_mcbsp0()
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 TEST_mcbsp0()
{
    /* Do an internal loopback test on McBSP0 */
    return 0;
}
/*F***************************************************************************
* NAME:  TEST_mcbsp1()
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
Int16 TEST_mcbsp1()
{
    /* Do an internal loopback test on McBSP1 */
    return 0;
}
/*---------------------------END McBSP FUNCTIONS----------------------------------------------*/

/*---------------------------DIAGNOSTIC UTILITY FUNCTIONS-------------------------------------*/

/*---------------------OVERALL DIAGNOSTIC TESTS FUNCTIONS-----------------------------*/
/*F***************************************************************************
* NAME:  TEST_intMemVersion()  
*
* DESCRIPTION:Gets the board version and CPLD version for the target.
*
* NOTES:CALLED BY DSP DIAGNOSTICS IN THE DIAGNOSTIC UTILITY
*   
*F***************************************************************************/
Int16 TEST_intMemVersion()
{
    Int16 status = 0;

    /* Store CPLD version */
    databuf[2] = (DSK6455_rget(DSK6455_VERSION) >> 4) & 0xf;
    
    /* Store board version */
    databuf[3] = DSK6455_rget(DSK6455_VERSION) & 0x7;
    
    return status;
}

/*F***************************************************************************
* NAME:  TEST_extMem()  
*
* DESCRIPTION:Test SDRAM by filling with both pattern and address values.
*
* NOTES:CALLED BY EXTERNAL MEMORY IN THE DIAGNOSTIC UTILITY
*   
*F***************************************************************************/
Int16 TEST_extMem()
{
    Int16 status = 0;

    /* Check external memory (byte 0x40000 to byte 0x1000000) */
    status |= MEM_test(0xe0000000, 0x1000000, 0);
    status |= MEM_test(0xe0000000, 0x1000000, 1);
    status |= MEM_walking(0xe0000000);
    status |= MEM_walking(0xe0000004);
    status |= MEM_bytestrobe(0xE0000000);

    return status;
}
/*F***************************************************************************
* NAME:  TEST_flash()  
*
* DESCRIPTION:
*
* NOTES:CALLED BY FLASH DIAGNOSTICS IN THE DIAGNOSTIC UTILITY
*   
*F***************************************************************************/
Int16 TEST_flash()
{
    Uint16 i, j, k, page;
    Uint32 startaddr, addr;

	/* Check Flash IDs */
    if (TEST_flashID())
        return 1;

    /* Program the Flash page by page, 256 words at a time */
    for (page = 0; page < 1; page++)
    {
        /* Erase all of the Flash */
        DSK6455_FLASH_erase(DSK6455_FLASH_BASE + DSK6455_FLASH_SECTORSIZE, DSK6455_FLASH_SECTORSIZE *8); 

        addr = DSK6455_FLASH_BASE + DSK6455_FLASH_SECTORSIZE;
        for (i = 0; i < 8; i++)
        {
            for (j = 0; j < 128; j++)
            {
                startaddr = addr;
                for (k = 0; k < 256; k++)
                {
                    buffer[k] = addr + i + page;
                    addr += 2;
                }
                DSK6455_FLASH_write((Uint32)buffer, startaddr, 512);
            }
        }
    }

    /* Verify the Flash page by page, 256 words at a time */
    for (page = 0; page < 1; page++)
    {
        addr = DSK6455_FLASH_BASE + DSK6455_FLASH_SECTORSIZE;
        for (i = 0; i < 8 ; i++)
        {
            for (j = 0; j < 128; j++)
            {
                startaddr = addr;
                DSK6455_FLASH_read(startaddr, (Uint32)buffer, 512);
                for (k = 0; k < 256; k++)
                {
                    if (buffer[k] != ((addr + i + page) & 0xffff))
                        return page + 1;
                    addr += 2;
                }
            }
        }
    }

    return 0;
}
/*F***************************************************************************
* NAME:  TEST_codec()  
*
* DESCRIPTION:
*
* NOTES:CALLED BY CODEC DIAGNOSTICS IN THE DIAGNOSTIC UTILITY
*   
*F***************************************************************************/
Int16 TEST_codec()
{
    DSK6455_AIC23_CodecHandle hCodec;
    Int16 i, j;

    /* Start the codec */
    hCodec = DSK6455_AIC23_openCodec(0, &config);

    /* Generate a 1KHz sine wave for 1 second */
    for (i = 0; i < 1000; i++)
    {
        for (j = 0; j < SINE_TABLE_SIZE; j++)
        {
            while (!DSK6455_AIC23_write32(hCodec, sinetable[j]));
            while (!DSK6455_AIC23_write32(hCodec, sinetable[j]));
        }
    }

    /* Close codec */
    DSK6455_AIC23_closeCodec(hCodec);

    return 0;
}
/*F***************************************************************************
* NAME:  TEST_userIO()  
*
* DESCRIPTION:
*
* NOTES:CALLED BY LED AND DIP SWITCH DIAGNOSTICS IN THE DIAGNOSTIC UTILITY
*   
*F***************************************************************************/
Int16 TEST_userIO()
{
    int i, j;
    
    for (j = 0; j < 1; j++)
    {
        for (i = 0; i < 4; i++)
        {
            if (DSK6455_DIP_get(i) == 1)
                DSK6455_LED_on(i);
            else
                DSK6455_LED_off(i);
        }
        TEST_sleep(10);        
    }
    
    return 0;
}
/*---------------------END OVERALL DIAGNOSTIC TESTS FUNCTIONS-------------------------*/

/*---------------------ADVANCED DIAGNOSTIC TESTS FUNCTIONS-----------------------------*/
/*F***************************************************************************
* NAME:  TEST_memPat()  
*
* DESCRIPTION:Test SDRAM by filling with pattern values.
*
* NOTES:CALLED WHEN TESTING MEMORY IN THE ADVANCED TAB
*   
*F***************************************************************************/
Int16 TEST_memPat()
{
    Int16 status = 0;
    
    /* Test external memory (SDRAM) */
    status |= MEM_test(0xe0000000, 0x1000000, 0);
        
    return status;
}
/*F***************************************************************************
* NAME:  TEST_memAddr()  
*
* DESCRIPTION:Test SDRAM by filling with address values.
*
* NOTES:CALLED WHEN TESTING MEMORY IN THE ADVANCED TAB 
*   
*F***************************************************************************/
Int16 TEST_memAddr()
{
    Int16 status = 0;
      
    /* Test external memory (SDRAM) */
    status |= MEM_test(0xe0000000, 0x1000000, 1);

    return status;
}
/*F***************************************************************************
* NAME:  TEST_dsp()  
*
* DESCRIPTION:Tests Timer,DMA,McBSP0 & McBSP1
*
* NOTES:CALLED WHEN TESTING DSP IN THE ADVANCED TAB 
*   
*F***************************************************************************/
Int16 TEST_dsp()
{
    Int16 status = 0;
    
    /* Call sub-tests */
    if (TEST_dma())
        status |= 0x01;
    
    if (TEST_timer())
        status |= 0x02;
  
    if (TEST_mcbsp0())
        status |= 0x04;
    
    if (TEST_mcbsp1())
        status |= 0x08;

    return status;
}
/*F***************************************************************************
* NAME:  TEST_readSwitch()  
*
* DESCRIPTION:
*
* NOTES:CALLED WHEN TESTING LED/SWT IN THE ADVANCED TAB 
*   
*F***************************************************************************/
Int16 TEST_readSwitch()
{
    Int16 i, n;
    
    n = 0;

    for (i = 3; i >= 0; i--)
        n = (n << 1) | DSK6455_DIP_get(i);

    LED_binary(n);        
 
    databuf[2] = n;
    
    return 0;
}
/*F***************************************************************************
* NAME:  TEST_codecLoopBack()  
*
* DESCRIPTION:
*
* NOTES:CALLED WHEN TESTING CODEC IN THE ADVANCED TAB 
*   
*F***************************************************************************/
Int16 TEST_codecLoopBack()
{
    DSK6455_AIC23_CodecHandle hCodec;
    Int16 i, j, data;
    Uint16 *pbuf;
        
    /* Start the codec */
    hCodec = DSK6455_AIC23_openCodec(0, &config);

    /* Set volume */
	DSK6455_AIC23_outGain(hCodec, cmdbuf[1] & 0x7f);

	/* Set mute */
	if (cmdbuf[2])
	{
		DSK6455_AIC23_mute(hCodec, TRUE);
	} else
	{
	    DSK6455_AIC23_mute(hCodec, FALSE);
	}

    if (cmdbuf[3])
    {
		/* Configure the codec for loopback mode*/
		DSK6455_AIC23_loopback(hCodec,1);  
       
    } else
    {	
	    DSK6455_AIC23_loopback(hCodec,0);  
	}
	
    /* Run for a second */
	for (i = 0; i < 1000; i++)
	{
		pbuf = databuf + 2;    
	    for (j = 0; j < SINE_TABLE_SIZE; j++)
	    {
	    	while (!DSK6455_AIC23_write32(hCodec, sinetable[j]));
	        while (!DSK6455_AIC23_read32(hCodec, (Uint32 *)&data));
	        *pbuf++ = data;
	        while (!DSK6455_AIC23_write32(hCodec, sinetable[(j << 1) % SINE_TABLE_SIZE]));
	        while (!DSK6455_AIC23_read32(hCodec, (Uint32 *)&data));
	        *pbuf++ = data;
	 	}        
	 }
   /* Close codec */
   DSK6455_AIC23_closeCodec(hCodec);
    
   return 0;
}
/*--------------------- END ADVANCED DIAGNOSTIC TESTS FUNCTIONS--------------------------*/

/*------------------------END OF DIAGNOSTIC UTILITY FUNCTIONS---------------------------------*/

/*F***************************************************************************
* NAME:  TEST_execute()  
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
void TEST_execute(Int16 (*funchandle)(), Int16 ledmask, Int16 insertdelay)
{
    Int16 status;

    /* Display test ID */
    LED_binary(ledmask);

    /* Call test function */
    status = funchandle();

    /* Pause so LEDs can be read */
    if (insertdelay)
        TEST_sleep(100);

    /* Check for test fail */
    if (status)
        LED_error(ledmask);
}
/*F***************************************************************************
* NAME:  TEST_executeDiag()  
*
* DESCRIPTION:
*
* NOTES:
*   
*F***************************************************************************/
void TEST_executeDiag(Int16 (*funchandle)(), Int16 ledmask, Int16 insertdelay)
{
    Int16 status;

    /* Call test function */
    status = funchandle();

    databuf[0] = 0x1234;
    databuf[1] = status;
}

/*------------------------------START OF MAIN FUNCTION--------------------------------*/
main()
{
    /* Call BSL init */
    DSK6455_init();

    /* Initialize DIP switch and LEDs */
    DSK6455_LED_init();
    DSK6455_DIP_init();
		
    
#if defined( BUILD_DSK_DIAGS )

	#if(0)
	    /* Iniitalize the command buffer for debug purposes */
	    cmdbuf[0] = 6;
	    cmdbuf[1] = 0x79;
	    cmdbuf[2] = 0;
	    cmdbuf[3] = 1;
	#endif
  
    /* Execute command for host */
    switch(*cmdbuf)
    {
        case 0:  /* Internal memory */
            TEST_executeDiag(TEST_intMemVersion, 0, 0);
            break;
            
        case 1:  /* External memory (typical) */
            TEST_executeDiag(TEST_extMem, 0, 0);
            break;

        case 2:  /* External memory (special); */
            TEST_executeDiag(TEST_flashID, 0, 0);
            break;
            
        case 3:  /* Codec test */
            TEST_executeDiag(TEST_codec, 0, 0);        
            break;
            
        case 4:  /* LED and timer test */
            TEST_executeDiag(TEST_timer, 0, 0);
            break;
            
        case 5:  /* DIP switch test */
            TEST_executeDiag(TEST_userIO, 0, 0);
            break;

        case 6:  /* Test codec external loopback */
            TEST_executeDiag(TEST_codecLoopBack, 0, 0);
            break;

        case 7:   /* Test memory (fill pattern) */
            TEST_executeDiag(TEST_memPat, 0, 0);
            break;

        case 8:  /* Test DSP */
            TEST_executeDiag(TEST_dsp, 0, 0);
            break;

        case 9:  /* DIP switch read */
            TEST_executeDiag(TEST_readSwitch, 0, 0);
            break;
                              
        case 10:  /* Test memory (address pattern) */
            TEST_executeDiag(TEST_memAddr, 0, 0);
            break;

        case 11:  /* External Flash */
            TEST_executeDiag(TEST_flash, 0, 0);
            break;
            
        default:
            break;
    }
#endif


#if defined (_DEBUG )
    TEST_execute(TEST_intMemVersion, 1, 0);
    TEST_execute(TEST_extMem,        2, 0);
    TEST_execute(TEST_flash,         3, 0);
    TEST_execute(TEST_codec,         4, 0);
    TEST_execute(TEST_userIO,        5, 0);        
    TEST_execute(TEST_memPat,        6, 0);    
    TEST_execute(TEST_memAddr,       7, 0);
    TEST_execute(TEST_dsp,           8, 0);
    TEST_execute(TEST_readSwitch,    9, 0);
    TEST_execute(TEST_codecLoopBack, 10, 0); 
    
    /* All tests complete, board passes */
    LED_blink(0xf, 3);

    /* Leave all LEDs on */
    LED_binary(0xf);
#endif   

    /* Set artificial breakpoint to return control to host */
    asm( " .long 0x1001E000" );

    while(1);   
        
}
/*------------------------------END OF MAIN FUNCTION--------------------------------*/
