------------------------------------------------------------------------------------
-- $Archive:: /VHDL/product/dsk6455/usr_cpld/dsk6455.vhd                           $
-- $Revision:: 4                                                                   $
-- $Date:: 6/01/06 1:18p                                                           $
-- $Author:: Tonyc                                                                 $
--
--
-- Copyright (c) 2002,2003,2004, Spectrum Digital Incorporated
-- All rights reserved
------------------------------------------------------------------------------------
------------------------------------------------------------------------------------
-- Start the real code                  
------------------------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;

entity dsk6455 is
  port
  (  
    CLKIN        : in    std_logic; -- 12 MHz clock in
    OPT_CLK1     : in    std_logic; -- 8 MHz clock in
    EMU_RSTn     : in    std_logic; -- Emulator reset from USB block
    PUSHBRS      : in    std_logic; -- Push button reset   
    HPIRSn       : in    std_logic; -- HPI reset from optional DC
    
    -- DSP Memory interface signals for internal CPLD registers.  From C64xx B port.
    DSP_DQ       : inout std_logic_vector( 7 downto 0 );    -- DSP Data bus
	  DSP_ADDR     : in    std_logic_vector( 3 downto 0 );    -- DSP Address bus
    DSP_WEn      : in    std_logic;                         -- DSP Write strobe
    DSP_ARNW     : in    std_logic;                         -- DSP Read strobe
    DSP_OEn      : in    std_logic;                         -- DSP Output enable
     
    DSP_CS2n     : in    std_logic; 
    DSP_CS3n     : in    std_logic; 
    DSP_CS4n     : in    std_logic; 
    DSP_CS5n     : in    std_logic; 
    
    -- User/Board Support
    USER_SW      : in    std_logic_vector( 3 downto 0 );  -- User swtiches
    USER_LED     : out   std_logic_vector( 3 downto 0 );  -- Uwer led
    PWB_REV      : in    std_logic_vector( 2 downto 0 );  -- PWB revision
    
    -- Daughter Card Support
    DC_STAT      : in    std_logic_vector( 1 downto 0 );   -- DC Status
    DC_CNTL      : out   std_logic_vector( 1 downto 0 );   -- DC Control
    DC_DBUF_DIR  : out   std_logic;                 -- DC Data buffer direction
    DC_DBUF_OEn  : out   std_logic;                 -- DC Data buffer output enable
    DC_CNTL_OEn  : out   std_logic;                 -- DC Control buffer enable
    DC_DETn      : in    std_logic;                 -- DC Detec            -- DC Reset 
    DC_RESETn    : out   std_logic;     
    -- McBSP Multiplexer Control
    MCBSP_SELA   : out   std_logic;                 -- Codec/DC McBsp 1 mux cntl
                             
    -- Misc. Stuff
    DSP_RSn_LED  : out   std_logic;                 -- DSP reset led
    --FLASH_PAGE0  : out   std_logic;                 -- Flash Address 19 
    --FLASH_PAGE1  : out   std_logic;                 -- Flash Address 20
    ABA0  				: in   std_logic;                 -- Flash Address 19 
    ABA1  				: in   std_logic;                 -- Flash Address 20

    CPLD_CLK_OUT : out   std_logic;                 -- Place holder
           
    -- CLOCKING FOR EMIF AND CPU 
    TEST_PIN        : out   std_logic;
    DSP_PLL_CLK     : out   std_logic; 
    DSPPLL_SW       : in    std_logic_vector( 3 downto 1 );
    CPU_DSPPLL_S    : out   std_logic_vector( 1 downto 0 );
    EMIF_PLL_CLK    : out   std_logic;

    -- CPUB, MISC2
    CPUB_PRESENTn   	: in    std_logic;
    CPUB_RESETn     	: out   std_logic;
    HURRICANE_RESETn  : in    std_logic;
    DSPA_PORZ         : out   std_logic;
    DSPA_RESETZ       : out   std_logic;
    DSPA_RESETSTATn   : in    std_logic;
    MII_RESETZ        : out   std_logic;
    BD_PWR_ON_RSZ     : in    std_logic;
    SVS_RSTn          : out   std_logic;
    BRD_RSTn          : out   std_logic

);
end dsk6455;

-----------------------------------------------------------------------------------
-- Include standard librariess
-----------------------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;
-- use work.std_arith.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

-----------------------------------------------------------------------------------
-- Include fpga specifics here if required.  
-- act3  is for Actel 54sx devices.  We normally use this for the hardwired
-- clock definition.
-----------------------------------------------------------------------------------
-- library act3;
-- use act3.components.all;

architecture behavior_dsk6455 of dsk6455 is      

constant CPLD_VERSION         : std_logic_vector(3 downto 0)  := "0011"; 

-----------------------------------------------------------------------------------
-- Add local components in here  
-----------------------------------------------------------------------------------
--component MyComponent
--port
--(
--);
--end component;

-----------------------------------------------------------------------------------
-- Add signals 
-----------------------------------------------------------------------------------

-- CPLD Register signals
signal    CpldReg0        : std_logic_vector( 7 downto 0 );  
signal    CpldReg1        : std_logic_vector( 7 downto 0 );  
signal    CpldReg4        : std_logic_vector( 7 downto 0 );  
signal    CpldReg6        : std_logic_vector( 7 downto 0 );  
signal    CpldReg7        : std_logic_vector( 7 downto 0 ); 
signal    OPT_CLK1_DIV    : std_logic;
signal    MuxD            : std_logic_vector( 7 downto 0 ); 

signal    ChipEnables     : std_logic_vector( 7 downto 0 );
signal    CpldRegCs0      : std_logic;
signal    CpldRegCs1      : std_logic;
signal    CpldRegCs2      : std_logic;
signal    CpldRegCs3      : std_logic;
signal    CpldRegCs4      : std_logic;
signal    CpldRegCs5      : std_logic;
signal    CpldRegCs6      : std_logic;
signal    CpldRegCs7      : std_logic;

signal    CpldClkOut      : std_logic;  

signal    DcCs0           : std_logic;
signal    DcCs1           : std_logic;
signal    CpldCs          : std_logic;
signal    FlashCs         : std_logic;

signal    HurricaneResetn : std_logic;
signal    PowOnRs         : std_logic;
signal    NewDspAddr      : std_logic_vector(3 downto 0 );

-----------------------------------------------------------------------------------
-- The implementation
-----------------------------------------------------------------------------------

begin 
-----------------------------------------------------------------------------------
-- Map the other components
-----------------------------------------------------------------------------------

-----------------------------------------------------------------------------------
-- Now define the logic
-----------------------------------------------------------------------------------
   
  -- Map signals to generic useage
  CpldCs 	<= DSP_CS2n;
  FlashCs <= DSP_CS3n;
  DcCs0  	<= DSP_CS4n;
  DcCs1  	<= DSP_CS5n;
 
  -- Generate a reset from the three sources.  
  --     
  -- Power on reset sources
  PowOnRs <= '0' when     EMU_RSTn         = '0' 
                       or BD_PWR_ON_RSZ    = '0'
                       or PUSHBRS          = '1' 
                       else '1';        
 
  
  -- SVS_RSTn is to/from USB regulator.  Add in board regulator signal
  SVS_RSTn     <= '0' when BD_PWR_ON_RSZ  = '0' else 'Z';

  -- Board reset, flash
  BRD_RSTn     <= '0' when PowOnRs  = '0' else '1';

  -- DSP Warm reset + power on for DSP.
  DSPA_RESETZ  <= '0' when PowOnRs  	     = '0' 
                        or HPIRSn          = '0' 
                        or HurricaneResetn = '0'
                        else '1';

  DSPA_PORZ    <= '0' when PowOnRs  = '0' else '1';

  -- Put the reset status into the led so that we can see if the dsp is hung
  -- in reset.
  --
  DSP_RSn_LED  <= '0' when PowOnRs  = '0' or DSPA_RESETSTATn = '0'  else '1';    

  OPT_CLK1_DIV <= '1'; -- Place holder
	                        
	  
  -- Generate a CPLD clockout from clock input.  This is a place holder just in
  -- case we need it later.
  --
  process( PowOnRs, CLKIN ) 
  begin
		if PowOnRs = '0' then
        CpldClkOut <= '0';
		elsif CLKIN'event and CLKIN = '1' then    
		    CpldClkOut <= not CpldClkOut;
		end if;
	end process;  

	CPLD_CLK_OUT <= CpldClkOut;
	
  -- #########################################################################
  -- Generic register addresss decode and register chip select generation.
  -- VHDL compiler will reduce any unused logic so we can be verbose.
	--
	NewDspAddr <= DSP_ADDR(1) & DSP_ADDR(0) & ABA1 & ABA0;
	
	process(  NewDspAddr )
	begin
		case NewDspAddr( 3 downto 0) is
    	when "0000"  => ChipEnables  <= "00000001"; 
      when "0001"  => ChipEnables  <= "00000010"; 
      when "0010"  => ChipEnables  <= "00000100";
      when "0011"  => ChipEnables  <= "00001000";     
      when "0100"  => ChipEnables  <= "00010000";   
      when "0101"  => ChipEnables  <= "00100000";  
      when "0110"  => ChipEnables  <= "01000000";   
      when "0111"  => ChipEnables  <= "10000000";        
			when others => ChipEnables  <= "00000000";
		end case;
	end process;     
  
  CpldRegCs0 <= '1' when ChipEnables(0) = '1' and CpldCs = '0' else '0';
  CpldRegCs1 <= '1' when ChipEnables(1) = '1' and CpldCs = '0' else '0';
  CpldRegCs2 <= '1' when ChipEnables(2) = '1' and CpldCs = '0' else '0';
  CpldRegCs3 <= '1' when ChipEnables(3) = '1' and CpldCs = '0' else '0';
  CpldRegCs4 <= '1' when ChipEnables(4) = '1' and CpldCs = '0' else '0';
  CpldRegCs5 <= '1' when ChipEnables(5) = '1' and CpldCs = '0' else '0';
  CpldRegCs6 <= '1' when ChipEnables(6) = '1' and CpldCs = '0' else '0';
  CpldRegCs7 <= '1' when ChipEnables(7) = '1' and CpldCs = '0' else '0';
	
	-- #########################################################################
  -- Generate logic for each CPLD register and assign it's write, read, and
  -- pin values if necessary.
  -- 
  -- All CPLD register writes occur on the rising edge DSP write strobe.
	--  
	 
	-- ========================================================================
	-- REG 0: User Register 
	-- Bit 3-0   Led 3-0
	-- Bit 7-4   Switch 3-0
  process( PowOnRs, DSP_WEn, CpldRegCs0, DSP_DQ ) 
  begin
		if PowOnRs = '0' then
        CpldReg0(3 downto 0 ) <= "0000";
		elsif DSP_WEn'event and DSP_WEn = '1' then    
		  if( CpldRegCs0 = '1' ) then
		    CpldReg0( 3 downto 0 ) <= DSP_DQ( 3 downto 0 );
		  end if;
		end if;
	end process;  
  
  CpldReg0(7 downto 4 )  <= USER_SW( 3 downto 0 ); 
  USER_LED( 3 downto 0 ) <= not CpldReg0(3 downto 0 );  
  
  -- ========================================================================
	-- REG 1: DC Register
	-- Bit 1-0   DC_CNTL 1-0
	-- Bit 2     NU  read 0
	-- Bit 3     DC_RESET
	-- Bit 5-4   DC_STAT 1-0
	-- Bit 6     NU  read 0
	-- Bit 7     DC_DETECT
	--
  process( PowOnRs, DSP_WEn, CpldRegCs1, DSP_DQ ) 
  begin
		if PowOnRs = '0' then
        CpldReg1(1 downto 0 ) <= "00";   
        CpldReg1(3)           <= '0';  -- not Reset by default
		elsif DSP_WEn'event and DSP_WEn = '1' then    
		  if( CpldRegCs1 = '1' ) then
		    CpldReg1( 1 downto 0 ) <= DSP_DQ( 1 downto 0 );
		    CpldReg1(3)            <= DSP_DQ(3);
		  end if;
		end if;
	end process;  
   

  CpldReg1(2)            <= '0';
  CpldReg1(5 downto 4 )  <= DC_STAT( 1 downto 0 ); 
  CpldReg1(6)            <= '0';
  CpldReg1(7)            <= not DC_DETn;
  
  DC_CNTL( 1 downto 0 )  <= CpldReg1( 1 downto 0 );
      
  -- HPIRSn not included in the DC_RESETn equation.  This should prevent the
  -- DC from holding itself in reset if HPIRSn is active.
  DC_RESETn              <= '0' when    CpldReg1(3)  = '1' 
	                                   or PowOnRs = '0' else '1';
       
  -- ========================================================================
	-- REG 4: Version Register
	-- Bit 2-0   PWB Revision 2-0
	-- Bit 3     NU read 0
	-- Bit 7-4   CPLD version
	CpldReg4(7 downto 0 ) <= CPLD_VERSION(3 downto 0 ) & '0' & PWB_REV(2 downto 0 );   
	
	-- ========================================================================
	-- REG 6: Misc. Register
	-- Bit 0     McBSP1 select 
	-- Bit 1     Flash Page/Flash Address 19       
	-- Bit 2     Flash Page/Flash Address 20
	-- Bit 3     DSPPLL_SELECT1 (READ) SW3-5
	-- Bit 4     DSPPLL_SELECT2 (READ) SW3-6
	-- Bit 5     DSPPLL_SELECT3 (READ) SW3-7
	-- Bit 6     DSPPLL_SELECT4 (READ) SW3-8
	-- Bit 7    NU
	--       
	process( PowOnRs, DSP_WEn, CpldRegCs6, DSP_DQ ) 
  begin
		if PowOnRs = '0' then     
		    CpldReg6(0) <= '0';
		    CpldReg6(1) <= '0';
		    CpldReg6(2) <= '0';
		    

		elsif DSP_WEn'event and DSP_WEn = '1' then    
		  if( CpldRegCs6 = '1' ) then  
			    CpldReg6(0) <= DSP_DQ(0);
		      CpldReg6(1) <= DSP_DQ(1);
		      CpldReg6(2) <= DSP_DQ(2);  
		   end if;
		end if;
	end process;  
	  
	-- Low  = DC/UTOPIA        
	-- High = Codec  
	MCBSP_SELA  <= '1' when CpldReg6(0) = '0' else '0'; 
	     		
 	CpldReg6(3) <= '0';
	CpldReg6(4) <= DSPPLL_SW(1);
	CpldReg6(5) <= DSPPLL_SW(2);
	CpldReg6(6) <= DSPPLL_SW(3);
 
  --------------------------------------------	   
  -- Mapping ICS521  S1/S0
  --------------------------------------------
  --    S1      S0       MULTIPLIER
  --     0       0           4         
  --     0       Z           5.333    
  --     0       1           5        
  --     Z       0           2.5                                   
  --     Z       Z           2
  --     Z       1           3.333
  --     1       0           6  
  --     1       Z           3
  --     1       1           8 
  -------------------------------------------- 
  -- Note Logically  Switch On = 0 Off = 1  	
  --------------------------------------------

  CPU_DSPPLL_S <=  "00";
  DSP_PLL_CLK  <=  '0';  
                                            
	CpldReg6(7) <= '0';  
	
  -- ========================================================================
	-- REG 7: MISC2
	-- Bit 0  CPUB_PRESENT
	-- Bit 1  CPUB_RESET
	-- Bit 2  MII_RESETn
	-- Bit 3  HURRICANE_RESETn enable
	-- Bit 4  
	-- Bit 5
	-- Bit 6
	-- Bit 7  DSPA_RESETSTATn
  process( PowOnRs, DSP_WEn, CpldRegCs7, DSP_DQ ) 
  begin
		if PowOnRs = '0' then
        CpldReg7(1)  <= '0';
        CpldReg7(2)  <= '0';
        CpldReg7(3)  <= '0';
        CpldReg7(4)  <= '0';
        CpldReg7(5)  <= '0';
        CpldReg7(6)  <= '0';
		elsif DSP_WEn'event and DSP_WEn = '1' then    
		  if( CpldRegCs7 = '1' ) then
		    CpldReg7(1)  <= DSP_DQ(1);
			  CpldReg7(2)  <= DSP_DQ(2);
		    CpldReg7(3)  <= DSP_DQ(3);
		    CpldReg7(4)  <= DSP_DQ(4);
		    CpldReg7(5)  <= DSP_DQ(5);
		    CpldReg7(6)  <= DSP_DQ(6);
			  end if;
		end if;
	end process;
	
	CpldReg7(0)      <= not CPUB_PRESENTn;
	CPUB_RESETn      <= '1' when CpldReg7(1) = '1' or PowOnRs = '0' else '0';
	MII_RESETZ       <= '0' when CpldReg7(2) = '1' or PowOnRs = '0' else '1';
	HurricaneResetn  <= '0' when CpldReg7(3) = '1' and HURRICANE_RESETn = '0' else '1';

	CpldReg7(7) <= DSPA_RESETSTATn;

	
	-- =======================================================================
	-- Mux the read data from all the registers and output for reads
	--
  process(  NewDspAddr,CpldReg0,CpldReg1,CpldReg4,CpldReg6,CpldReg7 )
	begin
		case NewDspAddr( 3 downto 0) is
    	  when "0000"  => MuxD  <= CpldReg0; 
        when "0001"  => MuxD  <= CpldReg1; 
        when "0100"  => MuxD  <= CpldReg4;   
        when "0110"  => MuxD  <= CpldReg6; 
        when "0111"  => MuxD  <= CpldReg7;
 		  when others => MuxD  <= "00000000";
    end case;
	end process;     	
	   
	DSP_DQ <= MuxD  when    CpldCs  = '0' 
	                 and  DSP_ARNW  = '1'
	                 and  DSP_OEn   = '0'  
	                 else "ZZZZZZZZ";
	                 
  -- #########################################################################
  -- Generate the Daughter card buffer control signals. DC buffers are only
  -- enabled if a daughter card is plugged in to mininize EMI.
  -- 
  -- DSP ARNW signal is low for write and low for write.  We flip this to match
  -- the 245 direction control.  
  --
  -- #########################################################################
  
  DC_DBUF_DIR <= not DSP_ARNW; -- low for write, high for read
  
  DC_DBUF_OEn <= '0' when DC_DETn    = '0'  
	                 and ( DcCs0 = '0' or  DcCs1 = '0' )
					         and ( DSP_OEn  = '0' or  DSP_WEn = '0'  )										     
					         else '1';     
										 
  DC_CNTL_OEn <= '0' when DC_DETn = '0' else '1';
    
  TEST_PIN <= FlashCs;
   
	EMIF_PLL_CLK <= CLKIN;

end behavior_dsk6455;

