/*
 * TI Booting and Flashing Utilities
 *
 * Generic SPI memory driver file
 *
 * Copyright (C) 2009 Texas Instruments Incorporated - http://www.ti.com/
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation version 2.
 *
 * This program is distributed "as is" WITHOUT ANY WARRANTY of any
 * kind, whether express or implied; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/* --------------------------------------------------------------------------
  AUTHOR      : Daniel Allred
 --------------------------------------------------------------------------- */

// General type include
#include "tistdtypes.h"

// Device specific CSL
#include "device.h"

// SPI module's header file 
#include "spi.h"

// This module's header file
#include "spi_mem.h"


/************************************************************
* Explicit External Declarations                            *
************************************************************/


/************************************************************
* Local Macro Declarations                                  *
************************************************************/


/************************************************************
* Local Function Declarations                               *
************************************************************/

static void LOCAL_xferAddrBytes(SPI_MemInfoHandle hSPIMemInfo, Uint32 addr);
static void LOCAL_readDataBytes(SPI_MemInfoHandle hSPIMemInfo, Uint32 byteCnt, Uint8 *data);

/************************************************************
* Local Variable Definitions                                *
************************************************************/


/************************************************************
* Global Variable Definitions                               *
************************************************************/

SPI_MemInfoObj gSPIMemInfo;
SPI_Config spiCfg;


/************************************************************
* Global Function Definitions                               *
************************************************************/

// Initialze SPI interface
SPI_MemInfoHandle SPI_MEM_open(Uint32 spiPeripheralNum)
{
  Uint8 spibuf;
  SPI_MemInfoHandle hSPIMemInfo;
      
  hSPIMemInfo = (SPI_MemInfoHandle) &gSPIMemInfo;

  // Transfer 8 bits at a time
  spiCfg.charLen = 8;
  
  // Use industry standard mode 3 (note that our SPI peripheral phase value is
  // inverted compared to all other industry players)
  spiCfg.phase = 0;
  spiCfg.polarity = 1;
#ifdef CONFIG_MACH_DAVINCI_DA850_EVM
  spiCfg.prescalar = 11;
#elif CONFIG_MACH_DAVINCI_DA830_EVM
  spiCfg.prescalar = 0;
#endif

  hSPIMemInfo->hSPIInfo = SPI_open(spiPeripheralNum,
    SPI_ROLE_MASTER,
    SPI_MODE_3PIN,
    &spiCfg);

  SPI_disableCS(hSPIMemInfo->hSPIInfo);
  
  // Assert chip select
  SPI_enableCS(hSPIMemInfo->hSPIInfo);

  // Send memory read command
  SPI_xferOneChar(hSPIMemInfo->hSPIInfo,SPI_MEM_CMD_READ);

  // Send 8-bit adresss, receive dummy
  SPI_xferOneChar(hSPIMemInfo->hSPIInfo,0x00);

  // Receive data from 8-bit device OR
  // Transmit next part of 16-bit address and receive dummy
  spibuf = SPI_xferOneChar(hSPIMemInfo->hSPIInfo,0x00);

  // Check for 8-bit memory
  if (spibuf != 0xFF)
    hSPIMemInfo->addrWidth = 8;
  else
  {
    // Receive data from 16-bit device OR
    // Transmit next part of 24-bit address and receive dummy
    spibuf = SPI_xferOneChar(hSPIMemInfo->hSPIInfo,0x00);

    // Check for 16-bit memory
    if (spibuf != 0xFF)
      hSPIMemInfo->addrWidth = 16;
    else
    {
      // Receive data from 24-bit device OR
      // Transmit dummy
      spibuf = SPI_xferOneChar(hSPIMemInfo->hSPIInfo,0x00);

      // Check for 24-bit memory
      if (spibuf != 0xFF)
        hSPIMemInfo->addrWidth = 24;
      else
      {
	  	hSPIMemInfo->addrWidth = 24;
        // Assume Atmel devices
      }
    }
  }

  // De-assert chip select
  SPI_disableCS(hSPIMemInfo->hSPIInfo);

  // Try to determine if this is flash or EEPROM by
  // issuing a DEVICE ID Read command

  // Assert chip select
  SPI_enableCS(hSPIMemInfo->hSPIInfo);

  // Send memory read command
  SPI_xferOneChar(hSPIMemInfo->hSPIInfo,SPI_MEM_CMD_JEDEC_ID);

  // Send dummy data, receive manufacture ID
  spibuf = SPI_xferOneChar(hSPIMemInfo->hSPIInfo,0x00);

  if (spibuf != 0x00)
  {
    hSPIMemInfo->memType = SPI_MEM_TYPE_FLASH;
    
    // Send dummy data, receive devicd ID1
    spibuf = SPI_xferOneChar(hSPIMemInfo->hSPIInfo,0x00);

    // Send dummy data, receive manufacture ID
    spibuf = SPI_xferOneChar(hSPIMemInfo->hSPIInfo,0x00);
  }
  else
  {
    hSPIMemInfo->memType = SPI_MEM_TYPE_EEPROM;
  }

  SPI_disableCS(hSPIMemInfo->hSPIInfo);

  return hSPIMemInfo;
}

// Routine to read data from SPI
Uint32 SPI_MEM_readBytes(SPI_MemInfoHandle hSPIMemInfo, Uint32 addr, Uint32 byteCnt, Uint8 *dest)
{
  SPI_enableCS(hSPIMemInfo->hSPIInfo);

  // Send memory read command
  SPI_xferOneChar(hSPIMemInfo->hSPIInfo,SPI_MEM_CMD_READ);

  // Send the address bytes
  LOCAL_xferAddrBytes(hSPIMemInfo,addr);

  // Receive data bytes
  LOCAL_readDataBytes(hSPIMemInfo, byteCnt, dest);

  SPI_disableCS(hSPIMemInfo->hSPIInfo);

  return E_PASS;
}

/************************************************************
* Local Function Definitions                                *
************************************************************/

static void LOCAL_xferAddrBytes(SPI_MemInfoHandle hSPIMemInfo, Uint32 addr)
{
  Uint32 i;

  for (i=0; i<hSPIMemInfo->addrWidth; i+=8)
  {
    Uint8 addrByte = ((addr >> (hSPIMemInfo->addrWidth - i - 8)) & 0xFF);
    SPI_xferOneChar(hSPIMemInfo->hSPIInfo,addrByte);
  }
}

static void LOCAL_readDataBytes(SPI_MemInfoHandle hSPIMemInfo, Uint32 byteCnt, Uint8 *data)
{
  Uint32 i;

  for (i=0; i< byteCnt; i++)
  {
    data[i] = SPI_xferOneChar(hSPIMemInfo->hSPIInfo,0x00);
  }
}


/***********************************************************
* End file                                                 *
***********************************************************/

/* --------------------------------------------------------------------------
  HISTORY
    v1.00 - DJA - 19-Aug-2008
      Initial release
-------------------------------------------------------------------------- */

