//------------------------------------------------------------------------------
// File mcasp.c
//------------------------------------------------------------------------------
// Copyright (c) 2010 Texas Instruments, Inc.
// All rights reserved
//------------------------------------------------------------------------------

#include "app.h"

#include "evmc6747.h"
#include "evmc6747_mcasp.h"
#include "aic3106.h"

//------------------------------------------------------------------------------
// Configuration Macros
//------------------------------------------------------------------------------

#define AIC3106_I2C_ADDR    0x18    // I2C address

//------------------------------------------------------------------------------
// Variables
//------------------------------------------------------------------------------

void *mcasp_xmt_register = (void *)(MCASP1_BASE + 0x214);
void *mcasp_rcv_register = (void *)(MCASP1_BASE + 0x280);

static MCASP_Handle mcasp;

//------------------------------------------------------------------------------
// Functions
//------------------------------------------------------------------------------

/* ------------------------------------------------------------------------ *
 *                                                                          *
 *  _AIC3106_rset(  regnum, regval )                                        *
 *                                                                          *
 *      Set codec register regnum to value regval                           *
 *                                                                          *
 * ------------------------------------------------------------------------ */

Int16 EVMC6747_AIC3106_rset( Uint16 regnum, Uint16 regval )
{
    Uint8 cmd[2];
    cmd[0] = regnum & 0x007F;       // 7-bit Device Address
    cmd[1] = regval;                // 8-bit Register Data

    return EVMC6747_I2C_write( AIC3106_I2C_ADDR, cmd, 2 );
}

void mcasp_close()
{
    // Close Codec
    EVMC6747_AIC3106_rset( AIC3106_PAGESELECT, 0 ); // Select Page 0
    EVMC6747_AIC3106_rset( AIC3106_RESET, 0x80 );   // Reset the AIC3106

    // Close McASP
    mcasp->regs->SRCTL0 = 0; // Serializers
    mcasp->regs->SRCTL1 = 0;
    mcasp->regs->SRCTL2 = 0;
    mcasp->regs->SRCTL3 = 0;
    mcasp->regs->SRCTL5 = 0;
    //mcasp->regs->SRCTL11 = 0;
    //mcasp->regs->SRCTL12 = 0;
    mcasp->regs->GBLCTL = 0;  // Global Reset
}

void mcasp_open()
{
    // Configure AIC3106
    EVMC6747_AIC3106_rset(  AIC3106_PAGESELECT, 0 );       // Select page 0
    EVMC6747_AIC3106_rset(  AIC3106_RESET, 0x80 );         // Reset AIC3106

    /* ------------------------------------------------------------------------ *
     *                                                                          *
     *  AIC3106 Setup                                                           *
     *                                                                          *
     *      AIC3106.MCLK = PLL1705.SCK02                                        *
     *      FS = ( AIC3106.MCLK * K ) / ( 2048 * P )                            *
     *                                                                          *
     *      For a FS=[48 kHz] & MCLK=[22.5792 MHz]                              *
     *          : 48kHz = ( 22.5792 MHz * K ) / ( 2048 * P )                    *
     *          : P = 2, K[J.D] = 8.7075                                        *
     *                                                                          *
     * ------------------------------------------------------------------------ */

    // Configure AIC3106 registers
    EVMC6747_AIC3106_rset(  3, 0x22 );  // 5 PLL A                            <- [PLL=OFF][Q=4][P=2]
    EVMC6747_AIC3106_rset(  4, 0x20 );  // 4 PLL B                            <- [J=8]
    EVMC6747_AIC3106_rset(  5, 0x6E );  // 5 PLL C                            <- [D=7075]
    EVMC6747_AIC3106_rset(  6, 0x23 );  // 6 PLL D                            <- [D=7075]
    EVMC6747_AIC3106_rset(  7, 0x0A );  // 7 Codec Datapath Setup             <- [FS=48 kHz][LeftDAC=LEFT][RightDAC=RIGHT]
    EVMC6747_AIC3106_rset(  8, 0x00 );  // 8  Audio Interface Control A       <- [BCLK=Slave][MCLK=Slave]
    EVMC6747_AIC3106_rset(  9, 0x00 );  // 9  Audio Interface Control B       <- [I2S mode][16 bit]
    EVMC6747_AIC3106_rset(  10, 0x00);  // 10 Audio Interface Control C       <- [Data offset=0]
    EVMC6747_AIC3106_rset(  15, 0 );    // 15  Left ADC PGA Gain              <- [Mute=OFF]
    EVMC6747_AIC3106_rset(  16, 0 );    // 16 Right ADC PGA Gain              <- [Mute=OFF]
    EVMC6747_AIC3106_rset(  19, 0x04 ); // 19  LINE1L to  Left ADC            <- [SingleEnd][Gain=0dB][Power=ON][SoftStep=OncePerFS]
    EVMC6747_AIC3106_rset(  22, 0x04 ); // 22  LINE1R to Right ADC            <- [SingleEnd][Gain=0dB][Power=ON][SoftStep=OncePerFS]
    EVMC6747_AIC3106_rset(  27, 0 );    // 27  Left AGC B                     <- [OFF]
    EVMC6747_AIC3106_rset(  30, 0 );    // 30 Right AGC B                     <- [OFF]
    EVMC6747_AIC3106_rset(  37, 0xE0 ); // 37 DAC Power & Output Dvr          <- [LeftDAC=ON][RightDAC=ON][HPLCOM=SingleEnd]
    EVMC6747_AIC3106_rset(  38, 0x10 ); // 38 High Power Output Dvr           <- [HPRCOM=SingleEnd][ShortCircuit=OFF]
    EVMC6747_AIC3106_rset(  43, 0 );    // 43  Left DAC Digital Volume        <- [Mute=OFF][Gain=0dB]
    EVMC6747_AIC3106_rset(  44, 0 );    // 44 Right DAC Digital Volume        <- [Mute=OFF][Gain=0dB]
    EVMC6747_AIC3106_rset(  47, 0x80 ); // 47 DAC_L1 to HPLOUT Volume         <- [Routed]
    EVMC6747_AIC3106_rset(  51, 0x09 ); // 51           HPLOUT Output         <- [Mute=OFF][Power=ON]
    EVMC6747_AIC3106_rset(  58, 0 );    // 58           HPLCOM Output         <- []
    EVMC6747_AIC3106_rset(  64, 0x80 ); // 64 DAC_R1 to HPROUT Volume         <- [Routed]
    EVMC6747_AIC3106_rset(  65, 0x09 ); // 65           HPROUT Output         <- [Mute=OFF][Power=ON]
    EVMC6747_AIC3106_rset(  72, 0 );    // 72           HPRCOM Output         <- []
    EVMC6747_AIC3106_rset(  82, 0x80 ); // 82 DAC_L1 to LEFT_LOP/M Volume     <- [Routed]
    EVMC6747_AIC3106_rset(  86, 0x09 ); // 83 LINE2R to LEFT_LOP/M Volume     <- []
    EVMC6747_AIC3106_rset(  92, 0x80 ); // 92 DAC_R1 to RIGHT_LOP/M Volume    <- [Routed]
    EVMC6747_AIC3106_rset(  93, 0x09 ); // 93           RIGHT_LOP/M Output    <- [Mute=OFF][Power=ON]
    EVMC6747_AIC3106_rset( 101, 0x01 ); // 101 GPIO Control Register B        <- [CODEC_CLKIN = CLKDIV_OUT]
    EVMC6747_AIC3106_rset( 102, 0 );    // 102 Clock Generation Control       <- [PLLCLK_IN and CLKDIV_IN use MCLK]

    // Initialize MCASP1
    mcasp = &MCASP_MODULE_1;

    /* ---------------------------------------------------------------- *
     *                                                                  *
     *  McASP1 is in MASTER mode.                                       *
     *      BCLK & WCLK come from McASP1                                *
     *      DIN is used by write16/write32                              *
     *      DOUT is usec by read16/read32                               *
     *                                                                  *
     * ---------------------------------------------------------------- */

    mcasp->regs->GBLCTL  = 0;       // Reset
    mcasp->regs->RGBLCTL = 0;       // Reset RX
    mcasp->regs->XGBLCTL = 0;       // Reset TX
    mcasp->regs->PWRDEMU = 1;       // Free-running

    // RX
    mcasp->regs->RMASK      = 0xffffffff; // No padding used
    mcasp->regs->RFMT       = 0x00008078; // MSB 16bit, 1-delay, no pad, CFGBus
    mcasp->regs->AFSRCTL    = 0x00000112; // 2TDM, 1bit Rising, INTERNAL FS, word
    mcasp->regs->ACLKRCTL   = 0x000000AF; // Rising INTERNAL CLK,(from tx side)
    mcasp->regs->AHCLKRCTL  = 0x00000000; // INT CLK (from tx side)
    mcasp->regs->RTDM       = 0x00000003; // Slots 0,1
    mcasp->regs->RINTCTL    = 0x00000000; // Not used
    mcasp->regs->RCLKCHK    = 0x00FF0008; // 255-MAX 0-MIN, div-by-256

    // TX
    mcasp->regs->XMASK      = 0xffffffff; // No padding used
    mcasp->regs->XFMT       = 0x00008078; // MSB 16bit, 1-delay, no pad, CFGBus
    mcasp->regs->AFSXCTL    = 0x00000112; // 2TDM, 1bit Rising edge INTERNAL FS, word
    mcasp->regs->ACLKXCTL   = 0x000000AF; // ASYNC, Rising INTERNAL CLK, div-by-16
    mcasp->regs->AHCLKXCTL  = 0x00000000; // EXT CLK
    mcasp->regs->XTDM       = 0x00000003; // Slots 0,1
    mcasp->regs->XINTCTL    = 0x00000000; // Not used
    mcasp->regs->XCLKCHK    = 0x00FF0008; // 255-MAX 0-MIN, div-by-256

    mcasp->regs->SRCTL5     = 0x000D;     // MCASP1.AXR1[5] --> DIN
    mcasp->regs->SRCTL0     = 0x000E;     // MCASP1.AXR1[0] <-- DOUT
    mcasp->regs->PFUNC      = 0;          // All MCASPs
    mcasp->regs->PDIR       = 0x14000020; // All inputs except AXR0[5], ACLKX1, AFSX1

    mcasp->regs->DITCTL     = 0x00000000; // Not used
    mcasp->regs->DLBCTL     = 0x00000000; // Not used
    mcasp->regs->AMUTE      = 0x00000000; // Not used

    // Starting sections of the McASP
    mcasp->regs->XGBLCTL |= GBLCTL_XHCLKRST_ON;                                    // HS Clk
    while ( ( mcasp->regs->XGBLCTL & GBLCTL_XHCLKRST_ON ) != GBLCTL_XHCLKRST_ON );
    mcasp->regs->RGBLCTL |= GBLCTL_RHCLKRST_ON;                                    // HS Clk
    while ( ( mcasp->regs->RGBLCTL & GBLCTL_RHCLKRST_ON ) != GBLCTL_RHCLKRST_ON );

    mcasp->regs->XGBLCTL |= GBLCTL_XCLKRST_ON;                                     // Clk
    while ( ( mcasp->regs->XGBLCTL & GBLCTL_XCLKRST_ON ) != GBLCTL_XCLKRST_ON );
    mcasp->regs->RGBLCTL |= GBLCTL_RCLKRST_ON;                                     // Clk
    while ( ( mcasp->regs->RGBLCTL & GBLCTL_RCLKRST_ON ) != GBLCTL_RCLKRST_ON );

    mcasp->regs->XSTAT = 0x0000ffff;        // Clear all
    mcasp->regs->RSTAT = 0x0000ffff;        // Clear all

    mcasp->regs->XGBLCTL |= GBLCTL_XSRCLR_ON;                                      // Serialize
    while ( ( mcasp->regs->XGBLCTL & GBLCTL_XSRCLR_ON ) != GBLCTL_XSRCLR_ON );
    mcasp->regs->RGBLCTL |= GBLCTL_RSRCLR_ON;                                      // Serialize
    while ( ( mcasp->regs->RGBLCTL & GBLCTL_RSRCLR_ON ) != GBLCTL_RSRCLR_ON );

    // Write a 0, so that no underrun occurs after releasing the state machine
    mcasp->regs->XBUF5 = 0;
    mcasp->regs->RBUF0 = 0;

    mcasp->regs->XGBLCTL |= GBLCTL_XSMRST_ON;                                       // State Machine
    while ( ( mcasp->regs->XGBLCTL & GBLCTL_XSMRST_ON ) != GBLCTL_XSMRST_ON );
    mcasp->regs->RGBLCTL |= GBLCTL_RSMRST_ON;                                       // State Machine
    while ( ( mcasp->regs->RGBLCTL & GBLCTL_RSMRST_ON ) != GBLCTL_RSMRST_ON );

    mcasp->regs->XGBLCTL |= GBLCTL_XFRST_ON;                                        // Frame Sync
    while ( ( mcasp->regs->XGBLCTL & GBLCTL_XFRST_ON ) != GBLCTL_XFRST_ON );
    mcasp->regs->RGBLCTL |= GBLCTL_RFRST_ON;                                        // Frame Sync
    while ( ( mcasp->regs->RGBLCTL & GBLCTL_RFRST_ON ) != GBLCTL_RFRST_ON );
}

//------------------------------------------------------------------------------
// End of File mcasp.c
//------------------------------------------------------------------------------


