
//------------------------------------------------------------------------------
// File audio.c
//------------------------------------------------------------------------------
// Copyright (c) 2010 Texas Instruments, Inc.
// All rights reserved
//------------------------------------------------------------------------------

#include <stdio.h>
#include "tistdtypes.h"

#include "app.h"
#include "edma3.h"

//-----------------------------------------------------------------------------
// Configuration Macros
//-----------------------------------------------------------------------------

#define SAMPLES_PER_BUF     128
#define BYTES_PER_SAMPLE    4
#define N_CHANNELS          2

#define BUF_LENGTH          SAMPLES_PER_BUF * N_CHANNELS
#define BUF_SIZE            SAMPLES_PER_BUF * N_CHANNELS * BYTES_PER_SAMPLE

//-----------------------------------------------------------------------------
// Static Variable Declarations
//-----------------------------------------------------------------------------

static Uint32 rcv_ping[BUF_LENGTH], rcv_pong[BUF_LENGTH],
              xmt_ping[BUF_LENGTH], xmt_pong[BUF_LENGTH];

//-----------------------------------------------------------------------------
// Functions
//-----------------------------------------------------------------------------

static void process(void *in_buf, void *out_buf)
{
    // process data from in_buf and but result in out_buf

    memcpy(out_buf, in_buf, BUF_SIZE); // (placeholder)
}

static void audio_run(void)
{
    while (1)
    {
        // ping
        SEM_pend(&rcv_ping_sem, SYS_FOREVER);
        SEM_pend(&xmt_ping_sem, SYS_FOREVER);
        process(rcv_ping, xmt_ping);

        // pong
        SEM_pend(&rcv_pong_sem, SYS_FOREVER);
        SEM_pend(&xmt_pong_sem, SYS_FOREVER);
        process(rcv_pong, xmt_pong);
    }
}

//-----------------------------------------------------------------------------
// Static TSK Function
//-----------------------------------------------------------------------------

int audio_tsk_fxn()
{
    LOG_printf(&trace, "[audio TSK]: starting audio TSK...");

    // configure and start EDMA channels
    LOG_printf(&trace, "[audio TSK]: setting up EDMA...");

    setup_edma_pingpong_rcv(mcasp_rcv_register, rcv_ping, rcv_pong,
                            BYTES_PER_SAMPLE, BUF_LENGTH);

    setup_edma_pingpong_xmt(xmt_ping, xmt_pong, mcasp_xmt_register,
                            BYTES_PER_SAMPLE, BUF_LENGTH);

    EdmaEnableChannel(EDMA_MCASPRXCH, 0);
    EdmaEnableChannel(EDMA_MCASPTXCH, 0);

    EdmaIntEnable(EDMA_RCV_PING_TCC);
    EdmaIntEnable(EDMA_RCV_PONG_TCC);
    EdmaIntEnable(EDMA_XMT_PING_TCC);
    EdmaIntEnable(EDMA_XMT_PONG_TCC);

    // setup codec and McASP
    LOG_printf(&trace, "[audio TSK]: initializing audio device...");
    mcasp_open();

    // run audio test
    LOG_printf(&trace, "[audio TSK]: running audio processing loop...");
    audio_run();

    // kill codec and McASP
    LOG_printf(&trace, "[audio TSK]: closing audio device...");
    mcasp_close();

    // close application
    LOG_printf(&trace, "[audio TSK]: audio TSK exited!");

    return 0;
}

//------------------------------------------------------------------------------
// End of File audio.c
//------------------------------------------------------------------------------

