//------------------------------------------------------------------------------
// File edma3.c
//------------------------------------------------------------------------------
// Copyright (c) 2010 Texas Instruments, Inc.
// All rights reserved
//------------------------------------------------------------------------------

#include <stdio.h>

#include "edma3.h"
#include "app.h"

//------------------------------------------------------------------------------
// Local Variables
//------------------------------------------------------------------------------

static CSL_Edma3ccRegsOvly edmaCcRegs = (CSL_Edma3ccRegsOvly)CSL_EDMA3CC_0_REGS;

//------------------------------------------------------------------------------
// Functions
//------------------------------------------------------------------------------

void edmaInit()
{
    edmaCcRegs->ECR  = 0xffffffff; // clear events  0 -> 31
    edmaCcRegs->SECR = 0xffffffff; // clear secondary events  0 -> 31
    edmaCcRegs->IECR = 0xffffffff; // disable all interrupts
    edmaCcRegs->ICR  = 0xffffffff; // clear all pending interrupts
}

void setup_edma_pingpong_xmt(void *src_ping, void *src_pong, void *dst,
                             Uint32 acnt, Uint32 bcnt)
{
    CSL_Edma3ccParamSetRegs param;

    // 1. setup channel PaRAM slot
    param.OPT = (EDMA_XMT_PING_TCC << 12) | (1 << 20);      // transfer complete interrupt enabled
    param.SRC = (Uint32)src_ping;
    param.A_B_CNT = (bcnt << 16) | acnt;                    // actual format: BCNT|ACNT
    param.DST = (Uint32)dst;
    param.SRC_DST_BIDX = (0 << 16) | acnt;                  // actual format: DSTBIDX|SCRBIDX
    param.LINK_BCNTRLD = (0 << 16) | (EDMA_XMTPONG * 0x20); // actual format: BCNTRLD|LINK
    param.SRC_DST_CIDX = 0;                                 // actual format: DSTCIDX|SRCCIDX
    param.CCNT = 1;
    edmaWritePaRAM(EDMA_MCASPTXCH, &param);

    // 2. setup ping PaRAM set (to be reloaded later)
    edmaWritePaRAM(EDMA_XMTPING, &param);

    // 3. setup pong PaRAM set (to be loaded and reloaded later)
    param.OPT = (EDMA_XMT_PONG_TCC << 12) | (1 << 20);      // transfer complete interrupt enabled
    param.LINK_BCNTRLD = (0 << 16) | (EDMA_XMTPING * 0x20); // actual format: BCNTRLD|LINK
    param.SRC = (Uint32)src_pong;
    edmaWritePaRAM(EDMA_XMTPONG, &param);
}

void setup_edma_pingpong_rcv(void *src, void *dst_ping, void *dst_pong,
                             Uint32 acnt, Uint32 bcnt)
{
    CSL_Edma3ccParamSetRegs param;

    // 1. setup channel PaRAM slot
    param.OPT = (EDMA_RCV_PING_TCC << 12) | (1 << 20);      // transfer complete interrupt enabled
    param.SRC = (Uint32)src;
    param.A_B_CNT = (bcnt << 16) | acnt;                    // actual format: BCNT|ACNT
    param.DST = (Uint32)dst_ping;
    param.SRC_DST_BIDX = (acnt << 16) | 0;                  // actual format: DSTBIDX|SCRBIDX
    param.LINK_BCNTRLD = (0 << 16) | (EDMA_RCVPONG * 0x20); // actual format: BCNTRLD|LINK
    param.SRC_DST_CIDX = 0;                                 // actual format: DSTCIDX|SRCCIDX
    param.CCNT = 1;
    edmaWritePaRAM(EDMA_MCASPRXCH, &param);

    // 2. setup ping PaRAM set (to be reloaded later)
    edmaWritePaRAM(EDMA_RCVPING, &param);

    // 3. setup pong PaRAM set (to be loaded and reloaded later)
    param.OPT = (EDMA_RCV_PONG_TCC << 12) | (1 << 20);      // transfer complete interrupt enabled
    param.LINK_BCNTRLD = (0 << 16) | (EDMA_RCVPING * 0x20); // actual format: BCNTRLD|LINK
    param.DST = (Uint32)dst_pong;
    edmaWritePaRAM(EDMA_RCVPONG, &param);
}

void edmaWritePaRAM(Int ParamNum, CSL_Edma3ccParamSetRegs *ptrParamInfo)
{
    edmaCcRegs->PARAMSET[ParamNum].OPT = ptrParamInfo->OPT;
    edmaCcRegs->PARAMSET[ParamNum].SRC = ptrParamInfo->SRC;
    edmaCcRegs->PARAMSET[ParamNum].A_B_CNT = ptrParamInfo->A_B_CNT;
    edmaCcRegs->PARAMSET[ParamNum].DST = ptrParamInfo->DST;
    edmaCcRegs->PARAMSET[ParamNum].SRC_DST_BIDX = ptrParamInfo->SRC_DST_BIDX;
    edmaCcRegs->PARAMSET[ParamNum].LINK_BCNTRLD = ptrParamInfo->LINK_BCNTRLD;
    edmaCcRegs->PARAMSET[ParamNum].SRC_DST_CIDX = ptrParamInfo->SRC_DST_CIDX;
    edmaCcRegs->PARAMSET[ParamNum].CCNT = ptrParamInfo->CCNT;
}

void edmaSetEvent(Int channelNumber)
{
    int mask;

    mask = 1 << channelNumber;
    edmaCcRegs->ESR = mask;
}

void EdmaIntClear(Int channelNumber)
{
    edmaCcRegs->ICR = 1 << channelNumber;
}

void EdmaIntEnable(Int intNumber)
{
    edmaCcRegs->IESR = 1 << intNumber;
    edmaCcRegs->DRA[0].DRAE &= ~(1 << intNumber);
    edmaCcRegs->DRA[1].DRAE |= 1 << intNumber;
}

// not a tested function
void EdmaIntDisable(Int intNumber)
{
    int mask;

    mask = 1 << intNumber;
    edmaCcRegs->DRA[0].DRAE &= ~mask;
}

void EdmaEnableChannel(Int channelNumber, Int QueueNumber)
{
    int mask;

    edmaCcRegs->EMCR = 1 << channelNumber;
    edmaCcRegs->SECR = 1 << channelNumber;

    QueueNumber &= 1; // only 0 or 1 are valid

    mask = 0x07 << (4 * (channelNumber & 0x07) );

    edmaCcRegs->DMAQNUM[channelNumber >> 3] &= ~mask;
    edmaCcRegs->DMAQNUM[channelNumber >> 3] |= QueueNumber << (4 * (channelNumber & 0x07) );

    edmaCcRegs->EESR = 1 << channelNumber;
}

void EdmaDisableChannel(Int channelNumber)
{
    edmaCcRegs->EECR = 1 << channelNumber;
}

void edmaWaitForEvent(int event)
{
    Uint32 mask = 1 << event;

    while((edmaCcRegs->IPR & mask) == NULL)
        asm(" nop");

    EdmaIntClear(event);
}

void edma3ccIsr()
{
    while(edmaCcRegs->IPR)
    {
        if (edmaCcRegs->IPR & (1 << EDMA_RCV_PING_TCC))    // receive channel (ping)
        {
            EdmaIntClear(EDMA_RCV_PING_TCC);
            SEM_post(&rcv_ping_sem);
            //LOG_printf(&trace, "[ISR]: receive ping!");
        }

        if (edmaCcRegs->IPR & (1 << EDMA_RCV_PONG_TCC))    // receive channel (pong)
        {
            EdmaIntClear(EDMA_RCV_PONG_TCC);
            SEM_post(&rcv_pong_sem);
            //LOG_printf(&trace, "[ISR]: receive pong!");
        }

        if (edmaCcRegs->IPR & (1 << EDMA_XMT_PING_TCC))    // transmit channel (ping)
        {
            EdmaIntClear(EDMA_XMT_PING_TCC);
            SEM_post(&xmt_ping_sem);
            //LOG_printf(&trace, "[ISR]: transmit ping!");
        }

        if (edmaCcRegs->IPR & (1 << EDMA_XMT_PONG_TCC))    // transmit channel (pong)
        {
            EdmaIntClear(EDMA_XMT_PONG_TCC);
            SEM_post(&xmt_pong_sem);
            //LOG_printf(&trace, "[ISR]: transmit pong!");
        }
    }

    return;
}

//------------------------------------------------------------------------------
// End of File edma3.c
//------------------------------------------------------------------------------

