/***************************************************************************
 *
 *
 *         **   **    **  ******  ********  ********  ********  **    **    
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ***************************************************************************
 *                                                                          
 *  Project     : EVM
 *  File        : aic33tone.c
 *  Description : Utility functions for audio codec
 *
 *                        Copyright (c) Lyrtech inc. 2007                        
 *
 ***************************************************************************
 *                                                                                          
 * "$Revision: 1.1 $"
 * "$Date: 2007/09/13 16:03:05 $"
 *
 ***************************************************************************/

/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include "soc.h"
#include "MCASP.h"
#include "EVM.h"
#include "evm_aic33.h"
#include "math.h"
#include "MCASP_setup.h"
#include "aic33tone.h"
#include "mydebug.h"


// This demo shows how to perform a digital audio tone generation
// on the EVM.
//
// All 4 codecs will generate a tone at the same time on there stereo lineouts.
// They are setup fs=48KHz by default (according to constant AUDIO48KHZ)
// The fisrt codec will be the clock master.
//
// The audio codec ADC and the DAC devices will be initialized using the functions
// of the EVM library. These devices are connected to the DSP through 
// the serializers of the McASP #0. 
//      ADC #1 & #2 -> McASP0 Serializer 0 (AXR0_0)
//      ADC #5 & #6 -> McASP0 Serializer 1 (AXR0_1)
//      DAC #1 & #2 -> McASP0 Serializer 2 (AXR0_2)
//      DAC #5 & #6 -> McASP0 Serializer 3 (AXR0_3)
//      ADC #3 & #4 -> McASP0 Serializer 4 (AXR0_4)
//      ADC #7 & #8 -> McASP0 Serializer 5 (AXR0_5)
//      DAC #3 & #4 -> McASP0 Serializer 6 (AXR0_6)
//      DAC #7 & #8 -> McASP0 Serializer 7 (AXR0_7)
//
//

/****************************************************************************
 *                             Local constants                              *
 ****************************************************************************/
// Comment following line to get 44.1 KHz audio instead of 48 KHz
#define AUDIO48KHZ 1

// Adjust number of elements in a 10msec period according to audio frequency
#ifdef AUDIO48KHZ
#define NSINEELEM 480
#else
#define NSINEELEM 441
#endif

#define SINEAMPL  22106
#define NPERIOD   10

// Timeout defines
// Maximum number of loop before timeout function
#define MAXLOOPTO 5000000

// Timeout error codes
#define ERR_TO1   11
#define ERR_TO2   12
#define ERR_TO3   13
#define ERR_TO4   14
#define ERR_TO5   15

/****************************************************************************
 *                             Local variables                              *
 ****************************************************************************/
// McASP 0 Module
//
static McaspHandle  hMcasp0;
static McaspObj     mcasp0Obj;
static McaspHwSetup mcasp0HwCfg = MY_MCASP0_HWSETUP_DEFAULTS;


#ifdef AUDIO48KHZ
// Audio codec default setup registers structure for I2S 16 bits, 48.0 KHz
static EVM_AIC33_Setup aic33setup = EVM_AIC33_16BITS_I2S_480;
#else
// Audio codec default setup registers structure for I2S 16 bits, 44.1 KHz
static EVM_AIC33_Setup aic33setup = EVM_AIC33_16BITS_I2S_441;
#endif


/****************************************************************************
 *                           Forward Declarations                           *
 ****************************************************************************/


/****************************************************************************
 *                       Private functions prototypes                       *
 ****************************************************************************/
static Int32 AIC33_Tone(void);
static void AIC33_CreateSineTable(Int32 *buff,Int32 nelem,Int32 ampl,Int32 nperiod,Int32 lshift);


/****************************************************************************
 *                             Public Functions                             *
 ****************************************************************************/


/**************************************************************************** 
 *
 * NAME  
 *      do_AIC33_Tone
 *
 * PURPOSE:
 *      Do a 1KHz audio tone test on all 4 stereo line outs. Then displays
 *      the result on the console. Needs that the debug buffers are ready
 *      for myprintf function for results output.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 do_AIC33_Tone(void)
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 do_AIC33_Tone(void)
{
    Int32 iResult;

    myprintf("Audio tone out test on all channels 1-8...\r\n",0,0);

    iResult = AIC33_Tone();

    // Displays the result
    if (!iResult)
    {
        myprintf("\r\nAudio tone out test on all channels 1-8... SUCCESS!\r\n\r\n",0,0);
    }
    else
    {
        myprintf("\r\nAudio tone out test on all channels 1-8... ERROR!\r\n\r\n",0,0);
    }

    return(iResult);
}

/****************************************************************************
 *                             Local Functions                              *
 ****************************************************************************/

/**************************************************************************** 
 *
 * NAME  
 *      AIC33_Tone
 *
 * PURPOSE:
 *      Output a tone on all codecs stereo lineout with a 1 KHz frequency.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 AIC33_Tone(void)
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
static Int32 AIC33_Tone(void)
{
    Int16 tmsec, sec;
    Int16 sample;
    EVM_AIC33_Handle aic33handle0,aic33handle1,aic33handle2,aic33handle3;
    Int32 sinetable[NSINEELEM+1];
    Int32 retcode=0;
    CSL_BitMask16 mask;
    Int32 loopto;
    
    // Create the 16 bits integer signed sinetable table without left shift
    AIC33_CreateSineTable(sinetable,NSINEELEM,SINEAMPL,NPERIOD,0);

    // Initialize EVM module
    // Must always be the first BSL functions called
    if ((retcode=EVM_init())!=EVM_INITOK)
    {
        // Problem cannot open evm
        return(retcode);
    }

    // Verify is correct evm binary bsl version
    if ((retcode=EVM_checkbslversion(EVM_BSLVERSION))!=EVM_INITOK)
    {
        // Problem unmatching binary bsl version
        return(retcode);
    }
    
    // Open all the audio codecs, one at a time
    aic33handle0 = EVM_AIC33_open( EVM_AIC33_DEV1, &aic33setup );

    if (aic33handle0 >= EVM_AIC33_NHANDLE)
    {
        // Problem cannot open codec
        return(aic33handle0);
    }
    
    // Set the codec EVM_AIC33_DEV1 has clock master codec
    retcode = EVM_AIC33_setmaster(aic33handle0);
    
    // Checks if we got an error while setting the codec master
    if (retcode!=EVM_AIC33_OK)
    {
        // Returns the error code
        return(retcode);
    }

    aic33handle1 = EVM_AIC33_open( EVM_AIC33_DEV2, &aic33setup );

    if (aic33handle1 >= EVM_AIC33_NHANDLE)
    {
        // Problem cannot open codec
        return(aic33handle1);
    }
    
    aic33handle2 = EVM_AIC33_open( EVM_AIC33_DEV3, &aic33setup );

    if (aic33handle2 >= EVM_AIC33_NHANDLE)
    {
        // Problem cannot open codec
        return(aic33handle2);
    }
    
    aic33handle3 = EVM_AIC33_open( EVM_AIC33_DEV4, &aic33setup );

    if (aic33handle3 >= EVM_AIC33_NHANDLE)
    {
        // Problem cannot open codec
        return(aic33handle3);
    }
    
    /*---------------------------------------------------------------*/
    /* Opening McASP Module                                          */
    /*---------------------------------------------------------------*/   
    McaspOpen (&mcasp0Obj,CSL_MCASP,hMcasp0);

    // Resets all mcasp registers
    McaspRegReset(hMcasp0);
    
    // Setup with correct default MCASP for tone generation
    McaspHwSetup( hMcasp0, &mcasp0HwCfg );

    /*---------------------------------------------------------------*/
    /* Take receive serial clock, high frequency clock and           */
    /* serializer out of reset                                       */
    /*---------------------------------------------------------------*/   
    hMcasp0->regs->RXSTAT = 0x01FF;
    mask =   CSL_MCASP_GBLCTL_RXSERCLR_MASK;
    McaspResetCtrl( hMcasp0, mask );

    /*---------------------------------------------------------------*/
    /* Take transmit serial clock, high frequency clock and          */
    /* serializer out of reset                                       */
    /*---------------------------------------------------------------*/   
    hMcasp0->regs->TXSTAT = 0x01FF;
    mask =    CSL_MCASP_GBLCTL_TXSERCLR_MASK;
    McaspResetCtrl( hMcasp0, mask );

    /*---------------------------------------------------------------*/
    /* Write to XBUFs if in polling mode                             */
    /* to prefill first sample to be ready when first sync is        */
    /* received after activation (Left channel).                     */
    /*---------------------------------------------------------------*/
    loopto=0;
    while ( !(hMcasp0->regs->TXSTAT & 0x0020) && loopto++<MAXLOOPTO);
    if (loopto>=MAXLOOPTO)
    {
        return(ERR_TO1);
    }
    hMcasp0->regs->TXBUF2  = 0x00000000;
    hMcasp0->regs->TXBUF3  = 0x00000000;
    hMcasp0->regs->TXBUF6  = 0x00000000;
    hMcasp0->regs->TXBUF7  = 0x00000000;
       
    // Wait for buffer fill has been serviced after filling previously...
    loopto=0;
    while ( (hMcasp0->regs->TXSTAT & 0x0020) && loopto++<MAXLOOPTO);
    if (loopto>=MAXLOOPTO)
    {
        return(ERR_TO2);
    }
    
    /*---------------------------------------------------------------*/
    /* Take receive and transmit state machine out of reset          */
    /*---------------------------------------------------------------*/
    McaspActivateSmRcvXmt(hMcasp0);

    /*---------------------------------------------------------------*/
    /* Take receive and transmit frame sync out of reset             */
    /*---------------------------------------------------------------*/
    McaspActivateFsRcvXmt (hMcasp0);

    /*---------------------------------------------------------------*/
    /* Write to XBUFs if in polling mode                             */
    /* to complete first sample to be ready when first sync is       */
    /* received after activation (Right channel).                    */
    /*---------------------------------------------------------------*/
    loopto=0;
    while ( !(hMcasp0->regs->TXSTAT & 0x0020) && loopto++<MAXLOOPTO);
    if (loopto>=MAXLOOPTO)
    {
        return(ERR_TO3);
    }
    hMcasp0->regs->TXBUF2  = 0x00000000;
    hMcasp0->regs->TXBUF3  = 0x00000000;
    hMcasp0->regs->TXBUF6  = 0x00000000;
    hMcasp0->regs->TXBUF7  = 0x00000000;

    // If we got no error
    if (retcode == 0)
    {
        /* Play Tone for 5 seconds*/
        for ( sec = 0 ; sec < 5 ; sec++ )
        {
            // For each ten msec
            for ( tmsec = 0 ; tmsec < 100 ; tmsec++ )
            {
                // Tone samples for 10 msec
                for ( sample = 0 ; sample < NSINEELEM ; sample++ )
                {
                    // wait transmit event for Left channel
                    loopto=0;
                    while ( !(hMcasp0->regs->TXSTAT & 0x0020) && loopto++<MAXLOOPTO);
                    if (loopto>=MAXLOOPTO)
                    {
                        return(ERR_TO4);
                    }

                    // write to the output serializers for each codec DAC
                    hMcasp0->regs->TXBUF2  = sinetable[sample];
                    hMcasp0->regs->TXBUF3  = sinetable[sample];
                    hMcasp0->regs->TXBUF6  = sinetable[sample];
                    hMcasp0->regs->TXBUF7  = sinetable[sample];            

                    // wait transmit event for Right channel
                    loopto=0;
                    while ( !(hMcasp0->regs->TXSTAT & 0x0020) && loopto++<MAXLOOPTO);
                    if (loopto>=MAXLOOPTO)
                    {
                        return(ERR_TO5);
                    }

                    // write to the output serializers for each codec DAC
                    hMcasp0->regs->TXBUF2  = sinetable[sample];
                    hMcasp0->regs->TXBUF3  = sinetable[sample];
                    hMcasp0->regs->TXBUF6  = sinetable[sample];
                    hMcasp0->regs->TXBUF7  = sinetable[sample];            
                }
            }
        }
    }

    // Close all audio codecs
    EVM_AIC33_close( &aic33handle3 );
    EVM_AIC33_close( &aic33handle2 );
    EVM_AIC33_close( &aic33handle1 );
    EVM_AIC33_close( &aic33handle0 );

    return (retcode);
}

/**************************************************************************** 
 *
 * NAME  
 *      AIC33_CreateSineTable
 *
 * PURPOSE:
 *      Create a nperiod period integer sine table in buffer
 *      with number of pts as specified,
 *      and amplitude of the sine periods specified
 *      The table is left shifted with the number of bits specified if lshift!=0
 *         
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      void AIC33_CreateSineTable(Int32 *buff,Int32 nelem,Int32 ampl,Int32 nperiod,Int32 lshift); 
 *
 *      buff    - (i)   buffer where will be built the sine table
 *      nelem   - (i)   number of element in the buff table needed 
 *      ampl    - (i)   amplitude of the sine table
 *      nperiod - (i)   number of sine period to be defined in the table
 *      lshift  - (i)   number of bits to left shift the sine table if needed
 *
 * RETURN VALUE
 *      NONE
 *      
 * REFERENCE
 *
 ****************************************************************************/    
static void AIC33_CreateSineTable(Int32 *buff,Int32 nelem,Int32 ampl,Int32 nperiod,Int32 lshift)
{
   Int32 i;
   double angle=0.0, angleinc;

   // Compute angle increment for the number of period desired and number of points desired
   angleinc = nperiod * 2 * 3.1415926535 / (nelem);
   for (i = 0; i < nelem; i++)
   {
       // Compute the value add 0.5 for pseudo rounding before truncation to Int32
       buff[i] = ((Int32)(((double)(ampl)*sin(angle))+0.5))<<lshift;
       angle += angleinc;
   }
}

