/***************************************************************************
 *
 *
 *         **   **    **  ******  ********  ********  ********  **    **    
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ***************************************************************************
 *                                                                          
 *  Project     : EVM
 *  File        : BT656.c
 *  Description : Utility functions for video port configuration
 *
 *                      Copyright (c) Lyrtech inc. 2007                        
 *
 ***************************************************************************
 *                                                                                          
 * "$Revision: 1.5 $"
 * "$Date: 2007/07/24 20:30:15 $"
 *
 ***************************************************************************/

/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include "vport.h"
#include "bt656.h"

/****************************************************************************
 *                             Local variables                              *
 ****************************************************************************/

/* Handle of vp that to be configured */
VP_Handle vpDisplayHandle;

/* No of frames that are displayed */
volatile Uint32 displayFrameCount = 0;

/* Underrun error flag */
volatile Uint32 dispUnderrun = 0;


/****************************************************************************
 *                             Public Functions                             *
 ****************************************************************************/


/**************************************************************************** 
 *
 * NAME  
 *      Bt656_8bit_Ncfd_NTSC
 *
 * PURPOSE:
 *      Setup dsp video port for display.
 *
 *      Description : 8-bit BT.656 non-continuous frame display
 *
 *      Some important field descriptions:
 *
 *      DMODE = 000, 8-bit BT.656 mode
 *      CON = 0
 *      FRAME = 1, display frame
 *      DF2 = 0
 *      DF1 = 0, (8-bit non-continuous frame display)
 *      SCALE = 0, no scaling
 *      RESMPL = 0, no resampling
 *      DPK = X, not used in 8-bit display
 *      RSYNC = X, used in Raw mode(Enable second synchronized raw
 *      data channel)
 *      RGBX = X, used in Raw mode(RGB extract enable. Perform
 *      3/4 FIFO unpacking)
 *      VCTL1S = 00, output HSYNC
 *      VCTL2S = 00, output VSYNC
 *      VCTL3S = 0, output CBLNK
 *      HXS = 0, VCTL1 is an output
 *      VXS = 0, VCTL2 is an output
 *      FXS = 0, VCTL3 is an output
 *      PVPSYN = 0, no previous port synchronization
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int16 Bt656_8bit_Ncfd_NTSC(Int16 portNumber)
 *
 *      portNumber - (i)    Video port number to use (1 for HD port)
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int16 Bt656_8bit_Ncfd_NTSC(Int16 portNumber)
{
    /* Open video port for display */
    VP_open(portNumber, VP_OPEN_RESET, vpDisplayHandle);

    if(vpDisplayHandle == INV)
        return(1);

    /* Enable video port functionality in VP Peripheral */
    /* Control Reg(PCR) */
    VP_FSETH(vpDisplayHandle , PCR, PEREN, VP_PCR_PEREN_ENABLE);


    /* Set this port to display mode */
    /* Revert VCLK1 polarity */
    /* clear VPHLT in VP_CTL to make video port function */
    VP_RSETH(vpDisplayHandle , VPCTL,VP_VPCTL_RMK(0,0,VP_VPCTL_VCLK2P_REVERSE,0,0,0,0,VP_VPCTL_DISP_DISPLAY,0));

    /* -------------------- */
    /* Set all other fields */
    /* -------------------- */

    /* set frame size */
    VP_RSETH(vpDisplayHandle , VDFRMSZ,VP_VDFRMSZ_RMK(VD_FRM_HEIGHT, VD_FRM_WIDTH));

    /* set horizontal blanking */
    VP_RSETH(vpDisplayHandle , VDHBLNK,VP_VDHBLNK_RMK(VD_HBLNK_STOP, VP_VDHBLNK_HBDLA_DELAY,VD_HBLNK_START));

    /* set vertical blanking start for field1 */
    VP_RSETH(vpDisplayHandle , VDVBLKS1,VP_VDVBLKS1_RMK(VD_VBLNK_YSTART1,VD_VBLNK_XSTART1));

    /* set vertical blanking end for field1 */
    VP_RSETH(vpDisplayHandle , VDVBLKE1,VP_VDVBLKE1_RMK(VD_VBLNK_YSTOP1, VD_VBLNK_XSTOP1));

    /* set vertical blanking start for field2 */
    VP_RSETH(vpDisplayHandle , VDVBLKS2,VP_VDVBLKS2_RMK(VD_VBLNK_YSTART2, VD_VBLNK_XSTART2));

    /* set vertical blanking end for field2 */
    VP_RSETH(vpDisplayHandle , VDVBLKE2,VP_VDVBLKE2_RMK(VD_VBLNK_YSTOP2, VD_VBLNK_XSTOP2));

    /* set vertical blanking bit register for field 1(VD_VBIT1) */
    VP_RSETH(vpDisplayHandle , VDVBIT1,VP_VDVBIT1_RMK(VD_VBIT_CLR1, VD_VBIT_SET1));

    /* set vertical blanking bit register for field 2(VD_VBIT2) */
    VP_RSETH(vpDisplayHandle , VDVBIT2,VP_VDVBIT2_RMK(VD_VBIT_CLR2, VD_VBIT_SET2));

    /* No image offsets in this example */
    /* set image size for field1 */
    VP_RSETH(vpDisplayHandle , VDIMGSZ1,VP_VDIMGSZ1_RMK(VD_IMG_VSIZE1, VD_IMG_HSIZE1));

    /* set image size for field2 */
    VP_RSETH(vpDisplayHandle , VDIMGSZ2,VP_VDIMGSZ1_RMK(VD_IMG_VSIZE2, VD_IMG_HSIZE2));

    /* set field1 timing */
    VP_RSETH(vpDisplayHandle , VDFLDT1,VP_VDFLDT1_RMK(VD_FIELD1_YSTART, VD_FIELD1_XSTART));

    /* set field2 timing */
    VP_RSETH(vpDisplayHandle , VDFLDT2,VP_VDFLDT2_RMK(VD_FIELD2_YSTART, VD_FIELD2_XSTART));

    /* set display field bit register(VD_FBIT) */
    VP_RSETH(vpDisplayHandle , VDFBIT,VP_VDFBIT_RMK(VD_FBIT_SET, VD_FBIT_CLR));

    /* set horizontal sync control (VCTL1S) */
    VP_RSETH(vpDisplayHandle , VDHSYNC,VP_VDHSYNC_RMK(VD_HSYNC_STOP, VD_HSYNC_START));

    /* set vertical sync start for field1 (VCTL2S) */
    VP_RSETH(vpDisplayHandle , VDVSYNS1,VP_VDVSYNS1_RMK(VD_VSYNC_YSTART1,VD_VSYNC_XSTART1));

    /* set vertical sync end for field1 (VCTL2S) */
    VP_RSETH(vpDisplayHandle , VDVSYNE1,VP_VDVSYNE1_RMK(VD_VSYNC_YSTOP1, VD_VSYNC_XSTOP1));

    /* set vertical sync start for field2 (VCTL2S) */
    VP_RSETH(vpDisplayHandle , VDVSYNS2,VP_VDVSYNS2_RMK(VD_VSYNC_YSTART2,VD_VSYNC_XSTART2));

    /* set vertical sync end for field2 (VCTL2S) */
    VP_RSETH(vpDisplayHandle , VDVSYNE2,VP_VDVSYNE2_RMK(VD_VSYNC_YSTOP2, VD_VSYNC_XSTOP2));

    /* Let clipping values to be their defaults (VD_CLIP) */
    /* No need to set DEF_VAL and VD_RELOAD in this example */
    /* set event register */
    VP_RSETH(vpDisplayHandle , VDDISPEVT,VP_VDDISPEVT_RMK(VD_DISPEVT2, VD_DISPEVT1));

    /* Vertical interrupts are not used in this example (VD_VINT) */
    /* set threshold value for DMA events */
    VP_RSETH(vpDisplayHandle, VDTHRLD,VP_VDTHRLD_RMK(VD_VDTHRLD2,VP_VDTHRLD_INCPIX_DEFAULT, VD_VDTHRLD1));

    /* ------------------------------- */
    /* Set display control reg(VD_CTL) */
    /* ------------------------------- */
    /* set display mode(DMODE) to 8-bit BT.656 */
    VP_FSETH(vpDisplayHandle , VDCTL, DMODE, VP_VDCTL_DMODE_BT656B);

    /* set non-continuous frame display */
    VP_FSETH(vpDisplayHandle , VDCTL, CON, VP_VDCTL_CON_DISABLE);
    VP_FSETH(vpDisplayHandle , VDCTL, FRAME, VP_VDCTL_FRAME_FRMDIS);
    VP_FSETH(vpDisplayHandle , VDCTL, DF2, VP_VDCTL_DF2_NONE);
    VP_FSETH(vpDisplayHandle , VDCTL, DF1, VP_VDCTL_DF1_NONE);

    /* Output field on VCTL2S */
    VP_FSETH(vpDisplayHandle , VDCTL, VCTL2S, VP_VDCTL_VCTL2S_FLD);

    /* let control outputs(VCTL1S, VCTL2S, VCTL3S, HXS, VXS, FXS) */
    /* be their defaults i.e. VCTLxS are output control signals */
    /* no scaling and no resampling in this example */
    /* no need to bother about 10-bit unpacking mode(DPK bit) */
    /* in this 8-bit example */

    return(0);
}

/**************************************************************************** 
 *
 * NAME  
 *      VPEnable
 *
 * PURPOSE:
 *      Enable an already configured video port.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int16 VPEnable(void)
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int16 VPEnable(void)
{
    if(vpDisplayHandle == INV)
        return(1);

    /* clear VPHLT in VP_CTL to make video port function */
    VP_FSETH(vpDisplayHandle , VPCTL, VPHLT, VP_VPCTL_VPHLT_CLEAR);

    /* -------------- */
    /* enable display */
    /* -------------- */
    /* set VDEN to enable display for loop-back */
    VP_FSETH(vpDisplayHandle, VDCTL, VDEN, VP_VDCTL_VDEN_ENABLE);

    /* clear BLKDIS in VD_CTL to enable display DMA events */
    VP_FSETH(vpDisplayHandle, VDCTL, BLKDIS, VP_VDCTL_BLKDIS_CLEAR);

    return(0);
}

/**************************************************************************** 
 *
 * NAME  
 *      VPDispPoll
 *
 * PURPOSE:
 *      Clears FRMD to continue display in this non-continuous mode and also
 *      clears other status bits. This routine must be called at least one
 *      time each display frame. This functionnality would usually being done
 *      by an interrupt routine.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      void VPDispPoll(void)
 *
 *      displayFrameCount (o) - will hold number of frame displayed since started
 *      dispUnderrun      (o) - will hold number of video underrun errors detected
 *
 * RETURN VALUE
 *      1 if frame display is completed, 0 if not
 *
 * REFERENCE
 *
 ****************************************************************************/
Uint32 VPDispPoll(void)
{
    Uint32 vpis = 0;
    Uint32 rescode=0;
    vpis = VP_RGETH(vpDisplayHandle , VPIS);

    if(vpis & _VP_VPIS_DCMP_MASK) /* frame display complete */
    {
        /* Clear frame complete bit FRMD to continue display */
        VP_FSETH(vpDisplayHandle , VDSTAT, FRMD,VP_VDSTAT_FRMD_CLEAR);
    
        /* clear DCMPA to enable next frame complete interrupts */
        VP_FSETH(vpDisplayHandle , VPIS, DCMP, VP_VPIS_DCMP_CLEAR);
        displayFrameCount++; /* increment displayed frame count */

        rescode=1;
    }

    if(vpis & _VP_VPIS_DUND_MASK) /* underrun error */
    {
        dispUnderrun++;
    
        /* clear DUND to enable next underrun interrupts */
        VP_FSETH(vpDisplayHandle , VPIS, DUND, VP_VPIS_DUND_CLEAR);
    }

    return(rescode);
}

/**************************************************************************** 
 *
 * NAME  
 *      CloseDisplayVideoPort
 *
 * PURPOSE:
 *      Close a video port opened by Bt656_8bit_Ncfd_NTSC.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int16 CloseDisplayVideoPort(void)
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int16 CloseDisplayVideoPort(void)
{
    if(vpDisplayHandle == INV)
        return(1);

    /* Close video port for display */
    VP_close(vpDisplayHandle);

   return(0);
}
