/***************************************************************************
 *
 *
 *         **   **    **  ******  ********  ********  ********  **    **    
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ***************************************************************************
 *                                                                          
 *  Project     : EVM
 *  File        : BT656capt.c
 *  Description : Utility functions for video port capture configuration
 *
 *                      Copyright (c) Lyrtech inc. 2007                        
 *
 ***************************************************************************
 *                                                                                          
 * "$Revision: 1.4 $"
 * "$Date: 2007/07/19 19:48:43 $"
 *
 ***************************************************************************/

/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include "vport.h"
#include "bt656capt.h"


/****************************************************************************
 *                             Local variables                              *
 ****************************************************************************/


VP_Handle vpCaptureHandle; /* handle of vp that to be configured */
volatile Uint32 capChaAFrameCount = 0; /* no of frames captured */
volatile Uint32 capChaBFrameCount = 0; /* no of frames captured */

/* Error flags */
volatile Uint32 capChaAOverrun = 0;
volatile Uint32 capChaASyncError = 0;
volatile Uint32 capChaAShortFieldDetect = 0;
volatile Uint32 capChaALongFieldDetect = 0;
volatile Uint32 capChaBOverrun = 0;
volatile Uint32 capChaBSyncError = 0;
volatile Uint32 capChaBShortFieldDetect = 0;
volatile Uint32 capChaBLongFieldDetect = 0;

/******************************************************************
 *
 * NAME  
 *      Bt656Capt_8bit_Ncfd_NTSC
 *
 * PURPOSE:
 *      Setup dsp video port for video capture channel A & B.
 *
 *      Description : 8-bit BT.656 non-continuous frame capture 
 * 
 *      Some important field descriptions: 
 * 
 *      CMODE = 000, 8-bit BT.656 mode 
 *      CON = 0 
 *      FRAME = 1, capture frame 
 *      CF2 = 0 
 *      CF1 = 0, (8-bit non-continuous frame capture) 
 *      SCALE = 0, no scaling 
 *      RESMPL= 0, no re-sampling 
 *      10BPK = X, not used in 8-bit capture 
 *      EXC = 0, use EAV/SAV codes 
 *      VRST = 1, end of vertical blanking 
 *      HRST = 0, start of horizontal blanking 
 *      FLDD = 0, 1st line EAV or FID input 
 *      FINV = 0, no field invert 
 *      RDFE = X, used in Raw mode only(Enable field identification) 
 *      SSE = X, used in Raw mode only(Startup synch enable) 
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int16 Bt656Capt_8bit_Ncfd_NTSC(Int16 portNumber)
 *
 *      portNumber - (i)    Video port number to use
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int16 Bt656Capt_8bit_Ncfd_NTSC(int portNumber)
{
    /* Open video port for capture */
    VP_open(portNumber, VP_OPEN_RESET,vpCaptureHandle);
    if(vpCaptureHandle == INV)
        return(1);

    /* Enable video port functionality in VP Peripheral */
    /* Control Reg(PCR) */
    VP_FSETH(vpCaptureHandle, PCR, PEREN, VP_PCR_PEREN_ENABLE);


    /* Set this port to capture mode */
    /* clear VPHLT in VP_CTL to make video port function */
    VP_RSETH(vpCaptureHandle , VPCTL,VP_VPCTL_RMK(0,0,0,0,0,0,0,VP_VPCTL_DISP_CAPTURE,VP_VPCTL_DCHNL_DUAL));

    /* ------------------------------- */
    /* Enable all interrupts channel A */
    /* ------------------------------- */
    /* Enable capture overrun interrupt(COVRA) for VP channel A */
    VP_FSETH(vpCaptureHandle, VPIE, COVRA, VP_VPIE_COVRA_ENABLE);

    /* Enable capture complete interrupt(CCMPA) for VP channel A */
    VP_FSETH(vpCaptureHandle, VPIE, CCMPA, VP_VPIE_CCMPA_ENABLE);

    /* Enable channel synchronization error interrupt(SERRA) for */
    /* VP channel A */
    VP_FSETH(vpCaptureHandle, VPIE, SERRA, VP_VPIE_SERRA_ENABLE);

    /* Enable short field detect interrupt(SFDA) for VP channel A */
    VP_FSETH(vpCaptureHandle, VPIE, SFDA, VP_VPIE_SFDA_ENABLE);

    /* ------------------------------- */
    /* Enable all interrupts channel B */
    /* ------------------------------- */
    /* Enable capture overrun interrupt(COVRB) for VP channel B */
    VP_FSETH(vpCaptureHandle, VPIE, COVRB, VP_VPIE_COVRB_ENABLE);

    /* Enable capture complete interrupt(CCMPB) for VP channel B */
    VP_FSETH(vpCaptureHandle, VPIE, CCMPB, VP_VPIE_CCMPB_ENABLE);

    /* Enable channel synchronization error interrupt(SERRB) for */
    /* VP channel B */
    VP_FSETH(vpCaptureHandle, VPIE, SERRB, VP_VPIE_SERRB_ENABLE);

    /* Enable short field detect interrupt(SFDB) for VP channel B */
    VP_FSETH(vpCaptureHandle, VPIE, SFDB, VP_VPIE_SFDB_ENABLE);

    /* Enable video port global interrupt enable */
    //VP_FSETH(vpCaptureHandle, VPIE, VIE, VP_VPIE_VIE_ENABLE);

    /* ------------------------------------ */
    /* Setup all other fields for A channel */
    /* ------------------------------------ */
    /* Enable short field detect */
    VP_FSETH(vpCaptureHandle, VCACTL, SFDE, VP_VCACTL_SFDE_ENABLE);

    /* Set last pixel to be captured in Field1 (VCA_STOP1 reg) */
    VP_RSETH(vpCaptureHandle, VCASTOP1,VP_VCASTOP1_RMK(VCA_YSTOP1, VCA_XSTOP1));

    /* Set last pixel to be captured in Field2 (VCA_STOP2 reg) */
    VP_RSETH(vpCaptureHandle, VCASTOP2,VP_VCASTOP2_RMK(VCA_YSTOP2, VCA_XSTOP2));

    /* Set first pixel to be captured in Field1 (VCA_STRT1 reg) */
    VP_RSETH(vpCaptureHandle, VCASTRT1, VP_VCASTRT1_RMK(VCA_YSTART1, VP_VCASTRT1_SSE_ENABLE, 0));

    /* Set first pixel to be captured in Field2 (VCA_STRT2 reg) */
    //VP_RSETH(vpCaptureHandle, VCASTRT2,VP_VCASTRT2_RMK(VCA_YSTART2, VCA_XSTART2));
    VP_RSETH(vpCaptureHandle, VCASTRT2,VP_VCASTRT2_RMK(VCA_YSTART2, 0));

    /* Set threshold values */
    VP_RSETH(vpCaptureHandle, VCATHRLD,VP_VCATHRLD_RMK(VCA_THRLD_FIELD2, VCA_THRLD_FIELD1));

    /* Set capture event-register values */
    VP_RSETH(vpCaptureHandle, VCAEVTCT,VP_VCAEVTCT_RMK(VCA_CAPEVT2,VCA_CAPEVT1));

    /* Vertical interrupts (VCA_INT) are not enabled in this */
    /* Set CMODE to 8-bit BT.656 */
    VP_FSETH(vpCaptureHandle, VCACTL, CMODE, VP_VCACTL_CMODE_BT656B);

    /* Set non-continuous frame capture */
    VP_FSETH(vpCaptureHandle, VCACTL, CON, VP_VCACTL_CON_DISABLE);
    VP_FSETH(vpCaptureHandle, VCACTL, FRAME, VP_VCACTL_FRAME_FRMCAP);
    VP_FSETH(vpCaptureHandle, VCACTL, CF2, VP_VCACTL_CF2_NONE);
    VP_FSETH(vpCaptureHandle, VCACTL, CF1, VP_VCACTL_CF1_NONE);

    /* Let FDD and FINV to be their defaults */
    /* Set VRST to end of vertical blanking */
    VP_FSETH(vpCaptureHandle, VCACTL, VRST, VP_VCACTL_VRST_V0EAV);

    /* Set HRST to start of horizontal blanking */
    VP_FSETH(vpCaptureHandle, VCACTL, HRST, VP_VCACTL_HRST_SAV);

    VP_FSETH(vpCaptureHandle, VCACTL, LFDE, VP_VCACTL_LFDE_ENABLE);


    /* ------------------------------------ */
    /* Setup all other fields for B channel */
    /* ------------------------------------ */
    /* Enable short field detect */
    VP_FSETH(vpCaptureHandle, VCBCTL, SFDE, VP_VCACTL_SFDE_ENABLE);

    /* Set last pixel to be captured in Field1 (VCA_STOP1 reg) */
    VP_RSETH(vpCaptureHandle, VCBSTOP1,VP_VCASTOP1_RMK(VCA_YSTOP1, VCA_XSTOP1));

    /* Set last pixel to be captured in Field2 (VCA_STOP2 reg) */
    VP_RSETH(vpCaptureHandle, VCBSTOP2,VP_VCASTOP2_RMK(VCA_YSTOP2, VCA_XSTOP2));

    /* Set first pixel to be captured in Field1 (VCA_STRT1 reg) */
    VP_RSETH(vpCaptureHandle, VCBSTRT1, VP_VCASTRT1_RMK(VCA_YSTART1, VP_VCASTRT1_SSE_ENABLE, 0));

    /* Set first pixel to be captured in Field2 (VCA_STRT2 reg) */
    VP_RSETH(vpCaptureHandle, VCBSTRT2,VP_VCASTRT2_RMK(VCA_YSTART2, 0));

    /* Set threshold values */
    VP_RSETH(vpCaptureHandle, VCBTHRLD,VP_VCATHRLD_RMK(VCA_THRLD_FIELD2, VCA_THRLD_FIELD1));

    /* Set capture event-register values */
    VP_RSETH(vpCaptureHandle, VCBEVTCT,VP_VCAEVTCT_RMK(VCA_CAPEVT2,VCA_CAPEVT1));

    /* Vertical interrupts (VCA_INT) are not enabled in this */
    /* Set CMODE to 8-bit BT.656 */
    VP_FSETH(vpCaptureHandle, VCBCTL, CMODE, VP_VCACTL_CMODE_BT656B);

    /* Set non-continuous frame capture */
    VP_FSETH(vpCaptureHandle, VCBCTL, CON, VP_VCACTL_CON_DISABLE);
    VP_FSETH(vpCaptureHandle, VCBCTL, FRAME, VP_VCACTL_FRAME_FRMCAP);
    VP_FSETH(vpCaptureHandle, VCBCTL, CF2, VP_VCACTL_CF2_NONE);
    VP_FSETH(vpCaptureHandle, VCBCTL, CF1, VP_VCACTL_CF1_NONE);

    /* Let FDD and FINV to be their defaults */
    /* Set VRST to end of vertical blanking */
    VP_FSETH(vpCaptureHandle, VCBCTL, VRST, VP_VCACTL_VRST_V0EAV);

    /* Set HRST to start of horizontal blanking */
    VP_FSETH(vpCaptureHandle, VCBCTL, HRST, VP_VCACTL_HRST_SAV);

    VP_FSETH(vpCaptureHandle, VCBCTL, LFDE, VP_VCACTL_LFDE_ENABLE);

    return(0);
}

/**************************************************************************** 
 *
 * NAME  
 *      VPCaptEnable
 *
 * PURPOSE:
 *      Enable and already configured video port.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int16 VPCaptEnable(void)
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int16 VPCaptEnable(void)
{
    if(vpCaptureHandle == INV)
        return(1);

    /* Clear VPHLT in VP_CTL to make video port function */
    VP_FSETH(vpCaptureHandle, VPCTL, VPHLT, VP_VPCTL_VPHLT_CLEAR);

    /* -------------- */
    /* enable capture */
    /* -------------- */

    /* set VCEN bit to enable capture */
    VP_FSETH(vpCaptureHandle, VCACTL, VCEN, VP_VCACTL_VCEN_ENABLE);

    /* clear BLKCAP in VCA_CTL to enable capture DMA events */
    VP_FSETH(vpCaptureHandle, VCACTL, BLKCAP,VP_VCACTL_BLKCAP_CLEAR);

    /* set VCEN bit to enable capture */
    VP_FSETH(vpCaptureHandle, VCBCTL, VCEN, VP_VCACTL_VCEN_ENABLE);

    /* clear BLKCAP in VCA_CTL to enable capture DMA events */
    VP_FSETH(vpCaptureHandle, VCBCTL, BLKCAP,VP_VCACTL_BLKCAP_CLEAR);

    return(0);
}

/**************************************************************************** 
 *
 * NAME  
 *      VPCapChanPoll
 *
 * PURPOSE:
 *      Enable and already configured video port. Usually and irq routine
 *      is used for this purpose.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int16 VPCaptEnable(void)
 *
 *      capChaAOverrun          (o) - capture overrun channel A
 *      capChaASyncError        (o) - sync error channel A
 *      capChaAShortFieldDetect (o) - short filed detection channel A
 *      capChaAFrameCount       (o) - channel A capture frame count since start
 *      capChaBOverrun          (o) - capture overrun channel B
 *      capChaBSyncError        (o) - sync error channel B
 *      capChaBShortFieldDetect (o) - short filed detection channel B
 *      capChaBFrameCount       (o) - channel B capture frame count since start
 *
 * RETURN VALUE
 *      bit 0: capture frame complete channel A
 *      bit 1: capture frame complete channel B
 *     
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 VPCapChanPoll(void)
{
    Uint32 vpis = 0;
    Int32 rescode=0;

    /* Get video port status register value */
    vpis = VP_RGETH(vpCaptureHandle, VPIS);

    if(vpis & _VP_VPIS_CCMPA_MASK) /* capture complete */
    {
        /* Clear frame complete bit in VCX_CTL to */
        /* continue capture in non-continuous mode */
        VP_FSETH(vpCaptureHandle, VCASTAT, FRMC,VP_VCASTAT_FRMC_CLEAR);

        /* Clear CCMPA to enable next frame complete */
        /* interrupts */
        VP_FSETH(vpCaptureHandle, VPIS, CCMPA,VP_VPIS_CCMPA_CLEAR);
        capChaAFrameCount++; /* increment captured frame count */

        // Means A chan capture complete
        rescode|=1;
    }

    if(vpis & _VP_VPIS_COVRA_MASK) /* overrun error */
    {
        capChaAOverrun++;
        VP_FSETH(vpCaptureHandle, VPIS, COVRA,VP_VPIS_COVRA_CLEAR);
    }

    if(vpis & _VP_VPIS_SERRA_MASK) /* synchronization error */
    {
        capChaASyncError++;
        VP_FSETH(vpCaptureHandle, VPIS, SERRA,VP_VPIS_SERRA_CLEAR);
    }

    if(vpis & _VP_VPIS_SFDA_MASK) /* short field detect */
    {
        capChaAShortFieldDetect++;
        VP_FSETH(vpCaptureHandle, VPIS, SFDA, VP_VPIS_SFDA_CLEAR);
    }

    if(vpis & _VP_VPIS_LFDA_MASK) /* long field detect */
    {
        capChaALongFieldDetect++;
        VP_FSETH(vpCaptureHandle, VPIS, LFDA, VP_VPIS_LFDA_CLEAR);
    }

    if(vpis & _VP_VPIS_CCMPB_MASK) /* capture complete */
    {
        /* Clear frame complete bit in VCX_CTL to */
        /* continue capture in non-continuous mode */
        VP_FSETH(vpCaptureHandle, VCBSTAT, FRMC,VP_VCBSTAT_FRMC_CLEAR);

        /* Clear CCMPB to enable next frame complete */
        /* interrupts */
        VP_FSETH(vpCaptureHandle, VPIS, CCMPB,VP_VPIS_CCMPB_CLEAR);
        capChaBFrameCount++; /* increment captured frame count */

        // Means B chan capture complete
        rescode|=2;
    }

    if(vpis & _VP_VPIS_COVRB_MASK) /* overrun error */
    {
        capChaBOverrun++;
        VP_FSETH(vpCaptureHandle, VPIS, COVRB,VP_VPIS_COVRB_CLEAR);
    }

    if(vpis & _VP_VPIS_SERRB_MASK) /* synchronization error */
    {
        capChaBSyncError++;
        VP_FSETH(vpCaptureHandle, VPIS, SERRB,VP_VPIS_SERRB_CLEAR);
    }

    if(vpis & _VP_VPIS_SFDB_MASK) /* short field detect */
    {
        capChaBShortFieldDetect++;
        VP_FSETH(vpCaptureHandle, VPIS, SFDB, VP_VPIS_SFDB_CLEAR);
    }

    if(vpis & _VP_VPIS_LFDB_MASK) /* long field detect */
    {
        capChaBLongFieldDetect++;
        VP_FSETH(vpCaptureHandle, VPIS, LFDB, VP_VPIS_LFDB_CLEAR);
    }

    return(rescode);
}

/**************************************************************************** 
 *
 * NAME  
 *      CloseDisplayVideoPort
 *
 * PURPOSE:
 *      Close a video port opened by Bt656_8bit_Ncfd_NTSC.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int16 CloseDisplayVideoPort(void)
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int16 CloseCaptureVideoPort(void)
{
    if(vpCaptureHandle == INV)
        return(1);

    /* Close video port for display */
    VP_close(vpCaptureHandle);

    return(0);
}
