/***************************************************************************
 *
 *
 *         **   **    **  ******  ********  ********  ********  **    **    
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ***************************************************************************
 *                                                                          
 *  Project     : EVM
 *  File        : MCASP.h
 *  Description : Basic MCASP functionnality for evm examples
 *
 *                Note that this source code is only intended to serve as
 *                an example of how to use the DM648 EVM Board Support
 *                Library (BSL). It should not be considered to be a
 *                complete software design reference for DM648 peripherals
 *                and was not written for this purpose. This software is
 *                provided as is and Lyrtech does not guarantee it or
 *                provide any support for it.
 *
 *                      Copyright (c) Lyrtech inc. 2007                        
 *
 ***************************************************************************
 *                                                                                          
 * "$Revision: 1.1 $"
 * "$Date: 2007/09/13 16:04:08 $"
 *
 ***************************************************************************/
#ifndef __MCASP_H__
#define __MCASP_H__

/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include "cslr_mcasp.h"
#include "csl_types.h"


/****************************************************************************
 *                             Global constants                             *
 ****************************************************************************/

/* Number of Serializers */
#define MCASP_NUM_OF_SERIALIZERS 10

/* FREE Tokens */
#define MCASP_PWRDEMU_FREE_ON    (0x00000001u)

/* 
 * McASP Module specific object structure 
 */
typedef struct McaspObj {
    /* Pointer to the register overlay structure for the peripheral */
    CSL_McaspRegsOvly   regs;
    /* Specifies a particular instance of McASP */
    Int16               perNo;
    /* Number of serializers */
    Int32               numOfSerializers;
    /* Support for DIT mode */
    Bool                ditStatus;
} McaspObj;

/*
 * This data type is used to return the handle to an
 * instance of McASP
 */
typedef struct McaspObj *McaspHandle;

/* 
 * McASP Module specific Hardware setup global structure 
 */
typedef struct McaspHwSetupGbl {
    /* Pin function register */
    Uint32  pfunc;
    /* Pin direction register */
    Uint32  pdir;
    /* Global control register - GBLCTL */
    Uint32  ctl;
    /* Decides whether McASP operates in DIT mode - DITCTL */
    Uint32  ditCtl;
    /* Digital loopback mode setup - DLBEN */
    Uint32  dlbMode;
    /* Mute control register - AMUTE */
    Uint32  amute;
    /* Setup serializer control register (XRSRCTL0-9) */
    Uint32  serSetup[MCASP_NUM_OF_SERIALIZERS];
} McaspHwSetupGbl;

/* 
 * Hardware setup data clock structure 
 */
typedef struct McaspHwSetupDataClk {
    /* Clock details ACLK(R/X)CTL */
    Uint32  clkSetupClk;
    /* High clock details AHCLK(R/X)CTL */
    Uint32  clkSetupHiClk;
    /* Configures receive/transmit clock failure detection R/XCLKCHK */
    Uint32  clkChk;
} McaspHwSetupDataClk;

/* 
 * Hardware setup data structure 
 */
typedef struct McaspHwSetupData {
    /* To mask or not to mask - R/XMASK */
    Uint32                      mask;
    /* Format details as per  - R/XFMT */
    Uint32                      fmt;
    /* Configure the rcv/xmt frame sync - AFSR/XCTL */
    Uint32                      frSyncCtl;
    /* Specifies which TDM slots are active - R/XTDM */
    Uint32                      tdm;
    /* Controls generation of McASP interrupts - R/XINTCTL */
    Uint32                      intCtl;
    /* Status register (controls writable fields of STAT register)-R/XSTAT */
    Uint32                      stat;
    /* Event control register - R/XEVTCTL */
    Uint32                      evtCtl;
    /* Clock settings for rcv/xmt */
    McaspHwSetupDataClk         clk;
} McaspHwSetupData;

/* 
 * Hardware setup structure 
 */
typedef struct McaspHwSetup {
    /* Value to be loaded in global control register (GLBCTL) */
    McaspHwSetupGbl     glb;
    /* Receiver settings */
    McaspHwSetupData    rx;
    /* Transmitter settings */
    McaspHwSetupData    tx;
    /* Power down emulation mode params - PWRDEMU */
    Uint32              emu;
} McaspHwSetup;


// MCASP macro definitions for basic McASP usage
//

#define McaspOpen(pMcaspObj,mcaspNum,mhMcasp)                                   \
{                                                                               \
        (pMcaspObj)->regs = (CSL_McaspRegsOvly) CSL_MCASP_0_REGS;               \
        (pMcaspObj)->perNo = (CSL_InstNum) (mcaspNum);                          \
        (pMcaspObj)->numOfSerializers = MCASP_NUM_OF_SERIALIZERS;               \
        (pMcaspObj)->ditStatus = TRUE;                                          \
        (mhMcasp) = (McaspHandle) (pMcaspObj);                                  \
}


#define McaspRegReset(hMcasp)                                                   \
{                                                                               \
    Uint32 serNum = 0;                                                          \
                                                                                \
    (hMcasp)->regs->GBLCTL = CSL_MCASP_GBLCTL_RESETVAL;                         \
    (hMcasp)->regs->PWREMUMGT = CSL_MCASP_PWREMUMGT_RESETVAL;                   \
    (hMcasp)->regs->RXMASK = CSL_MCASP_RXMASK_RESETVAL;                         \
    (hMcasp)->regs->RXFMT = CSL_MCASP_RXFMT_RESETVAL;                           \
    (hMcasp)->regs->RXFMCTL = CSL_MCASP_RXFMCTL_RESETVAL;                       \
    (hMcasp)->regs->ACLKRCTL = CSL_MCASP_ACLKRCTL_RESETVAL;                     \
    (hMcasp)->regs->AHCLKRCTL = CSL_MCASP_AHCLKRCTL_RESETVAL;                   \
    (hMcasp)->regs->RXTDM = CSL_MCASP_RXTDM_RESETVAL;                           \
    (hMcasp)->regs->EVTCTLR = CSL_MCASP_EVTCTLR_RESETVAL;                       \
    (hMcasp)->regs->RXCLKCHK = CSL_MCASP_RXCLKCHK_RESETVAL;                     \
    (hMcasp)->regs->TXMASK = CSL_MCASP_TXMASK_RESETVAL;                         \
    (hMcasp)->regs->TXFMT = CSL_MCASP_TXFMT_RESETVAL;                           \
    (hMcasp)->regs->TXFMCTL = CSL_MCASP_TXFMCTL_RESETVAL;                       \
    (hMcasp)->regs->ACLKXCTL = CSL_MCASP_ACLKXCTL_RESETVAL;                     \
    (hMcasp)->regs->AHCLKXCTL = CSL_MCASP_AHCLKXCTL_RESETVAL;                   \
    (hMcasp)->regs->TXTDM = CSL_MCASP_TXTDM_RESETVAL;                           \
    (hMcasp)->regs->EVTCTLX = CSL_MCASP_EVTCTLX_RESETVAL;                       \
    (hMcasp)->regs->TXCLKCHK = CSL_MCASP_TXCLKCHK_RESETVAL;                     \
    while (serNum < (hMcasp)->numOfSerializers)                                 \
    {                                                                           \
        (hMcasp)->regs->XRSRCTL0 = CSL_MCASP_XRSRCTL0_RESETVAL;                 \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
        (hMcasp)->regs->XRSRCTL1 = CSL_MCASP_XRSRCTL1_RESETVAL;                 \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
        (hMcasp)->regs->XRSRCTL2 = CSL_MCASP_XRSRCTL2_RESETVAL;                 \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
        (hMcasp)->regs->XRSRCTL3 = CSL_MCASP_XRSRCTL3_RESETVAL;                 \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
        (hMcasp)->regs->XRSRCTL4 = CSL_MCASP_XRSRCTL4_RESETVAL;                 \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
        (hMcasp)->regs->XRSRCTL5 = CSL_MCASP_XRSRCTL5_RESETVAL;                 \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
        (hMcasp)->regs->XRSRCTL6 = CSL_MCASP_XRSRCTL6_RESETVAL;                 \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
        (hMcasp)->regs->XRSRCTL7 = CSL_MCASP_XRSRCTL7_RESETVAL;                 \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
        (hMcasp)->regs->XRSRCTL8 = CSL_MCASP_XRSRCTL8_RESETVAL;                 \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
        (hMcasp)->regs->XRSRCTL9 = CSL_MCASP_XRSRCTL9_RESETVAL;                 \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
    }                                                                           \
    (hMcasp)->regs->PFUNC = CSL_MCASP_PFUNC_RESETVAL;                           \
    (hMcasp)->regs->PDIR = CSL_MCASP_PDIR_RESETVAL;                             \
    (hMcasp)->regs->TXDITCTL = CSL_MCASP_TXDITCTL_RESETVAL;                     \
    (hMcasp)->regs->LBCTL = CSL_MCASP_LBCTL_RESETVAL;                           \
    (hMcasp)->regs->AMUTE = CSL_MCASP_AMUTE_RESETVAL;                           \
    (hMcasp)->regs->RXSTAT = CSL_MCASP_RXSTAT_RESETVAL;                         \
    (hMcasp)->regs->REVTCTL = CSL_MCASP_REVTCTL_RESETVAL;                       \
    (hMcasp)->regs->TXSTAT = CSL_MCASP_TXSTAT_RESETVAL;                         \
    (hMcasp)->regs->XEVTCTL = CSL_MCASP_XEVTCTL_RESETVAL;                       \
    (hMcasp)->regs->PDOUT = CSL_MCASP_PDOUT_RESETVAL;                           \
    (hMcasp)->regs->PDIN = CSL_MCASP_PDIN_RESETVAL;                             \
    (hMcasp)->regs->PDCLR = CSL_MCASP_PDCLR_RESETVAL;                           \
}

#define McaspHwSetup(hMcasp,myHwSetup)                                          \
{                                                                               \
    Uint32 serNum = 0;                                                          \
                                                                                \
    /* Reset McASP to default values by setting GBLCTL = 0 */                   \
    (hMcasp)->regs->GBLCTL = (Uint32) CSL_MCASP_GBLCTL_RESETVAL;                \
                                                                                \
    /* Initialize the powerdown and emulation management register */            \
    CSL_FINS ((hMcasp)->regs->PWREMUMGT,MCASP_PWREMUMGT_FREE,(myHwSetup)->emu); \
                                                                                \
    /* Configure the RXMASK register */                                         \
    (hMcasp)->regs->RXMASK = (Uint32) (myHwSetup)->rx.mask;                     \
                                                                                \
    /* Configure RXFMT */                                                       \
    (hMcasp)->regs->RXFMT = (Uint32) (myHwSetup)->rx.fmt;                       \
                                                                                \
    /* Configure RXFMCTL */                                                     \
    (hMcasp)->regs->RXFMCTL = (Uint32) (myHwSetup)->rx.frSyncCtl;               \
                                                                                \
    /* Configure ACLKRCTL */                                                    \
    (hMcasp)->regs->ACLKRCTL = (Uint32) (myHwSetup)->rx.clk.clkSetupClk;        \
                                                                                \
    /* Configure AHCLKRCTL */                                                   \
    (hMcasp)->regs->AHCLKRCTL = (Uint32) (myHwSetup)->rx.clk.clkSetupHiClk;     \
                                                                                \
    /* Configure RXTDM */                                                       \
    (hMcasp)->regs->RXTDM = (Uint32) (myHwSetup)->rx.tdm;                       \
                                                                                \
    /* Configure EVTCTLR */                                                     \
    (hMcasp)->regs->EVTCTLR = (Uint32) (myHwSetup)->rx.intCtl;                  \
                                                                                \
    /* Configure RXCLKCHK */                                                    \
    (hMcasp)->regs->RXCLKCHK = (Uint32) (myHwSetup)->rx.clk.clkChk;             \
                                                                                \
    /* Configure TXMASK */                                                      \
    (hMcasp)->regs->TXMASK = (Uint32) (myHwSetup)->tx.mask;                     \
                                                                                \
    /* Configure TXFMT */                                                       \
    (hMcasp)->regs->TXFMT = (Uint32) (myHwSetup)->tx.fmt;                       \
                                                                                \
    /* Configure TXFMCTL */                                                     \
    (hMcasp)->regs->TXFMCTL = (Uint16) (myHwSetup)->tx.frSyncCtl;               \
                                                                                \
    /* Configure ACLKXCTL */                                                    \
    (hMcasp)->regs->ACLKXCTL = (Uint16) (myHwSetup)->tx.clk.clkSetupClk;        \
                                                                                \
    /* Configure AHCLKXCTL */                                                   \
    (hMcasp)->regs->AHCLKXCTL = (Uint16) (myHwSetup)->tx.clk.clkSetupHiClk;     \
                                                                                \
    /* Configure TXTDM */                                                       \
    (hMcasp)->regs->TXTDM = (Uint32) (myHwSetup)->tx.tdm;                       \
                                                                                \
    /* Configure EVTCTLX */                                                     \
    (hMcasp)->regs->EVTCTLX = (Uint16) (myHwSetup)->tx.intCtl;                  \
                                                                                \
    /* Configure TXCLKCHK */                                                    \
    (hMcasp)->regs->TXCLKCHK = (Uint32) (myHwSetup)->tx.clk.clkChk;             \
                                                                                \
    /* Configure serializer control registers 0-5 for McASP */                  \
    while (serNum < (hMcasp) ->numOfSerializers)                                \
    {                                                                           \
        (hMcasp)->regs->XRSRCTL0 = (Uint16) (myHwSetup)->glb.serSetup[serNum];  \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
        (hMcasp)->regs->XRSRCTL1 = (Uint16) (myHwSetup)->glb.serSetup[serNum];  \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
        (hMcasp)->regs->XRSRCTL2 = (Uint16) (myHwSetup)->glb.serSetup[serNum];  \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
        (hMcasp)->regs->XRSRCTL3 = (Uint16) (myHwSetup)->glb.serSetup[serNum];  \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
        (hMcasp)->regs->XRSRCTL4 = (Uint16) (myHwSetup)->glb.serSetup[serNum];  \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
        (hMcasp)->regs->XRSRCTL5 = (Uint16) (myHwSetup)->glb.serSetup[serNum];  \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
        (hMcasp)->regs->XRSRCTL6 = (Uint16) (myHwSetup)->glb.serSetup[serNum];  \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
        (hMcasp)->regs->XRSRCTL7 = (Uint16) (myHwSetup)->glb.serSetup[serNum];  \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
        (hMcasp)->regs->XRSRCTL8 = (Uint16) (myHwSetup)->glb.serSetup[serNum];  \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
        (hMcasp)->regs->XRSRCTL9 = (Uint16) (myHwSetup)->glb.serSetup[serNum];  \
        if (++serNum >= (hMcasp)->numOfSerializers)                             \
            break;                                                              \
    }                                                                           \
                                                                                \
    /* Configure pin function register */                                       \
    (hMcasp)->regs->PFUNC = (Uint32) (myHwSetup)->glb.pfunc;                    \
                                                                                \
    /* Configure pin direction register */                                      \
    (hMcasp)->regs->PDIR = (Uint32) (myHwSetup)->glb.pdir;                      \
                                                                                \
    /* Configure DITCTL */                                                      \
    if ((hMcasp)->ditStatus == TRUE)                                            \
    {                                                                           \
        (hMcasp)->regs->TXDITCTL = (Uint32) (myHwSetup)->glb.ditCtl;            \
    }                                                                           \
                                                                                \
    /* Configure LBCTL */                                                       \
    (hMcasp)->regs->LBCTL = (Uint32) (myHwSetup)->glb.dlbMode;                  \
                                                                                \
    /* Configure AMUTE */                                                       \
    (hMcasp)->regs->AMUTE = (Uint32) (myHwSetup)->glb.amute;                    \
                                                                                \
    /* Configure RXSTAT and TXSTAT */                                           \
    (hMcasp)->regs->RXSTAT = (Uint32) (myHwSetup)->rx.stat;                     \
    (hMcasp)->regs->TXSTAT = (Uint32) (myHwSetup)->tx.stat;                     \
                                                                                \
    /* Configure REVTCTL and XEVTCTL */                                         \
    (hMcasp)->regs->REVTCTL = (Uint32) (myHwSetup)->rx.evtCtl;                  \
    (hMcasp)->regs->XEVTCTL = (Uint32) (myHwSetup)->tx.evtCtl;                  \
                                                                                \
    /* Initialize the global control register */                                \
    (hMcasp)->regs->GBLCTL = (Uint32) (myHwSetup)->glb.ctl;                     \
}

#define McaspResetCtrl(hMcasp,selectMask)                                       \
{                                                                               \
        /* Activate the RXFSRST bit field of GBLCTL */                          \
        if ((selectMask) & CSL_MCASP_GBLCTL_RXFSRST_MASK)                       \
        {                                                                       \
            CSL_FINST ((hMcasp)->regs->GBLCTL, MCASP_GBLCTL_RXFSRST, ACTIVE);   \
            while (((hMcasp)->regs->GBLCTL & CSL_MCASP_GBLCTL_RXFSRST_MASK)     \
                    != CSL_MCASP_GBLCTL_RXFSRST_MASK);                          \
        }                                                                       \
                                                                                \
        /* Activate the RXSMRST bit field of GBLCTL */                          \
        if ((selectMask) & CSL_MCASP_GBLCTL_RXSMRST_MASK)                       \
        {                                                                       \
            CSL_FINST ((hMcasp)->regs->GBLCTL, MCASP_GBLCTL_RXSMRST, ACTIVE);   \
            while (((hMcasp)->regs->GBLCTL & CSL_MCASP_GBLCTL_RXSMRST_MASK)     \
                    != CSL_MCASP_GBLCTL_RXSMRST_MASK);                          \
        }                                                                       \
                                                                                \
        /* Activate the RXSERCLR bit field of GBLCTL */                         \
        if ((selectMask) & CSL_MCASP_GBLCTL_RXSERCLR_MASK)                      \
        {                                                                       \
            CSL_FINST ((hMcasp)->regs->GBLCTL, MCASP_GBLCTL_RXSERCLR, ACTIVE);  \
            while (((hMcasp)->regs->GBLCTL & CSL_MCASP_GBLCTL_RXSERCLR_MASK)    \
                    != CSL_MCASP_GBLCTL_RXSERCLR_MASK);                         \
        }                                                                       \
                                                                                \
        /* Activate the RXHCLKRST bit field of GBLCTL */                        \
        if ((selectMask) & CSL_MCASP_GBLCTL_RXHCLKRST_MASK)                     \
        {                                                                       \
            CSL_FINST ((hMcasp)->regs->GBLCTL, MCASP_GBLCTL_RXHCLKRST, ACTIVE); \
            while (((hMcasp)->regs->GBLCTL & CSL_MCASP_GBLCTL_RXHCLKRST_MASK)   \
                    != CSL_MCASP_GBLCTL_RXHCLKRST_MASK);                        \
        }                                                                       \
                                                                                \
        /* Activate the RXCLKRST bit field of GBLCTL */                         \
        if ((selectMask) & CSL_MCASP_GBLCTL_RXCLKRST_MASK)                      \
        {                                                                       \
            CSL_FINST ((hMcasp)->regs->GBLCTL, MCASP_GBLCTL_RXCLKRST, ACTIVE);  \
            while (((hMcasp)->regs->GBLCTL & CSL_MCASP_GBLCTL_RXCLKRST_MASK)    \
                    != CSL_MCASP_GBLCTL_RXCLKRST_MASK);                         \
        }                                                                       \
                                                                                \
        /* Activate the TXFSRST bit field of GBLCTL */                          \
        if ((selectMask) & CSL_MCASP_GBLCTL_TXFSRST_MASK)                       \
        {                                                                       \
            CSL_FINST ((hMcasp)->regs->GBLCTL, MCASP_GBLCTL_TXFSRST, ACTIVE);   \
            while (((hMcasp)->regs->GBLCTL & CSL_MCASP_GBLCTL_TXFSRST_MASK)     \
                    != CSL_MCASP_GBLCTL_TXFSRST_MASK);                          \
        }                                                                       \
                                                                                \
        /* Activate the TXSMRST bit field of GBLCTL */                          \
        if ((selectMask) & CSL_MCASP_GBLCTL_TXSMRST_MASK)                       \
        {                                                                       \
            CSL_FINST ((hMcasp)->regs->GBLCTL, MCASP_GBLCTL_TXSMRST, ACTIVE);   \
            while (((hMcasp)->regs->GBLCTL & CSL_MCASP_GBLCTL_TXSMRST_MASK)     \
                    != CSL_MCASP_GBLCTL_TXSMRST_MASK);                          \
        }                                                                       \
                                                                                \
        /* Activate the TXSERCLR bit field of GBLCTL */                         \
        if ((selectMask) & CSL_MCASP_GBLCTL_TXSERCLR_MASK)                      \
        {                                                                       \
            CSL_FINST ((hMcasp)->regs->GBLCTL, MCASP_GBLCTL_TXSERCLR, ACTIVE);  \
            while (((hMcasp)->regs->GBLCTL & CSL_MCASP_GBLCTL_TXSERCLR_MASK)    \
                    != CSL_MCASP_GBLCTL_TXSERCLR_MASK);                         \
        }                                                                       \
                                                                                \
        /* Activate the TXHCLKRST bit field of GBLCTL */                        \
        if ((selectMask) & CSL_MCASP_GBLCTL_TXHCLKRST_MASK)                     \
        {                                                                       \
            CSL_FINST ((hMcasp)->regs->GBLCTL, MCASP_GBLCTL_TXHCLKRST, ACTIVE); \
            while (((hMcasp)->regs->GBLCTL & CSL_MCASP_GBLCTL_TXHCLKRST_MASK)   \
                    != CSL_MCASP_GBLCTL_TXHCLKRST_MASK);                        \
        }                                                                       \
                                                                                \
        /* Activate the TXCLKRST bit field of GBLCTL */                         \
        if ((selectMask) & CSL_MCASP_GBLCTL_TXCLKRST_MASK)                      \
        {                                                                       \
            CSL_FINST ((hMcasp)->regs->GBLCTL, MCASP_GBLCTL_TXCLKRST, ACTIVE);  \
            while (((hMcasp)->regs->GBLCTL & CSL_MCASP_GBLCTL_TXCLKRST_MASK)    \
                    != CSL_MCASP_GBLCTL_TXCLKRST_MASK);                         \
        }                                                                       \
}


#define McaspActivateSmRcvXmt(hMcasp)                                           \
{                                                                               \
    CSL_BitMask16   selectMask;                                                 \
                                                                                \
    selectMask=CSL_MCASP_GBLCTL_RXSMRST_MASK|CSL_MCASP_GBLCTL_TXSMRST_MASK;     \
                                                                                \
    (hMcasp)->regs->GBLCTL = ((hMcasp)->regs->GBLCTL | selectMask);             \
    while (((hMcasp)->regs->GBLCTL & selectMask) != selectMask);                \
}


#define McaspActivateFsRcvXmt(hMcasp)                                           \
{                                                                               \
    CSL_BitMask16   selectMask;                                                 \
                                                                                \
    selectMask=CSL_MCASP_GBLCTL_RXFSRST_MASK|CSL_MCASP_GBLCTL_TXFSRST_MASK;     \
                                                                                \
    (hMcasp)->regs->GBLCTL = ((hMcasp)->regs->GBLCTL | selectMask);             \
    while (((hMcasp)->regs->GBLCTL & selectMask) != selectMask);                \
}


#endif /*  __MCASP_H__ */


