/***************************************************************************
 *
 *
 *         **   **    **  ******  ********  ********  ********  **    **    
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ***************************************************************************
 *                                                                          
 *  Project     : EVM
 *  File        : BT656.h
 *  Description : Utility functions for video port configuration
 *
 *                      Copyright (c) Lyrtech inc. 2007                        
 *
 ***************************************************************************
 *                                                                                          
 * "$Revision: 1.3 $"
 * "$Date: 2007/07/13 19:12:16 $"
 *
 ***************************************************************************/

#ifndef __BT656_H__ 
#define __BT656_H__

/****************************************************************************
 *                             Local constants                              *
 ****************************************************************************/

/**********************************************************/
/* Display parameter definitions based on 525/60 format */
/**********************************************************/

/* ----------------- */
/* Define frame size */
/* ----------------- */
#define VD_FRM_WIDTH 858 /* no of pixels per frame line */

/* including horizontal blanking */
#define VD_FRM_HEIGHT 525 /* total no of lines per frame */
#define VD_FRM_SIZE (VD_FRM_WIDTH * VD_FRM_HEIGHT)

/* ------------------- */
/* Horizontal blanking */
/* ------------------- */
#define VD_HBLNK_START 720 /* starting location of EAV */
#define VD_HBLNK_STOP 856 /* starting location of SAV */
#define VD_HBLNK_SIZE (VD_HBLNK_STOP - VD_HBLNK_START +2 /*EAV*/) /* (138) EAV, SAV inclusive */

/* ---------------------------- */
/* Vertical blanking for field1 */
/* ---------------------------- */
#define VD_VBLNK_XSTART1 720 /* pixel on which VBLNK active */

/* edge occurs for field1 */
#define VD_VBLNK_YSTART1 1 /* line on which VBLNK active */

/* edge occurs for field1 */
#define VD_VBLNK_XSTOP1 720 /* pixel on which VBLNK inactive */

/* edge occurs for field1 */

#define VD_VBLNK_YSTOP1 21 /* line on which VBLNK inactive */

/* edge occurs for field1 */

/* ---------------------------- */
/* Vertical blanking for field2 */
/* ---------------------------- */
#define VD_VBLNK_XSTART2 360 /* pixel on which VBLNK active */

/* edge occurs for field2 */
#define VD_VBLNK_YSTART2 263 /* line on which VBLNK active */

/* edge occurs for field2 */
#define VD_VBLNK_XSTOP2 360 /* pixel on which VBLNK inactive */

/* edge occurs for field2 */

#define VD_VBLNK_YSTOP2 284 /* line on which VBLNK inactive */

/* edge occurs for field2 */

/* ------------------------------------------------- */
/* Define vertical blanking bit(VD_VBITn) reg values */
/* ------------------------------------------------- */
#define VD_VBIT_SET1 1 /* first line with an EAV with V=1 */

/* indicating the start of Field1 */
/* vertical blanking */
#define VD_VBIT_CLR1 21 /* first line with an EAV with V=0 */

/* indicating the start of Field1 */
/* active display */
#define VD_VBLNK1_SIZE (VD_VBIT_CLR1 - VD_VBIT_SET1) /* 19 lines */

#define VD_VBIT_SET2 263 /* first line with an EAV with V=1 */

/* indicating the start of Field2 */
/* vertical blanking */
#define VD_VBIT_CLR2 284 /* first line with an EAV with V=0 */

/* indicating the start of Field2 */
/* active display */
#define VD_VBLNK2_SIZE (VD_VBIT_CLR2 - VD_VBIT_SET2) /* 19 lines */

/* ------------ */
/* Field timing */
/* ------------ */
#define VD_FIELD1_XSTART 720 /* pixel on the first line of */

/* Field1 on which FLD output */
/* is de-asserted */
#define VD_FIELD1_YSTART 1 /* line on which FLD is de-asserted */
#define VD_FIELD2_XSTART 360 /* pixel on the first line of */

/* Field1 on which FLD output */
/* is asserted */
#define VD_FIELD2_YSTART 263 /* line on which FLD is asserted */

/* ------------------------------------ */
/* Define field bit(VD_FBIT) reg values */
/* ------------------------------------ */
#define VD_FBIT_CLR 1 /* first line with an EAV with F=0 */

/* indicating Field 1 display */
#define VD_FBIT_SET 263 /* first line with an EAV with F=1 */

/* indicating Field 2 display */

/* -------------------------------- */
/* Define horizontal synchronization */
/* -------------------------------- */
#define VD_HSYNC_START 736
#define VD_HSYNC_STOP 800

/* ------------------------------------------ */
/* Define vertical synchronization for field1 */
/* ------------------------------------------ */
#define VD_VSYNC_XSTART1 (720+16)
#define VD_VSYNC_YSTART1 4

#define VD_VSYNC_XSTOP1 (720+16)
#define VD_VSYNC_YSTOP1 7

/* ------------------------------------------ */
/* Define vertical synchronization for field2 */
/* ------------------------------------------ */
#define VD_VSYNC_XSTART2 307

#define VD_VSYNC_YSTART2 266

#define VD_VSYNC_XSTOP2 307
#define VD_VSYNC_YSTOP2 269

/* ---------------------------------------- */
/* Define image offsets for both the fields */
/* which are zero in this example */
/* ---------------------------------------- */
#define VD_IMG_HOFF1 0
#define VD_IMG_VOFF1 0
#define VD_IMG_HOFF2 0
#define VD_IMG_VOFF2 0

/* ------------------------------------------------- */
/* Define image active vertical and horizontal sizes */
/* ------------------------------------------------- */
#define VD_IMG_HSIZE1 720 /* field1 horizontal image size */

#define VD_IMG_VSIZE1 240 /* field1 vertical image size */

#define VD_IMG_HSIZE2 720 /* field2 horizontal image size */

#define VD_IMG_VSIZE2 240 /* field2 vertical image size */

/* Manipulate field1 and field2 image sizes */
#define VD_IMAGE_SIZE1 (VD_IMG_HSIZE1 * VD_IMG_VSIZE1)
#define VD_IMAGE_SIZE2 (VD_IMG_HSIZE2 * VD_IMG_VSIZE2)

/* Define threshold values in double-words. Both fields should */
/* have same threshold value */
#define VD_VDTHRLD1 (VD_IMG_HSIZE1/8) /* line length in */
#define VD_VDTHRLD2 (VD_IMG_HSIZE2/8) /* double-words */

//#define VD_VDTHRLD2 VD_VDTHRLD1 /* double-words */
/* Define number of events to be generated for field1 and field2 */
#define VD_DISPEVT1 (VD_IMAGE_SIZE1 / (VD_VDTHRLD1 * 8))
#define VD_DISPEVT2 (VD_IMAGE_SIZE2 / (VD_VDTHRLD2 * 8))
#define DISPLAY_FRAME_COUNT 5 /* in this example */

/* -------------------------------------------- */
/* EDMA parameters for display Y event that are */
/* specific to this example. */
/* -------------------------------------------- */
/* VD_VDTHRLDn is in double-words and 32-bit element size */
#define VD_Y_EDMA_ELECNT (VD_VDTHRLD1 * 2)
#define VD_Y_EDMA_FRMCNT ((VD_DISPEVT1 + VD_DISPEVT2) *DISPLAY_FRAME_COUNT)


/****************************************************************************
 *                             Public Functions                             *
 ****************************************************************************/


/**************************************************************************** 
 *
 * NAME  
 *      Bt656_8bit_Ncfd_NTSC
 *
 * PURPOSE:
 *      Setup dsp video port for display.
 *
 *      Description : 8-bit BT.656 non-continuous frame display
 *
 *      Some important field descriptions:
 *
 *      DMODE = 000, 8-bit BT.656 mode
 *      CON = 0
 *      FRAME = 1, display frame
 *      DF2 = 0
 *      DF1 = 0, (8-bit non-continuous frame display)
 *      SCALE = 0, no scaling
 *      RESMPL = 0, no resampling
 *      DPK = X, not used in 8-bit display
 *      RSYNC = X, used in Raw mode(Enable second synchronized raw
 *      data channel)
 *      RGBX = X, used in Raw mode(RGB extract enable. Perform
 *      3/4 FIFO unpacking)
 *      VCTL1S = 00, output HSYNC
 *      VCTL2S = 00, output VSYNC
 *      VCTL3S = 0, output CBLNK
 *      HXS = 0, VCTL1 is an output
 *      VXS = 0, VCTL2 is an output
 *      FXS = 0, VCTL3 is an output
 *      PVPSYN = 0, no previous port synchronization
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int16 Bt656_8bit_Ncfd_NTSC(Int16 portNumber)
 *
 *      portNumber - (i)    Video port number to use (1 for HD port)
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int16 Bt656_8bit_Ncfd_NTSC(Int16 portNumber);


/**************************************************************************** 
 *
 * NAME  
 *      VPEnable
 *
 * PURPOSE:
 *      Enable an already configured video port.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int16 VPEnable(void)
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int16 VPEnable(void);


/**************************************************************************** 
 *
 * NAME  
 *      VPDispPoll
 *
 * PURPOSE:
 *      Clears FRMD to continue display in this non-continuous mode and also
 *      clears other status bits. This routine must be called at least one
 *      time each display frame. This functionnality would usually being done
 *      by an interrupt routine.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      void VPDispPoll(void)
 *
 *      displayFrameCount (o) - will hold number of frame displayed since started
 *      dispUnderrun      (o) - will hold number of video underrun errors detected
 *
 * RETURN VALUE
 *      1 if frame display is completed, 0 if not
 *
 * REFERENCE
 *
 ****************************************************************************/
Uint32 VPDispPoll(void);


/**************************************************************************** 
 *
 * NAME  
 *      CloseDisplayVideoPort
 *
 * PURPOSE:
 *      Close a video port opened by Bt656_8bit_Ncfd_NTSC.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int16 CloseDisplayVideoPort(void)
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int16 CloseDisplayVideoPort(void);


#endif // __BT656_H__
