/***************************************************************************
 *
 *
 *         **   **    **  ******  ********  ********  ********  **    **    
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ***************************************************************************
 *                                                                          
 *  Project     : EVM
 *  File        : BT656capt.h
 *  Description : Utility functions for video port capture configuration
 *
 *                      Copyright (c) Lyrtech inc. 2007                        
 *
 ***************************************************************************
 *                                                                                          
 * "$Revision: 1.4 $"
 * "$Date: 2007/07/24 20:30:15 $"
 *
 ***************************************************************************/

#ifndef __BT656CAPT_H__ 
#define __BT656CAPT_H__

/****************************************************************************
 *                             Local constants                              *
 ****************************************************************************/


/**********************************************************/
/* Capture parameter definitions based on 525/60 format */
/**********************************************************/
#define VCA_HBLNK_SIZE 138 /* (858-720),horizontal blanking */

/* (EAV/SAV inclusive) */
#define VCA_VBLNK1_SIZE 19 /* (20-1),v.blanking for field1 */
#define VCA_VBLNK2_SIZE 19 /* (283-264),v.blanking for field2 */
#define VCA_IMG_HSIZE1 720 /* field1 horizontal image size */

#define VCA_IMG_VSIZE1 240 /* (263-20+1), fld1 vertical image size */

#define VCA_IMG_HSIZE2 720 /* field2 horizontal image size */

#define VCA_IMG_VSIZE2 240 /* (525-283+1), fld2 vertical image size */

/* Define field image sizes */
#define VCA_IMAGE_SIZE1 (VCA_IMG_HSIZE1 * VCA_IMG_VSIZE1)
#define VCA_IMAGE_SIZE2 (VCA_IMG_HSIZE2 * VCA_IMG_VSIZE2)

/* ------------------------------------------------------- */
/* Define channel A capture window co-ordinates for Field1 */
/* ------------------------------------------------------- */
/* HRST = 0, start of horizontal blanking */
#define VCA_XSTART1 (VCA_HBLNK_SIZE - 2/*EAV*/)

/* VRST = 1, end of vertical blanking */
#define VCA_YSTART1 3

#define VCA_XSTOP1 (VCA_IMG_HSIZE1 - 1)

#define VCA_YSTOP1 (VCA_YSTART1 + VCA_IMG_VSIZE1 - 1)

/* ------------------------------------------------------- */
/* Define channel A capture window co-ordinates for Field2 */
/* ------------------------------------------------------- */
/* HRST = 0, start of horizontal blanking */
#define VCA_XSTART2 (VCA_HBLNK_SIZE - 2/*EAV*/)

/* VRST = 1, end of vertical blanking */
#define VCA_YSTART2 3

#define VCA_XSTOP2 (VCA_IMG_HSIZE2 - 1)

#define VCA_YSTOP2 (VCA_YSTART2 + VCA_IMG_VSIZE2 - 1)

/* Define threshold values in double-words. Both fields should */
/* same threshold value) */
#define VCA_THRLD_FIELD1 (VCA_IMG_HSIZE1/8) /* line length in */
#define VCA_THRLD_FIELD2 VCA_THRLD_FIELD1 /* double-words */

/* Define number of events to be generated for field1 and field2 */
#define VCA_CAPEVT1 (VCA_IMAGE_SIZE1 / (VCA_THRLD_FIELD1 * 8))
#define VCA_CAPEVT2 (VCA_IMAGE_SIZE2 / (VCA_THRLD_FIELD2 * 8))
#define CAPCHA_FRAME_COUNT 5 /* in this example */

/* -------------------------------------------- */
/* EDMA parameters for capture Y event that are */
/* specific to this example. */
/* -------------------------------------------- */
#define VCA_Y_EDMA_ELECNT (VCA_THRLD_FIELD1 * 2) /* because VCA_THRLD_FIELDn is in doublewords and element size is 32-bit */
#define VCA_Y_EDMA_FRMCNT ((VCA_CAPEVT1 + VCA_CAPEVT2) * CAPCHA_FRAME_COUNT)

/******************************************************************
 *
 * NAME  
 *      Bt656Capt_8bit_Ncfd_NTSC
 *
 * PURPOSE:
 *      Setup dsp video port for video capture channel A & B.
 *
 *      Description : 8-bit BT.656 non-continuous frame capture 
 * 
 *      Some important field descriptions: 
 * 
 *      CMODE = 000, 8-bit BT.656 mode 
 *      CON = 0 
 *      FRAME = 1, capture frame 
 *      CF2 = 0 
 *      CF1 = 0, (8-bit non-continuous frame capture) 
 *      SCALE = 0, no scaling 
 *      RESMPL= 0, no re-sampling 
 *      10BPK = X, not used in 8-bit capture 
 *      EXC = 0, use EAV/SAV codes 
 *      VRST = 1, end of vertical blanking 
 *      HRST = 0, start of horizontal blanking 
 *      FLDD = 0, 1st line EAV or FID input 
 *      FINV = 0, no field invert 
 *      RDFE = X, used in Raw mode only(Enable field identification) 
 *      SSE = X, used in Raw mode only(Startup synch enable) 
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int16 Bt656Capt_8bit_Ncfd_NTSC(Int16 portNumber)
 *
 *      portNumber - (i)    Video port number to use
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int16 Bt656Capt_8bit_Ncfd_NTSC(int portNumber);


/**************************************************************************** 
 *
 * NAME  
 *      VPCaptEnable
 *
 * PURPOSE:
 *      Enable and already configured video port.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int16 VPCaptEnable(void)
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int16 VPCaptEnable(void);


/**************************************************************************** 
 *
 * NAME  
 *      VPCapChanPoll
 *
 * PURPOSE:
 *      Enable and already configured video port. Usually and irq routine
 *      is used for this purpose.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int16 VPCaptEnable(void)
 *
 *      capChaAOverrun          (o) - capture overrun channel A
 *      capChaASyncError        (o) - sync error channel A
 *      capChaAShortFieldDetect (o) - short filed detection channel A
 *      capChaAFrameCount       (o) - channel A capture frame count since start
 *      capChaBOverrun          (o) - capture overrun channel B
 *      capChaBSyncError        (o) - sync error channel B
 *      capChaBShortFieldDetect (o) - short filed detection channel B
 *      capChaBFrameCount       (o) - channel B capture frame count since start
 *
 * RETURN VALUE
 *      bit 0: capture frame complete channel A
 *      bit 1: capture frame complete channel B
 *     
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 VPCapChanPoll(void);


/**************************************************************************** 
 *
 * NAME  
 *      CloseDisplayVideoPort
 *
 * PURPOSE:
 *      Close a video port opened by Bt656_8bit_Ncfd_NTSC.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int16 CloseDisplayVideoPort(void)
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int16 CloseCaptureVideoPort(void);


#endif // __BT656CAPT_H__
