/***************************************************************************
 *
 *
 *         **   **    **  ******  ********  ********  ********  **    **    
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ***************************************************************************
 *                                                                          
 *  Project     : EVM
 *  File        : EDMA3.h
 *  Description : Basic edma3 functionnality for evm examples
 *
 *                Note that this source code is only intended to serve as
 *                an example of how to use the DM648 EVM Board Support
 *                Library (BSL). It should not be considered to be a
 *                complete software design reference for DM648 peripherals
 *                and was not written for this purpose. This software is
 *                provided as is and Lyrtech does not guarantee it or
 *                provide any support for it.
 *
 *                      Copyright (c) Lyrtech inc. 2007                        
 *
 ***************************************************************************
 *                                              				                            
 * "$Revision: 1.3 $"
 * "$Date: 2007/07/24 19:56:15 $"
 *
 ***************************************************************************/
#ifndef __EDMA3_H__
#define __EDMA3_H__


/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include "cslr_edma3cc.h"
#include "csl_types.h"


/****************************************************************************
 *                             Global constants                             *
 ****************************************************************************/

/* Link to a Null Param set */
#define EDMA3_LINK_NULL                     0xFFFF 
/* Link to a Null Param set */
#define EDMA3_LINK_DEFAULT                  0xFFFF
/* A synchronized transfer  */
#define EDMA3_SYNC_A                        0
/* AB synchronized transfer */
#define EDMA3_SYNC_AB                       1
/* Normal Completion */
#define EDMA3_TCC_NORMAL                    0
/* Early  Completion */
#define EDMA3_TCC_EARLY                     1
/* Only for ease  */
#define EDMA3_FIFOWIDTH_NONE                0
/* 8 bit FIFO Width */
#define EDMA3_FIFOWIDTH_8BIT                0    
/* 16 bit FIFO Width */
#define EDMA3_FIFOWIDTH_16BIT               1    
/* 32 bit FIFO Width */
#define EDMA3_FIFOWIDTH_32BIT               2    
/* 64 bit FIFO Width */
#define EDMA3_FIFOWIDTH_64BIT               3    
/* 128 bit FIFO Width */
#define EDMA3_FIFOWIDTH_128BIT              4   
/* 256 bit FIFO Width */
#define EDMA3_FIFOWIDTH_256BIT              5    
/* Address Mode is incremental */
#define EDMA3_ADDRMODE_INCR                 0
/* Address Mode is such it wraps around after reaching FIFO width */
#define EDMA3_ADDRMODE_CONST                1


/* Bitwise OR of the below symbols are used for setting the Memory attributes 
   These are defined only if the Memory Protection feature exists */
#if CSL_EDMA3_MEMPROTECT
/* User Execute permission */
#define EDMA3_MEMACCESS_UX              0x0001  
/* User Write permission */  
#define EDMA3_MEMACCESS_UW              0x0002  
/* User Read permission */
#define EDMA3_MEMACCESS_UR              0x0004  
/* Supervisor Execute permission */  
#define EDMA3_MEMACCESS_SX              0x0008  
/* Supervisor Write permission */
#define EDMA3_MEMACCESS_SW              0x0010  
/* Supervisor Read permission */
#define EDMA3_MEMACCESS_SR              0x0020  
/* External Allowed ID. Requests with PrivID >= '6' are permitted 
 * if access type is allowed
 */
#define EDMA3_MEMACCESS_EXT            0x0200 
/* Allowed ID '0' */ 
#define EDMA3_MEMACCESS_AID0           0x0400  
/* Allowed ID '1' */
#define EDMA3_MEMACCESS_AID1           0x0800  
/* Allowed ID '2' */
#define EDMA3_MEMACCESS_AID2           0x1000  
/* Allowed ID '3' */
#define EDMA3_MEMACCESS_AID3           0x2000  
/* Allowed ID '4' */
#define EDMA3_MEMACCESS_AID4           0x4000  
/* Allowed ID '5' */
#define EDMA3_MEMACCESS_AID5           0x8000  
#endif

/* Intermediate transfer completion interrupt enable */  
#define EDMA3_ITCINT_EN             1
/* Intermediate transfer completion interrupt disable */ 
#define EDMA3_ITCINT_DIS            0
/* Intermediate transfer completion chaining enable */ 
#define EDMA3_ITCCH_EN              1
/* Intermediate transfer completion chaining disable */    
#define EDMA3_ITCCH_DIS             0
/* Transfer completion interrupt enable */  
#define EDMA3_TCINT_EN              1
/* Transfer completion interrupt disable */ 
#define EDMA3_TCINT_DIS             0
/* Transfer completion chaining enable */   
#define EDMA3_TCCH_EN               1
/* Transfer completion chaining disable */
#define EDMA3_TCCH_DIS              0
/* Enable Static */
#define EDMA3_STATIC_EN             1
/* Disable Static */
#define EDMA3_STATIC_DIS            0
/* Last trigger word in a QDMA parameter set */
#define EDMA3_TRIGWORD_DEFAULT        7
/* Trigger word option field */
#define EDMA3_TRIGWORD_OPT            0
/* Trigger word source  */
#define EDMA3_TRIGWORD_SRC            1
/* Trigger word AB count */
#define EDMA3_TRIGWORD_A_B_CNT        2
/* Trigger word destination */
#define EDMA3_TRIGWORD_DST            3
/* Trigger word src and dst B index */
#define EDMA3_TRIGWORD_SRC_DST_BIDX   4
/* Trigger word B count reload */
#define EDMA3_TRIGWORD_LINK_BCNTRLD   5
/* Trigger word src and dst C index */
#define EDMA3_TRIGWORD_SRC_DST_CIDX   6
/* Trigger word C count */
#define EDMA3_TRIGWORD_CCNT           7


/* Helper macro used for creating the options entry in the parameter ram */
#define EDMA3_OPT_MAKE(itcchEn, tcchEn, itcintEn, tcintEn, tcc, tccMode, \
                       fwid, stat, syncDim, dam, sam) \
(Uint32)(\
     CSL_FMKR(23,23,itcchEn) \
    |CSL_FMKR(22,22,tcchEn) \
    |CSL_FMKR(21,21,itcintEn) \
    |CSL_FMKR(20,20,tcintEn) \
    |CSL_FMKR(17,12,tcc) \
    |CSL_FMKR(11,11,tccMode) \
    |CSL_FMKR(10,8,fwid) \
    |CSL_FMKR(3,3,stat) \
    |CSL_FMKR(2,2,syncDim) \
    |CSL_FMKR(1,1,dam) \
    |CSL_FMKR(0,0,sam)) 
    
/* Used for creating the A,B Count entry in the parameter ram */
#define EDMA3_CNT_MAKE(aCnt,bCnt) \
(Uint32)(\
     CSL_FMK(EDMACC_A_B_CNT_ACNT,aCnt) \
    |CSL_FMK(EDMACC_A_B_CNT_BCNT,bCnt)\
    )

/* Used for creating the link and B count reload entry in the parameter ram */
#define EDMA3_LINKBCNTRLD_MAKE(link,bCntRld) \
(Uint32)(\
     CSL_FMK(EDMACC_LINK_BCNTRLD_LINK,(Uint32)link) \
    |CSL_FMK(EDMACC_LINK_BCNTRLD_BCNTRLD,bCntRld)\
    )

/* Used for creating the B index entry in the parameter ram */
#define EDMA3_BIDX_MAKE(src,dst) \
(Uint32)(\
     CSL_FMK(EDMACC_SRC_DST_BIDX_DSTBIDX,(Uint32)dst) \
    |CSL_FMK(EDMACC_SRC_DST_BIDX_SRCBIDX,(Uint32)src)\
    )

/* Used for creating the C index entry in the parameter ram */
#define EDMA3_CIDX_MAKE(src,dst) \
(Uint32)(\
     CSL_FMK(EDMACC_SRC_DST_CIDX_DSTCIDX,(Uint32)dst) \
    |CSL_FMK(EDMACC_SRC_DST_CIDX_SRCCIDX,(Uint32)src)\
    )

/* DMA Channel Default Setup  */
#define EDMA3_DMACHANNELSETUP_DEFAULT   {       \
   {CSL_EDMA3_QUE_0,0}, \
   {CSL_EDMA3_QUE_0,1}, \
   {CSL_EDMA3_QUE_0,2}, \
   {CSL_EDMA3_QUE_0,3}, \
   {CSL_EDMA3_QUE_0,4}, \
   {CSL_EDMA3_QUE_0,5}, \
   {CSL_EDMA3_QUE_0,6}, \
   {CSL_EDMA3_QUE_0,7}, \
   {CSL_EDMA3_QUE_0,8}, \
   {CSL_EDMA3_QUE_0,9}, \
   {CSL_EDMA3_QUE_0,10}, \
   {CSL_EDMA3_QUE_0,11}, \
   {CSL_EDMA3_QUE_0,12}, \
   {CSL_EDMA3_QUE_0,13}, \
   {CSL_EDMA3_QUE_0,14}, \
   {CSL_EDMA3_QUE_0,15}, \
   {CSL_EDMA3_QUE_0,16}, \
   {CSL_EDMA3_QUE_0,17}, \
   {CSL_EDMA3_QUE_0,18}, \
   {CSL_EDMA3_QUE_0,19}, \
   {CSL_EDMA3_QUE_0,20}, \
   {CSL_EDMA3_QUE_0,21}, \
   {CSL_EDMA3_QUE_0,22}, \
   {CSL_EDMA3_QUE_0,23}, \
   {CSL_EDMA3_QUE_0,24}, \
   {CSL_EDMA3_QUE_0,25}, \
   {CSL_EDMA3_QUE_0,26}, \
   {CSL_EDMA3_QUE_0,27}, \
   {CSL_EDMA3_QUE_0,28}, \
   {CSL_EDMA3_QUE_0,29}, \
   {CSL_EDMA3_QUE_0,30}, \
   {CSL_EDMA3_QUE_0,31}, \
   {CSL_EDMA3_QUE_0,32}, \
   {CSL_EDMA3_QUE_0,33}, \
   {CSL_EDMA3_QUE_0,34}, \
   {CSL_EDMA3_QUE_0,35}, \
   {CSL_EDMA3_QUE_0,36}, \
   {CSL_EDMA3_QUE_0,37}, \
   {CSL_EDMA3_QUE_0,38}, \
   {CSL_EDMA3_QUE_0,39}, \
   {CSL_EDMA3_QUE_0,40}, \
   {CSL_EDMA3_QUE_0,41}, \
   {CSL_EDMA3_QUE_0,42}, \
   {CSL_EDMA3_QUE_0,43}, \
   {CSL_EDMA3_QUE_0,44}, \
   {CSL_EDMA3_QUE_0,45}, \
   {CSL_EDMA3_QUE_0,46}, \
   {CSL_EDMA3_QUE_0,47}, \
   {CSL_EDMA3_QUE_0,48}, \
   {CSL_EDMA3_QUE_0,49}, \
   {CSL_EDMA3_QUE_0,50}, \
   {CSL_EDMA3_QUE_0,51}, \
   {CSL_EDMA3_QUE_0,52}, \
   {CSL_EDMA3_QUE_0,53}, \
   {CSL_EDMA3_QUE_0,54}, \
   {CSL_EDMA3_QUE_0,55}, \
   {CSL_EDMA3_QUE_0,56}, \
   {CSL_EDMA3_QUE_0,57}, \
   {CSL_EDMA3_QUE_0,58}, \
   {CSL_EDMA3_QUE_0,59}, \
   {CSL_EDMA3_QUE_0,60}, \
   {CSL_EDMA3_QUE_0,61}, \
   {CSL_EDMA3_QUE_0,62}, \
   {CSL_EDMA3_QUE_0,63} \
}


#define InstNum Int32

/*  This object contains the reference to the instance of Edma Module
 * opened using the @a edma3Open().
 *
 *  A pointer to this object is passed to all Edma Module level APIs.
 */
typedef struct Edma3Obj {
    /* This is a pointer to the Edma Channel Controller registers of the module
     *   requested
     */
    CSL_EdmaccRegsOvly regs;
    /* This is the instance of module number i.e EDMA3 */
    InstNum         instNum;
} Edma3Obj;

/*  EDMA handle */
typedef struct Edma3Obj *Edma3Handle;

/* Parameter Set Handle */
typedef volatile CSL_EdmaccParamentryRegs *Edma3ParamHandle;

/*  Edma ParamSetup Structure
 *
 *  An object of this type is allocated by the user and
 *  its address is passed as a parameter to the edma3ParamSetup().
 *  This structure is used to program the Param Set for EDMA/QDMA.
 *  The macros can be used to assign values to the fields of the structure.
 *  The setup structure should be setup using the macros provided OR
 *  as per the bit descriptions in the user guide..
 *  
 */
typedef struct Edma3ParamSetup {
    /* Options */
    Uint32          option;
    /* Specifies the source address */               
    Uint32          srcAddr;
    /* Lower 16 bits are A Count Upper 16 bits are B Count*/             
    Uint32          aCntbCnt;            
    /* Specifies the destination address */                                          
    Uint32          dstAddr;
    /* Lower 16 bits are source b index Upper 16 bits are
     * destination b index 
     */             
    Uint32          srcDstBidx;          
    /* Lower 16 bits are link of the next param entry Upper 16 bits are 
     * b count reload 
     */
    Uint32          linkBcntrld;         
    /* Lower 16 bits are source c index Upper 16 bits are destination 
     * c index 
     */             
    Uint32          srcDstCidx;          
    /* C count */                                           
    Uint32          cCnt;                
} Edma3ParamSetup;

/*  Edma Object Structure
 *
 *  An object of this type is allocated by the user and
 *  its address is passed as a parameter to the edma3ChannelOpen()
 *  The edma3ChannelOpen() updates all the members of the data structure
 *  and returns the objects address as a @a #Edma3ChannelHandle. The
 *  @a #Edma3ChannelHandle is used in all subsequent function calls.
 */
typedef struct Edma3ChannelObj {
    /* Pointer to the Edma Channel Controller module register 
     * Overlay structure 
     */
    CSL_EdmaccRegsOvly      regs;               
    /* Region number to which the channel belongs to */ 
    Int                     region;              
    /* EDMA instance whose channel is being requested */
    Int                     edmaNum;             
    /* Channel Number being requested */
    Int                     chaNum;              
} Edma3ChannelObj;

/*  Channel Handle
 *  All channel level API calls must be made with this handle. 
 */
typedef struct Edma3ChannelObj *Edma3ChannelHandle;

/*  Edma Control/Query Control Command structure for issuing commands 
 *  for Interrupt related APIs
 *  An object of this type is allocated by the user and
 *  its address is passed to the Control API. 
 */
typedef struct Edma3CmdIntr{
    /* Input field:- this field needs to be initialized by the user before
     * issuing the query/command 
     */
    Int              region;             
    /* Input/Output field:- this needs to be filled by the user in case 
     * of issuing a COMMAND or it will be filled in when used with 
     * a QUERY 
     */                
    CSL_BitMask32    intr;               
    /* Input/Output:- this needs to be filled by the user in case of issuing a                                               
     * COMMAND or it will be filled in when used with a QUERY 
     */                                             
    CSL_BitMask32    intrh;              
                                                  
                                                  
} Edma3CmdIntr;

/*  Edma Channel parameter structure used for opening a channel
 */
typedef struct { 
     /* Region Number */   
    Int regionNum;
    /* Channel number */                             
    Int chaNum;                                 
} Edma3ChannelAttr;

/*  Edma Channel Error . 
 *
 *  An object of this type is allocated by the user and
 *  its address is passed as a parameter to the edma3GetChannelError()
 *  /edma3GetHwStatus()/ edma3ChannelErrorClear()
 *  /edma3HwChannelControl().
 */
typedef struct Edma3ChannelErr {
    /* a TRUE indicates an event is missed on this channel.  */
    Bool    missed;      
    /* a TRUE indicates an event that no events on this channel will be
     * prioritized till this is cleared. This being TRUE does NOT necessarily
     * mean it is an error. ONLY if both missed and ser are set, this kind of 
     * error need  to be cleared.
     */
    Bool    secEvt;      
} Edma3ChannelErr;

/*  QDMA Edma Channel Setup
 *
 *  An array of such objects are allocated by the user and
 *  address initialized in the Edma3HwSetup structure which is passed
 *  edma3HwSetup()
 */
typedef struct Edma3HwQdmaChannelSetup {
    /* Que number for the channel */
    CSL_Edma3Que que;                            
    /* Parameter set mapping for the channel. */             
    Uint16  paramNum;                            
    /* Trigger word for the QDMA channels. */
    Uint8   triggerWord;                         
} Edma3HwQdmaChannelSetup;

/*  QDMA Edma Channel Setup
 *
 *  An array of such objects are allocated by the user and
 *  address initialized in the Edma3HwSetup structure which is passed
 *  edma3HwSetup()
 */

typedef struct Edma3HwDmaChannelSetup {
    /* Que number for the channel */
    CSL_Edma3Que que;                     
#ifdef CSL_EDMA3_CHMAPEXIST
    /* Parameter set mapping for the channel. This may not be initialized 
     * for Edma channels on devices that do not have CHMAPEXIST.  
     */
    Uint16  paramNum;                     
#endif
} Edma3HwDmaChannelSetup;


/*  Edma Hw Setup Structure 
 */
typedef struct {
    /* Edma Hw Channel setup */
    Edma3HwDmaChannelSetup *dmaChaSetup;
    /* QEdma Hw Channel setup */
    Edma3HwQdmaChannelSetup *qdmaChaSetup;
} Edma3HwSetup;


// EDMA3 macro definitions for basic edma usage
//
#define edma3Open(mpEdmaObj,medmaNum,mhEdma)                              \
    (mpEdmaObj)->regs = (CSL_EdmaccRegsOvly) CSL_EDMA3CC_0_REGS;          \
    (mpEdmaObj)->instNum = (InstNum)(medmaNum);                           \
    (mhEdma) = (Edma3Handle)(mpEdmaObj);                               


#define edma3ChannelOpen(pEdmaObj,medmaNum,pChAttr,hEdma)                 \
{                                                                         \
    (pEdmaObj)->regs = (CSL_EdmaccRegsOvly) (CSL_EDMA3CC_0_REGS);         \
    (pEdmaObj)->edmaNum = (medmaNum);                                     \
    (pEdmaObj)->chaNum = (pChAttr)->chaNum;                               \
    (pEdmaObj)->region = (pChAttr)->regionNum;                            \
    *(hEdma) = (Edma3ChannelHandle)(pEdmaObj);                            \
}

#define edma3HwSetup(mhmod,setup)                                         \
{                                                                         \
    Uint32        numCha;                                                 \
    Uint32        tempQnum  = 0;                                          \
    Uint32        ii;                                                     \
    for (numCha = 0 ; numCha < CSL_EDMA3_NUM_DMACH; numCha++)             \
    {                                                                     \
        (mhmod)->regs->DCHMAP[numCha] = CSL_FMK(EDMACC_DCHMAP_PAENTRY,(setup)->dmaChaSetup[numCha].paramNum); \
        ii = numCha % 8;                                                  \
        CSL_FINSR(tempQnum,(ii * 4) + 2,(ii * 4), (setup)->dmaChaSetup[numCha].que);                          \
        if (((ii + 1) % 8) == 0)                                          \
        {                                                                 \
            (mhmod)->regs->DMAQNUM[numCha/8] = tempQnum;                  \
            tempQnum = 0;                                                 \
        }                                                                 \
    }                                                                     \
}

#define edma3HwChannelSetupQue(hEdma, evtQue)                             \
{                                                                         \
    Uint32        _qNumIndex;                                             \
    Uint32        _qchMap;                                                \
    /*  Finding out the relevant DMAQNUM register and the correct */      \
    /* bit positions to write into */                                     \
    _qNumIndex = (hEdma)->chaNum >> 3;                                    \
    _qchMap = (hEdma)->chaNum - (_qNumIndex * 8);                         \
    CSL_FINSR((hEdma)->regs->DMAQNUM[_qNumIndex], _qchMap * 4 + 2,        \
              _qchMap * 4, (evtQue));                                     \
}  

#define edma3GetParamHandle(hEdma,paramNum)                               \
        (Edma3ParamHandle)(&(hEdma)->regs->PARAMENTRY[(paramNum)])

#define edma3ParamSetup(hParamHndl,setup)                                 \
{                                                                         \
    Uint32 *paramHndl = (Uint32*)(hParamHndl);                            \
    Uint32 i;                                                             \
    for (i = 0; i < 8 ; i++)                                              \
    {                                                                     \
        paramHndl[i] = ((Uint32*)(setup))[i];                             \
    }                                                                     \
}

#define edma3CmdIntrEnable(hModule,intrLo,intrHi)                         \
        (hModule)->regs->IESR = (intrLo);                                 \
        (hModule)->regs->IESRH = (intrHi);                                \
                

#define edma3GetChannelErrStatus(hChannel,errClr)                         \
{                                                                         \
    (errClr)->missed = FALSE;                                             \
    (errClr)->secEvt = FALSE;                                             \
    if ((hChannel)->chaNum < CSL_EDMA3_NUM_DMACH)                         \
    {                                                                     \
        if ((hChannel)->chaNum < 32)                                      \
        {                                                                 \
            if (CSL_FEXTR((hChannel)->regs->EMR,(hChannel)->chaNum,       \
                (hChannel)->chaNum))                                      \
                (errClr)->missed = TRUE;                                  \
            if (CSL_FEXTR((hChannel)->regs->SER,(hChannel)->chaNum,       \
                (hChannel)->chaNum))                                      \
                (errClr)->secEvt = TRUE;                                  \
        }                                                                 \
        else                                                              \
        {                                                                 \
            if (CSL_FEXTR((hChannel)->regs->EMRH,(hChannel)->chaNum-32,   \
                          (hChannel)->chaNum-32))                         \
                (errClr)->missed = TRUE;                                  \
            if (CSL_FEXTR((hChannel)->regs->SERH,(hChannel)->chaNum-32,   \
                          (hChannel)->chaNum-32))                         \
                (errClr)->secEvt = TRUE;                                  \
        }                                                                 \
    }                                                                     \
    else                                                                  \
    {                                                                     \
        if (CSL_FEXTR((hChannel)->regs->QEMR,(hChannel)->chaNum-CSL_EDMA3_NUM_DMACH, \
                      (hChannel)->chaNum-CSL_EDMA3_NUM_DMACH))            \
            (errClr)->missed = TRUE;                                      \
        if (CSL_FEXTR((hChannel)->regs->QSER,(hChannel)->chaNum-CSL_EDMA3_NUM_DMACH, \
                      (hChannel)->chaNum-CSL_EDMA3_NUM_DMACH))            \
            (errClr)->secEvt = TRUE;                                      \
    }                                                                     \
}                                                                         \

#define edma3ChannelErrorClear(mhChannel,merrClr)                         \
{                                                                         \
    if ((merrClr)->missed == TRUE)                                        \
    {                                                                     \
        if ((mhChannel)->chaNum < CSL_EDMA3_NUM_DMACH)                    \
        {                                                                 \
            if ((mhChannel)->chaNum < 32)                                 \
            {                                                             \
                (mhChannel)->regs->EMCR = 1 << (mhChannel)->chaNum;       \
                if ((merrClr)->secEvt == TRUE)                            \
                    (mhChannel)->regs->SECR = 1 << (mhChannel)->chaNum;   \
            }                                                             \
            else                                                          \
            {                                                             \
                (mhChannel)->regs->EMCRH = 1 << ((mhChannel)->chaNum - 32); \
                if ((merrClr)->secEvt == TRUE)                              \
                    (mhChannel)->regs->SECRH = 1 << ((mhChannel)->chaNum - 32); \
            }                                                             \
        }                                                                 \
        else                                                              \
        {                                                                 \
           (mhChannel)->regs->QEMR = 1 << ((mhChannel)->chaNum - CSL_EDMA3_NUM_DMACH); \
           if ((merrClr)->secEvt == TRUE)                                 \
               (mhChannel)->regs->QSECR = 1 << ((mhChannel)->chaNum - CSL_EDMA3_NUM_DMACH);  \
        }                                                                 \
    }                                                                     \
}

#define edma3ChannelEnable(hEdma)                                         \
    if ((hEdma)->chaNum >= CSL_EDMA3_NUM_DMACH)                           \
        (hEdma)->regs->QEESR = (1 << ((hEdma)->chaNum - CSL_EDMA3_NUM_DMACH));\
    else                                                                  \
        if ((hEdma)->chaNum < 32)                                         \
            (hEdma)->regs->EESR = (1 << (hEdma)->chaNum);                 \
        else                                                              \
            (hEdma)->regs->EESRH = (1 << ((hEdma)->chaNum - 32));

    
#define edma3ChannelDisable(hEdma)                                        \
        if ((hEdma)->chaNum >= CSL_EDMA3_NUM_DMACH)                       \
            (hEdma)->regs->QEECR = (1 << ((hEdma)->chaNum - CSL_EDMA3_NUM_DMACH)); \
        else                                                              \
            if ((hEdma)->chaNum < 32)                                     \
                (hEdma)->regs->EECR = (1 << (hEdma)->chaNum);             \
            else                                                          \
                (hEdma)->regs->EECRH = (1 << ((hEdma)->chaNum - 32));     \

#define edma3InterruptDisable(hModule,intrLo,intrHi)                                    \
        (hModule)->regs->IECR  = (intrLo);                                \
        (hModule)->regs->IECRH = (intrHi);                                \

#endif //__EDMA3_H__
