/***************************************************************************
 *
 *
 *         **   **    **  ******  ********  ********  ********  **    **    
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ***************************************************************************
 *                                                                          
 *  Project     : EVM
 *  File        : muxcontrol.c
 *  Description : Utility functions for mux control
 *
 *                      Copyright (c) Lyrtech inc. 2007                        
 *
 ***************************************************************************
 *                                                                                          
 * "$Revision: 1.4 $"
 * "$Date: 2007/07/13 20:09:00 $"
 *
 ***************************************************************************/

/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include "EVM.h"
#include "mydebug.h"


/****************************************************************************
 *                             Local constants                              *
 ****************************************************************************/


/****************************************************************************
 *                             Local variables                              *
 ****************************************************************************/


/****************************************************************************
 *                           Forward Declarations                           *
 ****************************************************************************/


/****************************************************************************
 *                       Private functions prototypes                       *
 ****************************************************************************/
Int32 Muxcontrol(void);


/****************************************************************************
 *                             Public Functions                             *
 ****************************************************************************/

/**************************************************************************** 
 *
 * NAME  
 *      do_Muxcontrol
 *
 * PURPOSE:
 *      Do a mux control test.
 *      This function shows how to lock evm resource that can be used later.
 *      This process IS NOT NEEDED when the related BSL module is used to
 *      control the needed resource: the BSL will lock all needed reources by
 *      itself. But it can be used for example when the BSL module associated
 *      with the needed peripheral is not used, but controled through another
 *      custom routine, driver, OS or BIOS functions. In this case, the
 *      resource can be locked by the BSL EVM_lockresource function: this will
 *      avoid resource conflicts, will take care of evm muxes and dsp pinmux
 *      initializations.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 do_Muxcontrol(void)
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 do_Muxcontrol(void)
{
    Int32 iResult;

    myprintf("Mux control test...\r\n",0,0);


    iResult = Muxcontrol();

    if (!iResult)
    {
        myprintf("Mux control test... SUCCESS!\r\n\r\n",0,0);
    }
    else
    {
        myprintf("Mux control test... ERROR!\r\n\r\n",0,0);
    }

    return(iResult);
}


/****************************************************************************
 *                             Local Functions                              *
 ****************************************************************************/


/**************************************************************************** 
 *
 * NAME  
 *      Muxcontrol
 *
 * PURPOSE:
 *      This function shows how to lock evm resource that can be used later.
 *      This process IS NOT NEEDED when the related BSL module is used to
 *      control the needed resource: the BSL will lock all needed reources by
 *      itself. But it can be used for example when the BSL module associated
 *      with the needed peripheral is not used, but controled through another
 *      custom routine, driver, OS or BIOS functions. In this case, the
 *      resource can be locked by the BSL EVM_lockresource function: this will
 *      avoid resource conflicts, will take care of evm muxes and dsp pinmux
 *      initializations.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 Muxcontrol(void)
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 Muxcontrol(void)
{
    Int32 retcode=0;
    
    // WARNING: When used inside multitask OS, BSL calls must be made in
    // an initalization routine (before multitasking is active), or in any
    // other task that will prevent to be preempt. All BSL functions are not
    // OS multitask awared...

    // Initialize the BSL & EVM
    // This is always necessary before using any other BSL functions / modules
    if ((retcode=EVM_init())!=EVM_INITOK)
    {
        // Problem cannot open evm
        return(1);
    }

    // Verify is correct evm binary bsl version
    if ((retcode=EVM_checkbslversion(EVM_BSLVERSION))!=EVM_INITOK)
    {
        // Problem unmatching binary bsl version
        return(6);
    }
    
    // Resources allocation step


    // Acquire a lock on resource SD video capture channel 1-4
    // and configure the MUX accordingly if success
    if ((retcode=EVM_lockresource(EVM_SDVIDEOINCH14_LNK))!=EVM_INITOK)
    {
        // Problem cannot lock the resource
        return(2);
    }

    // Acquire a lock on resource emif bus for flash memory
    // and configure the MUX accordingly if success
    // This one will fail because of previous lock that is not possible
    // with this one at same time...
    // To use this resource, the previous resource must be unlocked
    // first...
    if ((retcode=EVM_lockresource(EVM_FLASHMEM_LNK))==EVM_INITOK)
    {
        // Problem should not be able to lock this resource
        return(3);
    }

    
    
    // Do custom code for these peripherals outside of BSL (bios, driver)...
    // ...
    //



    // Resources deallocation step


    // The process of resource unlocking is mandatory if other conflicting
    // resources are needed later (like the flash memory in previous example)

    // Release the resource when not needed anymore
    if ((retcode=EVM_unlockresource(EVM_FLASHMEM_LNK))==EVM_INITOK)
    {
        // Problem is not supposed to be possible to unlock the resource
        // because was not locked before!
        return(4);
    }

    // Release the resource when not needed anymore
    if ((retcode=EVM_unlockresource(EVM_SDVIDEOINCH14_LNK))!=EVM_INITOK)
    {
        // Problem cannot unlock the resource
        return(5);
    }

    return (retcode);
}
