/***************************************************************************
 *
 *
 *         **   **    **  ******  ********  ********  ********  **    **    
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ***************************************************************************
 *                                                                          
 *  Project     : EVM
 *  File        : sdcolortest.c
 *  Description : Utility functions for SD video
 *
 *                      Copyright (c) Lyrtech inc. 2007                        
 *
 ***************************************************************************
 *                                                                                          
 * "$Revision: 1.6 $"
 * "$Date: 2007/07/24 20:30:15 $"
 *
 ***************************************************************************/

/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include "soc.h"
#include "edma3.h"
#include "EVM.h"
#include "evm_s8200.h"
#include "evm_a7105.h"
#include "evm_ecp.h"
#include "bt656.h"
#include "sdcolortest.h"
#include "mydebug.h"


/****************************************************************************
 *                             Local constants                              *
 ****************************************************************************/
// Uncomment folllowing line to get s-video output instead of composite on J13
//#define J13_SVIDEO

#define NPIXEL 720
#define NCOLOR (NPIXEL/2)

// Video display VP1 buffer address
#define YDSTVP1  (CSL_VP1_CHANNELA_DATA+0x00000080)
#define CBDSTVP1 (CSL_VP1_CHANNELA_DATA+0x000000A0)
#define CRDSTVP1 (CSL_VP1_CHANNELA_DATA+0x000000C0)

/****************************************************************************
 *                             Local variables                              *
 ****************************************************************************/

// HD video A7105 & S8200 default setup registers structure for color bar 480i bt656
static EVM_S8200_Setup S8200setup = EVM_S8200_SDTV480I_YCBCR704X480X60;

// Setup J13 output for conposite or svideo output
#ifdef J13_SVIDEO
static EVM_A7105_Setup A7105setup = EVM_A7105_SDTV480I_YCBCR704X480X60_SVIDEO;
#else
static EVM_A7105_Setup A7105setup = EVM_A7105_SDTV480I_YCBCR704X480X60_COMP;
#endif

/* Globals */
#pragma DATA_ALIGN(YSrc,32);
Uint8       YSrc[NPIXEL+2];
#pragma DATA_ALIGN(CbSrc,32);
Uint8       CbSrc[NCOLOR+1];
#pragma DATA_ALIGN(CrSrc,32);
Uint8       CrSrc[NCOLOR+1];

// Variables for EDMA3 setup
Edma3Handle                 hModule;
Edma3HwSetup                hwSetup;
Edma3Obj                    edmaObj;
Edma3ParamHandle            hParamCpyY;
Edma3ParamHandle            hParamCpyCb;
Edma3ParamHandle            hParamCpyCr;
Edma3ParamHandle            hParamBasicY;
Edma3ParamHandle            hParamBasicCb;
Edma3ParamHandle            hParamBasicCr;
Edma3ChannelObj             chObjY;
Edma3ChannelObj             chObjCb;
Edma3ChannelObj             chObjCr;
Edma3CmdIntr                regionIntr;
Edma3ChannelHandle          hChannelY;
Edma3ChannelHandle          hChannelCb;
Edma3ChannelHandle          hChannelCr;
Edma3ParamSetup             myParamSetup;
Edma3ChannelAttr            chAttr;
Edma3HwDmaChannelSetup      dmahwSetup[CSL_EDMA3_NUM_DMACH] = EDMA3_DMACHANNELSETUP_DEFAULT;
Edma3ChannelErr             myErrorsY;        
Edma3ChannelErr             myErrorsCb;        
Edma3ChannelErr             myErrorsCr;        


/****************************************************************************
 *                           Forward Declarations                           *
 ****************************************************************************/


/****************************************************************************
 *                       Private functions prototypes                       *
 ****************************************************************************/
Int32 SDvideo_ColorBar(void);
void GenerateYCbCr422ColorBar(Uint16 ncolor, Uint8 y[],Uint8 cb[],Uint8 cr[]);
Int32 ConfigureEDMA3VPDisp(Uint8 YBuff[],Uint8 CbBuff[] ,Uint8 CrBuff[]);
void ConfigureEDMA3VPStop(void);
void Double_Word_Byte_Swap(unsigned char *buffer_ptr, unsigned int no_of_bytes);


/****************************************************************************
 *                             Public Functions                             *
 ****************************************************************************/


/**************************************************************************** 
 *
 * NAME  
 *      do_SDvideo_ColorBar
 *
 * PURPOSE:
 *      Do a color bar test on HD and J13 video out from DSP. Then displays
 *      the result on the console. Needs that the debug buffer are ready
 *      for myprintf function for results output.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 do_SDvideo_ColorBar(void)
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 do_SDvideo_ColorBar(void)
{
    Int32 iResult;

    myprintf("SD video color bar test HD and J13 video out...\r\n",0,0);


    iResult = SDvideo_ColorBar();

    // Displays the result
    if (!iResult)
    {
        myprintf("\r\nSD video color bar test HD and J13 video out... SUCCESS!\r\n\r\n",0,0);
    }
    else
    {
        myprintf("\r\nSD video color bar test HD and J13 video out... ERROR!\r\n\r\n",0,0);
    }

    return(iResult);
}


/****************************************************************************
 *                             Local Functions                              *
 ****************************************************************************/


/**************************************************************************** 
 *
 * NAME  
 *      SDvideo_ColorBar
 *
 * PURPOSE:
 *      Setup a color bar test on HD and J13 video out from DSP. This test
 *      is implemented with a single display line buffer only so it
 *      can fit in dsp internal memory (no DDR2 is needed for video buffers).
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 SDvideo_ColorBar(void)
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 SDvideo_ColorBar(void)
{
    EVM_S8200_Handle S8200handle0;
    EVM_A7105_Handle A7105handle0;
    Int32 retcode=0;
    volatile Int32 iCount;
    
    // Generates 4:2:2 YCbCr color bar pattern in buffers
    GenerateYCbCr422ColorBar(NCOLOR, YSrc, CbSrc , CrSrc);

    // Need to be removed after silicon bug correction
    // Apply the first silicon VP bug byte swapping workaround for all buffers
    Double_Word_Byte_Swap(YSrc,  NPIXEL);
    Double_Word_Byte_Swap(CbSrc, NCOLOR);
    Double_Word_Byte_Swap(CrSrc, NCOLOR);

    // Initialize EVM module
    // Must always be the first BSL functions called
    if ((retcode=EVM_init())!=EVM_INITOK)
    {
        // Problem cannot open evm
        return(retcode);
    }

    // Verify is correct evm binary bsl version
    if ((retcode=EVM_checkbslversion(EVM_BSLVERSION))!=EVM_INITOK)
    {
        // Problem unmatching binary bsl version
        return(retcode);
    }
    
    // Select BT656 SDTV clock (27 MHz) for S8200 (COMMAND reg = 0x08)
    if ((retcode = EVM_ECP_writereg( EVM_ECP_ADDRESS, 0, 0x08))!=EVM_ECP_OK)
    {
        // Problem cannot send ECP cmd
        return(retcode);
    }

    // Configure Video port 1 as BT656 display video port NTSC
    // Doing here to allow the video clock reaching all video component in chain
    Bt656_8bit_Ncfd_NTSC(1);

    // Open the S8200 HD video digitizer in SDTV BT656 mode
    S8200handle0 = EVM_S8200_open( 0, &S8200setup );

    // Verify if we got a problem
    if (S8200handle0 >= EVM_S8200_NHANDLE)
    {
        // Problem cannot open 
        return(S8200handle0);
    }

    // Open the A7105 video digitizer in SDTV BT656 mode
    A7105handle0 = EVM_A7105_open( 0, &A7105setup );

    // Verify if we got a problem
    if (A7105handle0 >= EVM_A7105_NHANDLE)
    {
        // Problem cannot open 
        return(A7105handle0);
    }

    // Must configure EDMA here (and optionnaly irq if desired)
    retcode=ConfigureEDMA3VPDisp(YSrc, CbSrc , CrSrc);

    // Enable the video port
    VPEnable();

    // Wait a specific time to allow the pattern to be seen...
    if (retcode==0)
    {
        // Around 9 sec loop
        for (iCount=0;iCount<44812000;iCount++)
        {
            // Poll display flags to allow each frame display
            VPDispPoll();
        }
    }

    // Close all peripherals
    EVM_A7105_close( &A7105handle0 );
    EVM_S8200_close( &S8200handle0 );

    // Close current video display
    CloseDisplayVideoPort();

    ConfigureEDMA3VPStop();

    return (retcode);
}


/**************************************************************************** 
 *
 * NAME  
 *      GenerateYCbCr422ColorBar
 *
 * PURPOSE:
 *      Generate a YCbCr 4:2:2 color bar pattern in buffers.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      void GenerateYCbCr422ColorBar(Uint16 ncolor, Uint8 y[],Uint8 cb[],Uint8 cr[])
 *
 *      ncolor - (i) number of color sample on 1 video line (4:2:2 means Npixel/2)
 *      y      - (o) y intensity buffer
 *      cb     - (o) cb color buffer
 *      cr     - (o) cr color buffer
 *
 * RETURN VALUE
 *      None
 *
 * REFERENCE
 *
 ****************************************************************************/
void GenerateYCbCr422ColorBar(Uint16 ncolor, Uint8 y[],Uint8 cb[],Uint8 cr[])
{
    Int16 count;

    // Generate one line of the color bar pattern 4:2:2 in Ycbcr color space
    for (count=0;count<ncolor;count++)
    {
        switch(count/45)
        {
            // White 235:128:128
            case 0:
                y[count*2]=235;
                y[count*2+1]=235;
                cb[count]=128;
                cr[count]=128;
                break;

            // Yellow 210:16:146
            case 1:
                y[count*2]=210;
                y[count*2+1]=210;
                cb[count]=16;
                cr[count]=146;
                break;

            // Cyan 170:166:16
            case 2:
                y[count*2]=170;
                y[count*2+1]=170;
                cb[count]=166;
                cr[count]=16;
                break;

            // Green 145:54:34
            case 3:
                y[count*2]=145;
                y[count*2+1]=145;
                cb[count]=54;
                cr[count]=34;
                break;

            // Magenta 107:202:222
            case 4:
                y[count*2]=107;
                y[count*2+1]=107;
                cb[count]=202;
                cr[count]=222;
                break;

            // Red 82:90:240
            case 5:
                y[count*2]=82;
                y[count*2+1]=82;
                cb[count]=90;
                cr[count]=240;
                break;

            // Blue 41:240:110
            case 6:
                y[count*2]=41;
                y[count*2+1]=41;
                cb[count]=240;
                cr[count]=110;
                break;

            // Black 16:128:128
            case 7:
                y[count*2]=16;
                y[count*2+1]=16;
                cb[count]=128;
                cr[count]=128;
                break;
        }
    }
}
/**************************************************************************** 
 *
 * NAME  
 *      ConfigureEDMA3VPDisp
 *
 * PURPOSE:
 *      Configure the EDMA3 for video port event for line display mode.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      void ConfigureEDMA3VPDisp(Uint8 YBuff[],Uint8 CbBuff[] ,Uint8 CrBuff[])
 *
 *      YBuff  - (i) y intensity buffer
 *      CbBuff - (i) cb color buffer
 *      CrBuff - (i) cr color buffer
 *
 * RETURN VALUE
 *      0 if ok, !=0 if errors
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 ConfigureEDMA3VPDisp(Uint8 YBuff[],Uint8 CbBuff[] ,Uint8 CrBuff[])
{
    /* Module level open */
    edma3Open(&edmaObj,CSL_EDMA3,hModule);

    /* Module setup */
    hwSetup.dmaChaSetup = &dmahwSetup[0];
    hwSetup.qdmaChaSetup = NULL;
    edma3HwSetup(hModule,&hwSetup);
           
    /* Channel open */
    chAttr.regionNum = CSL_EDMA3_REGION_GLOBAL;
    chAttr.chaNum = CSL_EDMA3_CHA_VP1EVTYA;
    edma3ChannelOpen(&chObjY, CSL_EDMA3, &chAttr, &hChannelY);
    
    /* Change Channel Default queue setup from 0 to 0  */
    edma3HwChannelSetupQue(hChannelY,CSL_EDMA3_QUE_0);

    /* Channel open */
    chAttr.regionNum = CSL_EDMA3_REGION_GLOBAL;
    chAttr.chaNum = CSL_EDMA3_CHA_VP1EVTUA;
    edma3ChannelOpen(&chObjCb, CSL_EDMA3, &chAttr, &hChannelCb);
    
    /* Change Channel Default queue setup from 0 to 1  */
    edma3HwChannelSetupQue(hChannelCb,CSL_EDMA3_QUE_1);

    /* Channel open */
    chAttr.regionNum = CSL_EDMA3_REGION_GLOBAL;
    chAttr.chaNum = CSL_EDMA3_CHA_VP1EVTVA;
    edma3ChannelOpen(&chObjCr, CSL_EDMA3, &chAttr, &hChannelCr);
    
    /* Change Channel Default queue setup from 0 to 2  */
    edma3HwChannelSetupQue(hChannelCr,CSL_EDMA3_QUE_2);

    /* Obtain a handle to parameter set 0 */
    hParamBasicY=edma3GetParamHandle(hChannelY,CSL_EDMA3_CHA_VP1EVTYA);
    
    /* Obtain a handle to parameter set 3 */
    hParamCpyY=edma3GetParamHandle(hChannelY,CSL_EDMA3_CHA_ICREVT);
    
    /* Obtain a handle to parameter set 1 */
    hParamBasicCb=edma3GetParamHandle(hChannelCb,CSL_EDMA3_CHA_VP1EVTUA);
 
    /* Obtain a handle to parameter set 4 */
    hParamCpyCb=edma3GetParamHandle(hChannelCb,CSL_EDMA3_CHA_ICXEVT);
 
    /* Obtain a handle to parameter set 2 */
    hParamBasicCr=edma3GetParamHandle(hChannelCr,CSL_EDMA3_CHA_VP1EVTVA);
    
    /* Obtain a handle to parameter set 5 */
    hParamCpyCr=edma3GetParamHandle(hChannelCr,CSL_EDMA3_CHA_TEVTLO1);

    /* Setup the parameter entry parameters (Y buffer) */
    myParamSetup.option = EDMA3_OPT_MAKE(EDMA3_ITCCH_DIS, \
                                         EDMA3_TCCH_DIS, \
                                         EDMA3_ITCINT_DIS, \
                                         EDMA3_TCINT_EN, \
                                         CSL_EDMA3_CHA_VP1EVTYA,\
                                         EDMA3_TCC_NORMAL,\
                                         EDMA3_FIFOWIDTH_NONE, \
                                         EDMA3_STATIC_DIS, \
                                         EDMA3_SYNC_AB, \
                                         EDMA3_ADDRMODE_INCR, \
                                         EDMA3_ADDRMODE_INCR );
    myParamSetup.srcAddr = (Uint32)YBuff;         
    myParamSetup.aCntbCnt = EDMA3_CNT_MAKE(8,NPIXEL/8);       
    myParamSetup.dstAddr = (Uint32)YDSTVP1;        
    myParamSetup.srcDstBidx = EDMA3_BIDX_MAKE(8,0);     
    myParamSetup.linkBcntrld = EDMA3_LINKBCNTRLD_MAKE(hParamCpyY,0);     
    myParamSetup.srcDstCidx = EDMA3_CIDX_MAKE(0,0);     
    myParamSetup.cCnt = 1;
    edma3ParamSetup(hParamBasicY,&myParamSetup);
    edma3ParamSetup(hParamCpyY,&myParamSetup);

    /* Setup the parameter entry parameters (Cb buffer) */
    myParamSetup.option = EDMA3_OPT_MAKE(EDMA3_ITCCH_DIS, \
                                         EDMA3_TCCH_DIS, \
                                         EDMA3_ITCINT_DIS, \
                                         EDMA3_TCINT_EN, \
                                         CSL_EDMA3_CHA_VP1EVTUA,\
                                         EDMA3_TCC_NORMAL,\
                                         EDMA3_FIFOWIDTH_NONE, \
                                         EDMA3_STATIC_DIS, \
                                         EDMA3_SYNC_AB, \
                                         EDMA3_ADDRMODE_INCR, \
                                         EDMA3_ADDRMODE_INCR );
    myParamSetup.srcAddr = (Uint32)CbBuff;         
    myParamSetup.aCntbCnt = EDMA3_CNT_MAKE(8,NPIXEL/2/8);       
    myParamSetup.dstAddr = (Uint32)CBDSTVP1;        
    myParamSetup.srcDstBidx = EDMA3_BIDX_MAKE(8,0);     
    myParamSetup.linkBcntrld = EDMA3_LINKBCNTRLD_MAKE(hParamCpyCb,0);     
    myParamSetup.srcDstCidx = EDMA3_CIDX_MAKE(0,0);     
    myParamSetup.cCnt = 1;
    edma3ParamSetup(hParamBasicCb,&myParamSetup);
    edma3ParamSetup(hParamCpyCb,&myParamSetup);

    /* Setup the parameter entry parameters (Cr buffer) */
    myParamSetup.option = EDMA3_OPT_MAKE(EDMA3_ITCCH_DIS, \
                                         EDMA3_TCCH_DIS, \
                                         EDMA3_ITCINT_DIS, \
                                         EDMA3_TCINT_EN, \
                                         CSL_EDMA3_CHA_VP1EVTVA,\
                                         EDMA3_TCC_NORMAL,\
                                         EDMA3_FIFOWIDTH_NONE, \
                                         EDMA3_STATIC_DIS, \
                                         EDMA3_SYNC_AB, \
                                         EDMA3_ADDRMODE_INCR, \
                                         EDMA3_ADDRMODE_INCR );
    myParamSetup.srcAddr = (Uint32)CrBuff;         
    myParamSetup.aCntbCnt = EDMA3_CNT_MAKE(8,NPIXEL/2/8);       
    myParamSetup.dstAddr = (Uint32)CRDSTVP1;        
    myParamSetup.srcDstBidx = EDMA3_BIDX_MAKE(8,0);     
    myParamSetup.linkBcntrld = EDMA3_LINKBCNTRLD_MAKE(hParamCpyCr,0);     
    myParamSetup.srcDstCidx = EDMA3_CIDX_MAKE(0,0);     
    myParamSetup.cCnt = 1;
    edma3ParamSetup(hParamBasicCr,&myParamSetup);
    edma3ParamSetup(hParamCpyCr,&myParamSetup);

    /* Interrupt enable (Bits 0-1)  for the global region interrupts */
    regionIntr.region =  CSL_EDMA3_REGION_GLOBAL  ;   
    regionIntr.intr  =   (1 << CSL_EDMA3_CHA_VP1EVTYA) | (1 << CSL_EDMA3_CHA_VP1EVTUA) |
                            (1 << CSL_EDMA3_CHA_VP1EVTVA);   
    regionIntr.intrh  =  0x0000 ;
    edma3CmdIntrEnable(hModule,regionIntr.intr,regionIntr.intrh);

    // Get Channel Error Status 
    edma3GetChannelErrStatus(hChannelY,&myErrorsY);
    edma3GetChannelErrStatus(hChannelCb,&myErrorsCb);
    edma3GetChannelErrStatus(hChannelCr,&myErrorsCr);

    // Clear edma errors...
    edma3ChannelErrorClear(hChannelY,&myErrorsY);
    edma3ChannelErrorClear(hChannelCb,&myErrorsCb);
    edma3ChannelErrorClear(hChannelCr,&myErrorsCr);

    // Enable Channels(if the channel is meant for external event) 
    // This step is not required if the channel is chained to or manually triggered.
    edma3ChannelEnable(hChannelY);
    edma3ChannelEnable(hChannelCb);
    edma3ChannelEnable(hChannelCr);

    return(0);
}

/**************************************************************************** 
 *
 * NAME  
 *      ConfigureEDMA3VPStop
 *
 * PURPOSE:
 *      Stop edma3 capture and display channel activity.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      void ConfigureEDMA3VPStop(void)
 *
 * RETURN VALUE
 *      None
 *
 * REFERENCE
 *
 ****************************************************************************/
void ConfigureEDMA3VPStop(void)
{
    // Disable Display Channels 
    edma3ChannelDisable(hChannelY);
    edma3ChannelDisable(hChannelCb);
    edma3ChannelDisable(hChannelCr);

    /* Disable all interrupts that were used */
    edma3InterruptDisable(hModule,regionIntr.intr,regionIntr.intrh);
}

// To be removed when silicon issue is resolved
/* ======================================================================== */
/*  NAME                                                                    */
/*      double_word_byte_swap -- byte swap of a double word                 */
/*S                                                                        S*/
/*S AUTHOR                                                                 S*/
/*S     Pavan Shastry                                                      S*/
/*S                                                                        S*/
/*S REVISION HISTORY                                                       S*/
/*S     25-Apr-2007 Initial version of double_word_byte_swap .... P.ShastryS*/
/*S     08-May-2007 Correction of half buffer was only processed. Lyrtech  S*/
/*S                                                                        S*/
/*                                                                          */
/*  USAGE                                                                   */
/*      This routine has the following C prototype:                         */
/*                                                                          */
/*      void double_word_byte_swap                                          */
/*      (                                                                   */
/*          unsigned char *restrict   buffer_ptr,                           */
/*          unsigned int              no_of_bytes,                          */
/*      )                                                                   */
/*                                                                          */
/*      The double_word_byte_swap routine accepts a buffer of size          */
/*      no_of_bytes bytes. It does a byte swap of double word.              */
/*      See description below for more details.                             */
/*                                                                          */
/*  DESCRIPTION                                                             */
/*      This code is written to perform a byte swap operation of            */
/*      a double word, in the manner indicated below -                      */
/*                                                                          */
/*      Input  Double word: A7 A6 A5 A4  A3 A2 A1 A0                        */
/*      Output Double word: A0 A1 A2 A3  A4 A5 A6 A7                        */
/*                                                                          */
/*      The Computation is in-place. The input buffer is written with       */
/*      bytes swapped in the above manner.                                  */
/*                                                                          */
/*      The number of double words, for which byte swapping will            */
/*      be performed is equal to (no_of_bytes >> 3).                        */
/*                                                                          */
/*  ASSUMPTIONS                                                             */
/*      'no_of_bytes' is a multiple of 8.                                   */
/*                                                                          */
/*  NOTES                                                                   */
/*      1. If 'no_of_bytes' is not a multiple of 8, then largest            */
/*         multiple of 8, which is lesser than no_of_bytes will be          */
/*         assumed.                                                         */
/*      2. For best performance of this routine, use -o3 and No debug       */
/*         as compile options.                                              */ 
/*                                                                          */
/*  SOURCE                                                                  */
/*      None.                                                               */
/* ------------------------------------------------------------------------ */
/*            Copyright (c) 2007 Texas Instruments, Incorporated.           */
/*                           All Rights Reserved.                           */
/* ======================================================================== */
void Double_Word_Byte_Swap(unsigned char *buffer_ptr, unsigned int no_of_bytes)
{
   
   double * restrict read_ptr;
   double * restrict write_ptr;
   double input_dword;
   double output_word;
   unsigned int word0;
   unsigned int word1; 
   unsigned int dword_cntr;
   unsigned int dword_total_cnt;
   
   read_ptr  = (double *)buffer_ptr;
   write_ptr = (double *)buffer_ptr;
   
   /* ------------------------------------------------------------- */
   /* No of double words to be processed                            */
   /* ------------------------------------------------------------- */
   dword_total_cnt  = no_of_bytes >> 3;

   /* ------------------------------------------------------------- */
   /* Each iteration handles two double words                       */
   /* ------------------------------------------------------------- */
   for(dword_cntr = 0; dword_cntr < dword_total_cnt; dword_cntr+=2)
   {
       input_dword  = read_ptr[dword_cntr];

       word0        = _lo(input_dword);
       word1        = _hi(input_dword);
       word0        = _swap4(word0);
       word1        = _swap4(word1);
       word0        = _rotl(word0,16);
       word1        = _rotl(word1,16);
       output_word  = _itod(word0, word1);
       
       write_ptr[dword_cntr] = output_word;

       input_dword  = read_ptr[dword_cntr+1];

       word0        = _lo(input_dword);
       word1        = _hi(input_dword);
       word0        = _swap4(word0);
       word1        = _swap4(word1);
       word0        = _rotl(word0,16);
       word1        = _rotl(word1,16);
       output_word  = _itod(word0, word1);
       
       write_ptr[dword_cntr+1] = output_word;
   }
   
   /* ------------------------------------------------------------- */
   /* If the number of double words is odd, then the last double    */
   /* word is handled here.                                         */
   /* ------------------------------------------------------------- */
   if(dword_total_cnt & 0x01)
   {
       input_dword  = read_ptr[dword_cntr];

       word0        = _lo(input_dword);
       word1        = _hi(input_dword);
       word0        = _swap4(word0);
       word1        = _swap4(word1);
       word0        = _rotl(word0,16);
       word1        = _rotl(word1,16);
       output_word  = _itod(word0, word1);
       
       write_ptr[dword_cntr] = output_word;
   }  
}
