/***************************************************************************
 *
 *
 *         **   **    **  ******  ********  ********  ********  **    **    
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ***************************************************************************
 *                                                                          
 *  Project     : EVM
 *  File        : sdvideodigloop.c
 *  Description : Utility functions for SD video
 *
 *                      Copyright (c) Lyrtech inc. 2007                        
 *
 ***************************************************************************
 *                                                                                          
 * "$Revision: 1.7 $"
 * "$Date: 2007/07/24 20:30:15 $"
 *
 ***************************************************************************/

/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include "soc.h"
#include "edma3.h"
#include "evm.h"
#include "evm_s8200.h"
#include "evm_a7105.h"
#include "evm_ecp.h"
#include "bt656.h"
#include "bt656capt.h"
#include "evm_p5154.h"
#include "sdvideodigloop.h"
#include "mydebug.h"

/****************************************************************************
 *                             Local constants                              *
 ****************************************************************************/
// Uncomment folllowing line to get s-video output instead of composite on J13
//#define J13_SVIDEO

#define NPIXEL 720
#define NCOLOR (NPIXEL/2)
#define NFIELD1 240
#define NFIELD2 240
#define NLINES (NFIELD1+NFIELD2)

// Video display VP1 buffer address
#define YDSTVP1   (CSL_VP1_CHANNELA_DATA+0x00000080)
#define CBDSTVP1  (CSL_VP1_CHANNELA_DATA+0x000000A0)
#define CRDSTVP1  (CSL_VP1_CHANNELA_DATA+0x000000C0)

// Video capture VPx buffer address
#define YSRCVP0A  (CSL_VP0_CHANNELA_DATA+0x00000000)
#define CBSRCVP0A (CSL_VP0_CHANNELA_DATA+0x00000020)
#define CRSRCVP0A (CSL_VP0_CHANNELA_DATA+0x00000040)
#define YSRCVP1A  (CSL_VP1_CHANNELA_DATA+0x00000000)
#define CBSRCVP1A (CSL_VP1_CHANNELA_DATA+0x00000020)
#define CRSRCVP1A (CSL_VP1_CHANNELA_DATA+0x00000040)
#define YSRCVP2A  (CSL_VP2_CHANNELA_DATA+0x00000000)
#define CBSRCVP2A (CSL_VP2_CHANNELA_DATA+0x00000020)
#define CRSRCVP2A (CSL_VP2_CHANNELA_DATA+0x00000040)
#define YSRCVP3A  (CSL_VP3_CHANNELA_DATA+0x00000000)
#define CBSRCVP3A (CSL_VP3_CHANNELA_DATA+0x00000020)
#define CRSRCVP3A (CSL_VP3_CHANNELA_DATA+0x00000040)
#define YSRCVP4A  (CSL_VP4_CHANNELA_DATA+0x00000000)
#define CBSRCVP4A (CSL_VP4_CHANNELA_DATA+0x00000020)
#define CRSRCVP4A (CSL_VP4_CHANNELA_DATA+0x00000040)

#define YSRCVP0B  (CSL_VP0_CHANNELB_DATA+0x00000000)
#define CBSRCVP0B (CSL_VP0_CHANNELB_DATA+0x00000020)
#define CRSRCVP0B (CSL_VP0_CHANNELB_DATA+0x00000040)
#define YSRCVP1B  (CSL_VP1_CHANNELB_DATA+0x00000000)
#define CBSRCVP1B (CSL_VP1_CHANNELB_DATA+0x00000020)
#define CRSRCVP1B (CSL_VP1_CHANNELB_DATA+0x00000040)
#define YSRCVP2B  (CSL_VP2_CHANNELB_DATA+0x00000000)
#define CBSRCVP2B (CSL_VP2_CHANNELB_DATA+0x00000020)
#define CRSRCVP2B (CSL_VP2_CHANNELB_DATA+0x00000040)
#define YSRCVP3B  (CSL_VP3_CHANNELB_DATA+0x00000000)
#define CBSRCVP3B (CSL_VP3_CHANNELB_DATA+0x00000020)
#define CRSRCVP3B (CSL_VP3_CHANNELB_DATA+0x00000040)
#define YSRCVP4B  (CSL_VP4_CHANNELB_DATA+0x00000000)
#define CBSRCVP4B (CSL_VP4_CHANNELB_DATA+0x00000020)
#define CRSRCVP4B (CSL_VP4_CHANNELB_DATA+0x00000040)


/****************************************************************************
 *                             Local variables                              *
 ****************************************************************************/

// HD video P5154, A7105 & S8200 default setup registers structure for video
// loopback 480i bt656
static EVM_S8200_Setup S8200setup = EVM_S8200_SDTV480I_YCBCR704X480X60;

// Setup J13 output for conposite or svideo output
#ifdef J13_SVIDEO
static EVM_A7105_Setup A7105setup = EVM_A7105_SDTV480I_YCBCR704X480X60_SVIDEO;
#else
static EVM_A7105_Setup A7105setup = EVM_A7105_SDTV480I_YCBCR704X480X60_COMP;
#endif

static EVM_P5154_Setup P5154setup = EVM_P5154_SDTV480I_YCBCR704X480X60;


// Variables for EDMA3 setup
// Generals
static Edma3Handle                 hModule;
static Edma3HwSetup                hwSetup;
static Edma3Obj                    edmaObj;
static Edma3CmdIntr                regionIntr;
static Edma3ParamSetup             myParamSetup;
static Edma3ChannelAttr            chAttr;
static Edma3HwDmaChannelSetup      dmahwSetup[CSL_EDMA3_NUM_DMACH] = EDMA3_DMACHANNELSETUP_DEFAULT;

// Display
static Edma3ParamHandle            hParamCpyY;
static Edma3ParamHandle            hParamCpyCb;
static Edma3ParamHandle            hParamCpyCr;
static Edma3ParamHandle            hParamBasicY;
static Edma3ParamHandle            hParamBasicCb;
static Edma3ParamHandle            hParamBasicCr;
static Edma3ChannelObj             chObjY;
static Edma3ChannelObj             chObjCb;
static Edma3ChannelObj             chObjCr;
static Edma3ChannelHandle          hChannelY;
static Edma3ChannelHandle          hChannelCb;
static Edma3ChannelHandle          hChannelCr;
static Edma3ChannelErr             myErrorsY;        
static Edma3ChannelErr             myErrorsCb;        
static Edma3ChannelErr             myErrorsCr;        
// Capture
static Edma3ParamHandle            hParamCpyYCapt;
static Edma3ParamHandle            hParamCpyCbCapt;
static Edma3ParamHandle            hParamCpyCrCapt;
static Edma3ParamHandle            hParamBasicYCapt;
static Edma3ParamHandle            hParamBasicCbCapt;
static Edma3ParamHandle            hParamBasicCrCapt;
static Edma3ChannelObj             chObjYCapt;
static Edma3ChannelObj             chObjCbCapt;
static Edma3ChannelObj             chObjCrCapt;
static Edma3ChannelHandle          hChannelYCapt;
static Edma3ChannelHandle          hChannelCbCapt;
static Edma3ChannelHandle          hChannelCrCapt;
static Edma3ChannelErr             myErrorsYCapt;        
static Edma3ChannelErr             myErrorsCbCapt;        
static Edma3ChannelErr             myErrorsCrCapt;        

// Defines variables that will go in a custom section with loader
// This is only used for this example to keep everything in
// internal dsp memory but not the video buffers (in DDR2)
// This way it may help diag DDR2 by looking at video buffer
// and avoid code hang in DDR2. It is not necessary in "real life".
// Buffers are 32 butes aligned in memory...
asm("_YSrcB .usect \".favar\",345600,32");
asm("_CbSrcB .usect \".favar\",345600/2,32");
asm("_CrSrcB .usect \".favar\",345600/2,32");
asm(" .global _YSrcB");
asm(" .global _CbSrcB");
asm(" .global _CrSrcB");

/* Globals */
//#pragma DATA_ALIGN(YSrcB,32);
extern Uint8       YSrcB[NLINES][NPIXEL];
//#pragma DATA_ALIGN(CbSrcB,32);
extern Uint8       CbSrcB[NLINES][NCOLOR];
//#pragma DATA_ALIGN(CrSrcB,32);
extern Uint8       CrSrcB[NLINES][NCOLOR];



/****************************************************************************
 *                           Forward Declarations                           *
 ****************************************************************************/


/****************************************************************************
 *                       Private functions prototypes                       *
 ****************************************************************************/
void ConfigureEDMA3(void);
static void ConfigureEDMA3VPDisp(Uint8 *YBuff,Uint8 *CbBuff ,Uint8 *CrBuff);
void ConfigureEDMA3VPDispParamSet(Uint8 *YBuff,Uint8 *CbBuff ,Uint8 *CrBuff);
void ConfigureEDMA3VPCapt(Uint8 *YBuff,Uint8 *CbBuff ,Uint8 *CrBuff, Int16 VideoPort, Int16 VideoChan);
void ConfigureEDMA3VPCaptParamSet(Uint8 *YBuff,Uint8 *CbBuff ,Uint8 *CrBuff, Int16 VideoPort, Int16 VideoChan);
static void ConfigureEDMA3VPStop(void);
static void Double_Word_Byte_Swap(unsigned char *buffer_ptr, unsigned int no_of_bytes);
static void GenerateYCbCr422ColorBar(Uint16 ncolor, Uint8 y[],Uint8 cb[],Uint8 cr[]);
Int32 SDvideo_DigLoopBack(Uint32 vid);


/****************************************************************************
 *                             Public Functions                             *
 ****************************************************************************/

/**************************************************************************** 
 *
 * NAME  
 *      do_SDvideo_DigLoopBack
 *
 * PURPOSE:
 *      Digital loopback between SD Chx video IN and HD + J13 video OUT.
 *      This example can easily output on J13 in S-video or composite
 *      mode just by defining J13_SVIDEO at top of this file...
 *      This routine required trace buffer being initialized for myprintf.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 do_SDvideo_DigLoopBack(Uint32 ichan)
 *
 *      ichan  - (i) id of desired video channel EVM_P5154_CH1-EVM_P5154_CH8 [0-7]
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 do_SDvideo_DigLoopBack(Uint32 ichan)
{
    Int32 iResult;

    myprintf("SD video digital loopback SD in CH%1d -> HD & J13 video out test...\n",(ichan+1),0);


    iResult = SDvideo_DigLoopBack(ichan);

    if (!iResult)
    {
        myprintf("\nSD video digital loopback SD in CH%1d -> HD & J13 video out test... SUCCESS!\n\n",(ichan+1),0);
    }
    else
    {
        myprintf("\nSD video digital loopback SD in CH%1d -> HD & J13 video out test... ERROR!\n\n",(ichan+1),0);
    }

    return(iResult);
}



/****************************************************************************
 *                             Local Functions                              *
 ****************************************************************************/


/**************************************************************************** 
 *
 * NAME  
 *      SDvideo_DigLoopBack
 *
 * PURPOSE:
 *      Digital loopback between SD Chx video IN and HD + J13 video OUT.
 *      This example can easily output on J13 in S-video or composite
 *      mode just by defining J13_SVIDEO at top of this file...
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 SDvideo_DigLoopBack(Uint32 vid)
 *
 *      vid   - (i) id of desired video channel [0-7]
 *
 * RETURN VALUE
 *      0 if OK, !=0 in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 SDvideo_DigLoopBack(Uint32 vid)
{
    EVM_S8200_Handle S8200handle0;
    EVM_A7105_Handle A7105handle0;
    EVM_P5154_Handle P5154handle0;
    Int32 retcode=0;
    volatile Int32 iCount;
    Int32 iLine;
    Int16 P5154Videoport;
    Int16 P5154VideoChan;

    // Initialize EVM module
    // Must always be the first BSL functions called
    if ((retcode=EVM_init())!=EVM_INITOK)
    {
        // Problem cannot open evm
        return(retcode);
    }

    // Verify is correct evm binary bsl version
    if ((retcode=EVM_checkbslversion(EVM_BSLVERSION))!=EVM_INITOK)
    {
        // Problem unmatching binary bsl version
        return(retcode);
    }

    // Select BT656 SDTV clock (27 MHz) for S8200 (COMMAND reg = 0x08)
    if ((retcode = EVM_ECP_writereg( EVM_ECP_ADDRESS, 0, 0x08))!=EVM_ECP_OK)
    {
        // Problem cannot send ECP cmd
        return(retcode);
    }

    // Configure Video port 1 as BT656 display video port NTSC
    // Doing here to allow the video clock reaching all video component in chain
    Bt656_8bit_Ncfd_NTSC(1);

    // Open the S8200 HD video digitizer in SDTV BT656 mode
    S8200handle0 = EVM_S8200_open( 0, &S8200setup );

    // Verify if we got a problem
    if (S8200handle0 >= EVM_S8200_NHANDLE)
    {
        // Problem cannot open 
        return(S8200handle0);
    }

    // Open the A7105 video digitizer in SDTV BT656 mode
    A7105handle0 = EVM_A7105_open( 0, &A7105setup );

    // Verify if we got a problem
    if (A7105handle0 >= EVM_A7105_NHANDLE)
    {
        // Problem cannot open 
        return(A7105handle0);
    }


    // Open the P5154 video decoder in SDTV BT656 mode
    P5154handle0 = EVM_P5154_open( vid, &P5154setup );

    if (P5154handle0 >= EVM_P5154_NHANDLE)
    {
        // Problem cannot open 
        return(P5154handle0);
    }

    // Get the video port related to this video decoder
    if ((retcode=EVM_P5154_gvideoport ( P5154handle0, &P5154Videoport, &P5154VideoChan )) != EVM_P5154_OK)
    {
        // Problem cannot open 
        return(retcode);
    }

    // If more than one video channel is used, info returned from EVM_P5154_gvideoport
    // should be used to detect which video channel to use. Bt656Capt_8bit_Ncfd_NTSC
    // will always configure 2 video channels at the same time. If EVM_P5154_gvideoport
    // returns the same video port for two differents video input, this will only
    // need a single Bt656Capt_8bit_Ncfd_NTSC to configure...

    // Configure the related video port with the current video decoder (both chan A & B)
    Bt656Capt_8bit_Ncfd_NTSC(P5154Videoport);

    // Global EDMA3 initialization
    ConfigureEDMA3();

    // Should modify to a multiple buffer display/capt scheme to avoid
    // display fickering in some situations

    // Usual video processing would need multiple Y CB CR buffers to allow
    // processing of a capture frame while one is captured and one is displayed.
    // EDMA3 config routines may need to be modified to achieved this...

    // Capture EDMA3 configuration
    ConfigureEDMA3VPCapt(&YSrcB[0][0], &CbSrcB[0][0] , &CrSrcB[0][0], P5154Videoport, P5154VideoChan);

    // Display EDMA3 configuration
    ConfigureEDMA3VPDisp(&YSrcB[0][0], &CbSrcB[0][0] , &CrSrcB[0][0]);

    // Erase video buffer (only needed as a debug measure)
    memset(&YSrcB[0][0],0,sizeof(YSrcB));
    memset(&CbSrcB[0][0],0,sizeof(CbSrcB));
    memset(&CrSrcB[0][0],0,sizeof(CrSrcB));

    // Generate a color bar pattern to detect correct capture when displayed

    // Generation of a complete video frame
    for (iLine=0;iLine<NLINES;iLine+=1)
    {
        // Generates 4:2:2 YCbCr color bar pattern in buffers
        GenerateYCbCr422ColorBar(NCOLOR, &YSrcB[iLine][0], &CbSrcB[iLine][0] , &CrSrcB[iLine][0]);

        // Need to be removed after silicon bug correction
        // Apply the first silicon VP bug byte swapping workaround for all buffers
        Double_Word_Byte_Swap(&YSrcB[iLine][0],  NPIXEL);
        Double_Word_Byte_Swap(&CbSrcB[iLine][0], NCOLOR);
        Double_Word_Byte_Swap(&CrSrcB[iLine][0], NCOLOR);
    }

    // Enable the capture video port
    VPCaptEnable();

    // Enable the display video port
    VPEnable();

    // Wait a specific time to allow the pattern to be seen...
    if (retcode==0)
    {
        // Around 9 secs loop...
        for (iCount=0;iCount<18671691;iCount++)
        {
            // Important: if video data would need to be processed, the silicon bug
            // would requiered to do the  byte swapping process explained above...

            // Poll capt and display flags to allow each frame display
            // and to continue in non continuos video capt/disp mode
            // This is to control the capture and display process. Would
            // be usually done with interrupt routines. Display/capture
        	// continous mode may also be used instead...
            VPCapChanPoll();
            VPDispPoll();
        }
	}

    // Close capture video port
    CloseCaptureVideoPort();

    // Close all peripherals
    EVM_P5154_close( &P5154handle0 );
    EVM_A7105_close( &A7105handle0 );
    EVM_S8200_close( &S8200handle0 );
    // Close current video display
    CloseDisplayVideoPort();

    ConfigureEDMA3VPStop();

    return (retcode);
}

/**************************************************************************** 
 *
 * NAME  
 *      ConfigureEDMA3
 *
 * PURPOSE:
 *      Open the EDMA3 module and do initial configuration.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      void ConfigureEDMA3(void)
 *
 * RETURN VALUE
 *      None
 *
 * REFERENCE
 *
 ****************************************************************************/
void ConfigureEDMA3(void)
{
    /* Module level open */
    edma3Open(&edmaObj,CSL_EDMA3,hModule);

    /* Module setup */
    hwSetup.dmaChaSetup = &dmahwSetup[0];
    hwSetup.qdmaChaSetup = NULL;
    edma3HwSetup(hModule,&hwSetup);

    /* Interrupt enable (Bits 0-1)  for the global region interrupts */
    regionIntr.region =  CSL_EDMA3_REGION_GLOBAL;
    regionIntr.intr   =  0x0000;   
    regionIntr.intrh  =  0x0000;
}

/**************************************************************************** 
 *
 * NAME  
 *      ConfigureEDMA3VPDisp
 *
 * PURPOSE:
 *      Configure the EDMA3 for video port event display mode.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      void ConfigureEDMA3VPDisp(Uint8 YBuff[],Uint8 CbBuff[] ,Uint8 CrBuff[])
 *
 * RETURN VALUE
 *      None
 *
 * REFERENCE
 *
 ****************************************************************************/
static void ConfigureEDMA3VPDisp(Uint8 *YBuff,Uint8 *CbBuff ,Uint8 *CrBuff)
{
    Int yevent,cbevent,crevent,VideoPort=1;

    // Gets video port events related to edma
    switch(VideoPort)
    {
        case 0:
        yevent=CSL_EDMA3_CHA_VP0EVTYA;
        cbevent=CSL_EDMA3_CHA_VP0EVTUA;
        crevent=CSL_EDMA3_CHA_VP0EVTVA;
        break;

        case 1:
        yevent=CSL_EDMA3_CHA_VP1EVTYA;
        cbevent=CSL_EDMA3_CHA_VP1EVTUA;
        crevent=CSL_EDMA3_CHA_VP1EVTVA;
        break;

        case 2:
        yevent=CSL_EDMA3_CHA_VP2EVTYA;
        cbevent=CSL_EDMA3_CHA_VP2EVTUA;
        crevent=CSL_EDMA3_CHA_VP2EVTVA;
        break;

        case 3:
        yevent=CSL_EDMA3_CHA_VP3EVTYA;
        cbevent=CSL_EDMA3_CHA_VP3EVTUA;
        crevent=CSL_EDMA3_CHA_VP3EVTVA;
        break;

        case 4:
        yevent=CSL_EDMA3_CHA_VP4EVTYA;
        cbevent=CSL_EDMA3_CHA_VP4EVTUA;
        crevent=CSL_EDMA3_CHA_VP4EVTVA;
        break;

        default:
        printf ("Edma invalid video port.\n");    
        return;
    }

    /* Channel open */
    chAttr.regionNum = CSL_EDMA3_REGION_GLOBAL;
    chAttr.chaNum = yevent;
    edma3ChannelOpen(&chObjY, CSL_EDMA3, &chAttr, &hChannelY);
    
    /* Change Channel Default queue setup from 0 to 0  */
    edma3HwChannelSetupQue(hChannelY,CSL_EDMA3_QUE_0);

    /* Channel open */
    chAttr.regionNum = CSL_EDMA3_REGION_GLOBAL;
    chAttr.chaNum = cbevent;
    edma3ChannelOpen(&chObjCb, CSL_EDMA3, &chAttr, &hChannelCb );
    
    /* Change Channel Default queue setup from 0 to 1  */
    //edma3HwChannelSetupQue(hChannelCb,CSL_EDMA3_QUE_1);
    edma3HwChannelSetupQue(hChannelCb,CSL_EDMA3_QUE_0);

    /* Channel open */
    chAttr.regionNum = CSL_EDMA3_REGION_GLOBAL;
    chAttr.chaNum = crevent;
    edma3ChannelOpen(&chObjCr, CSL_EDMA3, &chAttr, &hChannelCr);
    
    /* Change Channel Default queue setup from 0 to 2  */
    //edma3HwChannelSetupQue(hChannelCr,CSL_EDMA3_QUE_2);
    edma3HwChannelSetupQue(hChannelCr,CSL_EDMA3_QUE_0);

    /* Obtain a handle to parameter set 0 */
    hParamBasicY = edma3GetParamHandle(hChannelY,yevent);
    
    /* Obtain a handle to parameter set 3 */
    hParamCpyY = edma3GetParamHandle(hChannelY,yevent+64);
    
    /* Obtain a handle to parameter set 1 */
    hParamBasicCb = edma3GetParamHandle(hChannelCb,cbevent);
 
    /* Obtain a handle to parameter set 4 */
    hParamCpyCb = edma3GetParamHandle(hChannelCb,cbevent+64);
 
    /* Obtain a handle to parameter set 2 */
    hParamBasicCr = edma3GetParamHandle(hChannelCr,crevent);
    
    /* Obtain a handle to parameter set 5 */
    hParamCpyCr = edma3GetParamHandle(hChannelCr,crevent+64);

    // Configure parameter set for display edma
    ConfigureEDMA3VPDispParamSet(YBuff,CbBuff,CrBuff);

    /* Interrupt enable (Bits 0-1)  for the global region interrupts */
    regionIntr.region =  CSL_EDMA3_REGION_GLOBAL;
    if (yevent<32)
        regionIntr.intr  |=  (1 << yevent);
    else
        regionIntr.intrh |=  (1 << (yevent-32));
    if (cbevent<32)
        regionIntr.intr  |=  (1 << cbevent);
    else
        regionIntr.intrh |=  (1 << (cbevent-32));
    if (crevent<32)
        regionIntr.intr  |=  (1 << crevent);
    else
        regionIntr.intrh |=  (1 << (crevent-32));
    edma3CmdIntrEnable(hModule,regionIntr.intr,regionIntr.intrh);

    // Get Channel Error Status 
    edma3GetChannelErrStatus(hChannelY,&myErrorsY);
    edma3GetChannelErrStatus(hChannelCb,&myErrorsCb);
    edma3GetChannelErrStatus(hChannelCr,&myErrorsCr);

    // Clear edma errors...
    edma3ChannelErrorClear(hChannelY,&myErrorsY);
    edma3ChannelErrorClear(hChannelCb,&myErrorsCb);
    edma3ChannelErrorClear(hChannelCr,&myErrorsCr);

    // Enable Channels(if the channel is meant for external event) 
    // This step is not required if the channel is chained to or manually triggered.
    edma3ChannelEnable(hChannelY);
    edma3ChannelEnable(hChannelCb);
    edma3ChannelEnable(hChannelCr);
}

/**************************************************************************** 
 *
 * NAME  
 *      ConfigureEDMA3VPDispParamSet
 *
 * PURPOSE:
 *      Configure paramset for the EDMA3 for video port event display mode.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      void ConfigureEDMA3VPDispParamSet(Uint8 YBuff[],Uint8 CbBuff[] ,Uint8 CrBuff[])
 *
 * RETURN VALUE
 *      None
 *
 * REFERENCE
 *
 ****************************************************************************/
void ConfigureEDMA3VPDispParamSet(Uint8 *YBuff,Uint8 *CbBuff ,Uint8 *CrBuff)
{
    Int yevent,cbevent,crevent,VideoPort=1;

    // Gets video port events related to edma
    switch(VideoPort)
    {
        case 0:
        yevent=CSL_EDMA3_CHA_VP0EVTYA;
        cbevent=CSL_EDMA3_CHA_VP0EVTUA;
        crevent=CSL_EDMA3_CHA_VP0EVTVA;
        break;

        case 1:
        yevent=CSL_EDMA3_CHA_VP1EVTYA;
        cbevent=CSL_EDMA3_CHA_VP1EVTUA;
        crevent=CSL_EDMA3_CHA_VP1EVTVA;
        break;

        case 2:
        yevent=CSL_EDMA3_CHA_VP2EVTYA;
        cbevent=CSL_EDMA3_CHA_VP2EVTUA;
        crevent=CSL_EDMA3_CHA_VP2EVTVA;
        break;

        case 3:
        yevent=CSL_EDMA3_CHA_VP3EVTYA;
        cbevent=CSL_EDMA3_CHA_VP3EVTUA;
        crevent=CSL_EDMA3_CHA_VP3EVTVA;
        break;

        case 4:
        yevent=CSL_EDMA3_CHA_VP4EVTYA;
        cbevent=CSL_EDMA3_CHA_VP4EVTUA;
        crevent=CSL_EDMA3_CHA_VP4EVTVA;
        break;

        default:
        printf ("Edma invalid video port.\n");    
        return;
    }

    /* Setup the parameter entry parameters (Y buffer) */
    myParamSetup.option = EDMA3_OPT_MAKE(EDMA3_ITCCH_DIS, \
                                         EDMA3_TCCH_DIS, \
                                         EDMA3_ITCINT_DIS, \
                                         EDMA3_TCINT_EN, \
                                         yevent,\
                                         EDMA3_TCC_NORMAL,\
                                         EDMA3_FIFOWIDTH_NONE, \
                                         EDMA3_STATIC_DIS, \
                                         EDMA3_SYNC_AB, \
                                         EDMA3_ADDRMODE_INCR, \
                                         EDMA3_ADDRMODE_INCR );
    myParamSetup.srcAddr = (Uint32)YBuff;         
    myParamSetup.aCntbCnt = EDMA3_CNT_MAKE(8,NPIXEL/8);       
    myParamSetup.dstAddr = (Uint32)YDSTVP1;        
    myParamSetup.srcDstBidx = EDMA3_BIDX_MAKE(8,0);     
    myParamSetup.linkBcntrld = EDMA3_LINKBCNTRLD_MAKE(hParamCpyY,0);     
    myParamSetup.srcDstCidx = EDMA3_CIDX_MAKE(NPIXEL,0);
    myParamSetup.cCnt = NLINES;
    edma3ParamSetup(hParamBasicY,&myParamSetup);
    edma3ParamSetup(hParamCpyY,&myParamSetup);

    /* Setup the parameter entry parameters (Cb buffer) */
    myParamSetup.option = EDMA3_OPT_MAKE(EDMA3_ITCCH_DIS, \
                                         EDMA3_TCCH_DIS, \
                                         EDMA3_ITCINT_DIS, \
                                         EDMA3_TCINT_EN, \
                                         cbevent,\
                                         EDMA3_TCC_NORMAL,\
                                         EDMA3_FIFOWIDTH_NONE, \
                                         EDMA3_STATIC_DIS, \
                                         EDMA3_SYNC_AB, \
                                         EDMA3_ADDRMODE_INCR, \
                                         EDMA3_ADDRMODE_INCR );
    myParamSetup.srcAddr = (Uint32)CbBuff;         
    myParamSetup.aCntbCnt = EDMA3_CNT_MAKE(8,NPIXEL/2/8);       
    myParamSetup.dstAddr = (Uint32)CBDSTVP1;        
    myParamSetup.srcDstBidx = EDMA3_BIDX_MAKE(8,0);     
    myParamSetup.linkBcntrld = EDMA3_LINKBCNTRLD_MAKE(hParamCpyCb,0);     
    myParamSetup.srcDstCidx = EDMA3_CIDX_MAKE(NPIXEL/2,0);     
    myParamSetup.cCnt = NLINES;
    edma3ParamSetup(hParamBasicCb,&myParamSetup);
    edma3ParamSetup(hParamCpyCb,&myParamSetup);

    /* Setup the parameter entry parameters (Cr buffer) */
    myParamSetup.option = EDMA3_OPT_MAKE(EDMA3_ITCCH_DIS, \
                                         EDMA3_TCCH_DIS, \
                                         EDMA3_ITCINT_DIS, \
                                         EDMA3_TCINT_EN, \
                                         crevent,\
                                         EDMA3_TCC_NORMAL,\
                                         EDMA3_FIFOWIDTH_NONE, \
                                         EDMA3_STATIC_DIS, \
                                         EDMA3_SYNC_AB, \
                                         EDMA3_ADDRMODE_INCR, \
                                         EDMA3_ADDRMODE_INCR );
    myParamSetup.srcAddr = (Uint32)CrBuff;         
    myParamSetup.aCntbCnt = EDMA3_CNT_MAKE(8,NPIXEL/2/8);       
    myParamSetup.dstAddr = (Uint32)CRDSTVP1;        
    myParamSetup.srcDstBidx = EDMA3_BIDX_MAKE(8,0);     
    myParamSetup.linkBcntrld = EDMA3_LINKBCNTRLD_MAKE(hParamCpyCr,0);     
    myParamSetup.srcDstCidx = EDMA3_CIDX_MAKE(NPIXEL/2,0);     
    myParamSetup.cCnt = NLINES;
    edma3ParamSetup(hParamBasicCr,&myParamSetup);
    edma3ParamSetup(hParamCpyCr,&myParamSetup);
}    

/**************************************************************************** 
 *
 * NAME  
 *      ConfigureEDMA3VPCapt
 *
 * PURPOSE:
 *      Configure the EDMA3 for video port event capture mode.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      void ConfigureEDMA3VPCapt(Uint8 *YBuff,Uint8 *CbBuff ,Uint8 *CrBuff, Int16 VideoPort, Int16 VideoChan)
 *
 * RETURN VALUE
 *      None
 *
 * REFERENCE
 *
 ****************************************************************************/
void ConfigureEDMA3VPCapt(Uint8 *YBuff,Uint8 *CbBuff ,Uint8 *CrBuff, Int16 VideoPort, Int16 VideoChan)
{
    Int yevent,cbevent,crevent;
    Int vpchan;

    if (VideoPort<0 || VideoPort>4 || (VideoChan!=1 && VideoChan!=2))
    {
        printf ("Wrong video port or channel!\n");    
        return;
    }

    vpchan=VideoPort*2+(VideoChan-1);

    // Gets video port/channel events related to edma
    switch(vpchan)
    {
        case 0:
        yevent=CSL_EDMA3_CHA_VP0EVTYA;
        cbevent=CSL_EDMA3_CHA_VP0EVTUA;
        crevent=CSL_EDMA3_CHA_VP0EVTVA;
        break;

        case 1:
        yevent=CSL_EDMA3_CHA_VP0EVTYB;
        cbevent=CSL_EDMA3_CHA_VP0EVTUB;
        crevent=CSL_EDMA3_CHA_VP0EVTVB;
        break;

        case 2:
        yevent=CSL_EDMA3_CHA_VP1EVTYA;
        cbevent=CSL_EDMA3_CHA_VP1EVTUA;
        crevent=CSL_EDMA3_CHA_VP1EVTVA;
        break;

        case 3:
        yevent=CSL_EDMA3_CHA_VP1EVTYB;
        cbevent=CSL_EDMA3_CHA_VP1EVTUB;
        crevent=CSL_EDMA3_CHA_VP1EVTVB;
        break;

        case 4:
        yevent=CSL_EDMA3_CHA_VP2EVTYA;
        cbevent=CSL_EDMA3_CHA_VP2EVTUA;
        crevent=CSL_EDMA3_CHA_VP2EVTVA;
        break;

        case 5:
        yevent=CSL_EDMA3_CHA_VP2EVTYB;
        cbevent=CSL_EDMA3_CHA_VP2EVTUB;
        crevent=CSL_EDMA3_CHA_VP2EVTVB;
        break;

        case 6:
        yevent=CSL_EDMA3_CHA_VP3EVTYA;
        cbevent=CSL_EDMA3_CHA_VP3EVTUA;
        crevent=CSL_EDMA3_CHA_VP3EVTVA;
        break;

        case 7:
        yevent=CSL_EDMA3_CHA_VP3EVTYB;
        cbevent=CSL_EDMA3_CHA_VP3EVTUB;
        crevent=CSL_EDMA3_CHA_VP3EVTVB;
        break;

        case 8:
        yevent=CSL_EDMA3_CHA_VP4EVTYA;
        cbevent=CSL_EDMA3_CHA_VP4EVTUA;
        crevent=CSL_EDMA3_CHA_VP4EVTVA;
        break;

        case 9:
        yevent=CSL_EDMA3_CHA_VP4EVTYB;
        cbevent=CSL_EDMA3_CHA_VP4EVTUB;
        crevent=CSL_EDMA3_CHA_VP4EVTVB;
        break;

        default:
        printf ("Edma invalid video port.\n");    
        return;
    }

    /* Channel open */
    chAttr.regionNum = CSL_EDMA3_REGION_GLOBAL;
    chAttr.chaNum = yevent;
    edma3ChannelOpen(&chObjYCapt, CSL_EDMA3, &chAttr, &hChannelYCapt);
    
    /* Change Channel Default queue setup from 0 to 0  */
    //edma3HwChannelSetupQue(hChannelYCapt,CSL_EDMA3_QUE_0);
    edma3HwChannelSetupQue(hChannelYCapt,CSL_EDMA3_QUE_1);

    /* Channel open */
    chAttr.regionNum = CSL_EDMA3_REGION_GLOBAL;
    chAttr.chaNum = cbevent;
    edma3ChannelOpen(&chObjCbCapt, CSL_EDMA3, &chAttr, &hChannelCbCapt);
    
    /* Change Channel Default queue setup from 0 to 1  */
    //edma3HwChannelSetupQue(hChannelCbCapt,CSL_EDMA3_QUE_1);
    edma3HwChannelSetupQue(hChannelCbCapt,CSL_EDMA3_QUE_1);

    /* Channel open */
    chAttr.regionNum = CSL_EDMA3_REGION_GLOBAL;
    chAttr.chaNum = crevent;
    edma3ChannelOpen(&chObjCrCapt, CSL_EDMA3, &chAttr, &hChannelCrCapt);
    
    /* Change Channel Default queue setup from 0 to 2  */
    //edma3HwChannelSetupQue(hChannelCrCapt,CSL_EDMA3_QUE_2);
    edma3HwChannelSetupQue(hChannelCrCapt,CSL_EDMA3_QUE_1);

    /* Obtain a handle to parameter set 0 */
    hParamBasicYCapt = edma3GetParamHandle(hChannelYCapt,yevent);
    
    /* Obtain a handle to parameter set 3 */
    hParamCpyYCapt = edma3GetParamHandle(hChannelYCapt,yevent+64);
    
    /* Obtain a handle to parameter set 1 */
    hParamBasicCbCapt = edma3GetParamHandle(hChannelCbCapt,cbevent);
 
    /* Obtain a handle to parameter set 4 */
    hParamCpyCbCapt = edma3GetParamHandle(hChannelCbCapt,cbevent+64);
 
    /* Obtain a handle to parameter set 2 */
    hParamBasicCrCapt = edma3GetParamHandle(hChannelCrCapt,crevent);
    
    /* Obtain a handle to parameter set 5 */
    hParamCpyCrCapt = edma3GetParamHandle(hChannelCrCapt,crevent+64);

    // Configure edma paramset for video capt
    ConfigureEDMA3VPCaptParamSet(YBuff,CbBuff,CrBuff,VideoPort,VideoChan);

    /* Interrupt enable (Bits 0-1)  for the global region interrupts */
    regionIntr.region =  CSL_EDMA3_REGION_GLOBAL  ;   
    if (yevent<32)
        regionIntr.intr  |=  (1 << yevent);
    else
        regionIntr.intrh |=  (1 << (yevent-32));
    if (cbevent<32)
        regionIntr.intr  |=  (1 << cbevent);
    else
        regionIntr.intrh |=  (1 << (cbevent-32));
    if (crevent<32)
        regionIntr.intr  |=  (1 << crevent);
    else
        regionIntr.intrh |=  (1 << (crevent-32));
    edma3CmdIntrEnable(hModule,regionIntr.intr,regionIntr.intrh);

    // Get Channel Error Status 
    edma3GetChannelErrStatus(hChannelYCapt,&myErrorsYCapt);
    edma3GetChannelErrStatus(hChannelCbCapt,&myErrorsCbCapt);
    edma3GetChannelErrStatus(hChannelCrCapt,&myErrorsCrCapt);

    // Clear edma errors...
    edma3ChannelErrorClear(hChannelYCapt,&myErrorsYCapt);
    edma3ChannelErrorClear(hChannelCbCapt,&myErrorsCbCapt);
    edma3ChannelErrorClear(hChannelCrCapt,&myErrorsCrCapt);

    // Enable Channels(if the channel is meant for external event) 
    // This step is not required if the channel is chained to or manually triggered.
    edma3ChannelEnable(hChannelYCapt);
    edma3ChannelEnable(hChannelCbCapt);
    edma3ChannelEnable(hChannelCrCapt);
}

/**************************************************************************** 
 *
 * NAME  
 *      ConfigureEDMA3VPCaptParamSet
 *
 * PURPOSE:
 *      Configure paramset for the EDMA3 for video port event display mode.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      void ConfigureEDMA3VPCaptParamSet(Uint8 *YBuff,Uint8 *CbBuff ,
 *                                        Uint8 *CrBuff, Int16 VideoPort, Int16 VideoChan)
 *
 * RETURN VALUE
 *      None
 *
 * REFERENCE
 *
 ****************************************************************************/
void ConfigureEDMA3VPCaptParamSet(Uint8 *YBuff,Uint8 *CbBuff ,Uint8 *CrBuff, Int16 VideoPort, Int16 VideoChan)
{
    Int yevent,cbevent,crevent;
    Int ysrc,cbsrc,crsrc;
    Int vpchan;

    if (VideoPort<0 || VideoPort>4 || (VideoChan!=1 && VideoChan!=2))
    {
        printf ("Wrong video port or channel!\n");    
        return;
    }

    vpchan=VideoPort*2+(VideoChan-1);

    // Gets video port/channel events related to edma
    switch(vpchan)
    {
        case 0:
        yevent=CSL_EDMA3_CHA_VP0EVTYA;
        cbevent=CSL_EDMA3_CHA_VP0EVTUA;
        crevent=CSL_EDMA3_CHA_VP0EVTVA;
        ysrc=YSRCVP0A;
        cbsrc=CBSRCVP0A;
        crsrc=CRSRCVP0A;
        break;

        case 1:
        yevent=CSL_EDMA3_CHA_VP0EVTYB;
        cbevent=CSL_EDMA3_CHA_VP0EVTUB;
        crevent=CSL_EDMA3_CHA_VP0EVTVB;
        ysrc=YSRCVP0B;
        cbsrc=CBSRCVP0B;
        crsrc=CRSRCVP0B;
        break;

        case 2:
        yevent=CSL_EDMA3_CHA_VP1EVTYA;
        cbevent=CSL_EDMA3_CHA_VP1EVTUA;
        crevent=CSL_EDMA3_CHA_VP1EVTVA;
        ysrc=YSRCVP1A;
        cbsrc=CBSRCVP1A;
        crsrc=CRSRCVP1A;
        break;

        case 3:
        yevent=CSL_EDMA3_CHA_VP1EVTYB;
        cbevent=CSL_EDMA3_CHA_VP1EVTUB;
        crevent=CSL_EDMA3_CHA_VP1EVTVB;
        ysrc=YSRCVP1B;
        cbsrc=CBSRCVP1B;
        crsrc=CRSRCVP1B;
        break;

        case 4:
        yevent=CSL_EDMA3_CHA_VP2EVTYA;
        cbevent=CSL_EDMA3_CHA_VP2EVTUA;
        crevent=CSL_EDMA3_CHA_VP2EVTVA;
        ysrc=YSRCVP2A;
        cbsrc=CBSRCVP2A;
        crsrc=CRSRCVP2A;
        break;

        case 5:
        yevent=CSL_EDMA3_CHA_VP2EVTYB;
        cbevent=CSL_EDMA3_CHA_VP2EVTUB;
        crevent=CSL_EDMA3_CHA_VP2EVTVB;
        ysrc=YSRCVP2B;
        cbsrc=CBSRCVP2B;
        crsrc=CRSRCVP2B;
        break;

        case 6:
        yevent=CSL_EDMA3_CHA_VP3EVTYA;
        cbevent=CSL_EDMA3_CHA_VP3EVTUA;
        crevent=CSL_EDMA3_CHA_VP3EVTVA;
        ysrc=YSRCVP3A;
        cbsrc=CBSRCVP3A;
        crsrc=CRSRCVP3A;
        break;

        case 7:
        yevent=CSL_EDMA3_CHA_VP3EVTYB;
        cbevent=CSL_EDMA3_CHA_VP3EVTUB;
        crevent=CSL_EDMA3_CHA_VP3EVTVB;
        ysrc=YSRCVP3B;
        cbsrc=CBSRCVP3B;
        crsrc=CRSRCVP3B;
        break;

        case 8:
        yevent=CSL_EDMA3_CHA_VP4EVTYA;
        cbevent=CSL_EDMA3_CHA_VP4EVTUA;
        crevent=CSL_EDMA3_CHA_VP4EVTVA;
        ysrc=YSRCVP4A;
        cbsrc=CBSRCVP4A;
        crsrc=CRSRCVP4A;
        break;

        case 9:
        yevent=CSL_EDMA3_CHA_VP4EVTYB;
        cbevent=CSL_EDMA3_CHA_VP4EVTUB;
        crevent=CSL_EDMA3_CHA_VP4EVTVB;
        ysrc=YSRCVP4B;
        cbsrc=CBSRCVP4B;
        crsrc=CRSRCVP4B;
        break;

        default:
        printf ("Edma invalid video port.\n");    
        return;
    }

    /* Setup the parameter entry parameters (Y buffer) */
    myParamSetup.option = EDMA3_OPT_MAKE(EDMA3_ITCCH_DIS, \
                                         EDMA3_TCCH_DIS, \
                                         EDMA3_ITCINT_DIS, \
                                         EDMA3_TCINT_EN, \
                                         yevent,\
                                         EDMA3_TCC_NORMAL,\
                                         EDMA3_FIFOWIDTH_NONE, \
                                         EDMA3_STATIC_DIS, \
                                         EDMA3_SYNC_AB, \
                                         EDMA3_ADDRMODE_INCR, \
                                         EDMA3_ADDRMODE_INCR );
    myParamSetup.srcAddr = ysrc;         
    myParamSetup.aCntbCnt = EDMA3_CNT_MAKE(8,NPIXEL/8);       
    myParamSetup.dstAddr = (Uint32)YBuff;        
    myParamSetup.srcDstBidx = EDMA3_BIDX_MAKE(0,8);     
    myParamSetup.linkBcntrld = EDMA3_LINKBCNTRLD_MAKE(hParamCpyYCapt,0);     
    myParamSetup.srcDstCidx = EDMA3_CIDX_MAKE(0,NPIXEL);
    myParamSetup.cCnt = NLINES;
    edma3ParamSetup(hParamBasicYCapt,&myParamSetup);
    edma3ParamSetup(hParamCpyYCapt,&myParamSetup);

    /* Setup the parameter entry parameters (Cb buffer) */
    myParamSetup.option = EDMA3_OPT_MAKE(EDMA3_ITCCH_DIS, \
                                         EDMA3_TCCH_DIS, \
                                         EDMA3_ITCINT_DIS, \
                                         EDMA3_TCINT_EN, \
                                         cbevent,\
                                         EDMA3_TCC_NORMAL,\
                                         EDMA3_FIFOWIDTH_NONE, \
                                         EDMA3_STATIC_DIS, \
                                         EDMA3_SYNC_AB, \
                                         EDMA3_ADDRMODE_INCR, \
                                         EDMA3_ADDRMODE_INCR );
    myParamSetup.srcAddr = cbsrc;         
    myParamSetup.aCntbCnt = EDMA3_CNT_MAKE(8,NPIXEL/2/8);       
    myParamSetup.dstAddr = (Uint32)CbBuff;        
    myParamSetup.srcDstBidx = EDMA3_BIDX_MAKE(0,8);     
    myParamSetup.linkBcntrld = EDMA3_LINKBCNTRLD_MAKE(hParamCpyCbCapt,0);     
    myParamSetup.srcDstCidx = EDMA3_CIDX_MAKE(0,NPIXEL/2);     
    myParamSetup.cCnt = NLINES;
    edma3ParamSetup(hParamBasicCbCapt,&myParamSetup);
    edma3ParamSetup(hParamCpyCbCapt,&myParamSetup);

    /* Setup the parameter entry parameters (Cr buffer) */
    myParamSetup.option = EDMA3_OPT_MAKE(EDMA3_ITCCH_DIS, \
                                         EDMA3_TCCH_DIS, \
                                         EDMA3_ITCINT_DIS, \
                                         EDMA3_TCINT_EN, \
                                         crevent,\
                                         EDMA3_TCC_NORMAL,\
                                         EDMA3_FIFOWIDTH_NONE, \
                                         EDMA3_STATIC_DIS, \
                                         EDMA3_SYNC_AB, \
                                         EDMA3_ADDRMODE_INCR, \
                                         EDMA3_ADDRMODE_INCR );
    myParamSetup.srcAddr = crsrc;
    myParamSetup.aCntbCnt = EDMA3_CNT_MAKE(8,NPIXEL/2/8);       
    myParamSetup.dstAddr = (Uint32)CrBuff;        
    myParamSetup.srcDstBidx = EDMA3_BIDX_MAKE(0,8);     
    myParamSetup.linkBcntrld = EDMA3_LINKBCNTRLD_MAKE(hParamCpyCrCapt,0);     
    myParamSetup.srcDstCidx = EDMA3_CIDX_MAKE(0,NPIXEL/2);     
    myParamSetup.cCnt = NLINES;
    edma3ParamSetup(hParamBasicCrCapt,&myParamSetup);
    edma3ParamSetup(hParamCpyCrCapt,&myParamSetup);
}


/**************************************************************************** 
 *
 * NAME  
 *      ConfigureEDMA3VPStop
 *
 * PURPOSE:
 *      Stop edma3 capture and display channel activity.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      void ConfigureEDMA3VPStop(void)
 *
 * RETURN VALUE
 *      None
 *
 * REFERENCE
 *
 ****************************************************************************/
static void ConfigureEDMA3VPStop(void)
{
    // Disable Capture channels
    edma3ChannelDisable(hChannelYCapt);
    edma3ChannelDisable(hChannelCbCapt);
    edma3ChannelDisable(hChannelCrCapt);

    // Disable Display Channels 
    edma3ChannelDisable(hChannelY);
    edma3ChannelDisable(hChannelCb);
    edma3ChannelDisable(hChannelCr);

    /* Disable all interrupts that were used */
    edma3InterruptDisable(hModule,regionIntr.intr,regionIntr.intrh);
}


/* ======================================================================== */
/*  NAME                                                                    */
/*      double_word_byte_swap -- byte swap of a double word                 */
/*S                                                                        S*/
/*S AUTHOR                                                                 S*/
/*S     Pavan Shastry                                                      S*/
/*S                                                                        S*/
/*S REVISION HISTORY                                                       S*/
/*S     25-Apr-2007 Initial version of double_word_byte_swap .... P.ShastryS*/
/*S     08-May-2007 Correction of half buffer was only processed. Lyrtech  S*/
/*S                                                                        S*/
/*                                                                          */
/*  USAGE                                                                   */
/*      This routine has the following C prototype:                         */
/*                                                                          */
/*      void double_word_byte_swap                                          */
/*      (                                                                   */
/*          unsigned char *restrict   buffer_ptr,                           */
/*          unsigned int              no_of_bytes,                          */
/*      )                                                                   */
/*                                                                          */
/*      The double_word_byte_swap routine accepts a buffer of size          */
/*      no_of_bytes bytes. It does a byte swap of double word.              */
/*      See description below for more details.                             */
/*                                                                          */
/*  DESCRIPTION                                                             */
/*      This code is written to perform a byte swap operation of            */
/*      a double word, in the manner indicated below -                      */
/*                                                                          */
/*      Input  Double word: A7 A6 A5 A4  A3 A2 A1 A0                        */
/*      Output Double word: A0 A1 A2 A3  A4 A5 A6 A7                        */
/*                                                                          */
/*      The Computation is in-place. The input buffer is written with       */
/*      bytes swapped in the above manner.                                  */
/*                                                                          */
/*      The number of double words, for which byte swapping will            */
/*      be performed is equal to (no_of_bytes >> 3).                        */
/*                                                                          */
/*  ASSUMPTIONS                                                             */
/*      'no_of_bytes' is a multiple of 8.                                   */
/*                                                                          */
/*  NOTES                                                                   */
/*      1. If 'no_of_bytes' is not a multiple of 8, then largest            */
/*         multiple of 8, which is lesser than no_of_bytes will be          */
/*         assumed.                                                         */
/*      2. For best performance of this routine, use -o3 and No debug       */
/*         as compile options.                                              */ 
/*                                                                          */
/*  SOURCE                                                                  */
/*      None.                                                               */
/* ------------------------------------------------------------------------ */
/*            Copyright (c) 2007 Texas Instruments, Incorporated.           */
/*                           All Rights Reserved.                           */
/* ======================================================================== */
static void Double_Word_Byte_Swap(unsigned char *buffer_ptr, unsigned int no_of_bytes)
{
   
   double * restrict read_ptr;
   double * restrict write_ptr;
   double input_dword;
   double output_word;
   unsigned int word0;
   unsigned int word1; 
   unsigned int dword_cntr;
   unsigned int dword_total_cnt;
   
   read_ptr  = (double *)buffer_ptr;
   write_ptr = (double *)buffer_ptr;
   
   /* ------------------------------------------------------------- */
   /* No of double words to be processed                            */
   /* ------------------------------------------------------------- */
   dword_total_cnt  = no_of_bytes >> 3;

   /* ------------------------------------------------------------- */
   /* Each iteration handles two double words                       */
   /* ------------------------------------------------------------- */
   for(dword_cntr = 0; dword_cntr < dword_total_cnt; dword_cntr+=2)
   {
       input_dword  = read_ptr[dword_cntr];

       word0        = _lo(input_dword);
       word1        = _hi(input_dword);
       word0        = _swap4(word0);
       word1        = _swap4(word1);
       word0        = _rotl(word0,16);
       word1        = _rotl(word1,16);
       output_word  = _itod(word0, word1);
       
       write_ptr[dword_cntr] = output_word;

       input_dword  = read_ptr[dword_cntr+1];

       word0        = _lo(input_dword);
       word1        = _hi(input_dword);
       word0        = _swap4(word0);
       word1        = _swap4(word1);
       word0        = _rotl(word0,16);
       word1        = _rotl(word1,16);
       output_word  = _itod(word0, word1);
       
       write_ptr[dword_cntr+1] = output_word;
   }
   
   /* ------------------------------------------------------------- */
   /* If the number of double words is odd, then the last double    */
   /* word is handled here.                                         */
   /* ------------------------------------------------------------- */
   if(dword_total_cnt & 0x01)
   {
       input_dword  = read_ptr[dword_cntr];

       word0        = _lo(input_dword);
       word1        = _hi(input_dword);
       word0        = _swap4(word0);
       word1        = _swap4(word1);
       word0        = _rotl(word0,16);
       word1        = _rotl(word1,16);
       output_word  = _itod(word0, word1);
       
       write_ptr[dword_cntr] = output_word;
   }  
}

/**************************************************************************** 
 *
 * NAME  
 *      GenerateYCbCr422ColorBar
 *
 * PURPOSE:
 *      Generate a YCbCr 4:2:2 color bar pattern in buffers.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      void GenerateYCbCr422ColorBar(Uint16 ncolor, Uint8 y[],Uint8 cb[],Uint8 cr[])
 *
 * RETURN VALUE
 *      None
 *
 * REFERENCE
 *
 ****************************************************************************/
static void GenerateYCbCr422ColorBar(Uint16 ncolor, Uint8 y[],Uint8 cb[],Uint8 cr[])
{
    Int16 count;

    // Generate one line of the color bar pattern 4:2:2 in Ycbcr color space
    for (count=0;count<ncolor;count++)
    {
        switch(count/45)
        {
            // White 235:128:128
            case 0:
                y[count*2]=235;
                y[count*2+1]=235;
                cb[count]=128;
                cr[count]=128;
                break;

            // Yellow 210:16:146
            case 1:
                y[count*2]=210;
                y[count*2+1]=210;
                cb[count]=16;
                cr[count]=146;
                break;

            // Cyan 170:166:16
            case 2:
                y[count*2]=170;
                y[count*2+1]=170;
                cb[count]=166;
                cr[count]=16;
                break;

            // Green 145:54:34
            case 3:
                y[count*2]=145;
                y[count*2+1]=145;
                cb[count]=54;
                cr[count]=34;
                break;

            // Magenta 107:202:222
            case 4:
                y[count*2]=107;
                y[count*2+1]=107;
                cb[count]=202;
                cr[count]=222;
                break;

            // Red 82:90:240
            case 5:
                y[count*2]=82;
                y[count*2+1]=82;
                cb[count]=90;
                cr[count]=240;
                break;

            // Blue 41:240:110
            case 6:
                y[count*2]=41;
                y[count*2+1]=41;
                cb[count]=240;
                cr[count]=110;
        	    break;

            // Black 16:128:128
            case 7:
                y[count*2]=16;
                y[count*2+1]=16;
                cb[count]=128;
                cr[count]=128;
                break;
        }
    }
}
