/***************************************************************************
 *
 *
 *         **   **    **  ******  ********  ********  ********  **    **    
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ***************************************************************************
 *                                                                          
 *  Project     : EVM
 *  File        : EVM.h
 *  Description : Utility functions for EVM board configuration
 *
 *                      Copyright (c) Lyrtech inc. 2007                        
 *
 ***************************************************************************
 *                                                                                          
 * "$Revision: 1.6 $"
 * "$Date: 2007/09/18 14:45:44 $"
 *
 ***************************************************************************/

#ifndef __EVM_EVM_H__ 
#define __EVM_EVM_H__


/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include <tistdtypes.h>


/****************************************************************************
 *                             Global constants                             *
 ****************************************************************************/
// EVM BSL VERSION
#define EVM_BSLVERSION "V0.3.0.0"
 
// Evm functions error status codes 
// ---------------------------------------------------------------------------
#define EVM_INITOK            0
#define EVM_I2CFAIL           0x0501 // Failure on an I2C command
#define EVM_RESOURCEBUSY      0x0502 // Mux resource unavailable locked by other peripheral
#define EVM_INVALIDRESOURCE   0x0503 // Asked resource do not exist
#define EVM_READMUXERROR      0x0504 // ECP mux register cannot be read
#define EVM_WRITEMUXERROR     0x0505 // ECP mux register cannot be written
#define EVM_RESOURCENOTOWN    0x0506 // Peripheral do not own resource and cannot release it
#define EVM_RESOURCENOTLOCK   0x0507 // The resource is not locked, cannot be unlocked
#define EVM_MCASP_MUX_WR_FAIL 0x0508 // ECP MCASP mux register cannot be written
#define EVM_UART_CTRL_WR_FAIL 0x0509 // ECP UART ctrl register cannot be written
#define EVM_VIDEO_MUX_WR_FAIL 0x050A // ECP VIDEO mux control register cannot be written
#define EVM_UART_MUX_RD_FAIL  0x050B // ECP uart/spi control register cannot be read
#define EVM_INCORRECT_VERSION 0x050C // Incorrect binary bsl version not matching .h
// ---------------------------------------------------------------------------

// Evm resources allocation link codes. This allocates the DSP/EVM board
// resources link (and MUX) to a peripheral category. This will avoid allow
// exclusion for modules utilization (some modules/functions types cannot work at same
// time of others)
enum EVM_ResourcesLnk {     // board MUX res  DSP res      Details and utilization
EVM_AUDIOCH18_LNK=0,        // MCASP/DCC      MCASP        Audio codecs ch 1 - 8 (AIC33)
EVM_MCASPDCC_LNK=1,         // MCASP/DCC      MCASP        MCASP/Daughter Card Connector
EVM_UART_LNK=2,             // UART/SPI       UART         Uart for external RS232 serial port
EVM_SPI_LNK=3,              // UART/SPI       SPI          On board SPI socket or Daughter Card Connector SPI
EVM_FLASHMEM_LNK=4,         // VP3 VP4        EMIF         Flash memory access
EVM_SDVIDEOINCH14_LNK=5,    // VP3 VP4        VP3 VP4      SD video ch 1 - 4 in (P5154)
EVM_EMIFDCC_LNK=6,          // VP3 VP4        EMIF         EMIF on Daughter Card Connector
EVM_SDVIDEOINCH58_LNK=7,    // VP0 VP2        VP0 VP2      Standard ch 5 - 8 video in (P5154)
EVM_HDVIDEOIN_VLYNQ_LNK=8   // VP0 VP2        VP0 VLYNQ    HD video in (P7000) and VLYNQ on Daugther Card Connector
};

#define EVM_Resources enum EVM_ResourcesLnk


/****************************************************************************
 *                             Public Functions                             *
 ****************************************************************************/


/**************************************************************************** 
 *
 * NAME  
 *      EVM_init
 *
 * PURPOSE:
 *      Init the evm configuration and resources for modules. This function
 *      must be the first function always use before using any other function
 *      of the BSL.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_init(void)
 *
 * RETURN VALUE
 *      Status of initialization, EVM_INITOK if no problem.
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_init(void);


/**************************************************************************** 
 *
 * NAME  
 *      EVM_lockresource
 *
 * PURPOSE:
 *      Lock a mux/dsp resources for specific module category utilization.
 *      EVM_unlockresource must be used for the same amount of time
 *      that this function has been used to completly unlock the resource.
 *      Usually, this function is internally used by the different evm peripherals,
 *      but can also be used alone to configure MUX when BSL peripherals modules
 *      are not used.
 *      This function will allow multiple lock for different peripherals when
 *      possible. In this case, multiple unlock must be done to release the resource.
 *      A resource lock that fails may indicates that there is another peripheral
 *      that is using the same resource.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_lockresource(EVM_Resources res_code)
 *
 *      res_code - (i) resource code for the desired resource to lock:
 *       EVM_AUDIOCH18_LNK        Audio codecs ch 1 - 8 (AIC33)
 *       EVM_MCASPDCC_LNK         MCASP/Daughter Card Connector
 *       EVM_UART_LNK             Uart for external RS232 serial port
 *       EVM_SPI_LNK              On board SPI socket or Daughter Card Connector SPI
 *       EVM_FLASHMEM_LNK         Flash memory access
 *       EVM_SDVIDEOINCH14_LNK    SD video ch 1 - 4 in (P5154)
 *       EVM_EMIFDCC_LNK          EMIF on Daughter Card Connector
 *       EVM_SDVIDEOINCH58_LNK    Standard ch 5 - 8 video in (P5154)
 *       EVM_HDVIDEOIN_VLYNQ_LNK  HD video in (P7000) and VLYNQ on Daugther Card Connector
 *
 * RETURN VALUE
 *      Status of operation, EVM_INITOK if no problem.
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_lockresource(EVM_Resources res_code);


/**************************************************************************** 
 *
 * NAME  
 *      EVM_unlockresource
 *
 * PURPOSE:
 *      Unlock a mux/dsp resources for specific module category utilization.
 *      EVM_lockresource must be used for the same amount of time
 *      that this function has been used to completly unlock the resource.
 *      Usually, this function is internally used by the different evm peripherals,
 *      but can also be used alone to release MUX when BSL peripherals modules
 *      are not used.
 *      This function will allow multiple unlock of a resource that has been
 *      locked more than one time by the peripherals modules. To release the
 *      resource, it must be unlock the same number of time it was locked by
 *      EVM_lockresource.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_unlockresource(EVM_Resources res_code)
 *
 *      res_code - (i) resource code for the desired resource to unlock
 *       EVM_AUDIOCH18_LNK        Audio codecs ch 1 - 8 (AIC33)
 *       EVM_MCASPDCC_LNK         MCASP/Daughter Card Connector
 *       EVM_UART_LNK             Uart for external RS232 serial port
 *       EVM_SPI_LNK              On board SPI socket or Daughter Card Connector SPI
 *       EVM_FLASHMEM_LNK         Flash memory access
 *       EVM_SDVIDEOINCH14_LNK    SD video ch 1 - 4 in (P5154)
 *       EVM_EMIFDCC_LNK          EMIF on Daughter Card Connector
 *       EVM_SDVIDEOINCH58_LNK    Standard ch 5 - 8 video in (P5154)
 *       EVM_HDVIDEOIN_VLYNQ_LNK  HD video in (P7000) and VLYNQ on Daugther Card Connector
 *
 * RETURN VALUE
 *      Status of operation, EVM_INITOK if no problem.
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_unlockresource(EVM_Resources res_code);


/**************************************************************************** 
 *
 * NAME  
 *      EVM_bslversion
 *
 * PURPOSE:
 *      Get the evm bsl version. It can be used to display the binary library
 *      version of the bsl.
 *      See function EVM_checkbslversion.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int8 *EVM_bslversion(void)
 *
 * RETURN VALUE
 *      A constant character string of the bsl library version.
 *
 * REFERENCE
 *
 ****************************************************************************/
Int8 *EVM_bslversion(void);


/**************************************************************************** 
 *
 * NAME  
 *      EVM_checkbslversion
 *
 * PURPOSE:
 *      Verify that we have a correct binary bsl version matching .h definitions
 *      Call this function with the constant EVM_BSLVERSION as input
 *      parameters to get a version check status.
 *      EVM_BSLVERSION is in the evm.h file.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_checkbslversion(Int8 *expectedversion)
 *
 *      expectedversion - (i) must be EVM_BSLVERSION version from .h
 *
 * RETURN VALUE
 *      EVM_INITOK if no problem is found, !EVM_INITOK if problem
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_checkbslversion(Int8 *expectedversion);


#endif // __EVM_EVM_H__
