/***************************************************************************
 *
 *
 *         **   **    **  ******  ********  ********  ********  **    **    
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ***************************************************************************
 *                                                                          
 *  Project     : EVM
 *  File        : EVM_P5154.h
 *  Description : Utility functions for P5154 SD video capture
 *
 *                      Copyright (c) Lyrtech inc. 2007                        
 *
 ***************************************************************************
 *                                                                                          
 * "$Revision: 1.3 $"
 * "$Date: 2007/07/13 19:12:16 $"
 *
 ***************************************************************************/

#ifndef __EVM_P5154_H__
#define __EVM_P5154_H__

/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include <tistdtypes.h>


/****************************************************************************
 *                             Global constants                             *
 ****************************************************************************/
// P5154 functions error status codes 
// ---------------------------------------------------------------------------
#define EVM_P5154_OK                0
#define EVM_P5154_NHANDLE           0x3001  // Cannot get a handle for the device
#define EVM_P5154_DEVICEINUSE       0x3002  // Device is already in use
#define EVM_P5154_INVALIDID         0x3003  // Invalid video channel ID
#define EVM_P5154_ERRLOCKRESOURCE   0x3004  // Cannot lock mux resource for video channel
#define EVM_P5154_INVALIDHANDLE     0x3005  // Invalid handle given
#define EVM_P5154_ERRUNLOCKRESOURCE 0x3006  // Cannot unlock mux resource for video channel
#define EVM_P5154_I2CREADREG        0x3007  // Problem reading device I2C register
#define EVM_P5154_I2CWRITEOFFSET    0x3008  // Problem writing I2C register offset
#define EVM_P5154_I2CWRITEDECSELECT 0x3009  // Problem writing I2C device decoder select
#define EVM_P5154_I2CWRITEREG       0x300A  // Problem writing device I2C register
// ---------------------------------------------------------------------------

// Video channel definitions used as ID to get handle
#define EVM_P5154_CH1 0
#define EVM_P5154_CH2 1
#define EVM_P5154_CH3 2
#define EVM_P5154_CH4 3
#define EVM_P5154_CH5 4
#define EVM_P5154_CH6 5
#define EVM_P5154_CH7 6
#define EVM_P5154_CH8 7


// Can be changed to add additionnals reg in setup
// Number of Registers to setup
#define EVM_P5154_NREGS    100

// Index definitions for setup matrix
#define EVM_P5154_REG_IND  0
#define EVM_P5154_REG_VAL  1

// End of setup definition
#define EVM_P5154_ENDSETUP 0xFF


// P5154 default setups default registers structures (EVM_P5154_Setup)
// There must be max EVM_P5154_NREGS table entries in these setups.
// Last valid setup entry must be followed by {EVM_P5154_ENDSETUP,EVM_P5154_ENDSETUP}

// P5154 SDTV 480i setup BT656 output format
// Setup : 704x480x60Hz - 15.734khz - 13.5MHz interlaced (525i) NTSC
// Register format: {regoff, value}
#define EVM_P5154_SDTV480I_YCBCR704X480X60 {                                                                                                     \
    {                                                                                                                                            \
    {0x7F,0x00},  /* Restart TVP5154 uP                                                                                                        */\
    {0x03,0x0D},  /* Enable YCbCr, sync, control outputs                                                                                       */\
    {0x17,0x1B},  /* Disable scaler powerdown for ROM2.0 and enable SAV/EAV codes                                                              */\
    {0x15,0x81},  /* Enable Stable Sync mode                                                                                                   */\
    {EVM_P5154_ENDSETUP,EVM_P5154_ENDSETUP}                                                                                                      \
                  /* As a safety measure, this means end of setup                                                                              */\
    }             /*  do not remove and keep it last element                                                                                   */\
}


// P5154 SDTV 576i setup BT656 output format
// Setup : 704x576x50Hz - 15.625khz - 13.5MHz interlaced (576i) PAL
// Register format: {regoff, value}
#define EVM_P5154_SDTV576I_YCBCR704X576X50 {                                                                                                     \
    {                                                                                                                                            \
    {0x7F,0x00},  /* Restart TVP5154 uP                                                                                                        */\
    {0x03,0x0D},  /* Enable YCbCr, sync, control outputs                                                                                       */\
    {0x05,0x0C},  /* Enable Clock 2 output                                                                                                     */\
    {0x17,0x1B},  /* Disable scaler powerdown for ROM2.0 and enable SAV/EAV codes                                                              */\
    {0x15,0x81},  /* Enable Stable Sync mode                                                                                                   */\
    {EVM_P5154_ENDSETUP,EVM_P5154_ENDSETUP}                                                                                                      \
                  /* As a safety measure, this means end of setup                                                                              */\
    }             /*  do not remove and keep it last element                                                                                   */\
}


// Codec Handle
typedef Int32 EVM_P5154_Handle;

// Register structure
typedef struct
{
    Uint8 regs[EVM_P5154_NREGS][2];
} EVM_P5154_Setup;


/****************************************************************************
 *                             Public Functions                             *
 ****************************************************************************/


/**************************************************************************** 
 *
 * NAME  
 *      EVM_P5154_open
 *
 * PURPOSE:
 *      Open a P5154 SD video channel capture device and return a handle.
 *      At opening, the P5154 video channel is initialized according to
 *      the specified setup matrix. When the channel is not used, the device must
 *      be closed by EVM_P5154_close. 
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      EVM_P5154_Handle EVM_P5154_open( Uint32 id, EVM_P5154_Setup *setup )
 *
 *      id    - (i) id of desired video channel [EVM_P5154_CH1-EVM_P5154_CH8]
 *      setup - (i) video setup structure
 *
 * RETURN VALUE
 *      Handle value, >= EVM_P5154_NHANDLE in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
EVM_P5154_Handle EVM_P5154_open( Uint32 id, EVM_P5154_Setup *setup );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_P5154_close
 *
 * PURPOSE:
 *      Close a video device previously opened by EVM_P5154_open.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_P5154_close( EVM_P5154_Handle *P5154handle )
 *
 *      P5154handle - (i) handle of the video channel returned when opened.
 *
 * RETURN VALUE
 *      EVM_P5154_OK if OK, !=EVM_P5154_OK if problems
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_P5154_close( EVM_P5154_Handle *P5154handle );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_P5154_setup
 *
 * PURPOSE:
 *      Setup the current P5154 video channel with default register values.
 *      A setup structure is given for this purpose.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_P5154_setup( EVM_P5154_Handle P5154handle, EVM_P5154_Setup *setup )
 *
 *      P5154handle - (i) handle of the video channel returned when opened.
 *
 *      setup       - (i) setup data structure.
 *
 * RETURN VALUE
 *      EVM_P5154_OK if OK, !=EVM_P5154_OK if problems
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_P5154_setup( EVM_P5154_Handle P5154handle, EVM_P5154_Setup *setup );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_P5154_rget
 *
 * PURPOSE:
 *      Read a specified register of the current P5154 video decoder handle.
 *      The register will be read from the decoder associated with the current
 *      handle, at the specified offset position.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_P5154_rget( EVM_P5154_Handle P5154handle, Uint16 regnum, Uint16* regval )
 *
 *      P5154handle - (i) handle of the P5154 video channel returned when opened.
 *
 *      regnum      - (i) register offset (8 bits).
 *
 *      regval      - (o) register value read (8 bits).
 *
 * RETURN VALUE
 *      EVM_P5154_OK if OK, !=EVM_P5154_OK if problems
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_P5154_rget( EVM_P5154_Handle P5154handle,  Uint16 regnum, Uint16* regval );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_P5154_rgetset
 *
 * PURPOSE:
 *      Read a  register, change a bitfield and then writeback the register.
 *      Bitmask values and bitval are binary value with 1 set at the bit position
 *      to modify in the register. It will work on the current video channel 
 *      decoder for the specified handle.
 *      Pseudocode operation:
 *       1- Value1 = Read P5154 channel register regnum of P5154handle
 *       2- Value1 = Value1 & (~bitmask)
 *       3- Value1 = Value1 | (bitmask & bitval)
 *       4- Write P5154 channel register regnum of P5154handle with Value1
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_P5154_rgetset( EVM_P5154_Handle P5154handle, Uint16 regnum, Uint16 bitmask, Uint16 bitval )
 *
 *      P5154handle - (i) handle of the P5154 video channel returned when opened.
 *
 *      regnum      - (i) register offset (8 bits).
 *
 *      bitmask     - (i) bit mask value (8 bits) with bit to keep set as 1.
 *
 *      bitval      - (i) bit values (8 bits) to be set at mask position.
 *
 * RETURN VALUE
 *      EVM_P5154_OK if OK, !=EVM_P5154_OK if problems
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_P5154_rgetset( EVM_P5154_Handle P5154handle, Uint16 regnum, Uint16 bitmask, Uint16 bitval );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_P5154_rset
 *
 * PURPOSE:
 *      Write a register with a specified value for the decoder of this handle.
 *      The register will be written from the decoder associated with the current
 *      handle, at the specified offset position.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_P5154_rset( EVM_P5154_Handle P5154handle, Uint16 regnum, Uint16 regval )
 *
 *      P5154handle - (i) handle of the P5154 video channel returned when opened.
 *
 *      regnum      - (i) register offset (8 bits).
 *
 *      regval      - (i) register value to be written (8 bits).
 *
 * RETURN VALUE
 *      EVM_P5154_OK if OK, !=EVM_P5154_OK if problems
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_P5154_rset( EVM_P5154_Handle P5154handle, Uint16 regnum, Uint16 regval );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_P5154_gvideoport
 *
 * PURPOSE:
 *      Returns the DSP video port and channels that is connected to the current
 *      video decoder of this P5154 handle. This function is used to identify
 *      which DSP video port resource is needed to be used with this P5154 video
 *      captured channel. It will identify if it is connected to DSP video port
 *      channel A or B. DSP video port channel would be initialized in dual
 *      channel mode to be used correctly with this information.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_P5154_gvideoport( EVM_P5154_Handle P5154handle, Int16 *P5154Videoport,
 *                              Int16 *P5154VideoChan)
 *
 *      P5154handle    - (i) handle of the P5154 video channel returned when opened.
 *
 *      P5154Videoport - (o) DSP video port connected to the current video decoder.
 *
 *      VideoChannel   - (o) video channel(s) connected to the current
 *                           DSP video port (1 for chanA and 2 for chanB).
 *
 * RETURN VALUE
 *      EVM_P5154_OK if OK, !=EVM_P5154_OK if problems
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_P5154_gvideoport( EVM_P5154_Handle P5154handle, Int16 *P5154Videoport, Int16 *P5154VideoChan);


#endif //__EVM_P5154_H__
