/***************************************************************************
 *
 *
 *         **   **    **  ******  ********  ********  ********  **    **    
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ***************************************************************************
 *                                                                          
 *  Project     : EVM
 *  File        : EVM_S8200.h
 *  Description : Utility functions for S8200 video DAC
 *
 *                      Copyright (c) Lyrtech inc. 2007                        
 *
 ***************************************************************************
 *                                                                                          
 * "$Revision: 1.3 $"
 * "$Date: 2007/07/13 19:12:16 $"
 *
 ***************************************************************************/

#ifndef __EVM_S8200_H__
#define __EVM_S8200_H__

/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include <tistdtypes.h>


/****************************************************************************
 *                             Global constants                             *
 ****************************************************************************/
// S8200 functions error status codes 
// ---------------------------------------------------------------------------
#define EVM_S8200_OK                0
#define EVM_S8200_NHANDLE           0x4001  // Cannot get a handle for the device
#define EVM_S8200_DEVICEINUSE       0x4002  // Device is already in use
#define EVM_S8200_INVALIDID         0x4003  // Invalid video channel ID
#define EVM_S8200_INVALIDHANDLE     0x4004  // Invalid handle given
#define EVM_S8200_I2CREADREG        0x4005  // Problem reading device I2C register
#define EVM_S8200_I2CWRITEOFFSET    0x4006  // Problem writing I2C register offset
#define EVM_S8200_I2CWRITEREG       0x4007  // Problem writing device I2C register
#define EVM_S8200_ERRRESETDEVICE    0x4008  // Problem cannot reset S8200 device
// ---------------------------------------------------------------------------

// Video channel definitions used as ID to get handle
#define EVM_S8200_CH1 0


// Number of Registers to setup
#define EVM_S8200_NREGS    100

// Index definitions for setup matrix
#define EVM_S8200_REG_IND  0
#define EVM_S8200_REG_VAL  1

// End of setup definition
#define EVM_S8200_ENDSETUP 0xFF


// S8200 default setups registers structures (EVM_S8200_Setup)
// There must be max EVM_S8200_NREGS table entries in these setups.
// Last valid setup entry must be followed by {EVM_S8200_ENDSETUP,EVM_S8200_ENDSETUP}

// S8200 Self generated Vesa Color Bar Test (needs a vesa CLKIN of 25.175MHz)
// Setup : 640x480x60Hz - 31.5khz - 2XPLL - 25.175MHz HS/VS +/+
//         N.B. the sync are inverted from standard VGA
// Register format: {regoff, value}
#define EVM_S8200_SELFCOLORBAR_GBR640X480X60 {                                                                                              \
    {                                                                                                                                       \
    {0x03,0xE3},     /* chip_ctl                                                                                                          */\
    {0x19,0x03},     /* csc_offset3                                                                                                       */\
    {0x1C,0xB0},     /* dman_cntl                                                                                                         */\
    {0x2A,0x8F},     /* dtg_spec_deh_lsb                                                                                                  */\
    {0x2B,0x00},     /* dtg_spec_deh_msb                                                                                                  */\
    {0x34,0x03},     /* dtg_total_pixel_msb                                                                                               */\
    {0x35,0x20},     /* dtg_total_pixel_lsb                                                                                               */\
    {0x36,0x80},     /* dtg_linecnt_msb                                                                                                   */\
    {0x37,0x01},     /* dtg_linecnt_lsb                                                                                                   */\
    {0x38,0x86},     /* dtg_mode                                                                                                          */\
    {0x39,0x27},     /* dtg_frame_field_msb                                                                                               */\
    {0x3A,0x0D},     /* dtg_frame_size_lsb                                                                                                */\
    {0x3B,0xFF},     /* dtg_field_size_lsb                                                                                                */\
    {0x3C,0x52},     /* dtg_vesa_cbar_size                                                                                                */\
    {0x4A,0x00},     /*                                                                                                                   */\
    {0x4F,0x00},     /* csm_mode                                                                                                          */\
    {0x70,0x60},     /* dtg_hlength_lsb                                                                                                   */\
    {0x71,0x00},     /* dtg_hdly_msb                                                                                                      */\
    {0x72,0x01},     /* dtg_hdly_lsb                                                                                                      */\
    {0x73,0x03},     /* dtg_vlength_lsb                                                                                                   */\
    {0x74,0x00},     /* dtg_vdly_msb                                                                                                      */\
    {0x75,0x01},     /* dtg_vdly_lsb                                                                                                      */\
    {0x76,0x00},     /* dtg_vlength2_lsb                                                                                                  */\
    {0x77,0x07},     /* dtg_vdly2_msb                                                                                                     */\
    {0x78,0xFF},     /* dtg_vdly2_lsb                                                                                                     */\
    {0x79,0x00},     /* dtg_hs_in_dly_msb                                                                                                 */\
    {0x7A,0x00},     /* dtg_hs_in_dly_lsb                                                                                                 */\
    {0x7B,0x00},     /* dtg_vs_in_dly_msb                                                                                                 */\
    {0x7C,0x00},     /* dtg_vs_in_dly_lsb                                                                                                 */\
    {0x82,0x5B},     /* pol_cntl                                                                                                          */\
    {EVM_S8200_ENDSETUP,EVM_S8200_ENDSETUP}                                                                                                 \
                     /* As a safety measure, this means end of setup                                                                      */\
    }                /*  do not remove and keep it last element                                                                           */\
}

// S8200 HDTV 720p setup (needs a CLKIN of 74.25MHz)
// Setup : 1280x720x60Hz - 45khz - 74.25MHz progressive
// Register format: {regoff, value}
#define EVM_S8200_HDTV720P_YCBCR1280X720X60 {                                                                                               \
    {                                                                                                                                       \
    /*{0x03,0x11},      chip_ctl                                                                                                          */\
    {0x03,0x01},     /* chip_ctl                                                                                                          */\
    /* CSC not used                                                                                                                       */\
    {0x19,0x03},     /* csc_offset3  - CSC bypassed                                                                                       */\
    {0x1C,0x5B},     /* dman_cntl  - 20 bits 4:2:2 format, 1Vp-p, 1X Dac mode, 4:2:2 interpol                                             */\
    /* composite sync amplitude control                                                                                                   */\
    {0x1D,0xFF},     /* dtg_y_sync1                                                                                                       */\
    {0x1E,0x49},     /* dtg_y_sync2                                                                                                       */\
    {0x1F,0xB6},     /* dtg_y_sync3                                                                                                       */\
    {0x20,0xFF},     /* dtg_cbcr_sync1                                                                                                    */\
    {0x21,0xFF},     /* dtg_cbcr_sync2                                                                                                    */\
    {0x22,0xFF},     /* dtg_cbcr_sync3                                                                                                    */\
    {0x23,0x13},     /* dtg_y_sync_upper                                                                                                  */\
    {0x24,0x15},     /* dtg_cbcr_sync_upper                                                                                               */\
    /* horizontal timing setup                                                                                                            */\
    {0x25,0x28},     /* dtg_spec_a                                                                                                        */\
    {0x26,0x6E},     /* dtg_spec_b                                                                                                        */\
    {0x27,0x28},     /* dtg_spec_c                                                                                                        */\
    {0x28,0x04},     /* dtg_spec_d                                                                                                        */\
    {0x29,0x00},     /* dtg_spec_d1                                                                                                       */\
    {0x2A,0x04},     /* dtg_spec_e                                                                                                        */\
    {0x2B,0xC0},     /* dtg_spec_h_msb                                                                                                    */\
    {0x2C,0x00},     /* dtg_spec_h_lsb                                                                                                    */\
    {0x2D,0x00},     /* dtg_spec_i_msb                                                                                                    */\
    {0x2E,0x00},     /* dtg_spec_i_lsb                                                                                                    */\
    {0x2F,0x6E},     /* dtg_spec_k_lsb                                                                                                    */\
    {0x30,0x00},     /* dtg_spec_k_msb                                                                                                    */\
    {0x31,0x00},     /* dtg_spec_k1                                                                                                       */\
    {0x32,0x00},     /* dtg_speg_g_lsb                                                                                                    */\
    {0x33,0x00},     /* dtg_speg_g_msb                                                                                                    */\
    {0x34,0x06},     /* dtg_total_pixel_msb                                                                                               */\
    {0x35,0x72},     /* dtg_total_pixel_lsb                                                                                               */\
    {0x36,0x80},     /* dtg_linecnt_msb                                                                                                   */\
    {0x37,0x01},     /* dtg_linecnt_lsb                                                                                                   */\
    {0x38,0x82},     /* dtg_mode  - 720p                                                                                                  */\
    {0x39,0x27},     /* dtg_frame_field_msb                                                                                               */\
    {0x3A,0xEE},     /* dtg_frame_size_lsb                                                                                                */\
    {0x3B,0xFF},     /* dtg_field_size_lsb                                                                                                */\
    {0x4A,0x00},     /* csm_mult_gy_msb                                                                                                   */\
    /* CSM setup not required if not used                                                                                                 */\
    {0x4F,0x00},     /* csm_mode disabled                                                                                                 */\
    /* discrete output sync control                                                                                                       */\
    {0x70,0x18},     /* dtg_hlength_lsb                                                                                                   */\
    {0x71,0x06},     /* dtg_hdly_msb                                                                                                      */\
    {0x72,0x49},     /* dtg_hdly_lsb                                                                                                      */\
    {0x73,0x1A},     /* dtg_vlength_lsb                                                                                                   */\
    {0x74,0x00},     /* dtg_vdly_msb                                                                                                      */\
    {0x75,0x01},     /* dtg_vdly_lsb                                                                                                      */\
    {0x76,0x00},     /* dtg_vlength2_lsb                                                                                                  */\
    {0x77,0xC7},     /* dtg_vdly2_msb                                                                                                     */\
    {0x78,0x07},     /* dtg_vdly2_lsb                                                                                                     */\
    /* discrete input sync control - use to align picture                                                                                 */\
    {0x79,0x00},     /* dtg_hs_in_dly_msb                                                                                                 */\
    {0x7A,0x3C},     /* dtg_hs_in_dly_lsb   - adjust horizontal position                                                                  */\
    {0x7B,0x00},     /* dtg_vs_in_dly_msb                                                                                                 */\
    {0x7C,0x01},     /* dtg_vs_in_dly_lsb  - adjust vertical position                                                                     */\
    {0x82,0x3B},     /* pol_cntl     YPbPr, Embedded timing                                                                               */\
    {EVM_S8200_ENDSETUP,EVM_S8200_ENDSETUP}                                                                                                 \
                     /* As a safety measure, this means end of setup                                                                      */\
    }                /*  do not remove and keep it last element                                                                           */\
}

// S8200 EDTV 480p setup (needs a CLKIN of 27MHz) ~SMPTE293M
// Setup : 720x480x60Hz - 31.468khz - 27MHz progressive
// Register format: {regoff, value}
#define EVM_S8200_EDTV480P_YCBCR720X480X60 {                                                                                                \
    {                                                                                                                                       \
    {0x03,0x11},     /* chip_ctl                                                                                                          */\
    /* CSC not used                                                                                                                       */\
    {0x19,0x03},     /* csc_offset3  - CSC bypassed                                                                                       */\
    {0x1C,0x5B},     /* dman_cntl  - 20 bits 4:2:2 format, 1Vp-p, 1X Dac mode, 4:2:2 interpol                                             */\
    {0x1D,0xFF},     /* dtg_y_sync1                                                                                                       */\
    {0x1E,0x50},     /* dtg_y_sync2                                                                                                       */\
    {0x1F,0xFF},     /* dtg_y_sync3                                                                                                       */\
    {0x20,0xFF},     /* dtg_cbcr_sync1                                                                                                    */\
    {0x21,0xFF},     /* dtg_cbcr_sync2                                                                                                    */\
    {0x22,0xFF},     /* dtg_cbcr_sync3                                                                                                    */\
    {0x23,0x11},     /* dtg_y_sync_upper                                                                                                  */\
    {0x24,0x15},     /* dtg_cbcr_sync_upper                                                                                               */\
    {0x25,0x3D},     /* dtg_spec_a                                                                                                        */\
    {0x26,0x10},     /* dtg_spec_b                                                                                                        */\
    {0x27,0x20},     /* dtg_spec_c                                                                                                        */\
    {0x28,0x7A},     /* dtg_spec_d                                                                                                        */\
    {0x29,0x00},     /* dtg_spec_d1                                                                                                       */\
    {0x2A,0xC0},     /* dtg_spec_e                                                                                                        */\
    {0x2B,0x01},     /* dtg_spec_h_msb                                                                                                    */\
    {0x2C,0x90},     /* dtg_spec_h_lsb                                                                                                    */\
    {0x2D,0x03},     /* dtg_spec_i_msb                                                                                                    */\
    {0x2E,0x1B},     /* dtg_spec_i_lsb                                                                                                    */\
    {0x2F,0x10},     /* dtg_spec_k_lsb                                                                                                    */\
    {0x30,0x00},     /* dtg_spec_k_msb                                                                                                    */\
    {0x31,0x00},     /* dtg_spec_k1                                                                                                       */\
    {0x32,0xAD},     /* dtg_speg_g_lsb                                                                                                    */\
    {0x33,0x01},     /* dtg_speg_g_msb                                                                                                    */\
    {0x34,0x03},     /* dtg_total_pixel_msb                                                                                               */\
    {0x35,0x5A},     /* dtg_total_pixel_lsb                                                                                               */\
    {0x36,0x80},     /* dtg_linecnt_msb                                                                                                   */\
    {0x37,0x08},     /* dtg_linecnt_lsb                                                                                                   */\
    {0x38,0x89},     /* dtg_mode                                                                                                          */\
    {0x39,0x27},     /* dtg_frame_field_msb                                                                                               */\
    {0x3A,0x0D},     /* dtg_frame_size_lsb                                                                                                */\
    {0x3B,0xFF},     /* dtg_field_size_lsb                                                                                                */\
    {0x4A,0x00},     /* csm_mult_gy_msb                                                                                                   */\
    {0x4F,0x00},     /* csm_mode OFF                                                                                                      */\
    {0x50,0x00},     /* dtg_bp1_2_msb                                                                                                     */\
    {0x51,0x02},     /* dtg_bp3_4_msb                                                                                                     */\
    {0x52,0x00},     /* dtg_bp5_6_msb                                                                                                     */\
    {0x58,0x07},     /* dtg_bp1_lsb                                                                                                       */\
    {0x59,0x0D},     /* dtg_bp2_lsb                                                                                                       */\
    {0x5A,0x2B},     /* dtg_bp3_lsb                                                                                                       */\
    {0x5B,0x0E},     /* dtg_bp4_lsb                                                                                                       */\
    {0x68,0x9A},     /* dtg_linetype1                                                                                                     */\
    {0x69,0x00},     /* dtg_linetype2                                                                                                     */\
    {0x6A,0x00},     /* dtg_linetype3                                                                                                     */\
    {0x70,0x40},     /* dtg_hlength_lsb                                                                                                   */\
    {0x71,0x03},     /* dtg_hdly_msb                                                                                                      */\
    {0x72,0x31},     /* dtg_hdly_lsb                                                                                                      */\
    {0x73,0x07},     /* dtg_vlength_lsb                                                                                                   */\
    {0x74,0x00},     /* dtg_vdly_msb                                                                                                      */\
    {0x75,0x07},     /* dtg_vdly_lsb                                                                                                      */\
    {0x76,0x00},     /* dtg_vlength2_lsb                                                                                                  */\
    {0x77,0xC7},     /* dtg_vdly2_msb                                                                                                     */\
    {0x78,0xFF},     /* dtg_vdly2_lsb                                                                                                     */\
    {0x79,0x00},     /* dtg_hs_in_dly_msb                                                                                                 */\
    {0x7A,0x12},     /* dtg_hs_in_dly_lsb                                                                                                 */\
    {0x7B,0x00},     /* dtg_vs_in_dly_msb                                                                                                 */\
    {0x7C,0x00},     /* dtg_vs_in_dly_lsb                                                                                                 */\
    {0x82,0x3B},     /* pol_cntl     YPbPr, Embedded timing                                                                               */\
    {EVM_S8200_ENDSETUP,EVM_S8200_ENDSETUP}                                                                                                 \
                     /* As a safety measure, this means end of setup                                                                      */\
    }                /*  do not remove and keep it last element                                                                           */\
}


// S8200 SDTV 480i setup (needs a CLKIN of 27MHz = 2 X Pclk) ~BT656 input format
// Setup : 704x480x60Hz - 15.734khz - 13.5MHz interlaced (525i)
// Optional BT656 data bus + clock is active
// Register format: {regoff, value}
#define EVM_S8200_SDTV480I_YCBCR704X480X60 {                                                                                                \
    {                                                                                                                                       \
    {0x03,0x11},     /* chip_ctl                                                                                                          */\
    /* CSC setup not required if not used                                                                                                 */\
    {0x19,0x17},     /* csc_offset3  CSC bypassed                                                                                         */\
    {0x1C,0xD4},     /* dman_cntl  - 10 bits 4:2:2 format BT656 mode, 1Vp-p, 1X Dac mode, 4:2:2 interpol, clk656 ON, BT656 bus active     */\
    /* composite sync amplitude control                                                                                                   */\
    {0x1D,0xFF},     /* dtg_y_sync1                                                                                                       */\
    {0x1E,0x49},     /* dtg_y_sync2                                                                                                       */\
    {0x1F,0xFF},     /* dtg_y_sync3                                                                                                       */\
    {0x20,0xFF},     /* dtg_cbcr_sync1                                                                                                    */\
    {0x21,0xFF},     /* dtg_cbcr_sync2                                                                                                    */\
    {0x22,0xFF},     /* dtg_cbcr_sync3                                                                                                    */\
    {0x23,0x11},     /* dtg_y_sync_upper                                                                                                  */\
    {0x24,0x15},     /* dtg_cbcr_sync_upper                                                                                               */\
    /* Horizontal timing parameters                                                                                                       */\
    {0x25,0x3D},     /* dtg_spec_a                                                                                                        */\
    {0x26,0x14},     /* dtg_spec_b                                                                                                        */\
    {0x27,0x1D},     /* dtg_spec_c                                                                                                        */\
    {0x28,0x7C},     /* dtg_spec_d                                                                                                        */\
    {0x29,0x00},     /* dtg_spec_d1                                                                                                       */\
    {0x2A,0x00},     /* dtg_spec_e                                                                                                        */\
    {0x2B,0x01},     /* dtg_spec_h_msb                                                                                                    */\
    {0x2C,0x6C},     /* dtg_spec_h_lsb                                                                                                    */\
    {0x2D,0x01},     /* dtg_spec_i_msb                                                                                                    */\
    {0x2E,0x8D},     /* dtg_spec_i_lsb                                                                                                    */\
    {0x2F,0x14},     /* dtg_spec_k_lsb                                                                                                    */\
    {0x30,0x00},     /* dtg_spec_k_msb                                                                                                    */\
    {0x31,0x0A},     /* dtg_spec_k1                                                                                                       */\
    {0x32,0xAD},     /* dtg_speg_g_lsb                                                                                                    */\
    {0x33,0x01},     /* dtg_speg_g_msb                                                                                                    */\
    {0x34,0x03},     /* dtg_total_pixel_msb                                                                                               */\
    {0x35,0x5A},     /* dtg_total_pixel_lsb                                                                                               */\
    {0x36,0x00},     /* dtg_linecnt_msb                                                                                                   */\
    {0x37,0x01},     /* dtg_linecnt_lsb                                                                                                   */\
    {0x38,0x84},     /* dtg_mode                                                                                                          */\
    {0x39,0x21},     /* dtg_frame_field_msb                                                                                               */\
    {0x3A,0x0D},     /* dtg_frame_size_lsb                                                                                                */\
    {0x3B,0x07},     /* dtg_field_size_lsb                                                                                                */\
    {0x4A,0x00},     /* csm_mult_gy_msb                                                                                                   */\
    /*CSM Setup - not required if not used                                                                                                */\
    {0x4F,0x00},     /* csm_mode                                                                                                          */\
    /* discrete output sync control                                                                                                       */\
    {0x70,0x7C},     /* dtg_hlength_lsb                                                                                                   */\
    {0x71,0x03},     /* dtg_hdly_msb                                                                                                      */\
    {0x72,0x31},     /* dtg_hdly_lsb                                                                                                      */\
    {0x73,0x12},     /* dtg_vlength_lsb                                                                                                   */\
    {0x74,0x00},     /* dtg_vdly_msb                                                                                                      */\
    {0x75,0x01},     /* dtg_vdly_lsb                                                                                                      */\
    {0x76,0x00},     /* dtg_vlength2_lsb                                                                                                  */\
    {0x77,0x00},     /* dtg_vdly2_msb                                                                                                     */\
    {0x78,0x00},     /* dtg_vdly2_lsb                                                                                                     */\
    /* discrete input sync control  - use these to align picture                                                                          */\
    {0x79,0x00},     /* dtg_hs_in_dly_msb                                                                                                 */\
    {0x7A,0x12},     /* dtg_hs_in_dly_lsb   use to adjust HSIN delay or horizontal position                                               */\
    {0x7B,0x00},     /* dtg_vs_in_dly_msb                                                                                                 */\
    {0x7C,0x00},     /* dtg_vs_in_dly_lsb   use to adjust VSIN delay or vertical position                                                 */\
    {0x82,0x3B},     /* pol_cntl     YPbPr, Embedded timing                                                                               */\
    {EVM_S8200_ENDSETUP,EVM_S8200_ENDSETUP}                                                                                                 \
                     /* As a safety measure, this means end of setup                                                                      */\
    }                /*  do not remove and keep it last element                                                                           */\
}


//  Handle
typedef Int32 EVM_S8200_Handle;

// Register structure
typedef struct
{
    Uint8 regs[EVM_S8200_NREGS][2];
} EVM_S8200_Setup;


/****************************************************************************
 *                             Public Functions                             *
 ****************************************************************************/

/**************************************************************************** 
 *
 * NAME  
 *      EVM_S8200_open
 *
 * PURPOSE:
 *      Open an S8200 HD video display device and return a handle.
 *      At opening, the S8200 video channel is initialized according to
 *      the specified setup matrix. When the channel is not used, the device must
 *      be closed by EVM_S8200_close. There is only one video display channel
 *      available. 
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      EVM_S8200_Handle EVM_S8200_open( Uint32 id, EVM_S8200_Setup *setup )
 *
 *      id    - (i) id of desired S8200 [EVM_S8200_CH1]
 *      setup - (i) S8200 video display setup structure
 *
 * RETURN VALUE
 *      Handle value, >= EVM_S8200_NHANDLE in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
EVM_S8200_Handle EVM_S8200_open( Uint32 id, EVM_S8200_Setup *setup );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_S8200_close
 *
 * PURPOSE:
 *      Close a video device previously opened by EVM_S8200_open.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_S8200_close( EVM_S8200_Handle *S8200handle )
 *
 *      S8200handle - (i) handle of the S8200 video returned when opened.
 *
 * RETURN VALUE
 *      EVM_S8200_OK if OK, !=EVM_S8200_OK if problems
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_S8200_close( EVM_S8200_Handle *S8200handle );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_S8200_setup
 *
 * PURPOSE:
 *      Setup the current S8200 HD video display channel with default register values.
 *      A setup structure is given for this purpose.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_S8200_setup( EVM_S8200_Handle S8200handle, EVM_S8200_Setup *setup )
 *
 *      S8200handle - (i) handle of the S8200 video channel returned when opened.
 *
 *      setup       - (i)  setup data structure.
 *
 * RETURN VALUE
 *      EVM_S8200_OK if OK, !=EVM_S8200_OK if problems
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_S8200_setup( EVM_S8200_Handle S8200handle, EVM_S8200_Setup *setup );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_S8200_rget
 *
 * PURPOSE:
 *      Read a specified register of the current S8200 video handle.
 *      The register will be read from the S8200 associated with the current
 *      handle, at the specified offset position.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_S8200_rget( EVM_S8200_Handle S8200handle, Uint16 regnum, Uint16* regval )
 *
 *      S8200handle - (i) handle of the S8200 returned when opened.
 *
 *      regnum      - (i) register offset (8 bits).
 *
 *      regval      - (o) register value read (8 bits).
 *
 * RETURN VALUE
 *      EVM_S8200_OK if OK, !=EVM_S8200_OK if problems
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_S8200_rget( EVM_S8200_Handle S8200handle,  Uint16 regnum, Uint16* regval );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_S8200_rgetset
 *
 * PURPOSE:
 *      Read a  register, change a bitfield and then writeback the register.
 *      Bitmask values and bitval are binary value with 1 set at the bit position
 *      to modify in the register. It will work on the current S8200 for
 *      the specified handle.
 *      Pseudocode operation:
 *       1- Value1 = Read S8200 register regnum of S8200handle
 *       2- Value1 = Value1 & (~bitmask)
 *       3- Value1 = Value1 | (bitmask & bitval)
 *       4- Write S8200 register regnum of S8200handle with Value1
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_S8200_rgetset( EVM_S8200_Handle S8200handle, Uint16 regnum, Uint16 bitmask, Uint16 bitval )
 *
 *      S8200handle - (i) handle of the S8200 returned when opened.
 *
 *      regnum      - (i) register offset (8 bits).
 *
 *      bitmask     - (i) bit mask value (8 bits) with bit to keep set as 1.
 *
 *      bitval      - (i) bit values (8 bits) to be set at mask position.
 *
 * RETURN VALUE
 *      EVM_S8200_OK if OK, !=EVM_S8200_OK if problems
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_S8200_rgetset( EVM_S8200_Handle S8200handle, Uint16 regnum, Uint16 bitmask, Uint16 bitval );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_S8200_rset
 *
 * PURPOSE:
 *      Write a register with a specified value for the S8200 of this handle.
 *      The register will be written from the S8200 associated with the current
 *      handle, at the specified offset position.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_S8200_rset( EVM_S8200_Handle S8200handle, Uint16 regnum, Uint16 regval )
 *
 *      S8200handle - (i) handle of the S8200 returned when opened.
 *
 *      regnum      - (i) register offset (8 bits).
 *
 *      regval      - (i) register value to be written (8 bits).
 *
 * RETURN VALUE
 *      EVM_S8200_OK if OK, !=EVM_S8200_OK if problems
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_S8200_rset( EVM_S8200_Handle S8200handle, Uint16 regnum, Uint16 regval );


#endif // __EVM_S8200_H__
