/***************************************************************************
 *
 *
 *         **   **    **  ******  ********  ********  ********  **    **    
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ***************************************************************************
 *                                                                          
 *  Project     : EVM
 *  File        : EVM_AIC33.h
 *  Description : Utility functions definitions for AIC33 audio codec
 *
 *                      Copyright (c) Lyrtech inc. 2007                        
 *
 ***************************************************************************
 *                                                                                          
 * "$Revision: 1.1 $"
 * "$Date: 2007/09/13 15:55:46 $"
 *
 ***************************************************************************/

#ifndef __EVM_AIC33_H__ 
#define __EVM_AIC33_H__

/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include <tistdtypes.h>

/****************************************************************************
 *                             Glocal constants                             *
 ****************************************************************************/
// AIC33 functions error status codes 
// ---------------------------------------------------------------------------
#define EVM_AIC33_OK                0
#define EVM_AIC33_NHANDLE           0x6001  // Cannot get a handle for the device
#define EVM_AIC33_DEVICEINUSE       0x6002  // Device is already in use
#define EVM_AIC33_INVALIDID         0x6003  // Invalid video channel ID
#define EVM_AIC33_INVALIDHANDLE     0x6004  // Invalid handle given
#define EVM_AIC33_I2CREADREG        0x6005  // Problem reading device I2C register
#define EVM_AIC33_I2CWRITEREG       0x6006  // Problem writing device I2C register
#define EVM_AIC33_ERRRESETDEVICE    0x6007  // Problem cannot reset AIC33 device
#define EVM_AIC33_ERRLOCKRESOURCE   0x6008  // Cannot lock mux resource for audio device
#define EVM_AIC33_ERRUNLOCKRESOURCE 0x6009  // Cannot unlock mux resource for audio device
#define EVM_AIC33_I2CWRITEPAGE      0x600A  // Cannot write codec registers page
#define EVM_AIC33_ALREADYAMASTER    0x600B  // Problem, there is already a codec master
#define EVM_AIC33_CANNOTCONFMASTER  0x600C  // Cannot write codec master register
#define EVM_AIC33_CANNOTWRITEMUTER  0x600D  // Cannot write mute control of right channel
#define EVM_AIC33_CANNOTWRITEMUTEL  0x600E  // Cannot write mute control of left channel
// ---------------------------------------------------------------------------

// Audio channel definitions used as ID to get handle
#define EVM_AIC33_DEV1 0
#define EVM_AIC33_DEV2 1
#define EVM_AIC33_DEV3 2
#define EVM_AIC33_DEV4 3

// Mute function mode
#define EVM_AIC33_MUTEON  1
#define EVM_AIC33_MUTEOFF 0
                     
// Can be changed to add additionnals reg in setup
// Number of Registers to setup
#define EVM_AIC33_NREGS    170

// Index definitions for setup matrix
#define EVM_AIC33_PGE_IND  0
#define EVM_AIC33_REG_IND  1
#define EVM_AIC33_REG_VAL  2

// End of setup definition
#define EVM_AIC33_ENDSETUP 0xFF


// Audio codec default setups registers structures
// The setup must provide default codec clock slave setup and use
// setmaster to define as master the desired codec
// There must be EVM_AIC33_NREGS table entries in these setups.
// Last setup entry must be {EVM_AIC33_ENDSETUP,EVM_AIC33_ENDSETUP,EVM_AIC33_ENDSETUP}                                                                                     \

// Data mode: I2S
// Data resolution: 16 Bits
// Frequency of operation: 44.10003KHz (error from 44.1KHz: +0.00007%)
//      FS = ( MCLK * K * R) / ( 2048 * P )
//      For a FS=44.10003 kHz & MCLK=11.633523 MHz
//          -> 44.10003kHz = ( 11.633523 MHz * K * R) / ( 2048 * P )
//          -> R = 1, P = 1, K[J.D] = 7.7635
// Register format: {page, regoff, value}
#define EVM_AIC33_16BITS_I2S_441 {                                                                                                     \
    {             /* Page 0                                                                                                          */\
    {0,  2,0x00}, /* 2  Codec Sample Rate Select        BV: [ADC=FS][DAC=FS]                                                         */\
                                                                                                                                       \
    /* For: [FS=44.10003 kHz][MCLK=11.633523 MHz]                                                                                    */\
    {0,  3,0x91}, /* 3  PLL Reg A                       BV: [PLL=ON][P=1][Q=2]                                                       */\
    {0,  4,0x1C}, /* 4  PLL Reg B                       BV: [J=7]                                                                    */\
    {0,  5,0x77}, /* 5  PLL Reg C                       BV: [D=7635 Bin: 01110111 010011 MSB]                                        */\
    {0,  6,0x4C}, /* 6  PLL Reg D                       BV: [D=7635 Bin: 01110111 010011 LSB]                                        */\
    {0,  7,0x8A}, /* 7  Codec Datapath Setup            BV: [FS=44.1 kHz][LeftDACPlays=left][RightDACPlays=right]                    */\
                                                                                                                                       \
    {0,  8,0x00}, /* 8  Audio Serial Data Reg A         BV: [BCLK=Slave][WCLK=Slave][3DEffects=OFF][DigitalMIC=OFF]                  */\
    {0,  9,0x00}, /* 9  Audio Serial Data Reg B         BV: [Mode=I2S][Lenght=16-bit][TransferMode=Continous]                        */\
    {0, 10,0x00}, /* 10 Audio Serial Data Reg C         BV: [DataOffset=0]                                                           */\
    {0, 11,0x01}, /* 11 Overflow Flag Register          BV: [PLL R=1]                                                                */\
    {0, 12,0x00}, /* 12 Digital Filter Ctrl Reg         BV: All filters/effects/emphasis disabled                                    */\
    {0, 13,0x00}, /* 13 Headset Detection Reg A         BV: [HeadsetDetect=OFF]                                                      */\
    {0, 14,0x00}, /* 14 Headset Detection Reg B         BV: []                                                                       */\
    {0, 15,0x00}, /* 15 Left  ADC PGA Gain Ctrl Reg     BV: [Mute=OFF]                                                               */\
    {0, 16,0x00}, /* 16 Right ADC PGA Gain Ctrl Reg     BV: [Mute=OFF]                                                               */\
    {0, 17,0xFF}, /* 17 MIC3L/R to Left  ADC Ctrl Reg   BV: [MIC3L=NoConnect][MIC3R=NoConnect]                                       */\
    {0, 18,0xFF}, /* 18 MIC3L/R to Right ADC Ctrl Reg   BV: [MIC3L=NoConnect][MIC3R=NoConnect]                                       */\
    {0, 19,0x04}, /* 19 LINE1L  to Left  ADC Ctrl Reg   BV: [LINE1L=SingleEnd][LINE1L=0dBGain][LeftADCPower=ON][SoftStep=OncePerFS]  */\
                                                                                                                                       \
    {0, 20,0x78}, /* 20 LINE2L  to Left  ADC Ctrl Reg   BV: [LINE2L=SingleEnd][LINE2L=NoConnect]                                     */\
    {0, 21,0x78}, /* 21 LINE1R  to Left  ADC Ctrl Reg   BV: [LINE1R=SingleEnd][LINE1R=NoConnect]                                     */\
    {0, 22,0x04}, /* 22 LINE1R  to Right ADC Ctrl Reg   BV: [LINE1R=SingleEnd][LINE1R=0dBGain][RightADCPower=ON][SoftStep=OncePerFS] */\
    {0, 23,0x78}, /* 23 LINE2R  to Right ADC Ctrl Reg   BV: [LINE2R=SingleEnd][LINE2R=NoConnect]                                     */\
    {0, 24,0x78}, /* 24 LINE1L  to Right ADC Ctrl Reg   BV: [LINE1L=SingleEnd][LINE1L=NoConnect]                                     */\
    {0, 25,0x00}, /* 25 MICBIAS Control Reg             BV: [Power=OFF]                                                              */\
    {0, 26,0x00}, /* 26 Left  AGC Control Reg A         BV: [LeftAGC=OFF]                                                            */\
    {0, 27,0x00}, /* 27 Left  AGC Control Reg B         BV: []                                                                       */\
    {0, 28,0x00}, /* 28 Left  AGC Control Reg C         BV: [LeftHystersis=OFF]                                                      */\
    {0, 29,0x00}, /* 29 Right AGC Control Reg A         BV: [RightAGC=OFF]                                                           */\
                                                                                                                                       \
    {0, 30,0x00}, /* 30 Right AGC Control Reg B         BV: []                                                                       */\
    {0, 31,0x00}, /* 31 Right AGC Control Reg C         BV: [RightHystersis=OFF]                                                     */\
    {0, 34,0x00}, /* 34 Left  AGC Noise Gate Debounce   BV: []                                                                       */\
    {0, 35,0x00}, /* 35 Right AGC Noise Gate Debounce   BV: []                                                                       */\
    {0, 37,0xE0}, /* 37 DAC Power & Output Dvr Ctrl Reg BV: [LeftDACPower=ON][RightDACPower=ON][HPLCOM=SingleEnd]                    */\
    {0, 38,0x10}, /* 38 High Power Output Dvr Ctrl Reg  BV: [HPRCOM=SingleEnd][ShortCircuit=OFF]                                     */\
                                                                                                                                       \
    {0, 40,0x00}, /* 40 Output Stage Ctrl Reg           BV: []                                                                       */\
    {0, 41,0x00}, /* 41 DAC Output Switching Ctrl Reg   BV: []                                                                       */\
    {0, 42,0x00}, /* 42 Output Driver Pop Reduction Reg BV: []                                                                       */\
    {0, 43,0x00}, /* 43 Left  DAC Digital Vol Reg       BV: [LeftDACMute=OFF]                                                        */\
    {0, 44,0x00}, /* 44 Right DAC Digital Vol Reg       BV: [RightDACMute=OFF]                                                       */\
    {0, 45,0x00}, /* 45 Line2L to HPLOUT Vol Reg        BV: []                                                                       */\
    {0, 46,0x00}, /* 46 PGA_L  to HPLOUT Vol Reg        BV: []                                                                       */\
    {0, 47,0x00}, /* 47 DAC_L1 to HPLOUT Vol Reg        BV: []                                                                       */\
    {0, 48,0x00}, /* 48 LINE2R to HPLOUT Vol Reg        BV: []                                                                       */\
    {0, 49,0x00}, /* 49 PGA_R  to HPLOUT Vol Reg        BV: []                                                                       */\
                                                                                                                                       \
    {0, 50,0x00}, /* 50 DAC_R1 to HPLOUT Vol Reg        BV: []                                                                       */\
    {0, 51,0x00}, /* 51           HPLOUT Output Reg     BV: []                                                                       */\
    {0, 52,0x00}, /* 52 LINE2L to HPLCOM Vol Reg        BV: []                                                                       */\
    {0, 53,0x00}, /* 53 PGA_L  to HPLCOM Vol Reg        BV: []                                                                       */\
    {0, 54,0x00}, /* 54 DAC_L1 to HPLCOM Vol Reg        BV: []                                                                       */\
    {0, 55,0x00}, /* 55 LINE2R to HPLCOM Vol Reg        BV: []                                                                       */\
    {0, 56,0x00}, /* 56 PGA_R  to HPLCOM Vol Reg        BV: []                                                                       */\
    {0, 57,0x00}, /* 57 DAC_R1 to HPLCOM Vol Reg        BV: []                                                                       */\
    {0, 58,0x00}, /* 58           HPLCOM Output Reg     BV: []                                                                       */\
    {0, 59,0x00}, /* 59 LINE2L to HPROUT Vol Reg        BV: []                                                                       */\
                                                                                                                                       \
    {0, 60,0x00}, /* 60 PGA_L  to HPROUT Vol Reg        BV: []                                                                       */\
    {0, 61,0x00}, /* 61 DAC_L1 to HPROUT Vol Reg        BV: []                                                                       */\
    {0, 62,0x00}, /* 62 LINE2R to HPROUT Vol Reg        BV: []                                                                       */\
    {0, 63,0x00}, /* 63 PGA_R  to HPROUT Vol Reg        BV: []                                                                       */\
    {0, 64,0x00}, /* 64 DAC_R1 to HPROUT Vol Reg        BV: []                                                                       */\
    {0, 65,0x00}, /* 65           HPROUT Output Reg     BV: []                                                                       */\
    {0, 66,0x00}, /* 66 LINE2L to HPRCOM Vol Reg        BV: []                                                                       */\
    {0, 67,0x00}, /* 67 PGA_L  to HPRCOM Vol Reg        BV: []                                                                       */\
    {0, 68,0x00}, /* 68 DAC_L1 to HPRCOM Vol Reg        BV: []                                                                       */\
    {0, 69,0x00}, /* 69 LINE2R to HPRCOM Vol Reg        BV: []                                                                       */\
                                                                                                                                       \
    {0, 70,0x00}, /* 70 PGA_R  to HPRCOM Vol Reg        BV: []                                                                       */\
    {0, 71,0x00}, /* 71 DAC_R1 to HPRCOM Vol Reg        BV: []                                                                       */\
    {0, 72,0x00}, /* 72           HPRCOM Output Reg     BV: []                                                                       */\
    {0, 73,0x00}, /* 73 LINE2L to MONO_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 74,0x00}, /* 74 PGA_L  to MONO_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 75,0x00}, /* 75 DAC_L1 to MONO_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 76,0x00}, /* 76 LINE2R to MONO_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 77,0x00}, /* 77 PGA_R  to MONO_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 78,0x00}, /* 78 DAC_R1 to MONO_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 79,0x00}, /* 79           MONO_LOP/M Output Reg BV: []                                                                       */\
                                                                                                                                       \
    {0, 80,0x00}, /* 80 LINE2L to LEFT_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 81,0x00}, /* 81 PGA_L  to LEFT_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 82,0x80}, /* 82 DAC_L1 to LEFT_LOP/M Vol Reg    BV: [Routed]                                                                 */\
    {0, 83,0x00}, /* 83 LINE2R to LEFT_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 84,0x00}, /* 84 PGA_R  to LEFT_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 85,0x00}, /* 85 DAC_R1 to LEFT_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 86,0x09}, /* 86           LEFT_LOP/M Output Reg BV: [Mute=OFF][Power=ON]                                                     */\
    {0, 87,0x00}, /* 87 LINE2L to RIGHT_LOP/M Vol Reg   BV: []                                                                       */\
    {0, 88,0x00}, /* 88 PGA_L  to RIGHT_LOP/M Vol Reg   BV: []                                                                       */\
    {0, 89,0x00}, /* 89 DAC_L1 to RIGHT_LOP/M Vol Reg   BV: []                                                                       */\
                                                                                                                                       \
    {0, 90,0x00}, /* 90 LINE2R to RIGHT_LOP/M Vol Reg   BV: []                                                                       */\
    {0, 91,0x00}, /* 91 PGA_R  to RIGHT_LOP/M Vol Reg   BV: []                                                                       */\
    {0, 92,0x80}, /* 92 DAC_R1 to RIGHT_LOP/M Vol Reg   BV: [Routed]                                                                 */\
    {0, 93,0x09}, /* 93           RIGHT_LOP/M Outpt Reg BV: [Mute=OFF][Power=ON]                                                     */\
    {0, 98,0x10}, /* 98 GPIO1 Control Reg               BV: [GPIO1=ADC word clk]                                                     */\
    {0, 99,0x00}, /* 99 GPIO2 Control Reg               BV: []                                                                       */\
                                                                                                                                       \
    {0,100,0x00}, /* 100 Additional GPIO Control Reg A  BV: []                                                                       */\
    {0,101,0x00}, /* 101 Additional GPIO Control Reg B  BV: []                                                                       */\
    {0,102,0x02}, /* 102 Clock Generation Control Reg   BV: []                                                                       */\
                  /* Page 1                                                                                                          */\
    {1,  1,0x6B}, /*  1 Left Channel Audio Effects Filter N0 Coefficient MSB                                                         */\
    {1,  2,0xE3}, /*  2 Left Channel Audio Effects Filter N0 Coefficient LSB                                                         */\
    {1,  3,0x96}, /*  3 Left Channel Audio Effects Filter N1 Coefficient MSB                                                         */\
    {1,  4,0x66}, /*  4 Left Channel Audio Effects Filter N1 Coefficient LSB                                                         */\
    {1,  5,0x67}, /*  5 Left Channel Audio Effects Filter N2 Coefficient MSB                                                         */\
    {1,  6,0x5D}, /*  6 Left Channel Audio Effects Filter N2 Coefficient LSB                                                         */\
    {1,  7,0x6B}, /*  7 Left Channel Audio Effects Filter N3 Coefficient MSB                                                         */\
    {1,  8,0xE3}, /*  8 Left Channel Audio Effects Filter N3 Coefficient LSB                                                         */\
    {1,  9,0x96}, /*  9 Left Channel Audio Effects Filter N4 Coefficient MSB                                                         */\
                                                                                                                                       \
    {1, 10,0x66}, /* 10 Left Channel Audio Effects Filter N4 Coefficient LSB                                                         */\
    {1, 11,0x67}, /* 11 Left Channel Audio Effects Filter N5 Coefficient MSB                                                         */\
    {1, 12,0x5D}, /* 12 Left Channel Audio Effects Filter N5 Coefficient LSB                                                         */\
    {1, 13,0x7D}, /* 13 Left Channel Audio Effects Filter D1 Coefficient MSB                                                         */\
    {1, 14,0x83}, /* 14 Left Channel Audio Effects Filter D1 Coefficient LSB                                                         */\
    {1, 15,0x84}, /* 15 Left Channel Audio Effects Filter D2 Coefficient MSB                                                         */\
    {1, 16,0xEE}, /* 16 Left Channel Audio Effects Filter D2 Coefficient LSB                                                         */\
    {1, 17,0x7D}, /* 17 Left Channel Audio Effects Filter D4 Coefficient MSB                                                         */\
    {1, 18,0x83}, /* 18 Left Channel Audio Effects Filter D4 Coefficient LSB                                                         */\
    {1, 19,0x84}, /* 19 Left Channel Audio Effects Filter D5 Coefficient MSB                                                         */\
                                                                                                                                       \
    {1, 20,0xEE}, /* 20 Left Channel Audio Effects Filter D5 Coefficient LSB                                                         */\
    {1, 21,0x39}, /* 21 Left Channel De-emphasis Filter N0 Coefficient MSB                                                           */\
    {1, 22,0x55}, /* 22 Left Channel De-emphasis Filter N0 Coefficient LSB                                                           */\
    {1, 23,0xF3}, /* 23 Left Channel De-emphasis Filter N1 Coefficient MSB                                                           */\
    {1, 24,0x2D}, /* 24 Left Channel De-emphasis Filter N1 Coefficient LSB                                                           */\
    {1, 25,0x53}, /* 25 Left Channel De-emphasis Filter D1 Coefficient MSB                                                           */\
    {1, 26,0x7E}, /* 26 Left Channel De-emphasis Filter D1 Coefficient LSB                                                           */\
    {1, 27,0x6B}, /* 27 Right Channel Audio Effects Filter N0 Coefficient MSB                                                        */\
    {1, 28,0xE3}, /* 28 Right Channel Audio Effects Filter N0 Coefficient LSB                                                        */\
    {1, 29,0x96}, /* 29 Right Channel Audio Effects Filter N1 Coefficient MSB                                                        */\
                                                                                                                                       \
    {1, 30,0x66}, /* 30 Right Channel Audio Effects Filter N1 Coefficient LSB                                                        */\
    {1, 31,0x67}, /* 31 Right Channel Audio Effects Filter N2 Coefficient MSB                                                        */\
    {1, 32,0x5D}, /* 32 Right Channel Audio Effects Filter N2 Coefficient LSB                                                        */\
    {1, 33,0x6B}, /* 33 Right Channel Audio Effects Filter N3 Coefficient MSB                                                        */\
    {1, 34,0xE3}, /* 34 Right Channel Audio Effects Filter N3 Coefficient LSB                                                        */\
    {1, 35,0x96}, /* 35 Right Channel Audio Effects Filter N4 Coefficient MSB                                                        */\
    {1, 36,0x66}, /* 36 Right Channel Audio Effects Filter N4 Coefficient LSB                                                        */\
    {1, 37,0x67}, /* 37 Right Channel Audio Effects Filter N5 Coefficient MSB                                                        */\
    {1, 38,0x5D}, /* 38 Right Channel Audio Effects Filter N5 Coefficient LSB                                                        */\
    {1, 39,0x7D}, /* 39 Right Channel Audio Effects Filter D1 Coefficient MSB                                                        */\
                                                                                                                                       \
    {1, 40,0x83}, /* 40 Right Channel Audio Effects Filter D1 Coefficient LSB                                                        */\
    {1, 41,0x84}, /* 41 Right Channel Audio Effects Filter D2 Coefficient MSB                                                        */\
    {1, 42,0xEE}, /* 42 Right Channel Audio Effects Filter D2 Coefficient LSB                                                        */\
    {1, 43,0x7D}, /* 43 Right Channel Audio Effects Filter D4 Coefficient MSB                                                        */\
    {1, 44,0x83}, /* 44 Right Channel Audio Effects Filter D4 Coefficient LSB                                                        */\
    {1, 45,0x84}, /* 45 Right Channel Audio Effects Filter D5 Coefficient MSB                                                        */\
    {1, 46,0xEE}, /* 46 Right Channel Audio Effects Filter D5 Coefficient LSB                                                        */\
    {1, 47,0x39}, /* 47 Right Channel De-emphasis Filter N0 Coefficient MSB                                                          */\
    {1, 48,0x55}, /* 48 Right Channel De-emphasis Filter N0 Coefficient LSB                                                          */\
    {1, 49,0xF3}, /* 49 Right Channel De-emphasis Filter N1 Coefficient MSB                                                          */\
                                                                                                                                       \
    {1, 50,0x2D}, /* 50 Right Channel De-emphasis Filter N1 Coefficient LSB                                                          */\
    {1, 51,0x53}, /* 51 Right Channel De-emphasis Filter D1 Coefficient MSB                                                          */\
    {1, 52,0x7E}, /* 52 Right Channel De-emphasis Filter D1 Coefficient LSB                                                          */\
    {1, 53,0x7F}, /* 53 3-D Attenuation Coefficient MSB                                                                              */\
    {1, 54,0xFF}, /* 54 3-D Attenuation Coefficient LSB                                                                              */\
    {EVM_AIC33_ENDSETUP,EVM_AIC33_ENDSETUP,EVM_AIC33_ENDSETUP}                                                                         \
                  /* As a safety measure, this means end of setup                                                                    */\
    }             /*  do not remove and keep it last element                                                                         */\
}


// Data mode: I2S
// Data resolution: 16 Bits
// Frequency of operation: 47.99964KHz (error from 48.0KHz: -0.0007%)
//      FS = ( MCLK * K * R) / ( 2048 * P )
//      For a FS=47.99964 kHz & MCLK=11.633523 MHz
//          -> 47.99964kHz = ( 11.633523 MHz * K * R) / ( 2048 * P )
//          -> R = 1, P = 1, K[J.D] = 8.4500
// Register format: {page, regoff, value}
#define EVM_AIC33_16BITS_I2S_480 {                                                                                                     \
    {             /* Page 0                                                                                                          */\
    {0,  2,0x00}, /* 2  Codec Sample Rate Select        BV: [ADC=FS][DAC=FS]                                                         */\
                                                                                                                                       \
    /* For: [FS=47.99964 kHz][MCLK=11.633523 MHz]                                                                                    */\
    {0,  3,0x91}, /* 3  PLL Reg A                       BV: [PLL=ON][P=1][Q=2]                                                       */\
    {0,  4,0x20}, /* 4  PLL Reg B                       BV: [J=8]                                                                    */\
    {0,  5,0x46}, /* 5  PLL Reg C                       BV: [D=4500 Bin: 01000110 010100 MSB]                                        */\
    {0,  6,0x50}, /* 6  PLL Reg D                       BV: [D=4500 Bin: 01000110 010100 LSB]                                        */\
    {0,  7,0x0A}, /* 7  Codec Datapath Setup            BV: [FS=48.0 kHz][LeftDACPlays=left][RightDACPlays=right]                    */\
                                                                                                                                       \
    {0,  8,0x00}, /* 8  Audio Serial Data Reg A         BV: [BCLK=Slave][WCLK=Slave][3DEffects=OFF][DigitalMIC=OFF]                  */\
    {0,  9,0x00}, /* 9  Audio Serial Data Reg B         BV: [Mode=I2S][Lenght=16-bit][TransferMode=Continous]                        */\
    {0, 10,0x00}, /* 10 Audio Serial Data Reg C         BV: [DataOffset=0]                                                           */\
    {0, 11,0x01}, /* 11 Overflow Flag Register          BV: [PLL R=1]                                                                */\
    {0, 12,0x00}, /* 12 Digital Filter Ctrl Reg         BV: All filters/effects/emphasis disabled                                    */\
    {0, 13,0x00}, /* 13 Headset Detection Reg A         BV: [HeadsetDetect=OFF]                                                      */\
    {0, 14,0x00}, /* 14 Headset Detection Reg B         BV: []                                                                       */\
    {0, 15,0x00}, /* 15 Left  ADC PGA Gain Ctrl Reg     BV: [Mute=OFF]                                                               */\
    {0, 16,0x00}, /* 16 Right ADC PGA Gain Ctrl Reg     BV: [Mute=OFF]                                                               */\
    {0, 17,0xFF}, /* 17 MIC3L/R to Left  ADC Ctrl Reg   BV: [MIC3L=NoConnect][MIC3R=NoConnect]                                       */\
    {0, 18,0xFF}, /* 18 MIC3L/R to Right ADC Ctrl Reg   BV: [MIC3L=NoConnect][MIC3R=NoConnect]                                       */\
    {0, 19,0x04}, /* 19 LINE1L  to Left  ADC Ctrl Reg   BV: [LINE1L=SingleEnd][LINE1L=0dBGain][LeftADCPower=ON][SoftStep=OncePerFS]  */\
                                                                                                                                       \
    {0, 20,0x78}, /* 20 LINE2L  to Left  ADC Ctrl Reg   BV: [LINE2L=SingleEnd][LINE2L=NoConnect]                                     */\
    {0, 21,0x78}, /* 21 LINE1R  to Left  ADC Ctrl Reg   BV: [LINE1R=SingleEnd][LINE1R=NoConnect]                                     */\
    {0, 22,0x04}, /* 22 LINE1R  to Right ADC Ctrl Reg   BV: [LINE1R=SingleEnd][LINE1R=0dBGain][RightADCPower=ON][SoftStep=OncePerFS] */\
    {0, 23,0x78}, /* 23 LINE2R  to Right ADC Ctrl Reg   BV: [LINE2R=SingleEnd][LINE2R=NoConnect]                                     */\
    {0, 24,0x78}, /* 24 LINE1L  to Right ADC Ctrl Reg   BV: [LINE1L=SingleEnd][LINE1L=NoConnect]                                     */\
    {0, 25,0x00}, /* 25 MICBIAS Control Reg             BV: [Power=OFF]                                                              */\
    {0, 26,0x00}, /* 26 Left  AGC Control Reg A         BV: [LeftAGC=OFF]                                                            */\
    {0, 27,0x00}, /* 27 Left  AGC Control Reg B         BV: []                                                                       */\
    {0, 28,0x00}, /* 28 Left  AGC Control Reg C         BV: [LeftHystersis=OFF]                                                      */\
    {0, 29,0x00}, /* 29 Right AGC Control Reg A         BV: [RightAGC=OFF]                                                           */\
                                                                                                                                       \
    {0, 30,0x00}, /* 30 Right AGC Control Reg B         BV: []                                                                       */\
    {0, 31,0x00}, /* 31 Right AGC Control Reg C         BV: [RightHystersis=OFF]                                                     */\
    {0, 34,0x00}, /* 34 Left  AGC Noise Gate Debounce   BV: []                                                                       */\
    {0, 35,0x00}, /* 35 Right AGC Noise Gate Debounce   BV: []                                                                       */\
    {0, 37,0xE0}, /* 37 DAC Power & Output Dvr Ctrl Reg BV: [LeftDACPower=ON][RightDACPower=ON][HPLCOM=SingleEnd]                    */\
    {0, 38,0x10}, /* 38 High Power Output Dvr Ctrl Reg  BV: [HPRCOM=SingleEnd][ShortCircuit=OFF]                                     */\
                                                                                                                                       \
    {0, 40,0x00}, /* 40 Output Stage Ctrl Reg           BV: []                                                                       */\
    {0, 41,0x00}, /* 41 DAC Output Switching Ctrl Reg   BV: []                                                                       */\
    {0, 42,0x00}, /* 42 Output Driver Pop Reduction Reg BV: []                                                                       */\
    {0, 43,0x00}, /* 43 Left  DAC Digital Vol Reg       BV: [LeftDACMute=OFF]                                                        */\
    {0, 44,0x00}, /* 44 Right DAC Digital Vol Reg       BV: [RightDACMute=OFF]                                                       */\
    {0, 45,0x00}, /* 45 Line2L to HPLOUT Vol Reg        BV: []                                                                       */\
    {0, 46,0x00}, /* 46 PGA_L  to HPLOUT Vol Reg        BV: []                                                                       */\
    {0, 47,0x00}, /* 47 DAC_L1 to HPLOUT Vol Reg        BV: []                                                                       */\
    {0, 48,0x00}, /* 48 LINE2R to HPLOUT Vol Reg        BV: []                                                                       */\
    {0, 49,0x00}, /* 49 PGA_R  to HPLOUT Vol Reg        BV: []                                                                       */\
                                                                                                                                       \
    {0, 50,0x00}, /* 50 DAC_R1 to HPLOUT Vol Reg        BV: []                                                                       */\
    {0, 51,0x00}, /* 51           HPLOUT Output Reg     BV: []                                                                       */\
    {0, 52,0x00}, /* 52 LINE2L to HPLCOM Vol Reg        BV: []                                                                       */\
    {0, 53,0x00}, /* 53 PGA_L  to HPLCOM Vol Reg        BV: []                                                                       */\
    {0, 54,0x00}, /* 54 DAC_L1 to HPLCOM Vol Reg        BV: []                                                                       */\
    {0, 55,0x00}, /* 55 LINE2R to HPLCOM Vol Reg        BV: []                                                                       */\
    {0, 56,0x00}, /* 56 PGA_R  to HPLCOM Vol Reg        BV: []                                                                       */\
    {0, 57,0x00}, /* 57 DAC_R1 to HPLCOM Vol Reg        BV: []                                                                       */\
    {0, 58,0x00}, /* 58           HPLCOM Output Reg     BV: []                                                                       */\
    {0, 59,0x00}, /* 59 LINE2L to HPROUT Vol Reg        BV: []                                                                       */\
                                                                                                                                       \
    {0, 60,0x00}, /* 60 PGA_L  to HPROUT Vol Reg        BV: []                                                                       */\
    {0, 61,0x00}, /* 61 DAC_L1 to HPROUT Vol Reg        BV: []                                                                       */\
    {0, 62,0x00}, /* 62 LINE2R to HPROUT Vol Reg        BV: []                                                                       */\
    {0, 63,0x00}, /* 63 PGA_R  to HPROUT Vol Reg        BV: []                                                                       */\
    {0, 64,0x00}, /* 64 DAC_R1 to HPROUT Vol Reg        BV: []                                                                       */\
    {0, 65,0x00}, /* 65           HPROUT Output Reg     BV: []                                                                       */\
    {0, 66,0x00}, /* 66 LINE2L to HPRCOM Vol Reg        BV: []                                                                       */\
    {0, 67,0x00}, /* 67 PGA_L  to HPRCOM Vol Reg        BV: []                                                                       */\
    {0, 68,0x00}, /* 68 DAC_L1 to HPRCOM Vol Reg        BV: []                                                                       */\
    {0, 69,0x00}, /* 69 LINE2R to HPRCOM Vol Reg        BV: []                                                                       */\
                                                                                                                                       \
    {0, 70,0x00}, /* 70 PGA_R  to HPRCOM Vol Reg        BV: []                                                                       */\
    {0, 71,0x00}, /* 71 DAC_R1 to HPRCOM Vol Reg        BV: []                                                                       */\
    {0, 72,0x00}, /* 72           HPRCOM Output Reg     BV: []                                                                       */\
    {0, 73,0x00}, /* 73 LINE2L to MONO_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 74,0x00}, /* 74 PGA_L  to MONO_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 75,0x00}, /* 75 DAC_L1 to MONO_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 76,0x00}, /* 76 LINE2R to MONO_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 77,0x00}, /* 77 PGA_R  to MONO_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 78,0x00}, /* 78 DAC_R1 to MONO_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 79,0x00}, /* 79           MONO_LOP/M Output Reg BV: []                                                                       */\
                                                                                                                                       \
    {0, 80,0x00}, /* 80 LINE2L to LEFT_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 81,0x00}, /* 81 PGA_L  to LEFT_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 82,0x80}, /* 82 DAC_L1 to LEFT_LOP/M Vol Reg    BV: [Routed]                                                                 */\
    {0, 83,0x00}, /* 83 LINE2R to LEFT_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 84,0x00}, /* 84 PGA_R  to LEFT_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 85,0x00}, /* 85 DAC_R1 to LEFT_LOP/M Vol Reg    BV: []                                                                       */\
    {0, 86,0x09}, /* 86           LEFT_LOP/M Output Reg BV: [Mute=OFF][Power=ON]                                                     */\
    {0, 87,0x00}, /* 87 LINE2L to RIGHT_LOP/M Vol Reg   BV: []                                                                       */\
    {0, 88,0x00}, /* 88 PGA_L  to RIGHT_LOP/M Vol Reg   BV: []                                                                       */\
    {0, 89,0x00}, /* 89 DAC_L1 to RIGHT_LOP/M Vol Reg   BV: []                                                                       */\
                                                                                                                                       \
    {0, 90,0x00}, /* 90 LINE2R to RIGHT_LOP/M Vol Reg   BV: []                                                                       */\
    {0, 91,0x00}, /* 91 PGA_R  to RIGHT_LOP/M Vol Reg   BV: []                                                                       */\
    {0, 92,0x80}, /* 92 DAC_R1 to RIGHT_LOP/M Vol Reg   BV: [Routed]                                                                 */\
    {0, 93,0x09}, /* 93           RIGHT_LOP/M Outpt Reg BV: [Mute=OFF][Power=ON]                                                     */\
    {0, 98,0x10}, /* 98 GPIO1 Control Reg               BV: [GPIO1=ADC word clk]                                                     */\
    {0, 99,0x00}, /* 99 GPIO2 Control Reg               BV: []                                                                       */\
                                                                                                                                       \
    {0,100,0x00}, /* 100 Additional GPIO Control Reg A  BV: []                                                                       */\
    {0,101,0x00}, /* 101 Additional GPIO Control Reg B  BV: []                                                                       */\
    {0,102,0x02}, /* 102 Clock Generation Control Reg   BV: []                                                                       */\
                  /* Page 1                                                                                                          */\
    {1,  1,0x6B}, /*  1 Left Channel Audio Effects Filter N0 Coefficient MSB                                                         */\
    {1,  2,0xE3}, /*  2 Left Channel Audio Effects Filter N0 Coefficient LSB                                                         */\
    {1,  3,0x96}, /*  3 Left Channel Audio Effects Filter N1 Coefficient MSB                                                         */\
    {1,  4,0x66}, /*  4 Left Channel Audio Effects Filter N1 Coefficient LSB                                                         */\
    {1,  5,0x67}, /*  5 Left Channel Audio Effects Filter N2 Coefficient MSB                                                         */\
    {1,  6,0x5D}, /*  6 Left Channel Audio Effects Filter N2 Coefficient LSB                                                         */\
    {1,  7,0x6B}, /*  7 Left Channel Audio Effects Filter N3 Coefficient MSB                                                         */\
    {1,  8,0xE3}, /*  8 Left Channel Audio Effects Filter N3 Coefficient LSB                                                         */\
    {1,  9,0x96}, /*  9 Left Channel Audio Effects Filter N4 Coefficient MSB                                                         */\
                                                                                                                                       \
    {1, 10,0x66}, /* 10 Left Channel Audio Effects Filter N4 Coefficient LSB                                                         */\
    {1, 11,0x67}, /* 11 Left Channel Audio Effects Filter N5 Coefficient MSB                                                         */\
    {1, 12,0x5D}, /* 12 Left Channel Audio Effects Filter N5 Coefficient LSB                                                         */\
    {1, 13,0x7D}, /* 13 Left Channel Audio Effects Filter D1 Coefficient MSB                                                         */\
    {1, 14,0x83}, /* 14 Left Channel Audio Effects Filter D1 Coefficient LSB                                                         */\
    {1, 15,0x84}, /* 15 Left Channel Audio Effects Filter D2 Coefficient MSB                                                         */\
    {1, 16,0xEE}, /* 16 Left Channel Audio Effects Filter D2 Coefficient LSB                                                         */\
    {1, 17,0x7D}, /* 17 Left Channel Audio Effects Filter D4 Coefficient MSB                                                         */\
    {1, 18,0x83}, /* 18 Left Channel Audio Effects Filter D4 Coefficient LSB                                                         */\
    {1, 19,0x84}, /* 19 Left Channel Audio Effects Filter D5 Coefficient MSB                                                         */\
                                                                                                                                       \
    {1, 20,0xEE}, /* 20 Left Channel Audio Effects Filter D5 Coefficient LSB                                                         */\
    {1, 21,0x39}, /* 21 Left Channel De-emphasis Filter N0 Coefficient MSB                                                           */\
    {1, 22,0x55}, /* 22 Left Channel De-emphasis Filter N0 Coefficient LSB                                                           */\
    {1, 23,0xF3}, /* 23 Left Channel De-emphasis Filter N1 Coefficient MSB                                                           */\
    {1, 24,0x2D}, /* 24 Left Channel De-emphasis Filter N1 Coefficient LSB                                                           */\
    {1, 25,0x53}, /* 25 Left Channel De-emphasis Filter D1 Coefficient MSB                                                           */\
    {1, 26,0x7E}, /* 26 Left Channel De-emphasis Filter D1 Coefficient LSB                                                           */\
    {1, 27,0x6B}, /* 27 Right Channel Audio Effects Filter N0 Coefficient MSB                                                        */\
    {1, 28,0xE3}, /* 28 Right Channel Audio Effects Filter N0 Coefficient LSB                                                        */\
    {1, 29,0x96}, /* 29 Right Channel Audio Effects Filter N1 Coefficient MSB                                                        */\
                                                                                                                                       \
    {1, 30,0x66}, /* 30 Right Channel Audio Effects Filter N1 Coefficient LSB                                                        */\
    {1, 31,0x67}, /* 31 Right Channel Audio Effects Filter N2 Coefficient MSB                                                        */\
    {1, 32,0x5D}, /* 32 Right Channel Audio Effects Filter N2 Coefficient LSB                                                        */\
    {1, 33,0x6B}, /* 33 Right Channel Audio Effects Filter N3 Coefficient MSB                                                        */\
    {1, 34,0xE3}, /* 34 Right Channel Audio Effects Filter N3 Coefficient LSB                                                        */\
    {1, 35,0x96}, /* 35 Right Channel Audio Effects Filter N4 Coefficient MSB                                                        */\
    {1, 36,0x66}, /* 36 Right Channel Audio Effects Filter N4 Coefficient LSB                                                        */\
    {1, 37,0x67}, /* 37 Right Channel Audio Effects Filter N5 Coefficient MSB                                                        */\
    {1, 38,0x5D}, /* 38 Right Channel Audio Effects Filter N5 Coefficient LSB                                                        */\
    {1, 39,0x7D}, /* 39 Right Channel Audio Effects Filter D1 Coefficient MSB                                                        */\
                                                                                                                                       \
    {1, 40,0x83}, /* 40 Right Channel Audio Effects Filter D1 Coefficient LSB                                                        */\
    {1, 41,0x84}, /* 41 Right Channel Audio Effects Filter D2 Coefficient MSB                                                        */\
    {1, 42,0xEE}, /* 42 Right Channel Audio Effects Filter D2 Coefficient LSB                                                        */\
    {1, 43,0x7D}, /* 43 Right Channel Audio Effects Filter D4 Coefficient MSB                                                        */\
    {1, 44,0x83}, /* 44 Right Channel Audio Effects Filter D4 Coefficient LSB                                                        */\
    {1, 45,0x84}, /* 45 Right Channel Audio Effects Filter D5 Coefficient MSB                                                        */\
    {1, 46,0xEE}, /* 46 Right Channel Audio Effects Filter D5 Coefficient LSB                                                        */\
    {1, 47,0x39}, /* 47 Right Channel De-emphasis Filter N0 Coefficient MSB                                                          */\
    {1, 48,0x55}, /* 48 Right Channel De-emphasis Filter N0 Coefficient LSB                                                          */\
    {1, 49,0xF3}, /* 49 Right Channel De-emphasis Filter N1 Coefficient MSB                                                          */\
                                                                                                                                       \
    {1, 50,0x2D}, /* 50 Right Channel De-emphasis Filter N1 Coefficient LSB                                                          */\
    {1, 51,0x53}, /* 51 Right Channel De-emphasis Filter D1 Coefficient MSB                                                          */\
    {1, 52,0x7E}, /* 52 Right Channel De-emphasis Filter D1 Coefficient LSB                                                          */\
    {1, 53,0x7F}, /* 53 3-D Attenuation Coefficient MSB                                                                              */\
    {1, 54,0xFF}, /* 54 3-D Attenuation Coefficient LSB                                                                              */\
    {EVM_AIC33_ENDSETUP,EVM_AIC33_ENDSETUP,EVM_AIC33_ENDSETUP}                                                                         \
                  /* As a safety measure, this means end of setup                                                                    */\
    }             /*  do not remove and keep it last element                                                                         */\
}


// Codec Handle
typedef Int32 EVM_AIC33_Handle;

// Register structure
typedef struct
{
    Uint8 regs[EVM_AIC33_NREGS][3];
} EVM_AIC33_Setup;


/****************************************************************************
 *                             Public Functions                             *
 ****************************************************************************/

/**************************************************************************** 
 *
 * NAME  
 *      EVM_AIC33_open
 *
 * PURPOSE:
 *      Open an audio codec device and return a handle.
 *      At opening, the AIC33 audio codec is initialized according to
 *      the specified setup matrix. When the device is not used, the device must
 *      be closed by EVM_AIC33_close. 
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      EVM_AIC33_Handle EVM_AIC33_open( Uint32 id, EVM_AIC33_Setup *setup )
 *
 *      id    - (i) id of desired audio codec [EVM_AIC33_DEV1-EVM_AIC33_DEV4]
 *      setup - (i) audo codec setup structure
 *
 *      N.B. In a multiple codec configuration, one of the device must be set
 *      as the master codec using the function EVM_AIC33_setmaster after
 *      a successful opening.
 *
 * RETURN VALUE
 *      Handle value, >= EVM_AIC33_NHANDLE in case of problem
 *
 * REFERENCE
 *
 ****************************************************************************/
EVM_AIC33_Handle EVM_AIC33_open( Uint32 id, EVM_AIC33_Setup *setup );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_AIC33_close
 *
 * PURPOSE:
 *      Close an audio codec device previously opened by EVM_AIC33_open.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_AIC33_close( EVM_AIC33_Handle *AIC33handle )
 *
 *      AIC33handle - (i) handle of the audio codec returned when opened.
 *
 *      N.B. In a multiple codecs configuration, the device that has been set
 *      the master codec with the function EVM_AIC33_setmaster must be the last
 *      one to be closed using this function (or a new codec master must be defined
 *      after closing it).
 *      Closing the master codec will stop all others from working correctly.
 *
 * RETURN VALUE
 *      EVM_AIC33_OK if OK, !=EVM_AIC33_OK if problems
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_AIC33_close( EVM_AIC33_Handle *AIC33handle );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_AIC33_setup
 *
 * PURPOSE:
 *      Setup the current audio codec with default register values.
 *      A setup structure is given for this purpose.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_AIC33_setup( EVM_AIC33_Handle AIC33handle, EVM_AIC33_Setup *setup )
 *
 *      AIC33handle - (i) handle of the audio codec returned when opened.
 *
 *      setup       - (i) codec setup data structure.
 *
 * RETURN VALUE
 *      EVM_AIC33_OK if OK, !=EVM_AIC33_OK if problems
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_AIC33_setup( EVM_AIC33_Handle AIC33handle, EVM_AIC33_Setup *setup );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_AIC33_rget
 *
 * PURPOSE:
 *      Read a specified register of the current AIC33 codec handle.
 *      The register will be read from the codec associated with the current
 *      handle, at the specified offset position, and codec registers page.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_AIC33_rget( EVM_AIC33_Handle AIC33handle,  Uint16 regpage, Uint16 regnum, Uint16* regval )
 *
 *      AIC33handle - (i) handle of the audio codec returned when opened.
 *
 *      regpage     - (i) codec register page (0 or 1).
 *
 *      regnum      - (i) codec register offset in page.
 *
 *      regval      - (o) codec register value read.
 *
 * RETURN VALUE
 *      EVM_AIC33_OK if OK, !=EVM_AIC33_OK if problems
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_AIC33_rget( EVM_AIC33_Handle AIC33handle,  Uint16 regpage, Uint16 regnum, Uint16* regval );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_AIC33_rgetset
 *
 * PURPOSE:
 *      Read a  register, change a bitfield and then writeback the register.
 *      Bitmask values and bitval are binary value with 1 set at the bit position
 *      to modify in the register. It will work on the current codec for
 *      the specified handle.
 *      Pseudocode operation:
 *       1- Value1 = Read AIC33 codec register regnum in page regpage of AIC33handle
 *       2- Value1 = Value1 & (~bitmask)
 *       3- Value1 = Value1 | (bitmask & bitval)
 *       4- Write AIC33 codec register regnum of AIC33handle with Value1
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_AIC33_rgetset( EVM_AIC33_Handle AIC33handle, Uint16 regpage, Uint16 regnum, Uint16 bitmask, Uint16 bitval )
 *
 *      AIC33handle - (i) handle of the audio codec returned when opened.
 *
 *      regpage     - (i) codec register page (0 or 1).
 *
 *      regnum      - (i) codec register offset in page.
 *
 *      bitmask     - (i) bit mask value with bit to keep set as 1.
 *
 *      bitval      - (i) bit values to be set at mask position.
 *
 * RETURN VALUE
 *      EVM_AIC33_OK if OK, !=EVM_AIC33_OK if problems
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_AIC33_rgetset( EVM_AIC33_Handle AIC33handle, Uint16 regpage, Uint16 regnum, Uint16 bitmask, Uint16 bitval );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_AIC33_rset
 *
 * PURPOSE:
 *      Write a register with a specified value for the codec of this handle.
 *      The register will be written from the codec associated with the current
 *      handle, at the specified offset position, and codec registers page.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_AIC33_rset( EVM_AIC33_Handle AIC33handle, Uint16 regpage, Uint16 regnum, Uint16 regval )
 *
 *      AIC33handle - (i) handle of the audio codec returned when opened.
 *
 *      regpage     - (i) codec register page (0 or 1).
 *
 *      regnum      - (i) codec register offset in page.
 *
 *      regval      - (i) register value to be written.
 *
 * RETURN VALUE
 *      EVM_AIC33_OK if OK, !=EVM_AIC33_OK if problems
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_AIC33_rset( EVM_AIC33_Handle AIC33handle, Uint16 regpage, Uint16 regnum, Uint16 regval );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_AIC33_setmaster
 *
 * PURPOSE:
 *      Specified that the audio codec is a clock master (means it will clock all
 *      3 other audio codecs). There must be 1 of the codecs set as a master.
 *      (All codecs must be set clock slave by default when the codecs are opened).
 *      It should be noted that only one codec can be set as master. If another
 *      codec is to be set as master, the actual master must be closed and then
 *      reopened in default slave mode to set the new codec master.
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_AIC33_setmaster( EVM_AIC33_Handle AIC33handle)
 *
 *      AIC33handle - (i) handle of the audio codec returned when opened.
 *
 * RETURN VALUE
 *      EVM_AIC33_OK if OK, !=EVM_AIC33_OK if problems
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_AIC33_setmaster( EVM_AIC33_Handle AIC33handle);


/**************************************************************************** 
 *
 * NAME  
 *      EVM_AIC33_setmute
 *
 * PURPOSE:
 *      Set the mute control of the lineout output of an audio codec
 *      (both channels Left and Right).
 *
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_AIC33_setmute ( EVM_AIC33_Handle AIC33handle, Uint16 mode )
 *
 *      AIC33handle - (i) handle of the audio codec returned when opened.
 *
 *      mode        - (i) EVM_AIC33_MUTEON or EVM_AIC33_MUTEOFF.
 *
 * RETURN VALUE
 *      EVM_AIC33_OK if OK, !=EVM_AIC33_OK if problems
 *
 * REFERENCE
 *
 ****************************************************************************/
Int32 EVM_AIC33_setmute ( EVM_AIC33_Handle AIC33handle, Uint16 mode );


#endif
