/***************************************************************************
 *
 *
 *         **   **    **  ******  ********  ********  ********  **    **    
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ***************************************************************************
 *                                                                          
 *  Project     : EVM
 *  File        : EVM_ECP.H
 *  Description : Utility functions for writing/reading ECP on the I2C bus
 *
 *                      Copyright (c) Lyrtech inc. 2007                        
 *
 ***************************************************************************
 *                                                                                          
 * "$Revision: 1.5 $"
 * "$Date: 2007/09/13 15:49:39 $"
 *
 ***************************************************************************/

#ifndef __EVM_ECP_H__
#define __EVM_ECP_H__

/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include <tistdtypes.h>


/****************************************************************************
 *                             Local constants                              *
 ****************************************************************************/

// Default ECP I2C address
#define EVM_ECP_ADDRESS              (0x70)  // MSP430(ECP) I2C address

// ECP registers definitions
#define EVM_ECP_CMD_REG              (0x00)  // ECP Command register
#define EVM_ECP_STATUS_REG           (0x01)  // ECP Status register
#define EVM_ECP_RST_IN_REG           (0x02)  // ECP Reset in register
#define EVM_ECP_RST_OUT_REG          (0x03)  // ECP Reset out register
#define EVM_ECP_PCI_CTRL_REG         (0x04)  // ECP PCI config register
#define EVM_ECP_REV_REG              (0x05)  // ECP Revision register
#define EVM_ECP_VIDEO_MUX_REG        (0x06)  // ECP Video Mux register
#define EVM_ECP_SWITCH_LED_REG       (0x07)  // ECP Switch & LED register
#define EVM_ECP_DCGPIO_DIR_CTRL_REG  (0x08)  // ECP DC GPIO direction control
#define EVM_ECP_DCGPIO_STATE_REG     (0x09)  // ECP DC GPIO pins state
#define EVM_ECP_TIMER_REG            (0x0A)  // ECP DSP Timer interface
#define EVM_ECP_PWR_CTRL_REG         (0x0B)  // ECP Power Supply control
#define EVM_ECP_BTMD_LOW_REG         (0x0C)  // ECP Bootmode low (read live)
#define EVM_ECP_BTMD_HIGH_REG        (0x0D)  // ECP Bootmode high (read live)
#define EVM_ECP_BTMD_OVR_LOW_REG     (0x0E)  // ECP Bootmode override low
#define EVM_ECP_BTMD_OVR_HIGH_REG    (0x0F)  // ECP Bootmode override high
#define EVM_ECP_UART_SPI_CTRL_REG    (0x10)  // ECP UART SPI control
#define EVM_ECP_VID_STATUS_REG       (0x11)  // ECP Video status
#define EVM_ECP_CLK_CTRL_REG         (0x12)  // ECP Clock control
#define EVM_ECP_SPDIF_CTRL_REG       (0x13)  // ECP SPDIF control
#define EVM_ECP_MCASP_MUX_CTRL_REG   (0x13)  // ECP McASP mux control
// Beta and later
#define EVM_ECP_PER_RST_CTRL_REG     (0x14)  // ECP Peripheral Resets control
#define EVM_ECP_PWR_STATUS_REG       (0x15)  // ECP Power supply status
#define EVM_ECP_MSP430_LOW_VOLT_REG  (0x16)  // ECP MSP430 Low Voltage count
#define EVM_ECP_MSP430_HIGH_VOLT_REG (0x17)  // ECP MSP430 High Voltage count
#define EVM_ECP_CORE_LOW_VOLT_REG    (0x18)  // ECP Core Low Voltage count
#define EVM_ECP_CORE_HIGH_VOLT_REG   (0x19)  // ECP Core High Voltage count
#define EVM_ECP_SGMII_LOW_VOLT_REG   (0x1A)  // ECP SGMII Low Voltage count
#define EVM_ECP_SGMII_HIGH_VOLT_REG  (0x1B)  // ECP SGMII High Voltage count
#define EVM_ECP_1_8V_LOW_VOLT_REG    (0x1C)  // ECP 1.8V Low Voltage count
#define EVM_ECP_1_8V_HIGH_VOLT_REG   (0x1D)  // ECP 1.8V High Voltage count
#define EVM_ECP_2_5V_LOW_VOLT_REG    (0x1E)  // ECP 2.5V Low Voltage count
#define EVM_ECP_2_5V_HIGH_VOLT_REG   (0x1F)  // ECP 2.5V High Voltage count
#define EVM_ECP_3_3V_LOW_VOLT_REG    (0x20)  // ECP 3.3V Low Voltage count
#define EVM_ECP_3_3V_HIGH_VOLT_REG   (0x21)  // ECP 3.3V High Voltage count
#define EVM_ECP_5_0V_LOW_VOLT_REG    (0x22)  // ECP 5.0V Low Voltage count
#define EVM_ECP_5_0V_HIGH_VOLT_REG   (0x23)  // ECP 5.0V High Voltage count
#define EVM_ECP_HDVID_CLK_CTRL_REG   (0x24)  // ECP HD video clock control
#define EVM_ECP_LAST_BTMD_LOW_REG    (0x25)  // ECP Last Bootmode used low
#define EVM_ECP_LAST_BTMD_HIGH_REG   (0x26)  // ECP Last Bootmode used high
                                    
                                    
// ECP functions error status codes  
// -------------------------------- -------------------------------------------
#define EVM_ECP_OK                0
#define EVM_ECP_WRITEREG_FAIL     0x1000
#define EVM_ECP_STATUSREGRD_FAIL  0x1001
#define EVM_ECP_READREG_FAIL      0x1002
// ECP Status register error code return in the following range:
// [EVM_ECP_STATUSREG_CODE_ST,EVM_ECP_STATUSREG_CODE_EN]
// This status code error is (after removing EVM_ECP_STATUSREG_CODE_ST):
// Bit7     Bit6     Bit5     Bit4     Bit3     Bit2     Bit1     Bit0
// Reserved Reserved Reserved Invalid  Port     Invalid  I2C      Busy
//                            Command  Error    Offset   Error
#define EVM_ECP_STATUSREG_CODE_ST 0x1100
#define EVM_ECP_STATUSREG_CODE_EN 0x11FF
// ---------------------------------------------------------------------------



/****************************************************************************
 *                           Forward Declarations                           *
 ****************************************************************************/


/****************************************************************************
 *                             Public Functions                             *
 ****************************************************************************/


/**************************************************************************** 
 *
 * NAME  
 *      EVM_ECP_writereg
 *
 * PURPOSE:
 *      This function writes a value to a register of the ECP device
 *      using the I2C bus.
 *         
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_ECP_writereg( Int32 devAddr, Uint8 reg, Uint8 val ); 
 *
 *      devAddr - (i)   I2C 7 bits device address of ECP
 *      reg     - (i)   Register offset to write
 *      val     - (i)   Value (byte) to write
 *
 * RETURN VALUE
 *      !=EVM_ECP_OK for failure, EVM_ECP_OK for success
 *      
 * REFERENCE
 *
 ****************************************************************************/    
Int32 EVM_ECP_writereg( Int32 devAddr, Uint8 reg, Uint8 val );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_ECP_readreg
 *
 * PURPOSE:
 *      This function reads the value of a register of the ECP device
 *      using the I2C bus.
 *         
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_ECP_readreg( Int32 devAddr, Uint8 reg, Uint8 *val ); 
 *
 *      devAddr - (i)   I2C 7 bits device address of ECP
 *      reg     - (i)   Register offset to read
 *      val     - (o)   Value (byte) read
 *
 * RETURN VALUE
 *      !=EVM_ECP_OK for failure, EVM_ECP_OK for success
 *      
 * REFERENCE
 *
 ****************************************************************************/    
Int32 EVM_ECP_readreg( Int32 devAddr, Uint8 reg, Uint8 *val );


#endif //__EVM_ECP_H__
