/***************************************************************************
 *
 *
 *         **   **    **  ******  ********  ********  ********  **    **    
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ***************************************************************************
 *                                                                          
 *  Project     : EVM
 *  File        : EVM_I2C.h
 *  Description : Utility functions for writing/reading on the I2C bus
 *
 *                      Copyright (c) Lyrtech inc. 2007                        
 *
 ***************************************************************************
 *                                              				                            
 * "$Revision: 1.7 $"
 * "$Date: 2007/09/18 14:50:48 $"
 *
 ***************************************************************************/

#ifndef __EVM_I2C_H__
#define __EVM_I2C_H__

/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include <tistdtypes.h>


/****************************************************************************
 *                             Public Constants                             *
 ****************************************************************************/
// I2c functions error status codes 
#define EVM_I2C_OK           0
#define EVM_I2C_FAIL         0x2001
#define EVM_I2C_BUSBUSY      0x2002    // Problem BUS busy before writing command
#define EVM_I2C_MASTERCLRERR 0x2003    // Problem Master mode clear do not happened
#define EVM_I2C_READYTXERR   0x2003    // Problem not ready to transmit after a while
#define EVM_I2C_REGACCRDYERR 0x2004    // Problem waiting register access ready
#define EVM_I2C_BUSNOTBUSY   0x2005    // Problem BUS not busy timeout
#define EVM_I2C_NAKTIMEOUT   0x2007    // Problem maximum number of NAK has been reach
#define EVM_I2C_DATARDYERR   0x2008    // Problem waiting for data ready
#define EVM_I2C_ENABLEI2CERR 0x2009    // Problem cannot enable i2c controler


/****************************************************************************
 *                             Public Functions                             *
 ****************************************************************************/


/**************************************************************************** 
 *
 * NAME  
 *      EVM_I2C_writereg
 *
 * PURPOSE:
 *      This function writes a value (byte) to a register of the specified
 *      device using the I2C bus.
 *         
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_I2C_writereg( Int32 devAddr, Uint8 reg, Uint8 val ); 
 *      
 *      devAddr - (i)   Id of the device to write to (i2c 7 bits slave address)
 *      reg     - (i)   Register offset within the device
 *      val     - (i)   Value to be written
 *
 * RETURN VALUE
 *      EVM_I2C_OK for success, !=EVM_I2C_OK if problem
 *      
 * REFERENCE
 *
 ****************************************************************************/    
Int32 EVM_I2C_writereg( Int32 devAddr, Uint8 reg, Uint8 val );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_I2C_writemem
 *
 * PURPOSE:
 *      This function writes N bytes to the specified device on i2c bus.
 *      This function is usually used for memory device type (16 bits address).
 *         
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_I2C_writemem( Int32 devAddr, Uint16 addr, Uint8 *buf, Int32 n ); 
 *
 *      devAddr - (i)   Id of the device to write to (i2c 7 bits slave address)
 *      addr    - (i)   Address (16 bits memory offset address) 
 *      buf     - (i)   Buffer of data to write
 *      n       - (i)   Number of bytes to write
 *
 * RETURN VALUE
 *      EVM_I2C_OK for success, !=EVM_I2C_OK if problem
 *      
 * REFERENCE
 *
 ****************************************************************************/    
Int32 EVM_I2C_writemem( Int32 devAddr, Uint16 addr, Uint8 *buf, Int32 n );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_I2C_writeregbuf
 *
 * PURPOSE:
 *      This function writes N bytes to the specified device register from
 *      a buffer.
 *         
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_I2C_writeregbuf( Int32 devAddr, Uint8 addr, Uint8 *buf, Int32 n ); 
 *
 *      devAddr - (i)   Id of the device to write to (i2c 7 bits slave address)
 *      addr    - (i)   Address (8 bits register offset in the device)
 *      buf     - (i)   Buffer of data to write
 *      n       - (i)   Number of bytes to write
 *
 * RETURN VALUE
 *      EVM_I2C_OK for success, !=EVM_I2C_OK if problem
 *      
 * REFERENCE
 *
 ****************************************************************************/    
Int32 EVM_I2C_writeregbuf( Int32 devAddr, Uint8 addr, Uint8 *buf, Int32 n );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_I2C_writeval
 *
 * PURPOSE:
 *      This function send a byte to the specified device using the I2C bus.
 *      It is used to send a single bytes to a i2c device.
 *         
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_I2C_writeval( Int32 devAddr, Uint8 val ); 
 *
 *      devAddr - (i)   Id of the device to write to (i2c 7 bits slave address)
 *      val     - (i)   Value of the byte to be written
 *
 * RETURN VALUE
 *      EVM_I2C_OK for success, !=EVM_I2C_OK if problem
 *      
 * REFERENCE
 *
 ****************************************************************************/    
Int32 EVM_I2C_writeval( Int32 devAddr, Uint8 val );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_I2C_readreg
 *
 * PURPOSE:
 *      This function reads the value (byte) of a register of the specified device
 *      using the I2C bus.
 *         
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_I2C_readreg( Int32 devAddr, Uint8 reg, Uint8 *val ); 
 *
 *      devAddr - (i)   Id of the device to write to (i2c 7 bits slave address)
 *      reg     - (i)   Register offset (8 bits)
 *      val     - (o)   Value read (byte)
 *
 * RETURN VALUE
 *      EVM_I2C_OK for success, !=EVM_I2C_OK if problem
 *      
 * REFERENCE
 *
 ****************************************************************************/    
Int32 EVM_I2C_readreg( Int32 devAddr, Uint8 reg, Uint8 *val );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_I2C_readmem
 *
 * PURPOSE:
 *      This function reads N bytes from the specified device. This function
 *      is used for a memory type device (16 bits address).
 *         
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_I2C_readmem( Int32 devAddr, Uint16 addr, Uint8 *buf, Int32 n ); 
 *
 *      devAddr - (i)   Id of the device to write to (i2c 7 bits slave address)
 *      addr    - (i)   Address offset to read (16 bits)
 *      buf     - (o)   Buffer of data that will get the read
 *      n       - (i)   Number of bytes to read
 *
 * RETURN VALUE
 *      EVM_I2C_OK for success, !=EVM_I2C_OK if problem
 *      
 * REFERENCE
 *
 ****************************************************************************/    
Int32 EVM_I2C_readmem( Int32 devAddr, Uint16 addr, Uint8 *buf, Int32 n );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_I2C_readregbuf
 *
 * PURPOSE:
 *      This function reads N bytes for register addr from the specified device.
 *      It can be used to read buffer from a 8 bits register offset.
 *         
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_I2C_readregbuf( Int32 devAddr, Uint8 addr, Uint8 *buf, Int32 n ); 
 *
 *      devAddr - (i)   Id of the device to write to (i2c 7 bits slave address)
 *      addr    - (i)   Address offset for the read (8 bits)
 *      buf     - (o)   Buffer of data to get the read values
 *      n       - (i)   Number of bytes to read
 *
 * RETURN VALUE
 *      EVM_I2C_OK for success, !=EVM_I2C_OK if problem
 *      
 * REFERENCE
 *
 ****************************************************************************/    
Int32 EVM_I2C_readregbuf( Int32 devAddr, Uint8 addr, Uint8 *buf, Int32 n );


/**************************************************************************** 
 *
 * NAME  
 *      EVM_I2C_readval
 *
 * PURPOSE:
 *      This function read a byte from the specified device using the I2C bus.
 *         
 * USAGE
 *      This routine is C-callable and can be called as:
 *
 *      Int32 EVM_I2C_readval( Int32 devAddr, Uint8 *val ); 
 *
 *      devAddr - (i)   Id of the device to read from (i2c 7 bits slave address)
 *      val     - (o)   Value (byte read)
 *
 * RETURN VALUE
 *      EVM_I2C_OK for success, !=EVM_I2C_OK if problem
 *      
 * REFERENCE
 *
 ****************************************************************************/    
Int32 EVM_I2C_readval( Int32 devAddr, Uint8 *val );


#endif //__EVM_I2C_H__
