/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** @file  csl_pllc.h
 *
 *  @brief PLLC functional layer API header file
 *
 *  Path: \(CSLPATH)\inc
 */

/** @mainpage PLLC CSL 3.x
 *
 * @section Introduction
 *
 * @subsection xxx Purpose and Scope
 * The purpose of this document is to identify a set of common CSL APIs for
 * the PLLC module across various devices. The CSL developer is expected to
 * refer to this document while designing APIs for these modules. Some of the
 * listed APIs may not be applicable to a given PLLC module. While other cases
 * this list of APIs may not be sufficient to cover all the features of a
 * particular PLLC Module. The CSL developer should use his discretion designing
 * new APIs or extending the existing ones to cover these.
 *
 * @subsection aaa Terms and Abbreviations
 *   -# CSL:  Chip Support Library
 *   -# API:  Application Programmer Interface
 *
 * @subsection References
 *    -# CSL-001-DES, CSL 3.x Design Specification Document Version 1.02
 *
 */

/* =============================================================================
 *  Revision History
 *  ===============
 *  10-Feb-2004 kpn CSL3X Upgradation.
 *  21-Apr-2006 RR   updated for faraday
 * =============================================================================
 */

#ifndef _CSL_PLLC_H_
#define _CSL_PLLC_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <cslr.h>
#include <soc.h>
#include <csl_error.h>
#include <csl_types.h>
#include <cslr_pllc.h>

/**
\defgroup CSL_PLLC_API PLLC
   \internal Based on <b>PLLC Spec Rev 3.4 </b>
*/
/**
\defgroup CSL_PLLC_DATASTRUCT Data Structures
\ingroup CSL_PLLC_API
*/
/**
\defgroup CSL_PLLC_DEFINE  Defines
\ingroup CSL_PLLC_API
*/
/**
\defgroup CSL_PLLC_ENUM  Enumerated Data Types
\ingroup CSL_PLLC_API
*/

/**
\defgroup CSL_PLLC_FUNCTION  Functions
\ingroup CSL_PLLC_API
*/

/*****************************************************************************\
          PLLC global macro declarations
\*****************************************************************************/

/** \defgroup CSL_PLLC_STATUS_DEFINE PLL Controller Status
 *  \ingroup CSL_PLLC_DEFINE
 *
 * @{ */

/** Set when GO operation (divide-ratio change and
 *   clock alignment) is in progress
 */
#define CSL_PLLC_STATUS_GO CSL_FMKT (PLLC_PLLSTAT_GOSTAT, INPROG)

/** Set when PLL core is locked */
#define CSL_PLLC_STATUS_LOCK CSL_FMKT (PLLC_PLLSTAT_LOCK, YES)

/** Set when OSCIN/CLKIN is assumed to be stable */
#define CSL_PLLC_STATUS_STABLE CSL_FMKT (PLLC_PLLSTAT_STABLE, YES)

/**
@} */

/** \defgroup CSL_PLLC_DCHANGESTAT_DEFINE PLLC Divider Ratio Modified Status
 *  \ingroup CSL_PLLC_DEFINE
 *
 * @{ */

/** SYSCLK13 divide ratio is modified */
#define CSL_PLLC_DCHANGESTAT_SYS13 CSL_FMKT (PLLC_DCHANGE_SYS13, YES)
/** SYSCLK12 divide ratio is modified */
#define CSL_PLLC_DCHANGESTAT_SYS12 CSL_FMKT (PLLC_DCHANGE_SYS12, YES)
/** SYSCLK10 divide ratio is modified */
#define CSL_PLLC_DCHANGESTAT_SYS10 CSL_FMKT (PLLC_DCHANGE_SYS10, YES)

/**
@} */

/** \defgroup CSL_PLLC_CLKSTAT_DEFINE PLLC Clock Status
 *  \ingroup CSL_PLLC_DEFINE
 *
 * @{ */

/** AUXCLK is ON */
#define CSL_PLLC_CLKSTAT_AUXON CSL_FMKT (PLLC_CKSTAT_AUXEN, ON)
/** OBSCLK is ON */
#define CSL_PLLC_CLKSTAT_OBSON CSL_FMKT (PLLC_CKSTAT_OBSEN, ON)
/** SYSCLKBP is ON */
#define CSL_PLLC_CLKSTAT_BPON CSL_FMKT (PLLC_CKSTAT_BPON, ON)

/**
@} */

/** \defgroup CSL_PLLC_CLKEN_DEFINE PLLC Clock Enable Status
 *  \ingroup CSL_PLLC_DEFINE
 *
 * @{ */

/** AUXCLK enable */
#define CSL_PLLC_CLKEN_AUXEN CSL_FMKT (PLLC_CKEN_AUXEN, ENABLE)
/** OBSCLK enable */
#define CSL_PLLC_CLKEN_OBSEN CSL_FMKT (PLLC_CKEN_OBSEN, ENABLE)

/**
@} */

/** \defgroup CSL_PLLC_SYSCLKSTAT_DEFINE PLLC SYSCLK Status
 *  \ingroup CSL_PLLC_DEFINE
 *
 * @{ */

/** SYSCLK13 is ON */
#define CSL_PLLC_SYSCLKSTAT_SYS14ON CSL_FMKT (PLLC_CKSTAT_SYS14ON, ON)
/** SYSCLK12 is ON */
#define CSL_PLLC_SYSCLKSTAT_SYS13ON CSL_FMKT (PLLC_CKSTAT_SYS13ON, ON)
/** SYSCLK11 is ON */
#define CSL_PLLC_SYSCLKSTAT_SYS12ON CSL_FMKT (PLLC_CKSTAT_SYS12ON, ON)
/** SYSCLK10 is ON */
#define CSL_PLLC_SYSCLKSTAT_SYS11ON CSL_FMKT (PLLC_CKSTAT_SYS11ON, ON)
/** SYSCLK9 is ON */
#define CSL_PLLC_SYSCLKSTAT_SYS10ON CSL_FMKT (PLLC_CKSTAT_SYS10ON, ON)
/** SYSCLK8 is ON */
#define CSL_PLLC_SYSCLKSTAT_SYS9ON CSL_FMKT (PLLC_CKSTAT_SYS9ON, ON)
/** SYSCLK7 is ON */
#define CSL_PLLC_SYSCLKSTAT_SYS8ON CSL_FMKT (PLLC_CKSTAT_SYS8ON, ON)
/** SYSCLK6 is ON */
#define CSL_PLLC_SYSCLKSTAT_SYS7ON CSL_FMKT (PLLC_CKSTAT_SYS7ON, ON)
/** SYSCLK5 is ON */
#define CSL_PLLC_SYSCLKSTAT_SYS6ON CSL_FMKT (PLLC_CKSTAT_SYS6ON, ON)
/** SYSCLK4 is ON */
#define CSL_PLLC_SYSCLKSTAT_SYS5ON CSL_FMKT (PLLC_CKSTAT_SYS5ON, ON)
/** SYSCLK3 is ON */
#define CSL_PLLC_SYSCLKSTAT_SYS4ON CSL_FMKT (PLLC_CKSTAT_SYS4ON, ON)
/** SYSCLK2 is ON */
#define CSL_PLLC_SYSCLKSTAT_SYS3ON CSL_FMKT (PLLC_CKSTAT_SYS3ON, ON)
/** SYSCLK1 is ON */
#define CSL_PLLC_SYSCLKSTAT_SYS2ON CSL_FMKT (PLLC_CKSTAT_SYS2ON, ON)
/** SYSCLK0 is ON */
#define CSL_PLLC_SYSCLKSTAT_SYS1ON CSL_FMKT (PLLC_CKSTAT_SYS1ON, ON)

/**
@} */


/** \defgroup CSL_PLLC_RESETSTAT_DEFINE PLLC Last Reset Status
 *  \ingroup CSL_PLLC_DEFINE
 *
 * @{ */

/** Power On Reset */
#define CSL_PLLC_RESETSTAT_POR CSL_FMKT (PLLC_RSTYPE_POR, YES)
/** External Warm Reset */
#define CSL_PLLC_RESETSTAT_XWRST CSL_FMKT (PLLC_RSTYPE_XWRST, YES)
/** Maximum Reset */
#define CSL_PLLC_RESETSTAT_MRST CSL_FMKT (PLLC_RSTYPE_MRST, YES)
/** System/Chip Reset */
#define CSL_PLLC_RESETSTAT_SRST CSL_FMKT (PLLC_RSTYPE_SRST, YES)

/**
@} */


/** \defgroup CSL_PLLC_CTRL_DEFINE PLLC Control Mask
 *  \ingroup CSL_PLLC_DEFINE
 *
 * @{ */

/** PreDiv, PLL, and PostDiv are bypassed. SYSCLK
 *   divided down directly from input reference
 *   clock refclk
 */
#define CSL_PLLC_CTRL_BYPASS CSL_FMKT (PLLC_PLLCTL_PLLEN, BYPASS)
/** PLL is used. SYSCLK divided down from PostDiv
 *   output
 */
#define CSL_PLLC_CTRL_ENABLE CSL_FMKT (PLLC_PLLCTL_PLLEN, PLL)
/** Selected PLL Operational */
#define CSL_PLLC_CTRL_OPERATIONAL CSL_FMKT (PLLC_PLLCTL_PLLPWRDN, NO)
/** Selected PLL Placed In Power Down State */
#define CSL_PLLC_CTRL_POWERDOWN CSL_FMKT (PLLC_PLLCTL_PLLPWRDN, YES)
/** PLL Reset Released */
#define CSL_PLLC_CTRL_RELEASE_RESET CSL_FMKT (PLLC_PLLCTL_PLLRST, NO)
/** PLL Reset Asserted */
#define CSL_PLLC_CTRL_RESET CSL_FMKT (PLLC_PLLCTL_PLLRST, YES)
/** PLL Disable Released */
#define CSL_PLLC_CTRL_RELEASE_DIS CSL_FMKT (PLLC_PLLCTL_PLLDIS, NO)
/** PLL Disable Asserted */
#define CSL_PLLC_CTRL_ASSERT_DIS CSL_FMKT (PLLC_PLLCTL_PLLRST, YES)
/** PLLEN Mux is controlled by PLLCTL.PLLEN.
 *   pllen_pi is dont care
 */
#define CSL_PLLC_CTRL_MUXCTRL_REGBIT CSL_FMKT (PLLC_PLLCTL_PLLENSRC, REGBIT)
/** PLLEN Mux is controlled by input pllen_pi.
 *   PLLCTL.PLLEN is dont care
 */
#define CSL_PLLC_CTRL_MUXCTRL_PORT CSL_FMKT (PLLC_PLLCTL_PLLENSRC, PORT)
/** PLL A is selected. PLL B is put in power
 *   down
 */
#define CSL_PLLC_CTRL_SELECT_PLLA CSL_FMKT (PLLC_PLLCTL_PLLSELB, PLLA)
/** PLL B is selected. PLL A is put in power
 *   down
 */
#define CSL_PLLC_CTRL_SELECT_PLLB CSL_FMKT (PLLC_PLLCTL_PLLSELB, PLLB)
/** oscin_pi is the reference clock */
#define CSL_PLLC_CTRL_SELECT_OSCIN CSL_FMKT (PLLC_PLLCTL_CLKMODE, OSCIN)
/** clkin_pi is the reference clock */
#define CSL_PLLC_CTRL_SELECT_CLKIN CSL_FMKT (PLLC_PLLCTL_CLKMODE, CLKIN)
/** A write of 1 to this bit signifies that the new
 *   divide ratios in PLLDIV[1:n] are taken into
 *   account at the nearest possible rising edge to
 *   phase align the clocks. The actual SYSCLKx to
 *   be aligned are selected in register ALNCTL
 */
#define CSL_PLLC_CTRL_ALIGN_PHASE (CSL_FMKT (PLLC_PLLCMD_GOSET, SET)<< 16)
/** A write of 1 (doesnt need to be a transition
 *   from 0 to 1) to this bit initiates oscillator
 *   power down command
 */
#define CSL_PLLC_CTRL_OSCPWRDN (CSL_FMKT (PLLC_PLLCMD_OSCPWRDN, SET)<<16)


/**
@} */

/** \defgroup CSL_PLLC_DIVEN_DEFINE PLLC Divider Enable
 *  \ingroup CSL_PLLC_DEFINE
 *
 * @{ */

/** Oscillator Divider OD1 Enable */
#define CSL_PLLC_DIVEN_OSCDIV1   (1 << 1)
/** Enable divider D0 for SYSCLK0 */
#define CSL_PLLC_DIVEN_PLLDIV0   (1 << 2)
/** Enable divider D1 for SYSCLK1 */
#define CSL_PLLC_DIVEN_PLLDIV1   (1 << 3)
/** Enable divider D2 for SYSCLK2 */
#define CSL_PLLC_DIVEN_PLLDIV2   (1 << 4)
/** Enable divider D3 for SYSCLK3 */
#define CSL_PLLC_DIVEN_PLLDIV3   (1 << 5)
/** Enable divider D4 for SYSCLK4 */
#define CSL_PLLC_DIVEN_PLLDIV4   (1 << 6)
/** Enable divider D4 for SYSCLK5 */
#define CSL_PLLC_DIVEN_PLLDIV5   (1 << 7)
/** Enable divider D10 for SYSCLK10 */
#define CSL_PLLC_DIVEN_PLLDIV10   (1 << 11)
/** Enable divider D12 for SYSCLK12 */
#define CSL_PLLC_DIVEN_PLLDIV12   (1 << 13)
/** Enable divider D13 for SYSCLK13 */
#define CSL_PLLC_DIVEN_PLLDIV13   (1 << 14)


/**
@} */

/** \defgroup CSL_PLLC_DIVSEL_DEFINE Divider Select for SYSCLKs
 *  \ingroup CSL_PLLC_DEFINE
 *
 * @{ */

/** Divider D0 for SYSCLK0 */
#define CSL_PLLC_DIVSEL_PLLDIV0     (1)
/** Divider D1 for SYSCLK1 */
#define CSL_PLLC_DIVSEL_PLLDIV1     (2)
/** Divider D2 for SYSCLK2 */
#define CSL_PLLC_DIVSEL_PLLDIV2     (3)
/** Divider D3 for SYSCLK3 */
#define CSL_PLLC_DIVSEL_PLLDIV3     (4)
/** Divider D4 for SYSCLK4 */
#define CSL_PLLC_DIVSEL_PLLDIV4     (5)
/** Divider D5 for SYSCLK5 */
#define CSL_PLLC_DIVSEL_PLLDIV5     (6)
/** Divider D10 for SYSCLK10 */
#define CSL_PLLC_DIVSEL_PLLDIV10    (11)
/** Divider D10 for SYSCLK12 */
#define CSL_PLLC_DIVSEL_PLLDIV12    (13)
/** Divider D13 for SYSCLK13 */
#define CSL_PLLC_DIVSEL_PLLDIV13    (14)


/**
@} */

/**************************************************************************\
* PLLC global typedef declarations                                         *
\**************************************************************************/

/** \defgroup CSL_PLLC_RSTDEF_ENUM Reset Definition
 *  \ingroup CSL_PLLC_ENUM
 @{*/
/** \brief  Enums for PLL XWRST pin function */
typedef enum {
    /** <b>: Normal mode.
      *   xwrst_pi_n causes the xwrst_pi_n operation</b>
      */
    CSL_PLLC_RSTDEF_NORMAL = CSL_PLLC_RSTDEF_XWFUNC_NORM,
    /** <b>: Fast mode.
      *   xwrst_pi_n causes the system/chip reset operation</b>
      */
    CSL_PLLC_RSTDEF_FAST = CSL_PLLC_RSTDEF_XWFUNC_FAST
} CSL_PllcResetDef;
/**
@} */

/** \defgroup CSL_PLLC_OSCDIVCTL_ENUM Oscillator Divide Control
 *  \ingroup CSL_PLLC_ENUM
 @{*/
/** \brief  Enums for Oscillator divide enable/ disable */
typedef enum {
    /** <b>: Oscillator Divider OD1 Disable</b> */
    CSL_PLLC_OSCDIV_DISABLE = CSL_PLLC_OSCDIV1_OD1EN_DISABLE,
    /** <b>: Oscillator Divider OD1 Enable</b> */
    CSL_PLLC_OSCDIV_ENABLE = CSL_PLLC_OSCDIV1_OD1EN_ENABLE
} CSL_PllcOscDivCtrl;
/**
@} */

/** \defgroup CSL_PLLC_PLLDIVCTL_ENUM PLL Divide Control
 *  \ingroup CSL_PLLC_ENUM
 @{*/
/** \brief  Enums for PLL divide enable/ disable */
typedef enum {
    /** <b>: PLL Divider Disable</b> */
    CSL_PLLC_PLLDIV_DISABLE = 0,
    /** <b>: PLL Divider Enable</b> */
    CSL_PLLC_PLLDIV_ENABLE = 1
} CSL_PllcDivCtrl;
/**
@} */

/** \defgroup CSL_PLLC_CONTROLCMD_ENUM  Control Commands
 *  \ingroup CSL_PLLC_CONTROL_API
@{*/
/** \brief Enumeration for control commands passed to \a CSL_pllcHwControl()
 *
 *  This is the set of commands that are passed to the \a CSL_pllcHwControl()
 *  with an optional argument type-casted to \a void* .
 *  The arguments to be passed with each enumeration (if any) are specified
 *  next to the enumeration
 */
typedef enum {
    /** \brief Control PLL based on the bits set in the input argument
     *  \param CSL_BitMask32
     *  \return CSL_SOK
     *  \sa CSL_PLLC_CTRL_DEFINE
     */
    CSL_PLLC_CMD_PLLCONTROL,
    /** \brief Enable the clocks as specified by input  bitmask
     *  \param CSL_BitMask32
     *  \return CSL_SOK
     *  \sa CSL_PLLC_CLKEN_DEFINE
     */
    CSL_PLLC_CMD_CLOCK_ENABLE,
    /** \brief Disable the clocks specified by input bitmask
     *  \param CSL_BitMask32
     *  \return CSL_SOK
     *  \sa CSL_PLLC_CLKEN_DEFINE
     */
    CSL_PLLC_CMD_CLOCK_DISABLE,
    /** \brief Define XWRST pin function
     *  \param CSL_PllcResetDef
     *  \return CSL_SOK
     *  \sa CSL_PllcResetDef
     */
    CSL_PLLC_CMD_DEFINE_RESET,
    /** \brief Set PLL multiplier value
     *  \param Uint32
     *  \return CSL_SOK
     */
    CSL_PLLC_CMD_SET_PLLM,
    /** \brief Set oscillator divide ratio
     *  \param Uint32
     *  \return CSL_SOK
     */
    CSL_PLLC_CMD_SET_OSCRATIO,
    /** \brief Set PLL divide ratio
     *  \param CSL_PllcDivRatio
     *  \return CSL_SOK
     *  \sa CSL_PllcDivRatio
     */
    CSL_PLLC_CMD_SET_PLLRATIO,
    /** \brief Enable/disable  oscillator divider
     *  \param CSL_PllcOscDivCtrl
     *  \return CSL_SOK
     *  \sa CSL_PllcOscDivCtrl
     */
    CSL_PLLC_CMD_OSCDIV_CONTROL,
    /** \brief Enable/disable  PLL divider
     *  \param CSL_PllcDivideControl
     *  \return CSL_SOK
     *  \sa CSL_PllcDivideControl
     */
    CSL_PLLC_CMD_PLLDIV_CONTROL,
    /** \brief Enable/disable  Wake Up functionality of wakeup pins
     *  \param CSL_BitMask16
     *  \return CSL_SOK
     */
    CSL_PLLC_CMD_WAKEUP
} CSL_PllcHwControlCmd;
/**
@} */

/** \defgroup CSL_PLLC_QUERYCMD_ENUM Query Commands
 *  \ingroup CSL_PLLC_QUERY_API
 @{ */
/** \brief Enumeration for queries passed to \a CSL_PllcGetHwStatus()
 *
 *  This is used to get the status of different operations.The arguments
 *  to be passed with each enumeration if any are specified next to
 *  the enumeration
 */
typedef enum {
    /** \brief Queries PLL Control Peripheral ID
     *  \param (Uint32*)
     *  \return CSL_SOK
     */
    CSL_PLLC_QUERY_PID,
    /** \brief Queries PLL Controller Status
     *  \param (CSL_BitMask32*)
     *  \return CSL_SOK
     *  \sa CSL_PLLC_STATUS_DEFINE
     */
    CSL_PLLC_QUERY_STATUS,
    /** \brief Queries PLLDIV Modified Status
     *  \param (CSL_BitMask32*)
     *  \return CSL_SOK
     *  \sa CSL_PLLC_DCHANGESTAT_DEFINE
     */
    CSL_PLLC_QUERY_DIVRATIO_CHANGE,
    /** \brief Queries Clock Status
     *  \param (CSL_BitMask32*)
     *  \return CSL_SOK
     *  \sa CSL_PLLC_CLKSTAT_DEFINE
     */
    CSL_PLLC_QUERY_CLKSTAT,
    /** \brief Queries PLL SYSCLK Status
     *  \param (CSL_BitMask32*)
     *  \return CSL_SOK
     *  \sa CSL_PLLC_SYSCLKSTAT_DEFINE
     */
    CSL_PLLC_QUERY_SYSCLKSTAT,
    /** \brief Queries CLK Enable Status
     *  \param (CSL_BitMask32*)
     *  \return CSL_SOK
     *  \sa CSL_PLLC_CLKEN_DEFINE
     */
    CSL_PLLC_QUERY_CLKENSTAT,
    /** \brief Queries Reset Type Status
     *  \param (CSL_BitMask32*)
     *  \return CSL_SOK
     *  \sa CSL_PLLC_RESETSTAT_DEFINE
     */
    CSL_PLLC_QUERY_RESETSTAT,
    /** \brief Queries Fuse Farm Error Status
     *  \param (Uint32*)
     *  \return CSL_SOK
     */
    CSL_PLLC_QUERY_EFUSEERR
} CSL_PllcHwStatusQuery;
/**
@} */

/**
\addtogroup CSL_PLLC_DATASTRUCT
@{ */
/**
 *  \brief Input parameters for setting up PLL Divide ratio
 *  \sa CSL_PLLC_DIVSEL_DEFINE
 */
typedef struct CSL_PllcDivRatio {
    /** \brief Divider number
     *  \param Uint32
     */
    Uint32 divNum;
    /** \brief Divider Ratio
     *  \param Uint32
     */
    Uint32 divRatio;
} CSL_PllcDivRatio;
/**
@} */

/**
\addtogroup CSL_PLLC_DATASTRUCT
@{ */
/**
 *  \brief Input parameters for enabling or disabling PLL Divide ratio
 *  \sa CSL_PLLC_DIVSEL_DEFINE
 */
typedef struct CSL_PllcDivideControl {
    /** \brief Divider Number
     *  \param Uint32
     */
    Uint32 divNum;
    /** \brief Divider Control (Enable/Disable)
     *  \param CSL_PllcDivCtrl
     */
    CSL_PllcDivCtrl divCtrl;
} CSL_PllcDivideControl;
/**
@} */

/**
\addtogroup CSL_PLLC_DATASTRUCT
@{ */
/**
 *  \brief Input parameters for setting up PLL Controller
 *
 *  Used to put PLLC known useful state
 *  /sa CSL_PLLC_DIVEN_DEFINE
 */
typedef struct CSL_PllcHwSetup {
    /** \brief Divider Enable/Disable
     *  \param CSL_BitMask32
     */
    CSL_BitMask32 divEnable;
    /** \brief PLL Multiplier
     *  \param Uint32
     */
    Uint32        pllM;
	/** \brief PLL Divider 0
     *  \param Uint32
     */
    Uint32        pllDiv0;
    /** \brief PLL Divider 1
     *  \param Uint32
     */
    Uint32        pllDiv1;
    /** \brief PLL Divider 2
     *  \param Uint32
     */
    Uint32        pllDiv2;
    /** \brief PLL Divider 3
     *  \param Uint32
     */
    Uint32        pllDiv3;
    /** \brief PLL Divider 4
     *  \param Uint32
     */
    Uint32        pllDiv4;
    /** \brief PLL Divider 5
     *  \param Uint32
     */
    Uint32        pllDiv5;
   /** \brief PLL Divider 10
     *  \param Uint32
     */
    Uint32        pllDiv10;
    /** \brief PLL Divider 12
     *  \param Uint32
     */
    Uint32        pllDiv12;
    /** \brief PLL Divider 13
     *  \param Uint32
     */
    Uint32        pllDiv13;
	/** \brief Oscillator Divider 1
     *  \param Uint32
     */
    Uint32        oscDiv1;
    /** \brief Setup that can be used for future implementation
     *  \param void*
     */
    void*         extendSetup;
} CSL_PllcHwSetup;
/**
@} */

/** \defgroup CSL_PLLC_DEFAULT_HWSETUP_DEFINE PLLC Default HwSetup Structure
 *  \ingroup CSL_PLLC_DEFINE
 *
 * @{ */
/** \brief Default hardware setup parameters
 */
#define CSL_PLLC_HWSETUP_DEFAULTS{  \
    0,                              \
    0,                              \
    0,                              \
    0,                              \
    0,                              \
    0,                              \
    0,                              \
    0,                              \
    0,                              \
    0,                              \
    0,                              \
    0,                              \
    NULL                            \
}
/**
@} */

/**
\addtogroup CSL_PLLC_DATASTRUCT
@{ */
/** \brief Config-structure
 *
 *  Used to configure the pllc using CSL_pllcHwSetupRaw()
 */
typedef struct {
    /** Reset Definition register */
    volatile Uint32 RSTDEF;

    /** PLL Control register */
    volatile Uint32 PLLCTL;

    /** OBSCLK Select register */
    volatile Uint32 OCSEL;

    /** PLL Secondary Control register */
    volatile Uint32 SECCTL;

    /** PLL Multiplier Control register */
    volatile Uint32 PLLM;

    /** PLL Controller Divider 1 register (for SYSCLK0) */
    volatile Uint32 PLLDIV1;

    /** PLL Controller Divider 2 register (for SYSCLK1) */
    volatile Uint32 PLLDIV2;

    /** PLL Controller Divider 3 register (for SYSCLK2) */
    volatile Uint32 PLLDIV3;

    /** Oscillator Divider1 register */
    volatile Uint32 OSCDIV1;

    /** Bypass Divider register */
    volatile Uint32 BPDIV;

    /** Wakeup register */
    volatile Uint32 WAKEUP;

    /** PLL Controller Command register */
    volatile Uint32 PLLCMD;

    /** Clock Enable Control register */
    volatile Uint32 CKEN;

    /** PLL Controller Divider 4 register  (for SYSCLK3)*/
    volatile Uint32 PLLDIV4;

    /** PLL Controller Divider 5 register (for SYSCLK4)*/
    volatile Uint32 PLLDIV5;

    /** PLL Controller Divider 6 register (for SYSCLK5)*/
    volatile Uint32 PLLDIV6;

    /** PLL Controller Divider 11 register (for SYSCLK10)*/
    volatile Uint32 PLLDIV11;

    /** PLL Controller Divider 13 register (for SYSCLK11)*/
    volatile Uint32 PLLDIV13;

	/** PLL Controller Divider 14 register (for SYSCLK13)*/
    volatile Uint32 PLLDIV14;

} CSL_PllcConfig;
/**
@} */

/** \defgroup CSL_PLLC_DEFAULT_CONFIG_DEFINE PLLC Default Config Structure
 *  \ingroup CSL_PLLC_DEFINE
 *
 * @{ */
/** \brief Default values for config structure
 */
#define CSL_PLLC_CONFIG_DEFAULTS {    \
    CSL_PLLC_RSTDEF_RESETVAL,         \
    CSL_PLLC_PLLCTL_RESETVAL,         \
    CSL_PLLC_OCSEL_RESETVAL,          \
    CSL_PLLC_SECCTL_RESETVAL,         \
    CSL_PLLC_PLLM_RESETVAL,           \
    CSL_PLLC_PLLDIV1_RESETVAL,        \
    CSL_PLLC_PLLDIV2_RESETVAL,        \
    CSL_PLLC_PLLDIV3_RESETVAL,        \
    CSL_PLLC_OSCDIV1_RESETVAL,        \
    CSL_PLLC_BPDIV_RESETVAL,          \
    CSL_PLLC_WAKEUP_RESETVAL,         \
    CSL_PLLC_PLLCMD_RESETVAL,         \
    CSL_PLLC_CKEN_RESETVAL,           \
    CSL_PLLC_PLLDIV4_RESETVAL,        \
    CSL_PLLC_PLLDIV5_RESETVAL,        \
    CSL_PLLC_PLLDIV6_RESETVAL,        \
    CSL_PLLC_PLLDIV11_RESETVAL,       \
    CSL_PLLC_PLLDIV13_RESETVAL,       \
	CSL_PLLC_PLLDIV14_RESETVAL        \
}
/**
@} */

/**
\addtogroup CSL_PLLC_DATASTRUCT
@{ */
/** \brief Module specific context information. Present implementation of pllc
 *         CSL doesn't have any context information.
 */
typedef struct {
    /** Context information of Pllc CSL.
     *  The below declaration is just a place-holder for future implementation.
     */
    Uint16  contextInfo;
} CSL_PllcContext;
/**
@} */

/**
\addtogroup CSL_PLLC_DATASTRUCT
@{ */
/** \brief Module specific parameters. Present implementation of pllc CSL
 *         doesn't have any module specific parameters.
 */
typedef struct {
    /** Bit mask to be used for module specific parameters. The below
     *  declaration is just a place-holder for future implementation.
     */
    CSL_BitMask16   flags;
} CSL_PllcParam;
/**
@} */

/**
\addtogroup CSL_PLLC_DATASTRUCT
@{ */
/** \brief This structure contains the base-address information for the
 *         peripheral instance of the PLLC
 */
typedef struct {
    /** Base-address of the configuration registers of the peripheral
     */
    CSL_PllcRegsOvly  regs;
} CSL_PllcBaseAddress;
/**
@} */

/**
\addtogroup CSL_PLLC_DATASTRUCT
@{ */
/** \brief This object contains the reference to the instance of PLLC
 *         opened using the @a CSL_pllcOpen()
 *
 *  The pointer to this is passed to all PLLC CSL APIs
 *  This structure has the fields required to configure PLLC for any test
 *  case/application. It should be initialized as per requirements of a
 *  test case/application and passed on to the setup function
 */
typedef struct CSL_PllcObj {
    /** This is a pointer to the registers of the instance of PLLC
     *   referred to by this object
     */
    CSL_PllcRegsOvly regs;
    /** This is the instance of PLLC being referred to by this object */
    CSL_InstNum  pllcNum;
} CSL_PllcObj;
/**
@} */

/**
\addtogroup CSL_PLLC_DATASTRUCT
@{ */
/** \brief This data type is used to return the handle to the pllc functions */
typedef struct CSL_PllcObj *CSL_PllcHandle;
/**
@} */

/*******************************************************************************
 * PLLC function declarations
 ******************************************************************************/

/**
\defgroup CSL_PLLC_INIT_API PLLC Init API
\ingroup CSL_PLLC_FUNCTION
@{*/
/** ============================================================================
 *   @n@b CSL_pllcInit
 *
 *   @b Description
 *   @n This is the initialization function for the pllc CSL. The function
 *      must be called before calling any other API from this CSL. This
 *      function is idem-potent. Currently, the function just return status
 *      CSL_SOK, without doing anything.
 *
 *   @b Arguments
 *   @verbatim
        pContext    Pointer to module-context. As pllc doesn't have
                    any context based information user is expected to pass NULL.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                        CSL_SOK - Always returns
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
            CSL_pllcInit(NULL);
     @endverbatim
 * =============================================================================
 */
CSL_Status CSL_pllcInit (
    CSL_PllcContext       *pContext
);
/**
@} */

/**
\defgroup CSL_PLLC_OPEN_API PLLC Open API
\ingroup CSL_PLLC_FUNCTION
@{*/
/** ============================================================================
 *   @n@b CSL_pllcOpen
 *
 *   @b Description
 *   @n This function returns the handle to the PLLC
 *      instance. This handle is passed to all other CSL APIs.
 *
 *   @b Arguments
 *   @verbatim
            pllcObj     Pointer to pllc object.

            pllcNum     Instance of pllc CSL to be opened.
                        There is only one instance of the pllc available.
                        So, the value for this parameter will be
                        CSL_PLLC always.

            pPllcParam  Module specific parameters.

            status      Status of the function call
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_PllcHandle
 *   @n                     Valid pllc handle will be returned if
 *                          status value is equal to CSL_SOK.
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n   1.    The status is returned in the status variable. If status
 *              returned is
 *   @li            CSL_SOK            - Valid pllc handle is returned
 *   @li            CSL_ESYS_FAIL      - The pllc instance is invalid
 *   @li            CSL_ESYS_INVPARAMS - Invalid parameter
 *   @n   2.    PLLC object structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *   @n    2. PLLC object structure
 *
 *   @b Example
 *   @verbatim
            CSL_status           status;
            CSL_PllcObj          pllcObj;
            CSL_PllcHandle       hPllc;
            ...

            hPllc = CSL_pllcOpen(&pllcObj, CSL_PLLC_0, NULL, &status);
            ...

    @endverbatim
 * =============================================================================
 */
CSL_PllcHandle CSL_pllcOpen (
    CSL_PllcObj               *pllcObj,
    CSL_InstNum                pllcNum,
    CSL_PllcParam             *pPllcParam,
    CSL_Status                *status
);
/**
@} */

/**
\defgroup CSL_PLLC_GETBASEADDRESS_API PLLC GetBaseAddress API
\ingroup CSL_PLLC_FUNCTION
@{*/
/** ============================================================================
 *   @n@b CSL_pllcGetBaseAddress
 *
 *   @b Description
 *   @n  Function to get the base address of the peripheral instance.
 *       This function is used for getting the base address of the peripheral
 *       instance. This function will be called inside the CSL_pllcOpen()
 *       function call. This function is open for re-implementing if the user
 *       wants to modify the base address of the peripheral object to point to
 *       a different location and there by allow CSL initiated write/reads into
 *       peripheral. MMR's go to an alternate location.
 *
 *   @b Arguments
 *   @verbatim
            pllcNum         Specifies the instance of the pllc to be opened.

            pPllcParam      Module specific parameters.

            pBaseAddress    Pointer to base address structure containing base
                            address details.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_OK             - Open call is successful
 *   @li                    CSL_ESYS_FAIL      - The instance number is invalid.
 *   @li                    CSL_ESYS_INVPARAMS - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Base address structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *   @n    2. Base address structure is modified.
 *
 *   @b Example
 *   @verbatim
         CSL_Status              status;
         CSL_PllcBaseAddress  baseAddress;

         ...

         status = CSL_pllcGetBaseAddress(CSL_PLLC_0, NULL,
                  &baseAddress);
         ...
    @endverbatim
 * =============================================================================
 */
CSL_Status CSL_pllcGetBaseAddress (
    CSL_InstNum                      pllcNum,
    CSL_PllcParam                   *pPllcParam,
    CSL_PllcBaseAddress             *pBaseAddress
);
/**
@} */

/**
\defgroup CSL_PLLC_CLOSE_API PLLC Close API
\ingroup CSL_PLLC_FUNCTION
@{*/
/** ============================================================================
 *   @n@b CSL_pllcClose
 *
 *   @b Description
 *   @n This function closes the specified instance of PLLC.
 *
 *   @b Arguments
 *   @verbatim
            hPllc            Handle to the pllc
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Close successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *
 *   @b Example
 *   @verbatim
            CSL_PllcHandle   hPllc;
            CSL_status       status;
            ...

            status = CSL_pllcClose(hPllc, &hwSetup);
     @endverbatim
 * =============================================================================
 */
CSL_Status CSL_pllcClose (
    CSL_PllcHandle         hPllc
);
/**
@} */

/**
\defgroup CSL_PLLC_SETUP_API PLLC Setup API
\ingroup CSL_PLLC_FUNCTION
@{*/
/** ============================================================================
 *   @n@b CSL_pllcHwSetup
 *
 *   @b Description
 *   @n It configures the pllc registers as per the values passed
 *      in the hardware setup structure.
 *
 *   @b Arguments
 *   @verbatim
            hPllc            Handle to the pllc

            hwSetup          Pointer to harware setup structure
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Hardware setup successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS  - Hardware structure is not
                                                  properly initialized
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  PLLC registers are configured according
 *       to the hardware setup parameters
 *
 *   @b Modifies
 *   @n PLLC registers
 *
 *   @b Example
 *   @verbatim
            CSL_PllcHandle   hPllc;
            CSL_PllcObj      pllcObj;
            CSL_PllcHwSetup  hwSetup;
            CSL_status       status;
            ...

            hPllc = CSL_pllcOpen(&pllcObj, CSL_PLLC_0, NULL, &status);
            ...

            status = CSL_pllcHwSetup(hPllc, &hwSetup);
     @endverbatim
 * =============================================================================
 */
CSL_Status CSL_pllcHwSetup (
    CSL_PllcHandle            hPllc,
    CSL_PllcHwSetup          *hwSetup
);
/**
@} */

/**
\defgroup CSL_PLLC_CONTROL_API PLLC Control API
\ingroup CSL_PLLC_FUNCTION
@{*/
/** ============================================================================
 *   @n@b CSL_pllcHwControl
 *
 *   @b Description
 *   @n Takes a command of PLLC with an optional argument & implements it.
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance

            cmd             The command to this API indicates the action to be
                            taken on PLLC.

            arg             An optional argument.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li            CSL_SOK               - Status info return successful.
 *   @li            CSL_ESYS_BADHANDLE    - Invalid handle
 *   @li            CSL_ESYS_INVCMD       - Invalid command
 *   @li            CSL_ESYS_INVPARAMS    - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n The hardware registers of PLLC.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle         hPllc;
        CSL_PllcHwControlCmd   cmd;
        void                      arg;
        ...

        status = CSL_pllcHwControl (hPllc, cmd, &arg);
     @endverbatim
 * =============================================================================
 */
CSL_Status CSL_pllcHwControl (
    CSL_PllcHandle              hPllc,
    CSL_PllcHwControlCmd        cmd,
    void                       *cmdArg
);
/**
@} */

/**
\defgroup CSL_PLLC_QUERY_API PLLC Query API
\ingroup CSL_PLLC_FUNCTION
@{*/
/** ============================================================================
 *   @n@b CSL_pllcGetHwStatus
 *
 *   @b Description
 *   @n Gets the status of the different operations of PLLC.
 *
 *   @b Arguments
 *   @verbatim
            hPllc         Handle to the PLLC instance

            query         The query to this API of PLLC which indicates the
                          status to be returned.

            response      Placeholder to return the status.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Status info return successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVQUERY   - Invalid query command
 *   @li                    CSL_ESYS_INVPARAMS  - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle          hPllc;
        CSL_PllcHwStatusQuery   query;
        void                    reponse;
        ...

        status = CSL_pllcGetHwStatus (hPllc, query, &response);
     @endverbatim
 * =============================================================================
 */
CSL_Status CSL_pllcGetHwStatus (
    CSL_PllcHandle                hPllc,
    CSL_PllcHwStatusQuery         query,
    void                         *response
);
/**
@} */

/**
\defgroup CSL_PLLC_HWSETUPRAW_API PLLC HwSetupRaw API
\ingroup CSL_PLLC_FUNCTION
@{*/
/** ============================================================================
 *   @n@b CSL_pllcHwSetupRaw
 *
 *   @b Description
 *   @n This function initializes the device registers with the register-values
 *      provided through the Config Data structure.
 *
 *   @b Arguments
 *   @verbatim
            hPllc        Handle to the PLLC instance

            config       Pointer to config structure
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li            CSL_SOK               - Configuration successful
 *   @li            CSL_ESYS_BADHANDLE    - Invalid handle
 *   @li            CSL_ESYS_INVPARAMS    - Configuration is not
                                            properly initialized
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The registers of the specified PLLC instance will be setup
 *       according to input configuration structure values.
 *
 *   @b Modifies
 *   @n Hardware registers of the specified PLLC instance.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle       hPllc;
        CSL_PllcConfig       config = CSL_PLLC_CONFIG_DEFAULTS;
        CSL_Status           status;
        ...

        status = CSL_pllcHwSetupRaw (hPllc, &config);
     @endverbatim
 * =============================================================================
 */
CSL_Status CSL_pllcHwSetupRaw (
    CSL_PllcHandle               hPllc,
    CSL_PllcConfig              *config
);
/**
@} */

/**
\defgroup CSL_PLLC_GETSETUP_API PLLC GetSetup API
\ingroup CSL_PLLC_FUNCTION
@{*/
/** ============================================================================
 *   @n@b CSL_pllcGetHwSetup
 *
 *   @b Description
 *   @n It retrives the hardware setup parameters of the pllc
 *      specified by the given handle.
 *
 *   @b Arguments
 *   @verbatim
            hPllc        Handle to the pllc

            hwSetup      Pointer to the harware setup structure
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Retrieving the hardware setup
 *                                                parameters is successful
 *   @li                    CSL_ESYS_BADHANDLE  - The handle is passed is
 *                                                invalid
 *   @li                    CSL_ESYS_INVPARAMS  - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The hardware setup structure is populated with the hardware setup
 *       parameters
 *
 *   @b Modifies
 *   @n hwSetup variable
 *
 *   @b Example
 *   @verbatim
            CSL_PllcHandle   hPllc;
            CSL_PllcHwSetup  hwSetup;
            ...

            status = CSL_pllcGetHwSetup(hPllc, &hwSetup);
     @endverbatim
 * =============================================================================
 */
CSL_Status CSL_pllcGetHwSetup (
    CSL_PllcHandle               hPllc,
    CSL_PllcHwSetup             *hwSetup
);
/**
@} */

#ifdef __cplusplus
}
#endif

#endif /* _CSL_PLLC_H_ */

