/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** @file  csl_pllcAux.h
 *
 *  @brief API Auxilary header file for PLLC CSL.
 *
 *  Path: \\(CSLPATH)\inc
 */

/* =============================================================================
 *  Revision History
 *  ===============
 *  10-Feb-2004 kpn  File Created.
 *  21-Apr-2006 RR   updated for faraday
 * =============================================================================
 */

#ifndef _CSL_PPLCAUX_H_
#define _CSL_PLLCAUX_H_

#include <csl_pllc.h>

#ifdef __cplusplus
extern "C" {
#endif

/*
 *  Status query functions of the pllc
 */

/** ============================================================================
 *   @n@b CSL_pllcGetPID
 *
 *   @b Description
 *   @n Gets the Peripheral ID of the pllc.
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  Uint32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle    hPllc;
        Uint32            response;
        ...

        response = CSL_pllcGetPID (hPllc);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
Uint32 CSL_pllcGetPID (
    CSL_PllcHandle      hPllc
)
{
    Uint32 response = 0;
    response = (Uint32) hPllc->regs->PID;
    return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetStatus
 *
 *   @b Description
 *   @n Gets the Status of the pll controller.
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  CSL_BitMask32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle    hPllc;
        CSL_BitMask32     response;
        ...

        response = CSL_pllcGetStatus (hPllc);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
CSL_BitMask32 CSL_pllcGetStatus (
    CSL_PllcHandle                hPllc
)
{
    CSL_BitMask32 response;
    response = (CSL_BitMask32) hPllc->regs->PLLSTAT;
    return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetDivRatioChange
 *
 *   @b Description
 *   @n Gets the Divider Ratio Change Status of the pllc SYSCLK.
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  CSL_BitMask32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle    hPllc;
        CSL_BitMask32     response;
        ...

        response = CSL_pllcGetDivRatioChange (hPllc);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
CSL_BitMask32 CSL_pllcGetDivRatioChange (
    CSL_PllcHandle                        hPllc
)
{
    CSL_BitMask32 response;
    response = (CSL_BitMask32) hPllc->regs->DCHANGE;
    return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetClkStatus
 *
 *   @b Description
 *   @n Gets the Clock Status of the pllc.
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  CSL_BitMask32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle    hPllc;
        CSL_BitMask32     response;
        ...

        response = CSL_pllcGetClkStatus (hPllc);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
CSL_BitMask32 CSL_pllcGetClkStatus (
    CSL_PllcHandle                   hPllc
)
{
    CSL_BitMask32 response;
    response = (CSL_BitMask32) hPllc->regs->CKSTAT;
    return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetClkEnableStatus
 *
 *   @b Description
 *   @n Gets the Clock Enable Control Status of the pllc.
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  CSL_BitMask32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle    hPllc;
        CSL_BitMask32     response;
        ...

        response = CSL_pllcGetClkEnableStatus (hPllc);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
CSL_BitMask32 CSL_pllcGetClkEnableStatus (
    CSL_PllcHandle                         hPllc
)
{
    CSL_BitMask32 response;
    response = (CSL_BitMask32) hPllc->regs->CKEN;
    return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetSysClkStatus
 *
 *   @b Description
 *   @n Gets the System Clock ON/OFF Status of the pllc.
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  CSL_BitMask32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle    hPllc;
        CSL_BitMask32     response;
        ...

        response = CSL_pllcGetSysClkStatus (hPllc);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
CSL_BitMask32 CSL_pllcGetSysClkStatus (
    CSL_PllcHandle                      hPllc
)
{
    CSL_BitMask32 response;
    response = (CSL_BitMask32) hPllc->regs->SYSTAT;
    return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetResetStatus
 *
 *   @b Description
 *   @n Gets the Reset Type Status of the pllc.
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  CSL_BitMask32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle    hPllc;
        CSL_BitMask32     response;
        ...

        response = CSL_pllcGetResetStatus (hPllc);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
CSL_BitMask32 CSL_pllcGetResetStatus (
    CSL_PllcHandle                     hPllc
)
{
    CSL_BitMask32 response;
    response = (Uint32) hPllc->regs->RSTYPE;
    return response;
}

/** ============================================================================
 *   @n@b CSL_pllcGetFusefarmErrorStatus
 *
 *   @b Description
 *   @n Gets the Fusefarm Error Status of the pllc.
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  Uint32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle    hPllc;
        Uint32            response;
        ...

        response = CSL_pllcGetFusefarmErrorStatus (hPllc);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
Uint32 CSL_pllcGetFusefarmErrorStatus (
    CSL_PllcHandle                      hPllc
)
{
    Uint32 response = 0;
    response = (Uint32) hPllc->regs->FUSERR;
    return response;
}

/*
 *  Control command functions of the pllc
 */

/** ============================================================================
 *   @n@b CSL_pllcCommandCtrl
 *
 *   @b Description
 *      Controls the pllc operation.
 *
 *   @b Arguments
 *   @verbatim
        hPllc        Handle to the PLLC instance
        loadVal      Value to be loaded to pllc PLLCMD register
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc operation is controlled according to value set.
 *
 *   @b Modifies
 *   @n pllc PLLCMD register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_BitMask32       loadVal;
        ...

        CSL_pllcCommandCtrl (hPllc, loadVal);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_pllcCommandCtrl (
    CSL_PllcHandle         hPllc,
    CSL_BitMask32          loadVal
)
{
    hPllc->regs->PLLCTL = (loadVal & 0xFFFF);
    loadVal = (loadVal & 0xFFFF0000)>>16;
    if (loadVal)
        hPllc->regs->PLLCMD = loadVal;
}

/** ============================================================================
 *   @n@b CSL_pllcClockEnable
 *
 *   @b Description
 *      Enables the miscellaneous output clocks.
 *
 *   @b Arguments
 *   @verbatim
        hPllc        Handle to the PLLC instance
        loadVal      Value to be loaded to pllc CKEN register
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The miscellaneous output clocks are enabled.
 *
 *   @b Modifies
 *   @n pllc CKEN register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_BitMask32       loadVal;
        ...

        CSL_pllcClockEnable (hPllc, loadVal);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_pllcClockEnable (
    CSL_PllcHandle         hPllc,
    CSL_BitMask32          loadVal
)
{
    hPllc->regs->CKEN = loadVal;
}

/** ============================================================================
 *   @n@b CSL_pllcClockDisable
 *
 *   @b Description
 *      Disables the miscellaneous output clocks.
 *
 *   @b Arguments
 *   @verbatim
        hPllc        Handle to the PLLC instance
        loadVal      Value to be loaded to pllc CKEN register
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The miscellaneous output clocks are disabled.
 *
 *   @b Modifies
 *   @n pllc CKEN register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_BitMask32       loadVal;
        ...

        CSL_pllcClockDisable (hPllc, loadVal);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_pllcClockDisable (
    CSL_PllcHandle          hPllc,
    CSL_BitMask32           loadVal
)
{
    hPllc->regs->CKEN &= ~(loadVal);
}

/** ============================================================================
 *   @n@b CSL_pllcDefineReset
 *
 *   @b Description
 *      Sets the operation caused by external warm reset pin.
 *
 *   @b Arguments
 *   @verbatim
        hPllc        Handle to the PLLC instance
        loadVal      Value to be loaded to pllc RSTDEF register
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc operation caused by reset is defined according to set value.
 *
 *   @b Modifies
 *   @n pllc RSTDEF register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_PllcResetDef    loadVal;
        ...

        CSL_pllcDefineReset (hPllc, loadVal);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_pllcDefineReset (
    CSL_PllcHandle         hPllc,
    CSL_PllcResetDef       loadVal
)
{
    CSL_FINS (hPllc->regs->RSTDEF, PLLC_RSTDEF_XWFUNC, loadVal);
}


/** ============================================================================
 *   @n@b CSL_pllcMultiplierCtrl
 *
 *   @b Description
 *      Controls the pllc Multiplier.
 *
 *   @b Arguments
 *   @verbatim
        hPllc        Handle to the PLLC instance
        loadVal      Value to be loaded to pllc PLLM register
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc multiplier is controlled accordingly.
 *
 *   @b Modifies
 *   @n pllc PLLM register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        Uint32              loadVal;
        ...

        CSL_pllcMultiplierCtrl (hPllc, loadVal);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_pllcMultiplierCtrl (
    CSL_PllcHandle            hPllc,
    Uint32                    loadVal
)
{
    CSL_FINS (hPllc->regs->PLLM, PLLC_PLLM_PLLM, loadVal);
}

/** ============================================================================
 *   @n@b CSL_pllcSetOscDivRatio
 *
 *   @b Description
 *      Set the pllc Oscillator Divider Ratio.
 *
 *   @b Arguments
 *   @verbatim
        hPllc        Handle to the PLLC instance
        loadVal      Value to be loaded to pllc OSCDIV1 register
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The pllc Oscillator Divider Ratio is set.
 *
 *   @b Modifies
 *   @n pllc OSCDIV1 register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        Uint32              loadVal;
        ...

        CSL_pllcSetOscDivRatio (hPllc, loadVal);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_pllcSetOscDivRatio (
    CSL_PllcHandle            hPllc,
    Uint32                    loadVal
)
{
    CSL_FINS (hPllc->regs->OSCDIV1, PLLC_OSCDIV1_RATIO, loadVal);
}

/** ============================================================================
 *   @n@b CSL_pllcSetPLLDivRatio
 *
 *   @b Description
 *      Sets the pllc Dividers ratios.
 *
 *   @b Arguments
 *   @verbatim
        hPllc        Handle to the PLLC instance
        loadVal      Value to be loaded to pllc divider registers
        status       Status variable
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc dividers ratios are set.
 *
 *   @b Modifies
 *   @n pllc divider registers.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_PllcDivRatio    loadVal;
        CSL_Status          status;
        ...

        CSL_pllcSetPLLDivRatio (hPllc, loadVal, &status);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_pllcSetPLLDivRatio (
    CSL_PllcHandle             hPllc,
    CSL_PllcDivRatio           loadVal,
    CSL_Status                *status
)
{
    switch (loadVal.divNum) {
        case CSL_PLLC_DIVSEL_PLLDIV10:
            CSL_FINS (hPllc->regs->PLLDIV11, PLLC_PLLDIV11_RATIO,
                      (Uint32)loadVal.divRatio);
            break;
        case CSL_PLLC_DIVSEL_PLLDIV12:
            CSL_FINS (hPllc->regs->PLLDIV13, PLLC_PLLDIV13_RATIO,
                      (Uint32)loadVal.divRatio);
            break;

		case CSL_PLLC_DIVSEL_PLLDIV13:
            CSL_FINS (hPllc->regs->PLLDIV14, PLLC_PLLDIV14_RATIO,
                      (Uint32)loadVal.divRatio);
            break;
        
        default:
            *status = CSL_ESYS_INVPARAMS;
            break;
    }
}

/** ============================================================================
 *   @n@b CSL_pllcOscDivCtrl
 *
 *   @b Description
 *      Controls the pllc Oscillator Divider.
 *
 *   @b Arguments
 *   @verbatim
        hPllc        Handle to the PLLC instance
        loadVal      Value to be loaded to pllc OSCDIV1 register
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc oscillator divider is controlled according to value set.
 *
 *   @b Modifies
 *   @n pllc OSCDIV1 register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_PllcOscDivCtrl  loadVal;
        ...

        CSL_pllcOscDivCtrl (hPllc, loadVal);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_pllcOscDivCtrl (
    CSL_PllcHandle        hPllc,
    CSL_PllcOscDivCtrl    loadVal
)
{
    CSL_FINS (hPllc->regs->OSCDIV1, PLLC_OSCDIV1_OD1EN, loadVal);
}

/** ============================================================================
 *   @n@b CSL_pllcPLLDivCtrl
 *
 *   @b Description
 *      Controls the pllc dividers.
 *
 *   @b Arguments
 *   @verbatim
        hPllc        Handle to the PLLC instance
        loadVal      Value to be loaded to pllc dividers register.
        status       Status variable
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc dividers are controlled.
 *
 *   @b Modifies
 *   @n pllc dividers register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle              hPllc;
        CSL_PllcDivideControl       loadVal;
        CSL_Status                  status;
        ...

        CSL_pllcPLLDivCtrl (hPllc, loadVal, &status);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_pllcPLLDivCtrl (
    CSL_PllcHandle         hPllc,
    CSL_PllcDivideControl  loadVal,
    CSL_Status            *status
)
{
    switch (loadVal.divNum) {
        case CSL_PLLC_DIVSEL_PLLDIV0:
            CSL_FINS (hPllc->regs->PLLDIV1, PLLC_PLLDIV1_D1EN,
                      (CSL_PllcDivCtrl)loadVal.divCtrl);
            break;

        case CSL_PLLC_DIVSEL_PLLDIV1:
            CSL_FINS (hPllc->regs->PLLDIV2, PLLC_PLLDIV2_D2EN,
                      (CSL_PllcDivCtrl)loadVal.divCtrl);
            break;

        case CSL_PLLC_DIVSEL_PLLDIV2:
            CSL_FINS (hPllc->regs->PLLDIV3, PLLC_PLLDIV3_D3EN,
                      (CSL_PllcDivCtrl)loadVal.divCtrl);
            break;

        case CSL_PLLC_DIVSEL_PLLDIV3:
            CSL_FINS (hPllc->regs->PLLDIV4, PLLC_PLLDIV4_D4EN,
                      (CSL_PllcDivCtrl)loadVal.divCtrl);
            break;

        case CSL_PLLC_DIVSEL_PLLDIV4:
            CSL_FINS (hPllc->regs->PLLDIV5, PLLC_PLLDIV5_D5EN,
                      (CSL_PllcDivCtrl)loadVal.divCtrl);
            break;

        case CSL_PLLC_DIVSEL_PLLDIV5:
            CSL_FINS (hPllc->regs->PLLDIV6, PLLC_PLLDIV6_D6EN,
                      (CSL_PllcDivCtrl)loadVal.divCtrl);
            break;

        case CSL_PLLC_DIVSEL_PLLDIV10:
            CSL_FINS (hPllc->regs->PLLDIV11, PLLC_PLLDIV11_D11EN,
                      (CSL_PllcDivCtrl)loadVal.divCtrl);
            break;

        case CSL_PLLC_DIVSEL_PLLDIV12:
            CSL_FINS (hPllc->regs->PLLDIV13, PLLC_PLLDIV13_D13EN,
                      (CSL_PllcDivCtrl)loadVal.divCtrl);
            break;

		case CSL_PLLC_DIVSEL_PLLDIV13:
            CSL_FINS (hPllc->regs->PLLDIV14, PLLC_PLLDIV14_D14EN,
                      (CSL_PllcDivCtrl)loadVal.divCtrl);
            break;

        default:
            *status = CSL_ESYS_INVPARAMS;
            break;
    }
}

/** ============================================================================
 *   @n@b CSL_pllcCmdWakeup
 *
 *   @b Description
 *      Controls the pllc wakeup operation.
 *
 *   @b Arguments
 *   @verbatim
        hPllc        Handle to the PLLC instance
        loadVal      Value to be loaded to pllc WAKEUP register
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc wakeup operation is controlled.
 *
 *   @b Modifies
 *   @n pllc WAKEUP register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_BitMask16       loadVal;
        ...

        CSL_pllcCmdWakeup (hPllc, loadVal);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_pllcCmdWakeup (
    CSL_PllcHandle       hPllc,
    CSL_BitMask16        loadVal
)
{
    hPllc->regs->WAKEUP = loadVal;
}

#ifdef __cplusplus
}
#endif

#endif /* _CSL_PLLCAUX_H_ */

