/* =============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *  ============================================================================
 */

/** ============================================================================
 *   @file  csl_rac_regsBEAux.h
 *
 *   @path  $(CSLPATH)\inc
 *
 *   @desc  This file contains all the API operation prototypes for RAC COMMON CSL
 *			for the BackEnd interface
 *
 */
 
/* =============================================================================
 *  Revision History
 *  ================
 *  31-May-2005   RDefosseux     File Created.
 *  23-Nov-2005   DBell          Added CSL_RAC_BEII_getInterruptStatus
 *
 * =============================================================================
 */
 
#ifndef _CSL_RAC_BE_AUX_H_
#define _CSL_RAC_BE_AUX_H_

#include <csl_rac_regs.h>
#include <csl_rac_regsBETypes.h>

#ifdef __cplusplus
extern "C" {
#endif

/* =============================================================================
 * Functions linked to BETI management
 * =============================================================================
 */

/**
 *  @func    CSL_RAC_BETI_enable
 *
 *  @brief   This function enables the RAC Back End Transfer Interface (BETI)
 *			 by writing 0x1 into the RAC_BETI_CTRL register. 
 *
 *  @arg     None
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_BETI_CTRL register.
 *
 *  @enter   None          
 *
 *  @leave   The Back End Transfer Interface is then enabled.
 *
 *  @see
 *
 */
void CSL_RAC_BETI_enable ();

/**
 *  @func    CSL_RAC_BETI_disable
 *
 *  @brief   This function disables the RAC Back End Transfer Interface (BETI)
 *			 by writing 0x0 into the RAC_BETI_CTRL register. 
 *
 *  @arg     None
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_BETI_CTRL register.
 *
 *  @enter   None          
 *
 *  @leave   The Back End Transfer Interface is then disabled.
 *
 *  @see
 *
 */
void CSL_RAC_BETI_disable ();

/**
 *  @func    CSL_RAC_BETI_getStatus
 *
 *  @brief   This function returns the BETI status indicating whether the
 *			 BETI is enabled or not.
 *
 *  @arg     None
 *
 *  @return  CSL_RAC_BETI_statusBit_Disable
 *               When the BETI is disabled.
 * 
 *  @return  CSL_RAC_BETI_statusBit_Enable
 *               When the BETI is enabled.
 *
 *  @desc    This function reads the RAC_BETI_STAT register. Reading this
 *			 register is relevant especially when disabling the BETI. When
 *			 disabling, a transfer can be in process and the BETI has to
 *			 complete it before being disabled.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
CSL_RAC_BETI_statusBit CSL_RAC_BETI_getStatus ();

/**
 *  @func    CSL_RAC_BETI_setWatchDog
 *
 *  @brief   This function configures the minimal activity watch dog
 *           in the BETI. 
 *
 *  @arg     nbCyclesToReload
 *           Half-word with the value to set in the register.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_BETI_MIN_WCFG register with
 *           the nbCyclesToReload value.
 *
 *  @enter   None          
 *
 *  @leave   The Back End watch-dogs are now configured.
 *
 *  @see
 *
 */
void CSL_RAC_BETI_setWatchDog (
	Uint16 nbCyclesToReload
);

/**
 *  @func    CSL_RAC_BETI_getWatchDogStatus
 *
 *  @brief   This function reads the content of the minimal activity watch dog
 *           decounter for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP minimal activity watch-dog decounter to read.
 *
 *  @return  Uint16
 *				Current value of the decounter.
 *
 *  @desc    This operation reads the corresponding RAC_BETI_MIN_WST[gccpId]
 *           register and extracts the decounter field.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_BETI_getWatchDogStatus (
	Uint8 gccpId
);

/**
 *  @func    CSL_RAC_BETI_getWatchDogInterruptStatus
 *
 *  @brief   This function reads the content of the minimal activity watch dog
 *           interrupt status for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP minimal activity watch-dog status to read.
 *
 *  @return  CSL_RAC_BETI_wdInterruptStatus_NoInt
 *               No interrupt has been generated.
 * 
 *  @return  CSL_RAC_BETI_wdInterruptStatus_Int
 *               An interrupt has been generated and forwarded to the BEII.
 * 
 *
 *  @desc    This operation reads the corresponding RAC_BETI_MIN_WST[gccpId] register and extracts the
 *			 interrupt status field.
 *
 *  @enter   None          
 *
 *  @leave   When the user reads the RAC_BETI_MIN_WST[gccpId] register, the
 *           interrupt status field is cleared by the H/W.
 *
 *  @see
 *
 */
CSL_RAC_BETI_wdInterruptStatus CSL_RAC_BETI_getWatchDogInterruptStatus (
	Uint8 gccpId
);

/**
 *  @func    CSL_RAC_BETI_getOdbtStatus
 *
 *  @brief   This function returns if the corresponding Output Descriptor Buffer
 *           has received at least one new output block descriptor.
 *
 *  @arg     odbtEntryId
 *           Which buffer status to get.
 *
 *  @return  CSL_RAC_BETI_odbtStatusBit_noNewObd
 *               When no new Output Block Descriptor has been generated for this
 *				 output buffer.
 * 
 *  @return  CSL_RAC_BETI_odbtStatusBit_newObd
 *               A new Output Block Descriptor has been written to the buffer.
 * 
 *  @desc    This operation reads the corresponding RAC_BETI_NEW_OBD register and extracts
 *           status bit for the given buffer.
 *
 *  @enter   None          
 *
 *  @leave   Clears the bit inside the register.
 *
 *  @see
 *
 */
CSL_RAC_BETI_odbtStatusBit CSL_RAC_BETI_getOdbtStatus (
	Uint8 odbtEntryId
);

/* =============================================================================
 * Functions linked to BEII management
 * =============================================================================
 */

/**
 *  @func    CSL_RAC_BEII_enable
 *
 *  @brief   This function enables the RAC Back End Interrupt Interface (BEII)
 *			 by writing 0x1 into the RAC_BEII_CTRL register. 
 *
 *  @arg     None
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_BEII_CTRL register.
 *
 *  @enter   None          
 *
 *  @leave   The Back End Interrupt Interface is then enabled.
 *
 *  @see
 *
 */
void CSL_RAC_BEII_enable ();

/**
 *  @func    CSL_RAC_BEII_disable
 *
 *  @brief   This function disables the RAC Back End Interrupt Interface (BEII)
 *			 by writing 0x0 into the RAC_BEII_CTRL register. 
 *
 *  @arg     None
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_BEII_CTRL register.
 *
 *  @enter   None          
 *
 *  @leave   The Back End Interrupt Interface is then disabled.
 *
 *  @see
 *
 */
void CSL_RAC_BEII_disable ();

/**
 *  @func    CSL_RAC_BEII_setMasterMask
 *
 *  @brief   This function configures the BEII Master Mask for the specified
 *           cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @arg     cpuIntContrlEnable
 *			 CSL_RAC_BEII_interrupt_Disable: no interrupt is sent to the specified interrupt controller.
 *			 CSL_RAC_BEII_interrupt_Enable: interrupts are forwarded to the specified interrupt controller.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_BEII_MINT_ENA register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_setMasterMask (
	Uint8                  cpuId,
	CSL_RAC_BEII_interrupt cpuIntContrlEnable
);

/**
 *  @func    CSL_RAC_BEII_getInterruptStatus
 *
 *  @brief   This function reads the BEII Interrupt Status for the specified
 *           cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @return  CSL_RAC_BEII_interruptStatus
 *
 *  @desc    This operation modifies the RAC_BEII_INT_STATx register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_getInterruptStatus (
	Uint8                          cpuId,
	CSL_RAC_BEII_interruptStatus * cpuIntStatus
);

/**
 *  @func    CSL_RAC_BEII_setCycleOverflowMask
 *
 *  @brief   This function configures the Cycle Overflow Mask for the specified
 *           cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @arg     gccpId
 *           Which Gccp source to enable.
 *
 *  @arg     interruptEnable
 *			 CSL_RAC_BEII_interrupt_Disable: no interrupt is sent to the specified interrupt controller.
 *			 CSL_RAC_BEII_interrupt_Enable: interrupts are forwarded to the specified interrupt controller.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the corresponding RAC_BEII_INT_MSK[cpuId] register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_setCycleOverflowMask (
	Uint8                  cpuId,
	Uint8                  gccpId,
	CSL_RAC_BEII_interrupt interruptEnable
);

/**
 *  @func    CSL_RAC_BEII_setFifoOverflowMask
 *
 *  @brief   This function configures the FIFO Overflow Mask for the specified
 *           cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @arg     gccpId
 *           Which Gccp source to enable.
 *
 *  @arg     interruptEnable
 *			 CSL_RAC_BEII_interrupt_Disable: no interrupt is sent to the specified interrupt controller.
 *			 CSL_RAC_BEII_interrupt_Enable: interrupts are forwarded to the specified interrupt controller.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the corresponding RAC_BEII_INT_MSK[cpuId] register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_setFifoOverflowMask (
	Uint8                  cpuId,
	Uint8                  gccpId,
	CSL_RAC_BEII_interrupt interruptEnable
);

/**
 *  @func    CSL_RAC_BEII_setSequencerIdleMask
 *
 *  @brief   This function configures the Sequencer Idle Mask for the specified
 *           cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @arg     gccpId
 *           Which Gccp source to enable.
 *
 *  @arg     interruptEnable
 *			 CSL_RAC_BEII_interrupt_Disable: no interrupt is sent to the specified interrupt controller.
 *			 CSL_RAC_BEII_interrupt_Enable: interrupts are forwarded to the specified interrupt controller.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the corresponding RAC_BEII_INT_MSK[cpuId] register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_setSequencerIdleMask (
	Uint8                  cpuId,
	Uint8                  gccpId,
	CSL_RAC_BEII_interrupt interruptEnable
);

/**
 *  @func    CSL_RAC_BEII_setBeWatchDogMask
 *
 *  @brief   This function configures the Back End Min Activity Watch Dog Mask
 *           for the specified cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @arg     gccpId
 *           Which Gccp source to enable.
 *
 *  @arg     interruptEnable
 *			 CSL_RAC_BEII_interrupt_Disable: no interrupt is sent to the specified interrupt controller.
 *			 CSL_RAC_BEII_interrupt_Enable: interrupts are forwarded to the specified interrupt controller.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the corresponding RAC_BEII_INT_MSK[cpuId] register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_setBeWatchDogMask (
	Uint8                  cpuId,
	Uint8                  gccpId,
	CSL_RAC_BEII_interrupt interruptEnable
);

/**
 *  @func    CSL_RAC_BEII_setBetiEotMask
 *
 *  @brief   This function configures the Back End End-Of-Transfer Mask
 *           for the specified cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @arg     interruptEnable
 *			 CSL_RAC_BEII_interrupt_Disable: no interrupt is sent to the specified interrupt controller.
 *			 CSL_RAC_BEII_interrupt_Enable: interrupts are forwarded to the specified interrupt controller.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the corresponding RAC_BEII_INT_MSK[cpuId] register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_setBetiEotMask (
	Uint8                  cpuId,
	CSL_RAC_BEII_interrupt interruptEnable
);

/**
 *  @func    CSL_RAC_BEII_setBetiObbtRdCrossingMask
 *
 *  @brief   This function configures the Back End OBBT Read Pointer Crossing Mask
 *           for the specified cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @arg     interruptEnable
 *			 CSL_RAC_BEII_interrupt_Disable: no interrupt is sent to the specified interrupt controller.
 *			 CSL_RAC_BEII_interrupt_Enable: interrupts are forwarded to the specified interrupt controller.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the corresponding RAC_BEII_INT_MSK[cpuId] register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_setBetiObbtRdCrossingMask (
	Uint8                  cpuId,
	CSL_RAC_BEII_interrupt interruptEnable
);

/**
 *  @func    CSL_RAC_BEII_setBetiOdbtRdCrossingMask
 *
 *  @brief   This function configures the Back End ODBT Read Pointer Crossing Mask
 *           for the specified cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @arg     interruptEnable
 *			 CSL_RAC_BEII_interrupt_Disable: no interrupt is sent to the specified interrupt controller.
 *			 CSL_RAC_BEII_interrupt_Enable: interrupts are forwarded to the specified interrupt controller.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the corresponding RAC_BEII_INT_MSK[cpuId] register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_setBetiOdbtRdCrossingMask (
	Uint8                  cpuId,
	CSL_RAC_BEII_interrupt interruptEnable
);

/**
 *  @func    CSL_RAC_BEII_setFeWatchDogMask
 *
 *  @brief   This function configures the Front End Watch Dog Mask
 *           for the specified cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @arg     interruptEnable
 *			 CSL_RAC_BEII_interrupt_Disable: no interrupt is sent to the specified interrupt controller.
 *			 CSL_RAC_BEII_interrupt_Enable: interrupts are forwarded to the specified interrupt controller.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the corresponding RAC_BEII_INT_MSK[cpuId] register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_setFeWatchDogMask (
	Uint8                  cpuId,
	CSL_RAC_BEII_interrupt interruptEnable
);

/**
 *  @func    CSL_RAC_BETI_getObbtRdCrossingStatus
 *
 *  @brief   This function reads the content of the OBBT read pointer crossing register. 
 *
 *  @arg     None
 *
 *  @return  
 *				All the OBBT Status.
 *
 *  @desc    This operation reads the RAC_BETI_OBBT_STAT.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_BETI_getObbtRdCrossingStatus ();

/**
 *  @func    CSL_RAC_BETI_getOdbtRdCrossingStatus
 *
 *  @brief   This function reads the content of the ODBT read pointer crossing register. 
 *
 *  @arg     None
 *
 *  @return  
 *				All the ODBT Status.
 *
 *  @desc    This operation reads the RAC_BETI_ODBT_STAT.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_BETI_getOdbtRdCrossingStatus ();

/**
 *  @func    CSL_RAC_BETI_getEotInterruptStatus
 *
 *  @brief   This function reads the content of the End-Of-Transfer Interrupt
 *			 Status register to know which Output Descriptor Buffer has a new
 *			 descriptor.
 *
 *  @arg     cpuId
 *           Which CPU interrupt register to read.
 *
 *  @return  
 *				All the EOT Interrupt registers.
 *
 *  @desc    This operation reads the corresponding RAC_BETI_EOT_STAT[cpuId] register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_BETI_getEotInterruptStatus (
	Uint8 cpuId
);

/* =============================================================================
 * Functions linked to VBUS Stats registers
 * =============================================================================
 */

/**
 *  @func    CSL_RAC_Stats_getCfgTotalAccess
 *
 *  @brief   This function reads the content of the statistics register for 
 *           the total number of data phases used by the Configuration interface.
 *
 *  @arg     None
 *
 *  @return  
 *				The value of the register.
 *
 *  @desc    This operation reads the corresponding RAC_CFG_TOT_REG register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_Stats_getCfgTotalAccess ();

/**
 *  @func    CSL_RAC_Stats_getCfgWriteAccess
 *
 *  @brief   This function reads the content of the statistics register for 
 *           the total number of write data phases used by the Configuration interface.
 *
 *  @arg     None
 *
 *  @return  
 *				The value of the register.
 *
 *  @desc    This operation reads the corresponding RAC_CFG_WRITE_REG register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */

Uint32 CSL_RAC_Stats_getCfgWriteAccess ();

/**
 *  @func    CSL_RAC_Stats_getCfgReadAccess
 *
 *  @brief   This function reads the content of the statistics register for 
 *           the total number of read data phases used by the Configuration interface.
 *
 *  @arg     None
 *
 *  @return  
 *				The value of the register.
 *
 *  @desc    This operation reads the corresponding RAC_CFG_READ_REG register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_Stats_getCfgReadAccess ();

/**
 *  @func    CSL_RAC_Stats_getSlaveTotalAccess
 *
 *  @brief   This function reads the content of the statistics register for 
 *           the total number of data phases used by the DMA slave interface.
 *
 *  @arg     None
 *
 *  @return  
 *				The value of the register.
 *
 *  @desc    This operation reads the corresponding RAC_SLV_TOT_REG register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_Stats_getSlaveTotalAccess ();

/**
 *  @func    CSL_RAC_Stats_getSlaveWriteAccess
 *
 *  @brief   This function reads the content of the statistics register for 
 *           the total number of write data phases used by the DMA Slave interface.
 *
 *  @arg     None
 *
 *  @return  
 *				The value of the register.
 *
 *  @desc    This operation reads the corresponding RAC_SLV_WRITE_REG register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */

Uint32 CSL_RAC_Stats_getSlaveWriteAccess ();

/**
 *  @func    CSL_RAC_Stats_getSlaveReadAccess
 *
 *  @brief   This function reads the content of the statistics register for 
 *           the total number of read data phases used by the DMA Slave interface.
 *
 *  @arg     None
 *
 *  @return  
 *				The value of the register.
 *
 *  @desc    This operation reads the corresponding RAC_SLV_READ_REG register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_Stats_getSlaveReadAccess ();

/**
 *  @func    CSL_RAC_Stats_getMasterLowPrioAccess
 *
 *  @brief   This function reads the content of the statistics register for 
 *           the total number of data phases used by the low priority master
 *           interface.
 *
 *  @arg     None
 *
 *  @return  
 *				The value of the register.
 *
 *  @desc    This operation reads the corresponding RAC_MST_LP_TOT_REG register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_Stats_getMasterLowPrioAccess ();

/**
 *  @func    CSL_RAC_Stats_getMasterHighPrioAccess
 *
 *  @brief   This function reads the content of the statistics register for 
 *           the total number of data phases used by the high priority master
 *           interface.
 *
 *  @arg     None
 *
 *  @return  
 *				The value of the register.
 *
 *  @desc    This operation reads the corresponding RAC_MST_HP_TOT_REG register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_Stats_getMasterHighPrioAccess ();




#ifdef __cplusplus
}
#endif

#endif
