/* =============================================================================
 *   @copy    Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *  ============================================================================
 */                                                         
 
/** ============================================================================
 *   @file  csl_rac_regsGCCPAux.h
 *
 *   @path  $(CSLPATH)\inc
 *
 *   @desc  This file contains all the API operation prototypes for RAC COMMON CSL
 *			for the GCCP
 *
 */                            

/* =============================================================================
 *  Revision History
 *  ================
 *  31-May-2005   RDefosseux     File Created.
 *  05-Jan-2006   RDefosseux     Removed some APIs for optimization
 *
 * =============================================================================
 */
 
#ifndef _CSL_RAC_GCCP_AUX_H_
#define _CSL_RAC_GCCP_AUX_H_

#include <csl_rac_regs.h>
#include <csl_rac_regsGCCPTypes.h>
#include <csl_rac_regsFETypes.h>

#ifdef __cplusplus
extern "C" {
#endif

/* =============================================================================
 * Functions linked to GCCP management
 * =============================================================================
 */

/**
 *  @func    CSL_RAC_GCCP_enable
 *
 *  @brief   This function enables the corresponding Generic Correlation Co-Processor
 *			 (GCCP) by writing 0x1 into the RAC_GCCP_SEQ_ENA register. 
 *
 *  @arg     gccpId
 *           Which GCCP to activate.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_GCCP_SEQ_ENA register.
 *
 *  @enter   None          
 *
 *  @leave   The GCCP Sequencer is then enabled.
 *
 *  @see
 *
 */
void CSL_RAC_GCCP_enable (
	Uint8 gccpId
);

/**
 *  @func    CSL_RAC_GCCP_disable
 *
 *  @brief   This function disables the corresponding Generic Correlation Co-Processor
 *			 (GCCP) by writing 0x1 into the RAC_GCCP_SEQ_ENA register. 
 *
 *  @arg     gccpId
 *           Which GCCP to disactivate.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_GCCP_SEQ_ENA register.
 *
 *  @enter   None          
 *
 *  @leave   The GCCP Sequencer is then disabled.
 *
 *  @see
 *
 */
void CSL_RAC_GCCP_disable (
	Uint8 gccpId
);

/**
 *  @func    CSL_RAC_GCCP_getActiveCycles
 *
 *  @brief   This function returns the number of active cycles in the previous
 *           iteration.
 *
 *  @arg     gccpId
 *           Which GCCP to get status from.
 *
 *  @return  The number of active cycles
 *
 *  @desc    This function reads the RAC_GCCP_SEQ_ACT register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint16 CSL_RAC_GCCP_getActiveCycles (
	Uint8 gccpId
);

/**
 *  @func    CSL_RAC_GCCP_getSequencerCycles
 *
 *  @brief   This function returns the number of sequencer-used cycles in the previous
 *           iteration.
 *
 *  @arg     gccpId
 *           Which GCCP to get status from.
 *
 *  @return  The number of sequencer cycles
 *
 *  @desc    This function reads the RAC_GCCP_SEQ_CYC register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint16 CSL_RAC_GCCP_getSequencerCycles (
	Uint8 gccpId
);

/**
 *  @func    CSL_RAC_GCCP_resetHighPriorityQueue
 *
 *  @brief   This function resets the High Priority Output Queue of the 
 *			 corresponding GCCP by writing into the RAC_GCCP_FIFO_RESET register. 
 *
 *  @arg     gccpId
 *           Which GCCP FIFO to reset.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_GCCP_FIFO_RESET register.
 *
 *  @enter   None          
 *
 *  @leave   The GCCP High Priority Output Queue is reset.
 *
 *  @see
 *
 */
void CSL_RAC_GCCP_resetHighPriorityQueue (
	Uint8 gccpId
);

/**
 *  @func    CSL_RAC_GCCP_resetLowPriorityQueue
 *
 *  @brief   This function resets the Low Priority Output Queue of the 
 *			 corresponding GCCP by writing into the RAC_GCCP_FIFO_RESET register. 
 *
 *  @arg     gccpId
 *           Which GCCP FIFO to reset.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_GCCP_FIFO_RESET register.
 *
 *  @enter   None          
 *
 *  @leave   The GCCP Low Priority Output Queue is reset.
 *
 *  @see
 *
 */
void CSL_RAC_GCCP_resetLowPriorityQueue (
	Uint8 gccpId
);

/**
 *  @func    CSL_RAC_GCCP_getReadTime
 *
 *  @brief   This function reads the GCCP[0] read timestamp register. 
 *
 *  @arg     timestamp
 *           structure with timestamp parameters.
 *
 *  @return  None
 *
 *  @desc    This operation reads the content of the GCCP[0] read timestamp
 *           register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_GCCP_getReadTime (
	CSL_RAC_FE_Timestamp_req * timestamp
);

/**
 *  @func    CSL_RAC_GCCP_getLowPrioDataLevel
 *
 *  @brief   This function returns the Low Priority Data component level 
 *           of the output queue for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP status to read.
 *
 *  @return  The current value of the level
 *
 *  @desc    This operation reads the corresponding RAC_GCCP_LDQ_CURR_LVL register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getLowPrioDataLevel (
	Uint8 gccpId
);

/**
 *  @func    CSL_RAC_GCCP_getLowPrioDataWatermark
 *
 *  @brief   This function reads the Low Priority Data component water-mark
 *           of the output queue for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP status to read.
 *
 *  @return  The current value of the water-mark
 *
 *  @desc    This operation reads the corresponding RAC_GCCP_LDQ_WTMK_LVL register.
 *
 *  @enter   None          
 *
 *  @leave   The register is reset with the current level value.
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getLowPrioDataWatermark (
	Uint8 gccpId
);

/**
 *  @func    CSL_RAC_GCCP_getHighPrioDataLevel
 *
 *  @brief   This function returns the High Priority Data component level 
 *           of the output queue for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP status to read.
 *
 *  @return  The current value of the level
 *
 *  @desc    This operation reads the corresponding RAC_GCCP_HDQ_CURR_LVL register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getHighPrioDataLevel (
	Uint8 gccpId
);

/**
 *  @func    CSL_RAC_GCCP_getHighPrioDataWatermark
 *
 *  @brief   This function reads the High Priority Data component water-mark
 *           of the output queue for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP status to read.
 *
 *  @return  Uint16
 *				Current value of the water-mark
 *
 *  @desc    This operation reads the corresponding RAC_GCCP_HDQ_WTMK_LVL register.
 *
 *  @enter   None          
 *
 *  @leave   The register is reset with the current level value.
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getHighPrioDataWatermark (
	Uint8 gccpId
);

/**
 *  @func    CSL_RAC_GCCP_getLowPrioControlLevel
 *
 *  @brief   This function returns the Low Priority Control component level 
 *           of the output queue for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP status to read.
 *
 *  @return  The current value of the level
 *
 *  @desc    This operation reads the corresponding RAC_GCCP_LCQ_CURR_LVL register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getLowPrioControlLevel (
	Uint8 gccpId
);

/**
 *  @func    CSL_RAC_GCCP_getLowPrioControlWatermark
 *
 *  @brief   This function reads the Low Priority Control component water-mark
 *           of the output queue for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP status to read.
 *
 *  @return  The current value of the water-mark
 *
 *  @desc    This operation reads the corresponding RAC_GCCP_LCQ_WTMK_LVL register.
 *
 *  @enter   None          
 *
 *  @leave   The register is reset with the current level value.
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getLowPrioControlWatermark (
	Uint8 gccpId
);

/**
 *  @func    CSL_RAC_GCCP_getHighPrioControlLevel
 *
 *  @brief   This function returns the High Priority Control component level 
 *           of the output queue for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP status to read.
 *
 *  @return  The current value of the level
 *
 *  @desc    This operation reads the corresponding RAC_GCCP_HCQ_CURR_LVL register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getHighPrioControlLevel (
	Uint8 gccpId
);

/**
 *  @func    CSL_RAC_GCCP_getHighPrioControlWatermark
 *
 *  @brief   This function reads the High Priority Control component water-mark
 *           of the output queue for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP status to read.
 *
 *  @return  The current value of the water-mark
 *
 *  @desc    This operation reads the corresponding RAC_GCCP_HCQ_WTMK_LVL register.
 *
 *  @enter   None          
 *
 *  @leave   The register is reset with the current level value.
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getHighPrioControlWatermark (
	Uint8 gccpId
);

/**
 *  @func    CSL_RAC_GCCP_setIctEntry
 *
 *  @brief   This function configures the Interpolation Coefficients Table
 *           by writing into one of the RAC_ICT registers. 
 *
 *  @arg     gccpId
 *           Which GCCP ICT to configure.
 *
 *  @arg     entryId
 *           Which GCCP ICT to configure. It ranges from 0 to 4.
 *
 *  @arg     coeff0
 *           Value to set into the coefficient #0 field. It should range from -64 to 64
 *
 *  @arg     coeff1
 *           Value to set into the coefficient #1 field. It should range from -64 to 64
 *
 *  @arg     coeff2
 *           Value to set into the coefficient #2 field. It should range from -64 to 64
 *
 *  @arg     coeff3
 *           Value to set into the coefficient #3 field. It should range from -64 to 64
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_GCCP_ICT register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_GCCP_setIctEntry (
	Uint8 gccpId,
	Uint8 entryId,
	Int8  coeff0,
	Int8  coeff1,
	Int8  coeff2,
	Int8  coeff3
);

/**
 *  @func    CSL_RAC_GCCP_setCgtEntry
 *
 *  @brief   This function configures the Code Generation Table by writing
 *           into one of the RAC_GCCP_CGT registers. 
 *
 *  @arg     gccpId
 *           Which GCCP CGT to configure.
 *
 *  @arg     entryId
 *           Which entry to configure.
 *
 *  @arg     partY
 *           Value to write into the register.
 *
 *  @return  None
 *
 *  @desc    This operation modifies one of the RAC_GCCP_CGT register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_GCCP_setCgtEntry (
	Uint8  gccpId,
	Uint8  entryId,
	Uint32 partY
);

/**
 *  @func    CSL_RAC_GCCP_getCgtEntry
 *
 *  @brief   This function reads one of the RAC_GCCP_CGT registers. 
 *
 *  @arg     gccpId
 *           Which GCCP CGT to configure.
 *
 *  @arg     entryId
 *           Which entry to configure.
 *
 *  @return  None
 *
 *  @desc    This operation returns the Y part for the given GCCP and entry ID.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getCgtEntry (
	Uint8  gccpId,
	Uint8  entryId
);

/**
 *  @func    CSL_RAC_GCCP_setPrtEntry
 *
 *  @brief   This function configures the Pilot Rotation Table by writing
 *           into one of the RAC_GCCP_PRT registers. 
 *
 *  @arg     gccpId
 *           Which GCCP PRT to configure.
 *
 *  @arg     entryId
 *           Which entry to configure.
 *
 *  @arg     wordId
 *           Which word to configure.
 *
 *  @arg     format
 *           Value to insert into the register.
 *
 *  @return  None
 *
 *  @desc    This operation modifies one of the RAC_GCCP_PRT register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_GCCP_setPrtEntry (
	Uint8  gccpId,
	Uint8  entryId,
	Uint8  wordId,
	Uint32 format
);

/**
 *  @func    CSL_RAC_GCCP_getPrtEntry
 *
 *  @brief   This function reads one of the RAC_GCCP_PRT registers. 
 *
 *  @arg     gccpId
 *           Which GCCP PRT to configure.
 *
 *  @arg     entryId
 *           Which entry to configure.
 *
 *  @arg     wordId
 *           Which slot to configure.
 *
 *  @return  None
 *
 *  @desc    This operation returns the slot format for the given slot ID.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getPrtEntry (
	Uint8  gccpId,
	Uint8  entryId,
	Uint8  wordId
);

void CSL_RAC_GCCP_getCycleOverflowStatus (
	Uint8                              gccpId,
	CSL_RAC_GCCP_cycleOverflowStatus * cycleOverflowStatus
);

/**
 *  @func    CSL_RAC_GCCP_getFifoOverflowStatus
 *
 *  @brief   This function returns the last active TRM address
 *
 *  @arg     gccpId
 *           Which GCCP to get status from.
 *
 *  @return  The last active TRM address before a FIFO overflow
 *
 *  @desc    This function reads the RAC_GCCP_FIFO_OVER register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_GCCP_getFifoOverflowStatus (
	Uint8 gccpId,
	CSL_RAC_GCCP_fifoOverflowStatus * fifoOverflowStatus
);

#ifdef __cplusplus
}
#endif

#endif
