/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */
 
/** ===========================================================================
 *   @file  csl_sgmii.c
 *
 *   @brief File containing functional layer API's of SGMII.
 *
 *   @path  $(CSLPATH)\src\SGMII
 *
 *   @desc  The function definition of the API's of SGMII.
 */ 

/* =============================================================================
 * Revision History
 * ===============
 *  01-May-2006 NS  updated the file for DOXYGEN compatibiliy
 *  12-Jan-2006 PSK file created 
 * =============================================================================
 */                                      

#include <csl_sgmii.h>

/** ============================================================================
 *   @n@b SGMII_reset
 *
 *   @b Description
 *   @n Module reset is achieved by calling SGMII_reset().
 *
 *   <b> Return Value </b>  Uint32
 *   @li Always returns a '0' if reset happens properly.
 *
 *   <b> Pre Condition </b>
 *   @n  None.
 *
 *   <b> Post Condition </b>
 *   @n  SGMII logic is reset
 *
 *   @b Modifies    
 *   @n  Memory mapped register SOFT_RESET is modified.
 *
 *   @b Example
 *   @verbatim

     SGMII_reset ();

     @endverbatim
 * =============================================================================
 */
Uint32 SGMII_reset ()
{
    CSL_FINST(SGMII_REGS->SOFT_RESET, SGMII_SOFT_RESET_SOFT_RESET, YES);
    while (SGMII_REGS->SOFT_RESET != 0x00000000);

    return 0;
}

/** ============================================================================
 *   @n@b SGMII_config
 *
 *   @b Description
 *   @n Module configuration is achieved by calling SGMII_config().
 *   @b Arguments
 *   @verbatim
                            
            config        Reference to struture which contains configuration for the SGMII module
                 
     @endverbatim
 *   <b> Return Value </b>  Uint32
 *   @li                    0                   - Configure successful
 *   @li                    SGMII_ERROR_INVALID - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  None.
 *
 *   <b> Post Condition </b>
 *   @n  Module configuration is achieved.
 *
 *   @b Modifies    
 *   @n SGMII registers are modified.
 *
 *   @b Example
 *   @verbatim
     ...
     SGMII_Config SgmiiCfg;
         
    SgmiiCfg.masterEn   = 0x0;
    SgmiiCfg.loopbackEn = 0x1;
    SgmiiCfg.speed      = 0x0;
    SgmiiCfg.txConfig   = 0x00000e13;
    SgmiiCfg.rxConfig   = 0x00081013;
    SgmiiCfg.auxConfig  = 0x0000000b;
    SgmiiCfg.diagSmSel = 0x0;
    SgmiiCfg.diagEdgeSel = 0x0;
    SgmiiCfg.modeOfOperation = 0x0;

     
     SGMII_reset();
     SGMII_config (&SgmiiCfg);
     ...
     @endverbatim
 * =============================================================================
 */
Uint32 SGMII_config (SGMII_Config *config)
{
    volatile Uint32 val;

    if (!config) {
        return SGMII_ERROR_INVALID; 
    }

    if (config->loopbackEn) {
        CSL_FINS(SGMII_REGS->CONTROL, SGMII_CONTROL_MR_AN_ENABLE, 0);
        CSL_FINST(SGMII_REGS->SOFT_RESET, SGMII_SOFT_RESET_RT_SOFT_RESET, YES);
        CSL_FINST(SGMII_REGS->CONTROL, SGMII_CONTROL_LOOPBACK, YES);
        CSL_FINST(SGMII_REGS->SOFT_RESET, SGMII_SOFT_RESET_RT_SOFT_RESET, NO);
    }else {
        if ((config->masterEn) && (config->modeOfOperation == SGMII_MODE_OF_OPERATION_WITH_AN)) {
            SGMII_REGS->MR_ADV_ABILITY = 0x9801; /* Advertise fullduplex gigabit */
            SGMII_REGS->CONTROL = CSL_FMKT(SGMII_CONTROL_MASTER, MASTER) |
                                  CSL_FMKT(SGMII_CONTROL_MR_AN_ENABLE, YES);
           // while (CSL_FEXT(SGMII_REGS->STATUS, SGMII_STATUS_MR_AN_COMPLETE) ) {
            //}
            //val = *(volatile Uint32 *)0x02C80160;
            if (config->speed == SGMII_OPERATE_SPEED_2_5MHZ) {
                *(volatile Uint32 *)0x02C80160 |= 0x00010000;
            }
            if (config->speed == SGMII_OPERATE_SPEED_25MHZ) {
                *(volatile Uint32 *)0x02C80160 |= 0x00018000;
            }
        }else if ((!(config->masterEn)) && (config->modeOfOperation == SGMII_MODE_OF_OPERATION_WITH_AN)) {
            SGMII_REGS->MR_ADV_ABILITY = 0x01; /* Advertise fullduplex gigabit */
            SGMII_REGS->CONTROL = CSL_FMKT(SGMII_CONTROL_MASTER, SLAVE) |
                                  CSL_FMKT(SGMII_CONTROL_MR_AN_ENABLE, YES);

            //while (CSL_FEXT(SGMII_REGS->STATUS, SGMII_STATUS_MR_AN_COMPLETE) ) {
            //}
            //val = *(volatile Uint32 *)0x02C80160;
            if (config->speed == SGMII_OPERATE_SPEED_2_5MHZ) {
                *(volatile Uint32 *)0x02C80160 |= 0x00010000;
            }
            else if (config->speed == SGMII_OPERATE_SPEED_25MHZ) {
                *(volatile Uint32 *)0x02C80160 |= 0x00018000;
            }
        }else if ((config->masterEn) && (config->modeOfOperation == SGMII_MODE_OF_OPERATION_WITHOUT_AN)) {
            SGMII_REGS->MR_ADV_ABILITY = 0x9801; /* Advertise fullduplex gigabit */
            SGMII_REGS->CONTROL = CSL_FMKT(SGMII_CONTROL_MASTER, MASTER) |
                                  CSL_FMK(SGMII_CONTROL_MR_AN_ENABLE, 0);
            //while (CSL_FEXT(SGMII_REGS->STATUS, SGMII_STATUS_LINK) ) {
            //}
            //val = *(volatile Uint32 *)0x02C80160;
            if (config->speed == SGMII_OPERATE_SPEED_2_5MHZ) {
                *(volatile Uint32 *)0x02C80160 |= 0x00010000;
            }
            if (config->speed == SGMII_OPERATE_SPEED_25MHZ) {
                *(volatile Uint32 *)0x02C80160 |= 0x00018000;
            }
        }else {
            SGMII_REGS->MR_ADV_ABILITY = 0x01; /* Advertise fullduplex gigabit */
            SGMII_REGS->CONTROL = CSL_FMKT(SGMII_CONTROL_MASTER, SLAVE) |
                                  CSL_FMKT(SGMII_CONTROL_MR_AN_ENABLE, YES);
            //while (CSL_FEXT(SGMII_REGS->STATUS, SGMII_STATUS_MR_AN_COMPLETE) ) {
            //}
            //val = *(volatile Uint32 *)0x02C80160;
            if (config->speed == SGMII_OPERATE_SPEED_2_5MHZ) {
                *(volatile Uint32 *)0x02C80160 |= 0x00010000;
            }
            if (config->speed == SGMII_OPERATE_SPEED_25MHZ) {
                *(volatile Uint32 *)0x02C80160 |= 0x00018000;
            }
        }
    }

    SGMII_REGS->TX_CFG = config->txConfig;
    SGMII_REGS->RX_CFG = config->rxConfig;
    SGMII_REGS->AUX_CFG = config->auxConfig;

    CSL_FINS(SGMII_REGS->DIAG_CONTROL, SGMII_DIAG_CONTROL_DIAG_SM_SEL, config->diagSmSel);
    CSL_FINS(SGMII_REGS->DIAG_CONTROL, SGMII_DIAG_CONTROL_DIAG_EDGE_SEL, config->diagEdgeSel);

    return 0;
}

/** ============================================================================
 *   @n@b SGMII_getStatus                 SGMII module Status Structure
 *
 *   @b Description
 *   @n Module status is obtained by calling SGMII_getStatus().
 *   @b Arguments
 *   @verbatim
                            
            status         Reference to an SGMII module Status Structure.
                 
     @endverbatim
 *
 *   <b> Return Value </b>  Uint32
 *   @li                    0                   - Status is obtained successfully.
 *   @li                    SGMII_ERROR_INVALID - Invalid parameter.
 *
 *   <b> Pre Condition </b>
 *   @n  None.
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies    
 *   @n Updates the argument "status".
 *
 *   @b Example
 *   @verbatim
     ...
    SGMII_Status status;
    SGMII_Config SgmiiCfg;
         
    SgmiiCfg.masterEn   = 0x0;
    SgmiiCfg.loopbackEn = 0x1;
    SgmiiCfg.speed      = 0x0;
    SgmiiCfg.txConfig   = 0x00000e13;
    SgmiiCfg.rxConfig   = 0x00081013;
    SgmiiCfg.auxConfig  = 0x0000000b;
    SgmiiCfg.diagSmSel = 0x0;
    SgmiiCfg.diagEdgeSel = 0x0;
    SgmiiCfg.modeOfOperation = 0x0;

     
     SGMII_reset();
     SGMII_config (&SgmiiCfg);
    
     SGMII_getStatus (&status);
     ...
     @endverbatim
 * =============================================================================
 */
Uint32 SGMII_getStatus (SGMII_Status *status)
{
    if (!status) {
        return SGMII_ERROR_INVALID;
    }
    status->txCfgStatus = SGMII_REGS->TX_CFG;
    status->rxCfgStatus = SGMII_REGS->RX_CFG;
    status->auxCfgStatus = SGMII_REGS->AUX_CFG;

    status->diagStatus = SGMII_REGS->DIAG_STATUS & 0xFFFF;

    return 0;
}

/** ============================================================================
 *   @n@b SGMII_getLinkPatnerStatus
 *
 *   @b Description
 *   @n Gets the status value of link partner.
 *
 *   <b> Return Value </b>  Uint32
 *   @li                    SGMII_ERROR_DEVICE - auto negotiation not complete 
 *
 *   <b> Pre Condition </b>
 *   @n  None.
 *
 *   <b> Post Condition </b>
 *   @n  On success returns auto negotiation status value
 *
 *   @b Modifies    
 *   @n None.
 *
 *   @b Example
 *   @verbatim
 
    SGMII_Config SgmiiCfg;
         
    SgmiiCfg.masterEn   = 0x0;
    SgmiiCfg.loopbackEn = 0x1;
    SgmiiCfg.speed      = 0x0;
    SgmiiCfg.txConfig   = 0x00000e13;
    SgmiiCfg.rxConfig   = 0x00081013;
    SgmiiCfg.auxConfig  = 0x0000000b;
    SgmiiCfg.diagSmSel = 0x0;
    SgmiiCfg.diagEdgeSel = 0x0;
    SgmiiCfg.modeOfOperation = 0x0;

     
     SGMII_reset();
     SGMII_config (&SgmiiCfg);
    
        
     SGMII_getLinkPartnerStatus ( );
     ...
     @endverbatim
 * =============================================================================
 */
Uint32 SGMII_getLinkPartnerStatus ()
{   

    if (CSL_FEXT(SGMII_REGS->STATUS, SGMII_STATUS_MR_AN_COMPLETE)){
        return (SGMII_REGS->MR_LP_ADV_ABILITY & 0xFFFF);
    }

    return SGMII_ERROR_DEVICE;
}

/** ============================================================================
 *   @n@b SGMII_getAnErrorStatus
 *
 *   @b Description
 *   @n The function returns Error status of Auto negotiation process.
 *
 *   <b> Return Value </b>  Uint32
 *   @li Error status of Auto negotiation process.
 *
 *   <b> Pre Condition </b>
 *   @n  None.
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies    
 *   @n None.
 *
 *   @b Example
 *   @verbatim
     
    SGMII_Config SgmiiCfg;
         
    SgmiiCfg.masterEn   = 0x0;
    SgmiiCfg.loopbackEn = 0x1;
    SgmiiCfg.speed      = 0x0;
    SgmiiCfg.txConfig   = 0x00000e13;
    SgmiiCfg.rxConfig   = 0x00081013;
    SgmiiCfg.auxConfig  = 0x0000000b;
    SgmiiCfg.diagSmSel = 0x0;
    SgmiiCfg.diagEdgeSel = 0x0;
    SgmiiCfg.modeOfOperation = 0x0;

     
    SGMII_reset();
    SGMII_config (&SgmiiCfg);

    SGMII_getAnErrorStatus ( );

     @endverbatim
 * =============================================================================
 */
Uint32 SGMII_getAnErrorStatus () 
{
    return CSL_FEXT(SGMII_REGS->STATUS, SGMII_STATUS_AN_ERROR);
}

/** ============================================================================
 *   @n@b SGMII_clearDiagnostics
 *
 *   @b Description
 *   @n Clears diagnostics register
 *
 *   <b> Return Value </b>  Uint32
 *   @li                0                   - Clears diagnostics register successfully.
 *
 *   <b> Pre Condition </b>
 *   @n  None.
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies    
 *   @n SGMII_clearDiagnostics() API modifies DIAG_CLEAR register.
 *
 *   @b Example
 *   @verbatim

     SGMII_reset();
     SGMII_clearDiagnostics ( );

     @endverbatim
 * =============================================================================
 */
Uint32 SGMII_clearDiagnostics ()
{
    CSL_FINST(SGMII_REGS->DIAG_CLEAR, SGMII_DIAG_CLEAR_DIAG_CLEAR, SET);

    return 0;
}
                 
/** ============================================================================
 *   @n@b SGMII_getStatusReg
 *
 *   @b Description
 *   @n The function returns the value read from STATUS register 
 *
 *   <b> Return Value </b>  Uint32
 *   @li Returns value read from STATUS register
 *
 *   <b> Pre Condition </b>
 *   @n  None.
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies    
 *   @n None.
 *
 *   @b Example
 *   @verbatim
 
     SGMII_reset();
     SGMII_getStatusReg ( );

     @endverbatim
 * =============================================================================
 */

Uint32 SGMII_getStatusReg ()
{
    Uint32 status;

    status = SGMII_REGS->STATUS;
    return status; 
}



