/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** ============================================================================
 *   @file  csl_aifGetHwStatus.c
 *
 *   @path  $(CSLPATH)\aif\src
 *
 *   @desc  Antenna Interface.
 *
 */

/*TO CHANGE*/
/*  @(#) PSP/CSL 3.00.01.00[5905] (2004-10-12)  */

/* =============================================================================
 * Revision History
 * ===============
 *  26-Oct-2004 Pd File Created.
 *  21-Feb-2006 Naveen Shetti updated the file for DOXYGEN compatibiliy
 *
 * =============================================================================
 */

#include <csl_aif.h>
#include <csl_aifGetHwStatusAux.h>

/** ============================================================================
 *   @n@b CSL_aifGetHwStatus
 *
 *   @b Description
 *   @n This function is used to get the value of various parameters of the
 *      aif instance. The value returned depends on the query passed.
 *
 *   @b Arguments
 *   @verbatim
            hAifLink      Handle to the aif instance
 
            Query         Query to be performed 
 
            response      Pointer to buffer to return the data requested by
                          the query passed
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK            - Successful completion of the
 *                                               query
 *
 *   @li                    CSL_ESYS_BADHANDLE - Invalid handle
 *
 *   @li                    CSL_ESYS_INVQUERY  - Query command not supported
 *
 *   @li                    CSL_ESYS_FAIL      - Generic failure
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *       Data requested by query is returned through the variable "response"
 *
 *   @b Modifies
 *   @n The input argument "response" is modified
 *
 *   @b Example
 *   @verbatim
        
        #define CSL_AIF_MAX_RX_MASTER_FRAME_OFFSET 		256   
 
        // handle for link 0
        CSL_AifHandle handleAifLink0;
        // other link related declarations
        ...
        // ctrl argument for hw command
        Bool ctrlArg;
        // query response
        Uint16 response;

        // Open handle for link 0 - for use 
        handleAifLink0 = CSL_aifOpen(&AifLinkObj0, CSL_AIF, &aifParam, &status);

        if ((handleAifLink0 == NULL) || (status != CSL_SOK)) 
        {
           printf ("\nError opening CSL_AIF");
           exit(1);
        }

        // Do config for link 0 
        ConfigLink0.globalSetup = &gblCfg;
        ...
         
        //Do setup for link - 0
        CSL_aifHwSetup(handleAifLink0, &ConfigLink0);

        ctrlArg = CSL_AIF_CTRL_RX_LINK_ENABLE;

        // Send hw control command to enable Tx/Rx of link 0 
        CSL_aifHwControl(handleAifLink0, CSL_AIF_CMD_ENABLE_DISABLE_RX_LINK, (void *)&ctrlArg);
        ...
        wait(100); //wait for a sufficient length of time, so Rx link has enough 
                     time to sync; 100 cycles wait time is arbitrarily chosen 
    
        // Get status of Rx master frame offset  
        CSL_aifGetHwStatus(handleAifLink0, CSL_AIF_QUERY_RM_LINK_RCVD_MSTR_FRAME_OFFSET, (void *)&response);
   
        if (response > CSL_AIF_MAX_RX_MASTER_FRAME_OFFSET)
        {
           printf("\nRx Master Frame Offset exceeds bounds");
        }

     @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_aifGetHwStatus, ".text:csl_section:aif");
CSL_Status  CSL_aifGetHwStatus(
   CSL_AifHandle                      hAifLink,
   CSL_AifHwStatusQuery               Query,
   void                               *response
)
{
    CSL_Status status = CSL_SOK;
    
    if (hAifLink == NULL)
        return CSL_ESYS_BADHANDLE;
    
    switch (Query) {
    
	/** returns AIF version */
    case CSL_AIF_QUERY_VERSION:
        CSL_aifGetVersion(hAifLink, (CSL_AifPidStatus *)response);
        break;
            	
	/* Return status of SD recieve loss of signal detect */
    case CSL_AIF_QUERY_SD_RX_LOS_DET:
    	*((Uint8 *)response) = CSL_aifGetSdRxLosDet (hAifLink);  
        break;
        
    /* Return the status of PLL lock */
    case CSL_AIF_QUERY_PLL_LOCK:
        *((Uint8 *)response) = CSL_aifGetPllLock (hAifLink);  
        break;
    
    /* Return status of the AIF Rx MAC s/m  */
    case CSL_AIF_QUERY_RM_LINK_SYNC_STATE:
        *((CSL_AifRxSyncState *)response) = CSL_aifGetRxMacSyncState (hAifLink);   
        break;

    /* Return status of the Rx Mac loss 0f signal */
    case CSL_AIF_QUERY_RM_LINK_LOS_OF_SIG:
        *(Uint8 *)response = CSL_aifGetRmLosOfSig (hAifLink);
        break;

    /* Return status of the RX MAC loss detection */
    case CSL_AIF_QUERY_RM_LINK_LOS_DET:
        *(Uint8 *)response = CSL_aifGetRmLosDet (hAifLink);
        break;

    /* Return status of the Rx MAC frame boundary range */
    case CSL_AIF_QUERY_RM_LINK_FRAME_BNDY_RANGE:
        *(Uint8 *) response = CSL_aifGetRmFrameBndyRange (hAifLink);
        break;

    /* Return status of the Rx MAC number of los detect occuerence in a master
     * frame 
     */
    case CSL_AIF_QUERY_RM_LINK_NUM_LOS:
        *(Uint32 *) response = CSL_aifGetRmNumLos (hAifLink);
        break;

    /* Return the time in vbus_clk tick from after pioffest that the frame 
     * boundary was recieved 
     */
    case CSL_AIF_QUERY_RM_LINK_RCVD_MSTR_FRAME_OFFSET:
        *((Uint16 *)response) = CSL_aifGetRmRcvdMstrFrameOffset (hAifLink);
        break;

    /* Return status of the AIF Tx MAC s/m */
    case CSL_AIF_QUERY_TM_LINK_SYNC_STATE:
        *(CSL_AifTxSyncState *)response = CSL_aifGetTxMacSyncState (hAifLink);
        break;
        
    /** Return status indicated when the frame strobe from the Frame Sync Module
     *  is not aligned with the frame byte counter  
     */
    case CSL_AIF_QUERY_TM_LINK_FRAME_NOT_ALIGNED:
    	*((Uint8 *)response) = CSL_aifGetTmFrameNotAlign (hAifLink);
        break;
        
    /** Return status indicated when data to be transmitted is detected as not
     *	aligned with Frame Sync + delta_offset (i.e. Frame boundary to be 
     *	transmitted is not aligned with frame boundary received internally)  
     */  
    case CSL_AIF_QUERY_TM_LINK_DATA_NOT_ALIGNED:
    	*((Uint8 *)response) = CSL_aifGetTmDataNotAlign (hAifLink);
        break; 
           
    /* Return status TX MAC FIFO Overflow flag  */
    case CSL_AIF_QUERY_TM_LINK_FIFO_OVERFLOW:
        *((Uint8 *)response) = CSL_aifGetTxMacFifoOvf (hAifLink);
        break;

    /* Return frame alignment error for each of two combiner  */
    case CSL_AIF_QUERY_CB01_ALIGN_ERR:
        *((Uint8 *)response) = CSL_aifGetCbAlignErr (hAifLink);
        break;

    /* Return CD ouput fifo overflow status perlink  */
    case CSL_AIF_QUERY_CB01_OUT_FIFO_OVF:
        *((Uint8 *)response) = CSL_aifGetCbOutFifoOvf (hAifLink);
        break;

    /* Return CD ouput fifo underflow status perlink  */
    case CSL_AIF_QUERY_CB01_OUT_FIFO_UNF:
        *((Uint8 *)response) = CSL_aifGetCbOutFifoUnf (hAifLink);
        break;

	/** Aggregator link Status */
    case CSL_AIF_QUERY_AG_LINK_STATUS:
    	CSL_aifGetAgLinkStatus (hAifLink,(CSL_AifAggregatorStatus *)response);
        break;

    /* Returns count of DMA inbound done register value  */
    case CSL_AIF_QUERY_DB_IN_LINK_DMA_DONE_COUNT:
        *((Uint16 *)response) = CSL_aifGetInDmaDoneCount (hAifLink);
        break;

    /* Returns count of DMA outbound done register value  */
    case CSL_AIF_QUERY_DB_OUT_LINK_DMA_DONE_COUNT:
        *((Uint16 *)response) = CSL_aifGetOutDmaDoneCount (hAifLink);
        break;

    /** Returns number of DMA bursts written into DB by PD available for read by
     * 	VBUS 
     */
    case CSL_AIF_QUERY_DB_IN_LINK_DMA_DEPTH: 
         *((Uint8 *)response) = CSL_aifGetInDmaDepth (hAifLink);
        break;
    
    /** Returns number of DMA bursts written into DB available for processing by
     * 	PE 
     */
    case CSL_AIF_QUERY_DB_OUT_LINK_DMA_DEPTH:
         *((Uint8 *)response) = CSL_aifGetOutDmaDepth (hAifLink);
        break;

    /** Returns Current state of the "lowest" Read/head pointer */
    case CSL_AIF_QUERY_DB_OUT_PKTSW_FIFO_RD_PTR:
         *((Uint16 *)response) = CSL_aifGetPktswFifoRdPtr (hAifLink);
        break;
    /** Returns current state of the circular write pointer of Oubtound Packet 
     *	Switched Memory 
     */
    case CSL_AIF_QUERY_DB_OUT_PKTSW_FIFO_WR_PTR:
         *((Uint16 *)response) = CSL_aifGetPktswFifoWrPtr (hAifLink);
        break;

    /** Returns Fifo Index 0-to-29 indicating which of 30 FIFOs has the lowest,
     *	non-read memory location 
     */
    case CSL_AIF_QUERY_DB_OUT_PKTSW_FIFO_RD_INDEX:
         *((Uint8 *)response) = CSL_aifGetPktswFifoRdIndex (hAifLink);
        break;

    /** Returns depth of Packet Switched Memory currently available for write */
    case CSL_AIF_QUERY_DB_OUT_PKTSW_FIFO_DEPTH: 
         *((Uint16 *)response) = CSL_aifGetPktswFifoDepth (hAifLink);
        break;

    /** Returns status of Outbound packet switched FIFO 0-29, 1'b1 indicates 
     *  FIFO is currently not empty 
     */
    case CSL_AIF_QUERY_DB_OUT_PKTSW_FIFO_NE:
         *((Uint32 *)response) = CSL_aifGetPktswFifoNe (hAifLink);
        break;
        
    /** Packet Swtiched FIFO #0-29, Current state of head and tail pointer */
    case CSL_AIF_QUERY_DB_OUT_PKTSW_FIFO_PTR_STATUS:
    	CSL_aifGetDbOutPktswFifoPtr (hAifLink,(CSL_AifDbFifoPtrStatus*)response);
        break;
    
    /** Returns Error/Alarm status of each link */
    case CSL_AIF_QUERY_EE_INTR_VECTOR_EVT_0:
        *((CSL_BitMask32*)response) = CSL_aifGetEeIntrVecEvt0(hAifLink);
        break;

    /** Returns Error/Alarm status of each link */
    case CSL_AIF_QUERY_EE_INTR_VECTOR_EVT_1:
        *((CSL_BitMask32*)response) = CSL_aifGetEeIntrVecEvt1(hAifLink);
        break;

    /** Returns Error/Alarm status of each link */
    case CSL_AIF_QUERY_EE_INTR_VECTOR_EVT_2:
        *((CSL_BitMask32*)response) = CSL_aifGetEeIntrVecEvt2(hAifLink);
        break;

    /** Returns Error/Alarm status of each link */
    case CSL_AIF_QUERY_EE_INTR_VECTOR_EVT_3:
        *((CSL_BitMask32*)response) = CSL_aifGetEeIntrVecEvt3(hAifLink);
        break;

    /** Returns Event0 mask status of each link */
    case CSL_AIF_QUERY_EE_LINK_MASK_EVT_0:
        CSL_aifGetEeLinkMaskEvt0(hAifLink,(CSL_AifExcEventQueryObj*)response);
        break;

    /** Returns Event1 mask status of each link */
    case CSL_AIF_QUERY_EE_LINK_MASK_EVT_1:
        CSL_aifGetEeLinkMaskEvt1(hAifLink,(CSL_AifExcEventQueryObj*)response);
        break;

    /** Returns Event2 mask status of each link */
    case CSL_AIF_QUERY_EE_LINK_MASK_EVT_2:
        CSL_aifGetEeLinkMaskEvt2(hAifLink,(CSL_AifExcEventQueryObj*)response);
        break;

    /** Returns Event3 mask status of each link */
    case CSL_AIF_QUERY_EE_LINK_MASK_EVT_3:
        CSL_aifGetEeLinkMaskEvt3(hAifLink,(CSL_AifExcEventQueryObj*)response);
        break;

    /** Returns  raw status of each link */
    case CSL_AIF_QUERY_EE_LINK_RAW_STATUS:
        CSL_aifGetEeLinkRawStat(hAifLink,(CSL_AifExcEventQueryObj*)response);
        break;
    
    /** Returns the antenna interface run status  */
    case CSL_AIF_QUERY_EE_AIF_RUN_STATUS_BIT:
        *((Uint8*)response) = CSL_aifGetEeAifRunStatBit(hAifLink);
        break;

    default:
        status = CSL_ESYS_INVQUERY;
	}

return status;
}
